#!/bin/bash
#
# evms_failover_rsct
#       Description: Manages Private containers on shared storage media.
#  	Original Author: Ram Pai(linuxram(us.ibm.com)
#       Modified by:     Guochun Shi (gshi@ncsa.uiuc.edu) 
#                                                            
#       Support: evms-devel@lists.sourceforge.net
#                
# usage: evms_failover <container> {start|stop|status}
#		<container> : name of private container 
#

ONLINE=1
OFFLINE=2
CSMDEVNODES=/dev/evms/.nodes/csm
ENGINE_ERR=13
PREFIX=/etc/rsct.d/


log() {

#   echo $1 >>/var/log/rsct_failover.log 
#    echo $1
}




usage() {
cat <<-EOT;
	usage: $0 <container> {start|stop|status}
		<container> : name of private container to be imported
EOT
}
I=`hostname`
container=$1
if [ -z $container ]; then
	usage
	exit 1
fi



evms_exec()
{
	#modify the container
	ret=$ENGINE_ERR # loop till the command is able to open
	 	#the engine and execute successfully.
	while [ $ret -eq $ENGINE_ERR ] 
	do
		#note READWRITE_CRITICAL mode is ***only**** to be used 
		#by this failover script. 
		ssh localhost "evms -d debug -m READWRITE_CRITICAL  -b<<-EOF &>/dev/null  
			$1
			commit
			exit
		EOF"
		ret=$?
		if [ $ret -eq $ENGINE_ERR ]; then
		   #sleep random interval upto a max time of 10secs.
		   #this is done to avoid collisions between similar
		   #evms -m READWRITE_CRTIICAL invocations on
		   #other cluster nodes.

		   sleep `echo 10 | awk '{srand();printf("%d",rand()*$1)}'` 
		fi
	done
	return $ret
}


evms_report_status()
{

	if [ -d $CSMDEVNODES/$1 ]; then
		return $ONLINE
	fi
	return $OFFLINE

}

evms_import()
{
	#modify the container
	node=`uname -n`
	evms_exec "Modify:csm/$1,NodeId=$node,Type=private" 
	evms_exec "commit"

	return $?
}


evms_deport()
{
	#deport the container
	evms_exec "Modify:csm/$1,Type=deported" &&
		evms_exec "commit" 
	return $?
}

cleanexit()
{
        exit $1
}

check_daemon(){

		node_rsrc_list=`ssh localhost "lsrsrc -dx IBM.PeerNode NodeNameList NodeList OpState"`
		#check if evmsd is running on all of online nodes
		log "${I}:checking evms daemon status on cluster nodes"

		allonline=1
		for i in $node_rsrc_list ; do
		    online=${i%:*}
		    online=${online##*:}
		    if [ $online -eq "1" ] ; then
			node=${i%%\"\}*}
			node=${node##*\{\"}
			ssh $node "${PREFIX}/evms_daemon.sh status"
			if [ $? -eq $OFFLINE ]; then
			    log "${I}: evms daemon is offline in $node"	
			    ssh $node "killall -9 evmsd &>/dev/null && killall -9 evms&>/dev/null"
			    ssh $node "evmsd -d debug &>/dev/null &"			
			    allonline=0
			else
			    log "${I}: evms daemon online in $node"        
			fi
		    fi
		done
		
		if [ $allonline -eq 1 ] ; then
		    return $ONLINE
		else
		    check_daemon
		    return $?
		fi
		
}


# Look for the 'start', 'stop' or 'status' argument
case "$2" in
#
# START: Import the container
#
start)
	# import the container
	log "${I}: Trying to import $1"
	check_daemon
	if [ $? -eq $ONLINE ] ; then
	    /sbin/evms_activate 	
	    evms_report_status $1 
	    if [ $? -ne $ONLINE ];then
	
		log "${I}: Importing the container."
		evms_import $1
		
		#for all other online nodes, update their configuration
		#find all online nodes
		node_list=`lsrpnode -odx`

		for i in $node_list ; do
		    node=${i%%:*}
		    ssh $node "/sbin/evms_activate &> /dev/null" 
		done

		evms_report_status $1 &> /dev/null 
		if [ $? -eq 0 ]; then
			ret=0                   
		else
			ret=1
		fi
	    else
		log " ${I}: I already own the container. Start aborted."
	    fi
 	fi
	ret=0
	;;

#
# STOP: Release the container
#
stop)
	# deport the container
	log "${I}: trying to deport $1"
	check_daemon
	if [ $? -eq $ONLINE ] ; then
	    /sbin/evms_activate	
	    evms_report_status $1
	    if [ $? -eq $ONLINE ];then

		log "${I}: Deporting the container."
		evms_deport $1 
		evms_report_status $1
		if [ $? -ne 0 ]; then
			ret=0
		else
			ret=1
		fi
		/sbin/evms_activate	
	    else
		log " ${I}: I don't own the container. Stop aborted."
	    fi
	 fi
	ret=0
	;;

#
# STATUS: is the container already imported on local node?
#
status)
	#  is the container owned by us?
	evms_report_status $1
	if [ $? -eq $ONLINE ] ; then
		ret=$ONLINE
		log "${I}: online"
	else
		ret=$OFFLINE
		log "${I}: offline"
	fi
	;;

*)
   	usage
	exit 1 
esac

# If you got to this point, chances are everything is O.K.
cleanexit $ret
