/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <string.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "dialog.h"
#include "selwin.h"
#include "views.h"
#include "enum.h"
#include "readable.h"
#include "remove.h"
#include "plugin.h"
#include "container.h"
#include "logging.h"

/**
 *	remove_segment_manager_button_activated - callback to remove/unassign a segment manager
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Remove button is activated to remove the
 *	segment manager plugin assigned to a storage object.
 */
int remove_segment_manager_button_activated(struct menu_item *item)
{
	int rc;
	struct selwin *selwin = item->user_data;

	rc = evms_unassign(get_selected_handle(selwin->clist));
	update_status(_("Remove"), rc);

	if (rc == 0) {
		((struct dialog_window *)selwin)->status = DLG_STATUS_CLOSING;
		refresh_views();
	}
	return 0;
}

/**
 *	filter_remove_segment_manager_objects - keep objects that can have segment manager unassigned
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can have a segment manager plugin unassigned/removed.
 */
int filter_remove_segment_manager_objects(object_handle_t handle, void *user_data)
{
	return evms_can_unassign(handle);
}

/**
 *	format_remove_segment_manager_item - return column strings for a row in remove segment manager clist
 *	@handle: the handle of the object having the segment manager plugin removed
 *	@not_used: extra info that we don't pay attention to
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a thing placed
 *	in the remove segment manager clist. We return the name of the object and the name
 *	of the plugin currently assigned.
 */
int format_remove_segment_manager_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(" "));
		switch (object->type) {
		case DISK:
		case REGION:
		case SEGMENT:
		case EVMS_OBJECT:
			g_ptr_array_add(text, g_strdup(object->info.object.name));
			g_ptr_array_add(text, get_plugin_name(get_parent_plugin_for_object(handle)));
			break;
		default:
			g_ptr_array_add(text, g_strdup(""));
			g_ptr_array_add(text, g_strdup(""));
			log_warning("%s: Type %u was not processed.\n", __FUNCTION__, object->type);
			break;
		}
		evms_free(object);
	}
	return rc;
}

/**
 *	show_remove_segment_manager_dialog - display dialog to remove/unassign a segment manager
 *	@handle: non-zero when invoked from context popup menu
 *
 *	This routine displays a selection window to allow the user to remove a segment manager
 *	from one of the storage objects listed.
 **/
int show_remove_segment_manager_dialog(object_handle_t handle)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Remove Segment Manager From Storage Object"),
					NULL, NULL,
					_("_Remove"),
					(menuitem_activate_cb)remove_segment_manager_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;

	set_clist_column_info(selwin->clist, 2, calc_clist_column_width(selwin->clist, 0.45),
				get_clist_column_end(selwin->clist, 1),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Storage Object"));
	print_clist_column_title(selwin->clist, 2, _("Plug-in"));

	if (handle == 0)
		clist_populate(selwin->clist, enum_data_objects,
				filter_remove_segment_manager_objects,
				format_remove_segment_manager_item, NULL, NULL);
	else
		clist_populate_handle(selwin->clist, handle,
				format_remove_segment_manager_item, NULL, NULL);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);

	process_modal_dialog(dialog);
	return 0;
}

/**
 *	context_remove_segment_manager_menuitem_activated - unassign segment manager from storage object
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to remove a segment manager from a storage object. The
 *	operation was initiated by selecting the "Remove segment manager from storage object"
 *	menu item on the popup context menu.
 */
int context_remove_segment_manager_menuitem_activated(struct menu_item *item)
{
	return show_remove_segment_manager_dialog(GPOINTER_TO_UINT(item->user_data));
}

/**
 *	actions_remove_segment_manager_menuitem_activated - unassign segment manager from storage object
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Actions->Remove->Segment Manager from Storage Object
 *	pulldown menu item.
 */
int actions_remove_segment_manager_menuitem_activated(struct menu_item *item)
{
	return show_remove_segment_manager_dialog(0);
}

/**
 *	remove_consumed_object_button_activated - callback to remove a consumed object from a container
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Remove button is activated to remove a
 *	storage object consumed by a container.
 */
int remove_consumed_object_button_activated(struct menu_item *item)
{
	int rc;
	struct selwin *selwin = item->user_data;

	handle_object_info_t * info;

	rc = evms_get_info(get_selected_handle(selwin->clist), &info);
	if (rc == 0) {
		if (info->info.object.consuming_container != 0) {
			gchar ha[sizeof(handle_array_t) + sizeof(object_handle_t)];
			handle_array_t * objects = (handle_array_t *) ha;

			objects->count = 1;
			objects->handle[0] = get_selected_handle(selwin->clist);
			rc = evms_shrink(info->info.object.consuming_container, objects, NULL);

		} else {
			log_error("%s: object %s does not belong to a container.\n", __FUNCTION__, info->info.object.name);
			rc = EINVAL;
		}

		evms_free(info);
	}
	update_status(_("Remove"), rc);

	if (rc == 0) {
		((struct dialog_window *)selwin)->status = DLG_STATUS_CLOSING;
		refresh_views();
	}
	return 0;
}

/**
 *	can_remove_from_container
 *	@handle: the thing's handle
 *
 *      This routine detremines if an object can be removed from a
 *      container by starting a shrink task on the container and
 *      seeing if the object appears in the acceptable objects list.
 */
int can_remove_from_container(object_handle_t handle)
{
	gint rc;
	handle_object_info_t * info;

	rc = evms_get_info(handle, &info);
	if (rc != 0) {
		return rc;
	}

	if (info->info.disk.consuming_container != 0) {
		task_handle_t task;
		rc = evms_create_task(info->info.disk.consuming_container,EVMS_Task_Shrink,&task);
		if (rc == 0) {
			handle_array_t *objects;
			rc == evms_get_acceptable_objects(task, &objects);
			if (rc == 0) {
				gint i;

				rc = ENOENT;
				for (i = 0; i < objects->count; i++) {
					if (objects->handle[i] == handle) {
						rc = 0;
						break;
					}
				}
				evms_free(objects);
			}
			evms_destroy_task(task);
		}
	} else {
		rc = ENOENT;
	}

	return rc;
}

/**
 *	filter_remove_consumed_objects - keep objects that can be removed from a container
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be removed from its consuming container.
 */
int filter_remove_consumed_objects(object_handle_t handle, void *user_data)
{
	return can_remove_from_container(handle);
}

/**
 *	format_remove_consumed_item - return column strings for a row in remove consumed object clist
 *	@handle: the handle of the object to be removed from a container
 *	@not_used: extra info that we don't pay attention to
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a thing placed
 *	in the remove consumed object clist. We return the name of the object and the name
 *	of the container currently consuming that object.
 */
int format_remove_consumed_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(" "));
		switch (object->type) {
		case DISK:
		case REGION:
		case SEGMENT:
		case EVMS_OBJECT:
			g_ptr_array_add(text, g_strdup(object->info.object.name));
			g_ptr_array_add(text, get_container_name(object->info.object.consuming_container));
			break;
		default:
			g_ptr_array_add(text, g_strdup(""));
			g_ptr_array_add(text, g_strdup(""));
			log_warning("%s: Type %u was not processed.\n", __FUNCTION__, object->type);
			break;
		}
		evms_free(object);
	}
	return rc;
}

/**
 *	show_remove_consumed_object_dialog - display dialog to remove an object from a container
 *	@handle: non-zero when invoked from context popup menu
 *
 *	This routine displays a selection window to allow the user to remove a storage object
 *	from the container that is consuming it.
 **/
int show_remove_consumed_object_dialog(object_handle_t handle)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Remove Storage Object from Consuming Container"),
					NULL, NULL,
					_("_Remove"),
					(menuitem_activate_cb)remove_consumed_object_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;

	set_clist_column_info(selwin->clist, 2, calc_clist_column_width(selwin->clist, 0.45),
				get_clist_column_end(selwin->clist, 1),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Storage Object"));
	print_clist_column_title(selwin->clist, 2, _("Consuming Container"));

	if (handle == 0)
		clist_populate(selwin->clist, enum_data_objects,
				filter_remove_consumed_objects,
				format_remove_consumed_item, NULL, NULL);
	else
		clist_populate_handle(selwin->clist, handle,
				format_remove_consumed_item, NULL, NULL);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);

	process_modal_dialog(dialog);
	return 0;
}

/**
 *	context_remove_consumed_object_menuitem_activated - remove a consumed object from a container
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to remove a storage object from a container that finds this
 *	acceptable to do. The operation was initiated by selecting the "Remove from Container"
 *	menu item on the popup context menu.
 */
int context_remove_consumed_object_menuitem_activated(struct menu_item *item)
{
	return show_remove_consumed_object_dialog(GPOINTER_TO_UINT(item->user_data));
}

/**
 *	actions_remove_consumed_object_menuitem_activated - remove a consumed object from a container
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Actions->Remove->Storage Object from Container"
 *	pulldown menu item.
 */
int actions_remove_consumed_object_menuitem_activated(struct menu_item *item)
{
	return show_remove_consumed_object_dialog(0);
}
