/* -*- c-file-style: "GNU" -*- */
/*
 * Copyright  CNRS, INRIA, Universit Bordeaux 1
 * See COPYING in top-level directory.
 */

#ifndef EZTRACE_CONVERT_MPI_H
#define EZTRACE_CONVERT_MPI_H

#define _GNU_SOURCE
#include <stdio.h>
#include "eztrace_convert.h"

/* Different kinds of MPI messages */
enum MPI_MSG_TYPE {
  mpi_msg_type_p2p,
  mpi_msg_type_coll,
};

/* todo: remove this struct */
union mpi_msg {
  struct {
    enum {
      mpi_p2p_msg_send,
      mpi_p2p_msg_recv
    } p2p_type;
  } p2p;
  struct {
    enum{
      mpi_coll_msg_barrier,
      mpi_coll_msg_bcast
      /* todo: add other kinds of collective operations */
    } coll_type;
  } coll;
};


enum mpi_request_type{
  mpi_req_none,
  mpi_req_send,
  mpi_req_recv
};

struct mpi_request {
  app_ptr ptr;
  struct eztrace_container_t* container;
  enum mpi_request_type req_type;
  struct mpi_p2p_msg_t *msg;
};

enum p2p_time_ids{
  start_send = 0,
  start_swait = 1,
  stop_send = 2,
  start_recv = 3,
  start_rwait = 4,
  stop_recv = 5,
  P2P_NB_TIMES
};

struct mpi_p2p_msg_t {
  char*id;
  int src;
  int dest;
  int len;
  uint32_t tag;
  int unexp;  /* set to 1 if the recv is detected before the send */
  uint64_t times[P2P_NB_TIMES];
  char* link_value;
  const char* sender_thread_id;
  const struct mpi_request* sender_request;
  const char* recver_thread_id;
  const struct mpi_request* recver_request;
};

enum coll_type_t{
  mpi_coll_barrier,
  mpi_coll_bcast,
  mpi_coll_gather,
  mpi_coll_scatter,
  mpi_coll_allgather,
  mpi_coll_alltoall,
  mpi_coll_reduce,
  mpi_coll_allreduce,
  mpi_coll_reduce_scatter,
  mpi_coll_scan,
  /* todo: add other kinds of collective operations */
};

enum coll_time_ids{
  start_coll = 0,
  stop_coll = 2,
  COLL_NB_TIMES
};

struct mpi_coll_msg_t {
  enum coll_type_t type;
  uint64_t** times;
  int comm_size;
  int len;
  char***link_id;
  char*** link_value;
  int unexp;			/* todo: useless ? */
  char** thread_ids;
};


/* This structure contains information about a MPI message in 
   the global view (ie. it is valid no matter which process acces it)
*/
/* todo: cleanup this mess. some fields are only used for p2p messages !
 */
struct mpi_msg_t{
  char* id;
  enum MPI_MSG_TYPE msg_type;
  union mpi_msg msg;
  int src;
  int dest;
  int len;
  int unexp;  /* set to 1 if the recv is detected before the send */
  uint32_t tag;
  uint64_t send_time;
  uint64_t recv_time;
  uint64_t start_wait_time;
  char* link_value;
  char* threadstr;
};

/* This structure contains information about a MPI_COMM_SPAWN */
struct mpi_spawn_t{
  int nb_children;		/* number of processes spawned */
  uint64_t start_time;		/* date at which MPI_COMM_SPAWN was invoked */
  int ppid;			/* parent process id */
  struct trace_t *parent_trace;
};

/* This structure contains information on a persistent request */
struct mpi_pers_req_t{
  int process_id;
  enum mpi_request_type type;
  app_ptr buffer;
  int len;
  int src;
  int dest;
  uint32_t tag;
  struct mpi_request* mpi_req;
};

/* This structure contains informations on a MPI request.
 * it is only valid for one process (cf the rank field)
 */
struct __mpi_request_t {
  struct mpi_msg_t *msg;   /* pointer to the message corresponding to this request */
  union mpi_msg type;
  app_ptr req; /* address of the request in the MPI process */
  int rank;    /* the rank of the MPI process that uses this req */
};

/* This structure contains relevant informations used in the
 * communication matrix
 */
struct comm_info {
  int nb;
  int src;
  int dest;
  uint64_t size;
};

struct message_size {
  int nb;
  int size;
};

/* generate a MPI message id */
#define CREATE_P2P_MSG_ID(msg)						\
  {int __attribute__((unused)) ret = asprintf(&msg->id, "%d_%d_%20u_%p", msg->src, msg->dest, msg->tag, msg);}


/* generate the value of a p2p link */
#define CREATE_P2P_LINK_VALUE(msg)					\
  { int __attribute__((unused)) ret = asprintf(&(msg)->link_value, "src=%d, dest=%d, len=%d, tag=%x", \
					       (msg)->src, (msg)->dest, (msg)->len, (msg)->tag);}

/* generate the value of a coll link */
#define CREATE_COLL_MSG_ID(_msg_, _src_, _dest_)			\
  { int __attribute__((unused)) ret = asprintf(&(msg)->link_id[_src_][_dest_], "%d_%d_%p", _src_, _dest_, _msg_); }

#define CREATE_COLL_LINK_VALUE(_str_, _src_, _dest_)			\
  { int __attribute__((unused))ret = asprintf(&(_str_)->link_value[_src_][_dest_], "src=%d, dest=%d", \
					      _src_, _dest_); }

static inline struct comm_info *__create_comm_info(int src, int dest, int size, int nb){
  struct comm_info *msg = (struct comm_info*) malloc(sizeof(struct comm_info));
  msg->src = src;
  msg->dest = dest;
  msg->size = size;
  msg->nb = nb;
  return msg;
}

static inline struct message_size *__create_message_size(int size, int nb){
  struct message_size *msg = (struct message_size*) malloc(sizeof(struct message_size));
  msg->size = size;
  msg->nb = nb;
  return msg;
}

typedef enum {
 MPI_SEND_ID             ,
 MPI_RECV_ID             ,
 MPI_BSEND_ID            ,
 MPI_SSEND_ID            ,
 MPI_RSEND_ID            ,
 MPI_ISEND_ID            ,
 MPI_IBSEND_ID           ,
 MPI_ISSEND_ID           ,
 MPI_IRSEND_ID           ,
 MPI_IRECV_ID            ,
 MPI_SENDRECV_ID         ,
 MPI_SENDRECV_REPLACE_ID ,
 MPI_START_ID            ,
 MPI_STARTALL_ID         ,
 MPI_WAIT_ID             ,
 MPI_TEST_ID             ,
 MPI_WAITANY_ID          ,
 MPI_TESTANY_ID          ,
 MPI_WAITALL_ID          ,
 MPI_TESTALL_ID          ,
 MPI_WAITSOME_ID         ,
 MPI_TESTSOME_ID         ,
 MPI_PROBE_ID            ,
 MPI_IPROBE_ID           ,
 MPI_BARRIER_ID          ,
 MPI_BCAST_ID            ,
 MPI_GATHER_ID           ,
 MPI_GATHERV_ID          ,
 MPI_SCATTER_ID          ,
 MPI_SCATTERV_ID         ,
 MPI_ALLGATHER_ID        ,
 MPI_ALLGATHERV_ID       ,
 MPI_ALLTOALL_ID         ,
 MPI_ALLTOALLV_ID        ,
 MPI_REDUCE_ID           ,
 MPI_ALLREDUCE_ID        ,
 MPI_REDUCE_SCATTER_ID   ,
 MPI_SCAN_ID             ,
 MPI_GET_ID              ,
 MPI_PUT_ID              ,
 MPI_ID_SIZE
} MPI_id_t;

#endif	/*  EZTRACE_CONVERT_MPI_H */
