# This file is part of the Falcon repository manager
# Copyright (C) 2005-2008 Dennis Kaarsemaker
# See the file named COPYING in the root of the source tree for license details
#
# iso.py - CD image creation

import falcon
import os, statvfs, getpass, time, tempfile, shutil

isocache = []

diskdefines = """#define DISKNAME  Falcon %s iso - created by %s on %s
#define TYPE  binary
#define TYPEbinary  1
#define ARCH  %s
#define ARCH%s  1
#define DISKNUM  1
#define DISKNUM1  1
#define TOTALNUM  1
#define TOTALNUM1  1"""

def create_tree(topdir):
    """Creates directory structure and symlinks to .deb files"""
    basedir = tempfile.mkdtemp(prefix='falcon-iso-', dir='/tmp')
    debdir = os.path.join(basedir,'pool','falcon','iso')
    os.makedirs(debdir)
    # Make apt-cdrom happy
    falcon.util.writefile(os.path.join(basedir,'README.diskdefines'),diskdefines % (falcon.conf.falcon_version,
                           getpass.getuser(), time.strftime("%Y/%m/%d %H:%S"),
                           falcon.conf.iso_architecture, falcon.conf.iso_architecture))
    os.mkdir(os.path.join(basedir,'.disk'))
    falcon.util.writefile(os.path.join(basedir,'.disk','base_installable'),'')
    falcon.util.writefile(os.path.join(basedir,'.disk','udeb_include'),'')
    falcon.util.writefile(os.path.join(basedir,'.disk','base_components'),'iso\n')
    falcon.util.writefile(os.path.join(basedir,'.disk','info'),
                          'Falcon %s iso - created by %s on %s' % (falcon.conf.falcon_version, getpass.getuser(), time.strftime("%Y/%m/%d %H:%S")))
    
    for path in isocache:
        (dir, file) = os.path.split(path)
        os.symlink(os.path.join(os.getcwd(),path),os.path.join(debdir, file))
    return basedir

def check_size(topdir):
    """Checks required diskspace"""
    # Required diskspace
    required = 0
    _end = '%s.deb' % falcon.conf.iso_architecture
    _end2 = '%s.udeb' % falcon.conf.iso_architecture
    for (dir, dirs, files) in os.walk(topdir):
        for file in files:
            if file.endswith('all.deb') or file.endswith(_end) or file.endswith('all.udeb') or file.endswith(_end2):
                required += os.path.getsize(os.path.join(dir,file))
                f = os.path.join(dir, file).replace(topdir,'')
                if f[0] == '/':
                    f = f[1:]
                isocache.append(f)
    # Available diskspace in /tmp
    st = os.statvfs('/tmp')
    available = st[statvfs.F_BAVAIL] * st[statvfs.F_FRSIZE]
    
    return required / (1024*1024.0) + 8, available / (1024 * 1024.0)

def create_iso(path):
    """Creates an .iso image from a directory tree"""
    # Remove all that's unneeded
    shutil.rmtree(os.path.join(path, '.falcon'))
    shutil.rmtree(os.path.join(path, 'dists','falcon','iso','source'))
    isofile = tempfile.mkstemp(prefix='falcon-iso-%s-' % getpass.getuser(), suffix='.iso', dir = '/tmp')
    os.close(isofile[0])
    isofile = isofile[1]
    args = ['genisoimage', '-allow-leading-dots', '-J', '-r', '-f', '-V', "Falcon iso created on %s" % time.strftime("%Y/%m/%d")]

    falcon.util.run(args + ['-o', isofile, path])
    return isofile

def cleanup(path):
    """Recursively delete a temporary dir"""
    if path.startswith('/tmp/falcon-iso-'):
        shutil.rmtree(path)
