/* $Id: trace.c 4321 2009-01-27 13:02:52Z potyra $
 *
 * Utilities for writing to a VCD file.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "trace.h"
#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <errno.h>
#include "libfauhdli.h"
#include "glue-log.h"
#include <string.h>
#include <inttypes.h>

#define INTERPRETER_VERSION "fauhdli 0.0alpha1"
#define RESOLUTION_UNIT "ns"

struct trace_sig_wrapper {
	const struct signal *sig;
	char id_code;
	enum type_kind type;
};

static const char *
vcd_get_hex(universal_integer num)
{
	static char ret[65] = { '\0' };
	int bit;
	int dst = 0;
	bool seen = false;

	for (bit = 63; bit >= 0; bit--) {
		if ((num & (1ULL << bit)) == 0) {
			ret[dst] = '0';
		} else {
			ret[dst] = '1';
			seen = true;
		}

		if (seen) {
			dst++;
		}
	}

	ret[dst] = '\0';
	if (dst == 0) {
		ret[0] = '0';
		ret[1] = '\0';
	}

	return ret;
}

static const char *
vcd_get_var_type(enum type_kind k) 
{
	switch (k) {
	case TYPE_INT:
		return "integer";
	
	case TYPE_FLOAT:
		return "real";

	default:
		assert(false);
	}
}

static void
vcd_add_sig(
	FILE *f, 
	const char *name, 
	char id_code, 
	enum type_kind type, 
	int display_bits
)
{
	size_t sz;

	switch (type) {
	case TYPE_INT:
		sz = sizeof(universal_integer) * 8;
		break;
	
	case TYPE_FLOAT:
		sz = sizeof(universal_real) * 8;
		break;

	default:
		assert(0);
	}

	if (display_bits != -1) {
		sz = (size_t)display_bits;
	}

	fprintf(f, "$var %s %zd %c %s $end\n",
		vcd_get_var_type(type),
		sz,
		id_code,
		name
	);
}

static void
vcd_write_header(FILE *f)
{
	fprintf(f, "$version %s\n", INTERPRETER_VERSION);
	fprintf(f, "$end\n");
	fprintf(f, "$timescale 1 %s\n", RESOLUTION_UNIT);
	fprintf(f, "$end\n");
}

static void
vcd_end_header(FILE *f)
{
	fprintf(f, "$end\n");
}

static void
vcd_dump_var(FILE *f, const struct trace_sig_wrapper *tw)
{
	switch (tw->type) {
	case TYPE_INT:
		fprintf(f, "b%s %c\n",
			vcd_get_hex(tw->sig->value.univ_int),
			tw->id_code);
		break;

	case TYPE_FLOAT:
		/* TODO */
		assert(0);
		break;

	default:
		assert(0);
	}

}

static void
trace_end_header(struct trace_t *s)
{
	const struct slist_entry *i;

	vcd_end_header(s->output);
	s->header_written = true;

	/* write initial values */
	fprintf(s->output, "$dumpvars\n");

	for (i = s->traced_sigs->first; i != NULL; i = i->next) {
		const struct trace_sig_wrapper *tw = 
			(const struct trace_sig_wrapper *)i->data;

		vcd_dump_var(s->output, tw);
	}
	fprintf(s->output, "$end\n");
}

struct trace_t *
trace_create(const char *trace_file)
{
	struct trace_t *ret = malloc(sizeof(struct trace_t));
	assert(ret != NULL);

	ret->traced_sigs = slist_create();
	ret->output = fopen(trace_file, "w");
	if (ret->output == NULL) {
		faum_log(FAUM_LOG_ERROR, "fauhdli", "tracer",
			"cannot created output file %s: %s\n", trace_file, 
			strerror(errno));
		assert(0);
	}
	ret->ident_code = 33; /* ident code is in range 33-126 */
	ret->header_written = false;

	vcd_write_header(ret->output);
	return ret;
}

void
trace_destroy(struct trace_t *s) 
{
	int ret;
	struct slist_entry *i;

	assert(s != NULL);
	for (i = s->traced_sigs->first; i != NULL; i = i->next) {
		free(i->data);
	}

	slist_destroy(s->traced_sigs);

	ret = fclose(s->output);
	assert(ret == 0);

	free(s);
}

void
trace_add_signal(
	struct trace_t *s,
	const struct signal *sig,
	const char *name,
	enum type_kind type,
	int display_bits
)
{
	struct trace_sig_wrapper *tw = 
		malloc(sizeof(struct trace_sig_wrapper));
	assert(tw != NULL);
	assert(! s->header_written);

	tw->id_code = s->ident_code;
	if (tw->id_code > 126) {
		faum_log(FAUM_LOG_WARNING, "fauhdli", "tracer", 
			"Not tracing signal %s, out ident codes.\n", name);
		free(tw);
		return;
	}

	s->ident_code++;
	tw->type = type;
	tw->sig = sig;

	vcd_add_sig(s->output, name, tw->id_code, type, display_bits);
	slist_add(s->traced_sigs, tw);
}

void
trace_sig_event(struct trace_t *s, const struct signal *sig)
{
	const struct slist_entry *i;
	const struct trace_sig_wrapper *tw = NULL;

	if (! s->header_written) {
		trace_end_header(s);
	}
	
	for (i = s->traced_sigs->first; i != NULL; i = i->next) {
		const struct trace_sig_wrapper *t = 
			(const struct trace_sig_wrapper *)i->data;

		if (t->sig == sig) {
			tw = (const struct trace_sig_wrapper *)i->data;
			break;
		}
	}

	if (tw == NULL) {
		/* signal not traced. */
		return;
	}

	vcd_dump_var(s->output, tw);
}

void
trace_time_advance(struct trace_t *s, universal_integer sim_time)
{
	if (! s->header_written) {
		trace_end_header(s);
	}

	fprintf(s->output, "#%" PRIi64 "\n", sim_time);
}
