/* $Id: mouse_matcher.c,v 1.18 2009-02-05 16:48:24 potyra Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdbool.h>
#include <inttypes.h>
#include <string.h>
#include <dirent.h>
#include <stdlib.h>

#include "glue-shm.h"
#include "glue-log.h"

#include "mouse_matcher.h"

#define COMP "mouse_matcher"

#define ARRAY_SIZE(a) (sizeof(a) / sizeof(a[0]))

struct match_coords {
	uint16_t x;
	uint16_t y;
	bool visible;
	void * _cpssp;
};


struct cpssp {
	/* Config */

	/* Ports */
	struct sig_string *port_pointerdir;
	struct sig_boolean *port_match_state;
	struct sig_boolean *port_event;
	struct sig_integer *port_x;
	struct sig_integer *port_y;
	struct sig_match *port_slot0;
	struct sig_match *port_slot1;
	struct sig_match *port_slot2;
	struct sig_match *port_slot3;
	struct sig_match *port_slot4;
	struct sig_match *port_slot5;
	struct sig_match *port_slot6;
	struct sig_match *port_slot7;
	struct sig_match *port_slot8;
	struct sig_match *port_slot9;
	struct sig_match *port_slot10;
	struct sig_match *port_slot11;

	/* Signals */

	/* State */

	/** maximum used slot number */
	unsigned short num_slots_used;
	/** matching active */
	bool active;

	/** match coordinates of each slot */
	struct match_coords matches[12];

	/** last stored value */
	struct match_coords old_coords;

	/* Processes */
};

static void
mouse_matcher_match_resolve(
	const struct cpssp *cpssp, 
	uint16_t *x, 
	uint16_t *y, 
	bool *visible
)
{
	int i;

	*x = 0;
	*y = 0;
	*visible = false;

	for (i = 0; i < ARRAY_SIZE(cpssp->matches); i++) {
		if (cpssp->matches[i].visible 
		 && i < cpssp->num_slots_used) {
			*visible = true;
			*x = cpssp->matches[i].x;
			*y = cpssp->matches[i].y;
			return;
		}
	}
}

static void
mouse_matcher_send_result(struct cpssp *cpssp)
{
	uint16_t x;
	uint16_t y;
	bool visible;

	if (! cpssp->active) {
		return;
	}

	mouse_matcher_match_resolve(cpssp, &x, &y, &visible);

	if (   visible != cpssp->old_coords.visible
	    || x != cpssp->old_coords.x
	    || y != cpssp->old_coords.y) {
	
		/* important: set x and y first, active next and finally 
		 * event.
		 * Reason: x and y coordinates should be valid when active
		 *         or event hits expect
		 */
		sig_integer_set(cpssp->port_x, cpssp, x);
		sig_integer_set(cpssp->port_y, cpssp, y);
		sig_boolean_set(cpssp->port_match_state, cpssp, visible);
		sig_boolean_set(cpssp->port_event, cpssp, 1);
		sig_boolean_set(cpssp->port_event, cpssp, 0);

		cpssp->old_coords.x = x;
		cpssp->old_coords.y = y;
		cpssp->old_coords.visible = visible;
	}
}

static void
mouse_matcher_match_event(
	void *s,
	bool visible,
	uint16_t x, uint16_t y,
	uint16_t w, uint16_t h
)
{
	struct match_coords *m = (struct match_coords *)s;
	struct cpssp *cpssp = (struct cpssp *)m->_cpssp;

	m->x = x;
	m->y = y;
	m->visible = visible;

	mouse_matcher_send_result(cpssp);

}

static int
#ifdef DARWIN
match_ppm_png_name(struct dirent *d)
#else
match_ppm_png_name(const struct dirent *d)
#endif
{
	char *str = NULL;

	assert(d != NULL);
	str = strrchr(d->d_name, '.');
	if (! str) {
		return 0;
	}
	return (strcmp(str, ".ppm") == 0) || (strcmp(str, ".png") == 0);
}

static struct sig_match*
mouse_matcher_get_slot(struct cpssp *cpssp, unsigned int idx)
{
	switch(idx) {
	case 0:
		return cpssp->port_slot0;
	case 1:
		return cpssp->port_slot1;
	case 2:
		return cpssp->port_slot2;
	case 3:
		return cpssp->port_slot3;
	case 4:
		return cpssp->port_slot4;
	case 5:
		return cpssp->port_slot5;
	case 6:
		return cpssp->port_slot6;
	case 7:
		return cpssp->port_slot7;
	case 8:
		return cpssp->port_slot8;
	case 9:
		return cpssp->port_slot9;
	case 10:
		return cpssp->port_slot10;
	case 11:
		return cpssp->port_slot11;
	default:
		assert(false);
		return NULL;
	}
}

static void
mouse_matcher_activate_matching(struct cpssp *cpssp, const char *dir)
{
	int n;
	unsigned int idx = 0;
	struct dirent **namelist = NULL;
	char fullname[500];

	n = scandir(dir, &namelist, match_ppm_png_name, alphasort);
	if (n < 0) {
		faum_log(FAUM_LOG_WARNING, COMP, "", 
			"Couldn't open cursor image directory %s\n", 
			dir);
		return;
	}

	if (n == 0) {
		/* no images */
		faum_log(FAUM_LOG_WARNING, COMP, "",
			"No cursor images in directory %s\n",
			dir);

		return;
	}

	/* got mouse patterns */
	while (n--) {
		assert(namelist[n] != NULL);
		snprintf(fullname, sizeof(fullname), "%s/%s", 
			dir, namelist[n]->d_name);
		fullname[sizeof(fullname) - 1] = '\0';

		if (ARRAY_SIZE(cpssp->matches) <= idx) {
			faum_log(FAUM_LOG_WARNING, COMP, 
				"ignoring mouse watch %s: out of slots\n",
				fullname);
			free(namelist[n]);
			continue;
		}

		sig_match_add_match(mouse_matcher_get_slot(cpssp, idx), 
					&cpssp->matches[idx],
					fullname);
		idx++;
		free(namelist[n]);
	}
	free(namelist);

	cpssp->num_slots_used = idx;
}

static void
mouse_matcher_deactivate_matching(struct cpssp *cpssp)
{
	unsigned int i;

	for (i = 0; i < cpssp->num_slots_used; i++) {
		sig_match_remove_match(mouse_matcher_get_slot(cpssp, i),
				&cpssp->matches[i]);

		cpssp->matches[i].visible = false;
		cpssp->matches[i].x = 0;
		cpssp->matches[i].y = 0;
	}

	mouse_matcher_send_result(cpssp);

	cpssp->num_slots_used = 0;
}

static void
mouse_matcher_pointerdir_set(void *_cpssp, const char *str)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	if (*str) {
		if (cpssp->active) {
			faum_log(FAUM_LOG_WARNING, COMP, "",
				"matching already active.\n");
			return;
		}

		mouse_matcher_activate_matching(cpssp, str);
		cpssp->active = true;
	} else {
		if (! cpssp->active) {
			faum_log(FAUM_LOG_WARNING, COMP, "",
				"matching already not active.\n");
			return;
		}
		mouse_matcher_deactivate_matching(cpssp);
		cpssp->active = false;
	}
}

void
mouse_matcher_init(
	unsigned int nr,
	struct sig_string *port_pointerdir,
	struct sig_boolean *port_match_state,
	struct sig_boolean *port_event,
	struct sig_integer *port_x,
	struct sig_integer *port_y,
	struct sig_match *port_slot0,
	struct sig_match *port_slot1,
	struct sig_match *port_slot2,
	struct sig_match *port_slot3,
	struct sig_match *port_slot4,
	struct sig_match *port_slot5,
	struct sig_match *port_slot6,
	struct sig_match *port_slot7,
	struct sig_match *port_slot8,
	struct sig_match *port_slot9,
	struct sig_match *port_slot10,
	struct sig_match *port_slot11
)
{
	struct cpssp *cpssp;
	int i;

	static struct sig_string_funcs setf = { 
		.set = mouse_matcher_pointerdir_set
	};

	static struct sig_match_funcs matchf = {
		.event = mouse_matcher_match_event
	};

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	cpssp->active = false;
	cpssp->num_slots_used = 0;
	memset(cpssp->matches, 0, sizeof(cpssp->matches));
	memset(&cpssp->old_coords, 0, sizeof(cpssp->old_coords));

	cpssp->port_pointerdir = port_pointerdir;
	cpssp->port_match_state = port_match_state;
	cpssp->port_event = port_event;
	cpssp->port_x = port_x;
	cpssp->port_y = port_y;
	cpssp->port_slot0 = port_slot0;
	cpssp->port_slot1 = port_slot1;
	cpssp->port_slot2 = port_slot2;
	cpssp->port_slot3 = port_slot3;
	cpssp->port_slot4 = port_slot4;
	cpssp->port_slot5 = port_slot5;
	cpssp->port_slot6 = port_slot6;
	cpssp->port_slot7 = port_slot7;
	cpssp->port_slot8 = port_slot8;
	cpssp->port_slot9 = port_slot9;
	cpssp->port_slot10 = port_slot10;
	cpssp->port_slot11 = port_slot11;

	for (i = 0; i < ARRAY_SIZE(cpssp->matches); i++) {
		cpssp->matches[i]._cpssp = cpssp;
	}

	/* Call */
	sig_match_connect(cpssp->port_slot0, &cpssp->matches[0], &matchf);
	sig_match_connect(cpssp->port_slot1, &cpssp->matches[1], &matchf);
	sig_match_connect(cpssp->port_slot2, &cpssp->matches[2], &matchf);
	sig_match_connect(cpssp->port_slot3, &cpssp->matches[3], &matchf);
	sig_match_connect(cpssp->port_slot4, &cpssp->matches[4], &matchf);
	sig_match_connect(cpssp->port_slot5, &cpssp->matches[5], &matchf);
	sig_match_connect(cpssp->port_slot6, &cpssp->matches[6], &matchf);
	sig_match_connect(cpssp->port_slot7, &cpssp->matches[7], &matchf);
	sig_match_connect(cpssp->port_slot8, &cpssp->matches[8], &matchf);
	sig_match_connect(cpssp->port_slot9, &cpssp->matches[9], &matchf);
	sig_match_connect(cpssp->port_slot10, &cpssp->matches[10], &matchf);
	sig_match_connect(cpssp->port_slot11, &cpssp->matches[11], &matchf);

	/* Out */
	sig_integer_connect_out(port_x, cpssp, 0);
	sig_integer_connect_out(port_y, cpssp, 0);

	/* In */
	sig_string_connect(port_pointerdir, cpssp, &setf);
}

void
mouse_matcher_create(unsigned int nr, const char *name)
{
	struct cpssp *cpssp;

	shm_create(COMP, nr, sizeof(*cpssp));
	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	shm_unmap(cpssp, sizeof(*cpssp));
}

void
mouse_matcher_destroy(unsigned int nr)
{
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	shm_unmap(cpssp, sizeof(*cpssp));
	shm_destroy(COMP, nr);
}
