/* $Id: glue-main.h,v 1.26 2009-03-12 12:12:58 vrsieh Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __GLUE_MAIN_H_INCLUDED
#define __GLUE_MAIN_H_INCLUDED

#define TIME_HZ		(1ULL << 32)
#define TIME_INTERVAL	(TIME_HZ / 1024)

#include <inttypes.h>

struct process {
	union {
		struct {
			void *sp; /* Must be first! Hardcoded in glue-main.c! */
			struct process *prev;
			struct process *next;
			enum {
				PROCESS_SLEEPING,
				PROCESS_RUNNING,
			} state;
			uint64_t inst_offset;
			uint32_t inst_cnt;
			uint32_t inst_limit;
			uint64_t inst_hz;
		};
		char stack[0x10000];
	};
};

extern const char *progname;
extern const char *basedir;
extern int loglevel;
extern int launched;
extern int master;
extern int slave;

extern int failure;

/** Get real-time.
 *  @return real-time in TIME_HZ ticks.
 */
extern unsigned long long
time_real(void);

/** Get virtual time.
 *  @return virtual time in TIME_HZ ticks.
 */
extern unsigned long long
time_virt(void);

/** Stop time.
 *  Used while simulation is stalled/continued.
 */
extern void
time_stop(void);

/** Continue time.
 *  Used while simulation is stalled/continued.
 */
extern void
time_cont(void);

/** Register a timer to call func with parameter data as soon as the
 *  simulation time tsc is reached.
 *  @param tsc virtual timestamp counter, 1 second == 1 * TIME_HZ.
 *  @param func callback that will be called.
 *  @param data parameter to the callback.
 */
extern void
time_call_at(unsigned long long tsc, void (*func)(void *data), void *data);

/** Register a timer to call func with parameter data after interval delay
 *  has passed (simulation time).
 *  @param delay delay before func gets called, 1 second == 1 * TIME_HZ.
 *  @param func callback that will be called.
 *  @param data paramter to the callback.
 */
extern void
time_call_after(unsigned long long delay, void (*func)(void *data), void *data);

/** delete a registered timer from time_call_at/time_call_after.
 *  @param func callback that would have been called.
 *  @param data data that would have been the argument (must match
 *         data from time_call_at/time_call_after).
 *  @return 0 for success, 1 if the timer was not found.
 */
extern int
time_call_delete(void (*func)(void *), void *data);

/** Init a new process.
 *  @param proc process control block.
 *  @param func function to be called in the newly created process context.
 *  @param param parameter passed to func on process startup.
 */
extern void
sched_process_init(struct process *proc, void (*func)(void *), void *param);

/** Return to scheduler.
 */
extern void
sched_to_scheduler(void);

/** Remove current process from run queue.
 */
extern void
sched_sleep(void);

/** Add process to run queue.
 *  @param p process to be added to run queue.
 */
extern void
sched_wakeup(struct process *p);

/** Delay execution for some time.
 *  @param delay delay time.
 */
extern void
sched_delay(unsigned long long delay);

/** Stop simulation node.
 */
extern void
frontend_quit(void);

#endif /* __GLUE_MAIN_H_INCLUDED */
