/*
 * $Id: sig_usb_bus.c,v 1.17 2009-01-27 17:06:42 potyra Exp $
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "fixme.h"

#include "glue-shm.h"

#include "sig_boolean.h"
#include "sig_usb_bus.h"

int
sig_usb_bus_reset_set(
	struct sig_usb_bus_main *sig,
	void *s,
	int val
)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->reset_set
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->reset_set(sig->member[nr].s, val);
	}
	return 0;
}

int
sig_usb_bus_speed_set(
	struct sig_usb_bus_main *sig,
	void *s,
	int val
)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->speed_set
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->speed_set(sig->member[nr].s, val);
	}
	return 0;
}

int
sig_usb_bus_send_token(
	struct sig_usb_bus_main *sig,
	void *s,
	int pid,
	int addr,
	int endp
)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->recv_token
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->recv_token(sig->member[nr].s,
				pid, addr, endp);
	}
	return 0;
}

int
sig_usb_bus_send_sof(
	struct sig_usb_bus_main *sig,
	void *s,
	int frame_num
)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->recv_sof
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->recv_sof(sig->member[nr].s, frame_num);
	}
	return 0;
}

int
sig_usb_bus_send_data(
	struct sig_usb_bus_main *sig,
	void *s,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->recv_data
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->recv_data(sig->member[nr].s,
				pid, length, data, crc16);
	}
	return 0;
}

int
sig_usb_bus_send_handshake(struct sig_usb_bus_main *sig, void *s, int pid)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->recv_handshake
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->recv_handshake(sig->member[nr].s, pid);
	}
	return 0;
}

void
sig_usb_bus_main_connect(
	struct sig_usb_bus_main *sig,
	void *s,
	const struct sig_usb_bus_main_funcs *f
)
{
	assert(sig->member_count
			< sizeof(sig->member) / sizeof(sig->member[0]));

	sig->member[sig->member_count].s = s;
	sig->member[sig->member_count].f = f;
	sig->member_count++;
}

struct sig_usb_bus_main *
sig_usb_bus_main_init(const char *name, int nr)
{
	struct sig_usb_bus_main *sig;

	sig = shm_map(name, nr, sizeof(*sig), 0);

	sig->member_count = 0;

	return sig;
}

void
sig_usb_bus_main_create(const char *name, int nr)
{
	shm_create(name, nr, sizeof(struct sig_usb_bus_main));
}

void
sig_usb_bus_main_destroy(const char *name, int nr)
{
	shm_destroy(name, nr);
}

static void
sig_usb_bus_main_s0_reset_set(void *_f, int val)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_reset_set(f->s1, f, val);
}

static void
sig_usb_bus_main_s0_speed_set(void *_f, int val)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_speed_set(f->s1, f, val);
}

static void
sig_usb_bus_main_s0_recv_token(void *_f, int pid, int addr, int endp)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_token(f->s1, f, pid, addr, endp);
}

static void
sig_usb_bus_main_s0_recv_sof(void *_f, int frame_num)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_sof(f->s1, f, frame_num);
}

static void
sig_usb_bus_main_s0_recv_data(
	void *_f,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_data(f->s1, f, pid, length, data, crc16);
}

static void
sig_usb_bus_main_s0_recv_handshake(void *_f, int pid)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_handshake(f->s1, f, pid);
}

static void
sig_usb_bus_main_s1_reset_set(void *_f, int val)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_reset_set(f->s0, f, val);
}

static void
sig_usb_bus_main_s1_speed_set(void *_f, int val)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_speed_set(f->s0, f, val);
}

static void
sig_usb_bus_main_s1_recv_token(void *_f, int pid, int addr, int endp)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_token(f->s0, f, pid, addr, endp);
}

static void
sig_usb_bus_main_s1_recv_sof(void *_f, int frame_num)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_sof(f->s0, f, frame_num);
}

static void
sig_usb_bus_main_s1_recv_data(
	void *_f,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_data(f->s0, f, pid, length, data, crc16);
}

static void
sig_usb_bus_main_s1_recv_handshake(void *_f, int pid)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_handshake(f->s0, f, pid);
}

struct sig_usb_bus_main_merge *
sig_usb_bus_main_merge(
	struct sig_usb_bus_main *s0,
	struct sig_usb_bus_main *s1
)
{
	static const struct sig_usb_bus_main_funcs s0_funcs = {
		.reset_set = sig_usb_bus_main_s0_reset_set,
		.speed_set = sig_usb_bus_main_s0_speed_set,
		.recv_token = sig_usb_bus_main_s0_recv_token,
		.recv_sof = sig_usb_bus_main_s0_recv_sof,
		.recv_data = sig_usb_bus_main_s0_recv_data,
		.recv_handshake = sig_usb_bus_main_s0_recv_handshake,
	};
	static const struct sig_usb_bus_main_funcs s1_funcs = {
		.reset_set = sig_usb_bus_main_s1_reset_set,
		.speed_set = sig_usb_bus_main_s1_speed_set,
		.recv_token = sig_usb_bus_main_s1_recv_token,
		.recv_sof = sig_usb_bus_main_s1_recv_sof,
		.recv_data = sig_usb_bus_main_s1_recv_data,
		.recv_handshake = sig_usb_bus_main_s1_recv_handshake,
	};
	struct sig_usb_bus_main_merge *m;

	m = malloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	sig_usb_bus_main_connect(s0, m, &s0_funcs);
	m->s1 = s1;
	sig_usb_bus_main_connect(s1, m, &s1_funcs);

	return m;
}

void
sig_usb_bus_main_split(struct sig_usb_bus_main_merge *m)
{
	fixme();
}

struct sig_usb_bus *
sig_usb_bus_init(const char *name, unsigned int nr)
{
	struct sig_usb_bus *c;
	char n[1000];

	c = shm_map(name, nr, sizeof(*c), 0);
	c->type = SIG_GEN_USB_BUS;

	sprintf(n, "%s-power", name);
	c->power = sig_boolean_init(n, nr);
	sprintf(n, "%s-bus", name);
	c->bus = sig_usb_bus_main_init(n, nr);

	return c;
}

void
sig_usb_bus_create(const char *name, int nr)
{
	char n[1000];

	shm_create(name, nr, sizeof(struct sig_usb_bus));

	sprintf(n, "%s-power", name);
	sig_boolean_create(n, nr);
	sprintf(n, "%s-bus", name);
	sig_usb_bus_main_create(n, nr);
}

void
sig_usb_bus_destroy(const char *name, int nr)
{
	char n[1000];

	shm_destroy(name, nr);

	sprintf(n, "%s-power", name);
	sig_boolean_destroy(n, nr);
	sprintf(n, "%s-bus", name);
	sig_usb_bus_main_destroy(n, nr);
}
