/* $Id: acpi-tables.c,v 1.65 2009-01-28 14:37:54 potyra Exp $ 
 *
 * ACPI Tables for FAUmachine BIOS. just testing at the moment.
 *
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */
#include "build_config.h"

#ifdef CONFIG_ACPI_SUPPORT /* compile only if enabled */

#include "compiler.h"

#include "assert.h"
#include "debug.h"
#include "stdio.h"

#include "acpi-tables.h"
#include "chipset.h"

/* oem id field of acpi tables, 6 chars */
#define FAUM_OEMID "FAUM  "
/* oem model id, 8 chars */
#define FAUM_MODELID "faum0001"

/* ==================== REAL-MODE MODE ==================== */
#if defined(RUNTIME_RM) || defined(INIT_RM)

struct acpi_rsdp {
	char signature[8];
	unsigned char checksum1;
	char oemid[6];
	char revision;
	uint32_t rsdtaddress;
	uint32_t length;
	uint32_t xsdtaddress_lo;
	uint32_t xsdtaddress_hi;
	unsigned char checksum2;
	char reserved[3];
};

#endif
/* ==================== REAL-MODE RUNTIME ==================== */
#ifdef RUNTIME_RM

/* Root System Description Pointer */
struct acpi_rsdp faum_rsdp __attribute__ ((aligned (16))) = {
	"RSD PTR ", /* signature */
	0x00, /* checksum1, calculated at init */
	FAUM_OEMID, /* oemid */
	0x02, /* revision */
	0, /* rsdtaddress, filled in at init */
	sizeof(struct acpi_rsdp), /* table length */
	0, /* xsdtaddress_lo, filled in at init */
	0, /* xsdtaddress_hi */
	0x00, /* checksum2, calculated at init */
	"\0\0\0" /* reserved */
};

#else

extern struct acpi_rsdp faum_rsdp;

#endif /* RUNTIME_RM */
/* =================== REAL-MODE INIT ==================== */
#ifdef INIT_RM

CODE16;

#include "string.h"
#include "var.h"

/*
 * Differentiated System Description Table Header
 */
struct acpi_dsdt {
	/* header */
	char signature[4];
	uint32_t length;
	char revision;
	unsigned char checksum;
	char oemid[6];
	char oemtableid[8];
	uint32_t oemrevision;
	char creatorid[4];
	uint32_t creatorrevision;
	/* data section */
	/* empty at the moment */
} PACKED;

/* 
 * Include DSDT, generated by Intel ASL Compiler (iasl)
 * iasl generates a "unsigned char AmlCode[]" in "faum-dsdt.h", which
 * is renamed to faum_dsdt by the preprocessor.
 */
#define AmlCode faum_dsdt
static const
#include "faum-dsdt.h"
#undef AmlCode

/* Fixed ACPI Control Structure */
static const struct acpi_facs {
	char signature[4];
	uint32_t length;
	uint32_t hardware_signature;
	uint32_t firmware_waking_vector;
	uint32_t global_lock;
	uint32_t flags;
	uint64_t x_firmware_waking_vector;
	char version;
	char reserved[31];
} PACKED faum_facs = {
	"FACS", /* signature */
	sizeof(struct acpi_facs), /* length */
	0x23232323, /* hw sig (FIXME: must be calculated at bios init) */
	0x00000000, /* 32 bit firmware waking vector */
	0x00000000, /* global lock */
	0x00000000, /* feature flags: S4BIOS unsupported */
	0, /* 64 bit firmware waking vector */
	1 - 2, /* 1-version of the table */
	{ 0 } /*reserved, must be 0 */
};

/* Generic Address Format (GAS) data type */
struct acpi_gas {
	unsigned char address_space_id;
	unsigned char register_bit_width;
	unsigned char register_bit_offset;
	unsigned char access_size;
	uint32_t address_lo;
	uint32_t address_hi;
} PACKED;

/* Fixed ACPI Description Table */
static const struct acpi_fadt {
	/* header */
	char signature[4];
	uint32_t length;
	char revision;
	unsigned char checksum;
	char oemid[6];
	char oemtableid[8];
	uint32_t oemrevision;
	char creatorid[4];
	uint32_t creatorrevision;
	/* data section */
	uint32_t facsaddress;
	uint32_t dsdtaddress;
	char reserved;
	char preferredpmprofile;
	uint16_t sci_int;
	uint32_t smi_cmd;
	char acpienable;
	char acpidisable;
	char s4bios_req;
	char pstate_cnt;
	uint32_t pm1a_evt_blk;
	uint32_t pm1b_evt_blk;
	uint32_t pm1a_cnt_blk;
	uint32_t pm1b_cnt_blk;
	uint32_t pm2_cnt_blk;
	uint32_t pm_tmr_blk;
	uint32_t gpe0_blk;
	uint32_t gpe1_blk;
	char pm1_evt_len;
	char pm1_cnt_len;
	char pm2_cnt_len;
	char pm_tmr_len;
	char gpe0_blk_len;
	char gpe1_blk_len;
	char gpe1_base;
	char cst_cnt;
	uint16_t p_lvl2_lat;
	uint16_t p_lvl3_lat;
	uint16_t flush_size;
	uint16_t flush_stride;
	char duty_offset;
	char duty_width;
	char day_alrm;
	char mon_alrm;
	char century;
	uint16_t iapc_boot_arch;
	char reserved2;
	uint32_t flags;
	struct acpi_gas reset_reg;
	char reset_value;
	char reserved3[3];
	uint32_t x_firmware_ctl_lo;
	uint32_t x_firmware_ctl_hi;
	uint32_t x_dsdt_lo;
	uint32_t x_dsdt_hi;
	struct acpi_gas x_pm1a_evt_blk;
	struct acpi_gas x_pm1b_evt_blk;
	struct acpi_gas x_pm1a_cnt_blk;
	struct acpi_gas x_pm1b_cnt_blk;
	struct acpi_gas x_pm2_cnt_blk;
	struct acpi_gas x_pm_tmr_blk;
	struct acpi_gas x_gpe0_blk;
	struct acpi_gas x_gpe1_blk;
} PACKED faum_fadt = {
	"FACP", /* header */
	sizeof(struct acpi_fadt), /* table length */
	0x04, /* revision */
	0x00, /* checksum, calculated at init */
	FAUM_OEMID, /* oemid */
	FAUM_MODELID, /* oem table id (model id) */
	0x00000001, /* oem revision */
	"knil", /* creator id */
	0x01006863, /* creator revision */
	0, /* facsaddress, filled in at init */
	0, /* dsdtaddress, filled in at init */
	0, /* reserved */
	0, /* preferredpmprofile */
	9, /* sci_int */
	0x000000b2, /* smi_cmd */
			/* System port address of the SMI Command Port. [...]
			 * must be zero on system that does not support System
			 * Management mode. */
	0x01, /* acpienable */
			/* The value to write to SMI_CMD to disable SMI
			 * ownership of the ACPI hardware registers. */
	0x00, /* acpidisable */
	0, /* s4bios_req */ /* 0 = unsupported */
	0, /* pstate_cnt */
	0x00004000, /* pm1a_evt_blk */
	0, /* pm1b_evt_blk */
	0x00004004, /* pm1a_cnt_blk */
	            /* is 0x00004040 in gigabyte bios, probably typo */
	0, /* pm1b_cnt_blk */
	0, /* pm2_cnt_blk */ /* implemented by real 82443BX but not by us */
	0x00004008, /* pm_tmr_blk */
	0x0000400c, /* gpe0_blk */
	0, /* gpe1_blk */
	4, /* pm1_evt_len */
	2, /* pm1_cnt_len */
	0, /* pm2_cnt_len */ /* see pm2_cnt_blk */
	4, /* pm_tmr_len */
	4, /* gpe0_blk_len */
	0, /* gpe1_blk_len */
	0, /* gpe1_base */
	0, /* cst_cnt */
	101, /* p_lvl2_lat */
	1001, /* p_lvl3_lat */
	0, /* flush_size */
	0, /* flush_stride */
	0, /* duty_offset */
	0, /* duty_width */
	0, /* day_alrm */
	0, /* mon_alrm */
	0, /* century */
	  (1 <<  0) /* legacy_devices */ /* FIXME: check */
	| (1 <<  1) /* 8042 */
	| (0 <<  2) /* vga_not_present */
	/* bits 3-15 reserved */
	, /* iapc_boot_arch */
	0, /* reserved2 */
	  (0 <<  0) /* winbvd */
	| (0 <<  1) /* winbvd_flush */
	| (0 <<  2) /* proc_c1 */
	| (0 <<  3) /* p_lvl2_up */
	| (0 <<  4) /* pwr_button */ /* 0 fixed, 1 control or n/a */
	| (1 <<  5) /* slp_button */ /* 0 fixed, 1 control or n/a */
	| (0 <<  6) /* fix_rtc */
	| (0 <<  7) /* rtc_s4 */
	| (0 <<  8) /* tmr_val_ext */
	| (0 <<  9) /* dck_cap */
	| (0 << 10) /* reset_reg_sup */
	| (0 << 11) /* sealed_case */
	| (0 << 12) /* headless */
	| (0 << 13) /* cpu_sw_slp */
	| (0 << 14) /* pci_exp_wak */
	| (0 << 15) /* use_platform_clock */
	| (0 << 16) /* s4_rtc_sts_valid */
	| (0 << 17) /* remote_power_on_capable */
	| (0 << 18) /* force_apic_cluster_model */
	| (0 << 19) /* force_apic_physical_destination_mode */
	/* bits 20-31 reserved */
	, /* flags */
	{ 0 }, /* reset_reg */
	0, /* reset_value */
	"\0\0\0", /* reserved3[3] */
	0, /* x_firmware_ctl_lo, filled in at init */
	0, /* x_firmware_ctl_hi */
	0, /* x_dsdt_lo, filled in at init */
	0, /* x_dsdt_hi */
	{	0x01, /* address_space_id */
		32, /* register_bit_width */
		0, /* register_bit_offset */
		0, /* access_size */
		0x00004000, /* address_lo */
		0, /* address_hi */
	}, /* x_pm1a_evt_blk */
	{ 0 }, /* x_pm1b_evt_blk */
	{	0x01, /* address_space_id */
		16, /* register_bit_width */
		0, /* register_bit_offset */
		0, /* access_size */
		0x00004004, /* address_lo */
		0, /* address_hi */
	}, /* x_pm1a_cnt_blk */
	{ 0 }, /* x_pm1b_cnt_blk */
	{ 0 }, /* x_pm2_cnt_blk */ /* see pm2_cnt_blk */
	{ 	0x01, /* address_space_id */
		24, /* register_bit_width */
		0, /* register_bit_offset */
		0, /* access_size */
		0x00004008, /* address_lo */
		0, /* address_hi */
	}, /* x_pm_tmr_blk */
	{ 	0x01, /* address_space_id */
		32, /* register_bit_width */
		0, /* register_bit_offset */
		0, /* access_size */
		0x0000400c, /* address_lo */
		0, /* address_hi */
	}, /* x_gpe0_blk */
	{ 0 } /* x_gpe1_blk */
};

/*
 * Multiple APIC Description Table (MADT)
 * Page 112
 */
static const struct acpi_madt {
	char signature[4];
	uint32_t length;
	char revision;
	char checksum;
	char oemid[6];
	char oemtableid[8];
	uint32_t oemrevision;
	char creatorid[4];
	uint32_t creatorrevision;
	/* data section */
	uint32_t local_apic_address;
	uint32_t flags;
	/* APIC Structure[n] */
} PACKED faum_madt = {
	"API0", /* header */
	0, /* table length; filled at init */
	0x04, /* revision */
	0x00, /* checksum, calculated at init */
	FAUM_OEMID, /* oemid */
	FAUM_MODELID, /* oem table id (model id) */
	0x00000001, /* oem revision */
	"voss", /* creator id */
	0x01006863, /* creator revision */
	/* Data Section */
	0xfee00000, /* local apic address */
	0 /* Reserved */
	| 1 << 0 /* Dual-8259 setup */
	/* APIC Structure[n] */
};
/* Page 113 */
struct acpi_madt_local_apic {
	uint8_t type;
	uint8_t length;
	uint8_t processor_id;
	uint8_t apic_id;
	uint32_t flags;
} PACKED;
/* Page 114 */
struct acpi_madt_io_apic {
	uint8_t type;
	uint8_t length;
	uint8_t io_apic_id;
	uint8_t reserved;
	uint32_t io_apic_address;
	uint32_t global_system_interrupt_base;
} PACKED;
/* Page 115 */
struct acpi_madt_interrupt_override {
	uint8_t type;
	uint8_t length;
	uint8_t bus;
	uint8_t source;
	uint32_t global_system_interrupt;
	uint16_t flags;
} PACKED;
/* ... */

/* Extended System Description Table */
static const struct acpi_xsdt {
	/* header */
	char signature[4];
	uint32_t length;
	char revision;
	unsigned char checksum;
	char oemid[6];
	char oemtableid[8];
	uint32_t oemrevision;
	char creatorid[4];
	uint32_t creatorrevision;
	/* data section */
	uint32_t fadtaddress_lo;
	uint32_t fadtaddress_hi;
	uint32_t madtaddress_lo;
	uint32_t madtaddress_hi;
} PACKED faum_xsdt = {
	"XSDT", /* signature */
	sizeof(struct acpi_xsdt), /* table length */
	0x01, /* revision */
	0x00, /* checksum, calculated later */
	FAUM_OEMID, /* oemid */
	FAUM_MODELID, /* oem table id (model id) */
	0x00000001, /* oem revision */
	"knil", /* creator id */
	0x01006863, /* creator revision */
	0, /* address of FADT, filled in at init */
	0, /* fadtaddress_hi */
	0, /* address of MADT, filled in at init */
	0 /* madtaddress_hi */
};

/* Root System Description Table */
static const struct acpi_rsdt {
	/* header */
	char signature[4];
	uint32_t length;
	char revision;
	unsigned char checksum;
	char oemid[6];
	char oemtableid[8];
	uint32_t oemrevision;
	char creatorid[4];
	uint32_t creatorrevision;
	/* data section */
	uint32_t fadtaddress;
	uint32_t madtaddress;
} PACKED faum_rsdt = {
	"RSDT", /* header */
	sizeof(struct acpi_rsdt), /* table length */
	0x01, /* revision */
	0x00, /* checksum, calculated later */
	FAUM_OEMID, /* oemid */
	FAUM_MODELID, /* oem table id (model id) */
	0x00000001, /* oem revision */
	"knil", /* creator id */
	0x01006863, /* creator revision */
	0, /* address of FADT, filled in at init */
	0 /* address of MADT, filled in at init */
};

void
acpi_tables_init(void)
{
	struct acpi_rsdp * rsdpaddr;
	struct acpi_rsdt * rsdtaddr;
	struct acpi_xsdt * xsdtaddr;
	struct acpi_fadt * fadtaddr;
	struct acpi_madt * madtaddr;
	unsigned int madt_length;
	struct acpi_madt_local_apic *madtaddr_local_apic;
	struct acpi_madt_io_apic *madtaddr_io_apic;
	struct acpi_madt_interrupt_override *madtaddr_interrupt_override;
	struct acpi_facs * facsaddr;
	struct acpi_dsdt * dsdtaddr;
	unsigned int size;
	int i;
	unsigned char sum;

	/* Depends on actual board configuration - FIXME */
	madt_length = sizeof(struct acpi_madt)
		+ 2 * sizeof(struct acpi_madt_local_apic)
		+ 1 * sizeof(struct acpi_madt_io_apic)
		+ 1 * sizeof(struct acpi_madt_interrupt_override);

	/* check if tables will fit in acpi memory */
	size = sizeof(faum_rsdt) + sizeof(faum_xsdt)
		+ sizeof(faum_fadt) + madt_length
		+ sizeof(faum_dsdt);

	if ((ACPI_MEM_SIZE - ACPI_NVS_SIZE) < size) {
		dprintf("ACPI-tables won't fit in ACPI-memory! ");
		dprintf("needed: %u, got: %u\n", size, (ACPI_MEM_SIZE
							- ACPI_NVS_SIZE));
		assert(0);
	}

#if 0
	rsdpaddr = &faum_rsdp;
#else
	asm volatile (
		""
		: "=r" (rsdpaddr)
		: "0" (&faum_rsdp)
	);
#endif
	assert(((unsigned int) rsdpaddr & 0xf) == 0);

	facsaddr = (struct acpi_facs *) (var_get(mem_size) - ACPI_MEM_SIZE);

	rsdtaddr = (struct acpi_rsdt *) (var_get(mem_size) - ACPI_MEM_SIZE
							   + ACPI_NVS_SIZE);

	xsdtaddr = (struct acpi_xsdt *) (((unsigned int) rsdtaddr)
					 + sizeof(faum_rsdt));
	fadtaddr = (struct acpi_fadt *) (((unsigned int) xsdtaddr)
					 + sizeof(faum_xsdt));
	madtaddr = (struct acpi_madt *) (((unsigned int) fadtaddr)
					 + sizeof(faum_fadt));
	dsdtaddr = (struct acpi_dsdt *) (((unsigned int) madtaddr)
					 + madt_length);

	/*
	 * Fill in RSDP.
	 */
	if (rsdpaddr == &faum_rsdp) {
		/* Enable write access to BIOS. */
		MODE_PREFIX(chipset_bios_readwrite)();
	} else {
		memcpy(rsdpaddr, &faum_rsdp, sizeof(faum_rsdp));
	}

	rsdpaddr->rsdtaddress = (uint32_t) rsdtaddr;
	rsdpaddr->xsdtaddress_lo = (uint32_t) xsdtaddr;
	rsdpaddr->xsdtaddress_hi = (uint32_t) 0;
	sum = 0;
	for (i = 0; i < 20; i++) {
		sum += *((unsigned char *) rsdpaddr + i);
	}
	rsdpaddr->checksum1 -= sum;
	sum = 0;
	for (i = 0; i < rsdpaddr->length; i++) {
		sum += *((unsigned char *) rsdpaddr + i);
	}
	rsdpaddr->checksum2 -= sum;

	if (rsdpaddr == &faum_rsdp) {
		/* Disable write access to BIOS. */
		MODE_PREFIX(chipset_bios_readonly)();
	}

	/*
	 * Fill in RSDT.
	 */
	memcpy(rsdtaddr, &faum_rsdt, sizeof(faum_rsdt));
	rsdtaddr->fadtaddress = (uint32_t) fadtaddr;
	rsdtaddr->madtaddress = (uint32_t) madtaddr;
	sum = 0;
	for (i = 0; i < rsdtaddr->length; i++) {
		sum += *((unsigned char *) rsdtaddr + i);
	}
	rsdtaddr->checksum -= sum;

	/*
	 * Fill in XSDT.
	 */
	memcpy(xsdtaddr, &faum_xsdt, sizeof(faum_xsdt));
	xsdtaddr->fadtaddress_lo = (uint32_t) fadtaddr;
	xsdtaddr->madtaddress_lo = (uint32_t) madtaddr;
	sum = 0;
	for (i = 0; i < xsdtaddr->length; i++) {
		sum += *((unsigned char *) xsdtaddr + i);
	}
	xsdtaddr->checksum -= sum;

	/*
	 * Fill in FADT.
	 */
	memcpy(fadtaddr, &faum_fadt, sizeof(faum_fadt));
	fadtaddr->dsdtaddress = (uint32_t) dsdtaddr;
	fadtaddr->x_dsdt_lo = (uint32_t) dsdtaddr;
	fadtaddr->facsaddress = (uint32_t) facsaddr;
	fadtaddr->x_firmware_ctl_lo = (uint32_t) facsaddr;
	sum = 0;
	for (i = 0; i < fadtaddr->length; i++) {
		sum += *((unsigned char *) fadtaddr + i);
	}
	fadtaddr->checksum -= sum;

	/*
	 * Fill in DSDT.
	 */
	memcpy(dsdtaddr, faum_dsdt, sizeof(faum_dsdt));

	/*
	 * Fill in MADT.
	 * Page 111.
	 */
	memcpy(madtaddr, &faum_madt, sizeof(faum_madt));
	/* Local APIC Entries */
	madtaddr_local_apic = (struct acpi_madt_local_apic *) (madtaddr + 1);
	for (i = 0; i < 2; i++) { /* FIXME */
		madtaddr_local_apic->type = 0; /* Processor local APIC */
		madtaddr_local_apic->length = sizeof(*madtaddr_local_apic);
		madtaddr_local_apic->processor_id = i;
		madtaddr_local_apic->apic_id = i;
		madtaddr_local_apic->flags = (i == 0); /* FIXME */
		madtaddr_local_apic++;
	}
	/* I/O-APIC Entries */
	madtaddr_io_apic = (struct acpi_madt_io_apic *) madtaddr_local_apic;
	madtaddr_io_apic->type = 1; /* I/O-APIC */
	madtaddr_io_apic->length = sizeof(*madtaddr_io_apic);
	madtaddr_io_apic->io_apic_id = 8; /* FIXME */
	madtaddr_io_apic->reserved = 0;
	madtaddr_io_apic->io_apic_address = 0xfec00000;
	madtaddr_io_apic->global_system_interrupt_base = 0;
	madtaddr_io_apic++;
	/* Interrupt Source Overrides */
	madtaddr_interrupt_override
		= (struct acpi_madt_interrupt_override *) madtaddr_io_apic;
	madtaddr_interrupt_override->type = 2; /* Interrupt Override */
	madtaddr_interrupt_override->length = sizeof(*madtaddr_interrupt_override);
	madtaddr_interrupt_override->bus = 0; /* ISA */
	madtaddr_interrupt_override->source = 0;
	madtaddr_interrupt_override->global_system_interrupt = 2;
	madtaddr_interrupt_override->flags = 0; /* Conforms to everything... */
	madtaddr_interrupt_override++;

	/* Length */
	madtaddr->length = (uint32_t) madtaddr_interrupt_override
			- (uint32_t) madtaddr;
	/* Checksum */
	sum = 0;
	for (i = 0; i < madtaddr->length; i++) {
		sum += *((unsigned char *) madtaddr + i);
	}
	madtaddr->checksum -= sum;

	/* handle FACS */
	/*
	 * FIXME: check [x_]firmware_waking_vector in FACS 
	 *        - transfer control to OSPM if non-zero
	 *        - FACS should not be overwritten if waking from STD/STR!
	 */
	memcpy(facsaddr, &faum_facs, sizeof(faum_facs));
	/* FIXME: no memcpy, but fill in values by hand here */
}

#endif /* INIT_RM */

#endif /* CONFIG_ACPI_SUPPORT */
