/* $Id: pci_intel_eepro100.c,v 1.36 2009-01-28 12:59:21 potyra Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "random100.h"
#include "glue-shm.h"

#include "sig_integer.h"

#include "chip_intel_82557.h"

#include "pci_intel_eepro100.h"

#define COMP "pci_intel_eepro100"

struct cpssp {
	/* Config */

	/* Ports */
	struct sig_pci_bus *port_pci;
	struct sig_eth *port_eth;
	struct sig_boolean *port_opt_busy_led;

	/* Signals */
	struct sig_eth *sig_eth;

	/* State */
	unsigned int recv_loss;
	unsigned int send_loss;

	/* Processes */
	unsigned int chip;
};

static void
pci_intel_eepro100_recv_loss(void *_cpssp, int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	
	cpssp->recv_loss = val;
}

static void
pci_intel_eepro100_send_loss(void *_cpssp, int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	
	cpssp->send_loss = val;
}

static void
pci_intel_eepro100_connect(void *_cpssp, const char *port, void *_sig)
{
	static const struct sig_integer_funcs recv_loss_funcs = {
		.set = pci_intel_eepro100_recv_loss,
	};
	static const struct sig_integer_funcs send_loss_funcs = {
		.set = pci_intel_eepro100_send_loss,
	};
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	struct sig_integer *sig = (struct sig_integer *) _sig;

	if (strcmp(port, "recv_loss") == 0) {
		sig_integer_connect_in(sig, cpssp, &recv_loss_funcs);
	} else if (strcmp(port, "send_loss") == 0) {
		sig_integer_connect_in(sig, cpssp, &send_loss_funcs);
	} else {
		assert(0);
	}
}

static void
pci_intel_eepro100_disconnect(void *_cpssp, const char *port)
{
	// struct cpssp *cpssp = (struct cpssp *) _cpssp;

	/* FIXME */
	if (strcmp(port, "recv_loss") == 0) {
		// sig_integer_disconnect_in(sig, cpssp);
	} else if (strcmp(port, "send_loss") == 0) {
		// sig_integer_disconnect_in(sig, cpssp);
	} else {
		assert(0);
	}
}

static void
pci_intel_eepro100_in_recv(void *_cpssp, const void *buf, unsigned int buflen)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (random100() < cpssp->recv_loss) {
		/* Packet lost due to fault injection. */
		return;
	}

	sig_eth_send(cpssp->sig_eth, cpssp, buf, buflen);
}

static void
pci_intel_eepro100_out_recv(void *_cpssp, const void *buf, unsigned int buflen)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (random100() < cpssp->send_loss) {
		/* Packet lost due to fault injection. */
		return;
	}

	sig_eth_send(cpssp->port_eth, cpssp, buf, buflen);
}

static void
pci_intel_eepro100_recv_loss_set(
	void *_cpssp,
	unsigned long long loc0,
	unsigned long long loc1,
	unsigned int val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->recv_loss = val;
}

static void
pci_intel_eepro100_send_loss_set(
	void *_cpssp,
	unsigned long long loc0,
	unsigned long long loc1,
	unsigned int val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->send_loss = val;
}

void
pci_intel_eepro100_init(
	unsigned int nr,
	struct sig_manage *manage,
	struct sig_pci_bus *port_pci,
	struct sig_eth *port_eth,
	struct sig_boolean *port_opt_busy_led,
	struct sig_fault *fault_recv_loss,
	struct sig_fault *fault_send_loss
)
{
	static const struct sig_manage_funcs manage_funcs = {
		.connect = pci_intel_eepro100_connect,
		.disconnect = pci_intel_eepro100_disconnect,
	};
	static const struct sig_eth_funcs in_funcs = {
		.recv = pci_intel_eepro100_in_recv,
	};
	static const struct sig_eth_funcs out_funcs = {
		.recv = pci_intel_eepro100_out_recv,
	};
	static const struct sig_fault_funcs recv_loss_funcs = {
		.set = pci_intel_eepro100_recv_loss_set,
	};
	static const struct sig_fault_funcs send_loss_funcs = {
		.set = pci_intel_eepro100_send_loss_set,
	};
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	cpssp->port_eth = port_eth;

	cpssp->sig_eth = sig_eth_init(COMP "-sig_eth", nr);

	chip_intel_82557_init(cpssp->chip,
			port_pci->p5V, port_pci->n_reset,
			port_pci->idsel, port_pci->main,
			port_pci->intA, cpssp->sig_eth, port_opt_busy_led);

	sig_manage_connect(manage, cpssp, &manage_funcs);

	sig_eth_connect(cpssp->port_eth, cpssp, &in_funcs);
	sig_eth_connect(cpssp->sig_eth, cpssp, &out_funcs);

	sig_fault_connect(fault_recv_loss, cpssp, &recv_loss_funcs);
	sig_fault_connect(fault_send_loss, cpssp, &send_loss_funcs);
}

void
pci_intel_eepro100_create(unsigned int nr, const char *name, const char *mac)
{
	static char mac_default[] = "00:D0:B7:00:00:01";
	struct cpssp *cpssp;

	shm_create(COMP, nr, sizeof(*cpssp));
	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	sig_eth_create(COMP "-sig_eth", nr);

	if (mac == NULL) {
		mac = "00:00:00:00:00:00";
	}

	if (strcmp(mac, "00:00:00:00:00:00") == 0) {
		mac = mac_default;
	}
	cpssp->chip = chip_intel_82557_create(mac);
	if (mac == mac_default) {
		mac_default[sizeof(mac_default) - 2]++;
	}

	shm_unmap(cpssp, sizeof(*cpssp));
}

void
pci_intel_eepro100_destroy(unsigned int nr)
{
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	cpssp->recv_loss = 0;
	cpssp->send_loss = 0;

	chip_intel_82557_destroy(cpssp->chip);

	sig_eth_destroy(COMP "-sig_eth", nr);

	shm_unmap(cpssp, sizeof(*cpssp));
	shm_destroy(COMP, nr);
}
