/* $Id: glue-vhdl.c,v 1.149 2010-02-11 18:11:19 potyra Exp $
 *
 *  Glue layer implementing the foreign interface of fauhdli for FAUmachine,
 *  so that fauhdli can access FAUmachine signals and components.
 *  Must not be called by FAUmachine itself.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue-vhdl.h"
#include "glue-dist.h"
#include "glue-log.h"
#include "glue-main.h"
#include "../node-pc/system.h"
#include "sig_boolean.h"
#include "sig_integer.h"
#include "sig_gen.h"
#include "sig_string.h"
#include "sig_std_logic.h"
#include "simsetup.h"
#include "structure.h"

/* IDs are generated by system.c in a more random way, now -- FIXME */
/* maximum possible signals. Must match system.c! */
#define GLUE_VHDL_MAX_SIGS 10000
/* maximum possible components. Must match system.c! */
#define GLUE_VHDL_MAX_COMP 10000
/* maximum possible architectures. Must match system.c! */
#define GLUE_VHDL_MAX_ARCH 10000

/** mapping of one vhdl signal to FAUmachine signal */
struct glue_vhdl_signal_t {
	/** type of signal */
	enum sig_gen_type type;
	/** link back to glue_vhdl instance */
	struct glue_vhdl *self;
	/** driver pointer (for callback) */
	void *_drv;
	/** is the signal entry used? */
	bool used;
};

/** parameter passing type */
struct glue_vhdl_param_t {
	/** name of the parameter */
	const char *name;
	/** passed argument */
	union fauhdli_value arg;
};

/** glue vhdl instance state */
struct glue_vhdl {
	/** all signals */
	struct glue_vhdl_signal_t signals[GLUE_VHDL_MAX_SIGS];
	/** wether any component is used/created from us? */
	bool comp_in_use[GLUE_VHDL_MAX_COMP];
	/** wether any component is used/created from us? */
	bool arch_in_use[GLUE_VHDL_MAX_ARCH];
	/** all parameters for a foreign procedure call */
	struct glue_vhdl_param_t params[100];
	/** interpreter instance (as from glue_vhdl_enable_event_wakeups) */
	void *interpreter;
	/** number of parameters present */
	unsigned int num_params;
	/** fauhdli instance (used for fauhdli_set_drv) */
	void *fauhdli;

	/** cache of page_name -> page_id mapping */
	struct {
		/** name of page */
		const char *page_name;
		/** corresponding page id */
		unsigned int page_id;
	} page_cache[10];
	/** number of first free slot in page_cache */
	unsigned int page_cache_count;

	/** cache of host -> node_id mapping */
	struct {
		/** host entry in simulation.setup */
		const char *host;
		/** corresponding node id */
		unsigned int node_id;
	} node_cache[10];
	/** number of first free slot in node_cache */
	unsigned int node_cache_count;

};

/** callback to proxy to boolean changes to fauhdli 
 *  @param _s glue_vhdl_signal_t entry.
 *  @param val boolean value.
 */
static void
glue_vhdl_data_get_boolean(void *_s, unsigned int val)
{
	struct glue_vhdl_signal_t *s = (struct glue_vhdl_signal_t *)_s;
	union fauhdli_value f_val;

	f_val.univ_int = val;

	assert(s->_drv != NULL);

	fauhdli_set_driver(s->self->fauhdli, s->_drv, f_val);
}

static void
glue_vhdl_data_get_integer(void *_s, int val)
{
	struct glue_vhdl_signal_t *s = (struct glue_vhdl_signal_t *)_s;
	union fauhdli_value f_val;

	f_val.univ_int = val;

	assert(s->_drv != NULL);
	fauhdli_set_driver(s->self->fauhdli, s->_drv, f_val);
}

/* FIXME looks exactly the same as boolean */
static void
glue_vhdl_data_get_std_logic(void *_s, unsigned int val)
{
	struct glue_vhdl_signal_t *s = (struct glue_vhdl_signal_t *)_s;
	union fauhdli_value f_val;

	f_val.univ_int = val;

	assert(s->_drv != NULL);
	fauhdli_set_driver(s->self->fauhdli, s->_drv, f_val);
}

/** determine signal kind by name
 *  TODO move to sig_gen.c?
 *  @param name of signal type.
 *  @return signal kind
 */
static enum sig_gen_type
glue_vhdl_determine_type(const char *name)
{
	if (strcmp(name, "agp_bus") == 0) {
		return SIG_GEN_AGP_BUS;
	} else if (strcmp(name, "agp_bus_main") == 0) {
		return SIG_GEN_AGP_BUS_MAIN;
	} else if (strcmp(name, "boolean") == 0) {
		return SIG_GEN_BOOLEAN;
	} else if (strcmp(name, "boolean_or") == 0) {
		return SIG_GEN_BOOLEAN_OR;
	} else if (strcmp(name, "cs") == 0) {
		return SIG_GEN_CS;
	} else if (strcmp(name, "cstring") == 0) {
		return SIG_GEN_STRING;
	} else if (strcmp(name, "dio24") == 0) {
		return SIG_GEN_DIO24;
	} else if (strcmp(name, "dio48") == 0) {
		return SIG_GEN_DIO48;
	} else if (strcmp(name, "eth") == 0) {
		return SIG_GEN_ETH;
	} else if (strcmp(name, "fault") == 0) {
		return SIG_GEN_FAULT;
	} else if (strcmp(name, "floppy") == 0) {
		return SIG_GEN_FLOPPY;
	} else if (strcmp(name, "host_bus") == 0) {
		return SIG_GEN_HOST_BUS;
	} else if (strcmp(name, "host_bus_main") == 0) {
		return SIG_GEN_HOST_BUS_MAIN;
	} else if (strcmp(name, "i2c_bus") == 0) {
		return SIG_GEN_I2C_BUS;
	} else if (strcmp(name, "icc_bus") == 0) {
		return SIG_GEN_ICC_BUS;
	} else if (strcmp(name, "ide_bus") == 0) {
		return SIG_GEN_IDE_BUS;
	} else if (strcmp(name, "integer") == 0) {
		return SIG_GEN_INTEGER;
	} else if (strcmp(name, "isa_bus") == 0) {
		return SIG_GEN_ISA_BUS;
	} else if (strcmp(name, "isa_bus_dma") == 0) {
		return SIG_GEN_ISA_BUS_DMA;
	} else if (strcmp(name, "isa_bus_main") == 0) {
		return SIG_GEN_ISA_BUS_MAIN;
	} else if (strcmp(name, "magneto_optical") == 0) {
		return SIG_GEN_MAGNETO_OPTICAL;
	} else if (strcmp(name, "match") == 0) {
		return SIG_GEN_MATCH;
	} else if (strcmp(name, "mem_bus") == 0) {
		return SIG_GEN_MEM_BUS;
	} else if (strcmp(name, "mem_bus_main") == 0) {
		return SIG_GEN_MEM_BUS_MAIN;
	} else if (strcmp(name, "opt_rgb") == 0) {
		return SIG_GEN_OPT_RGB;
	} else if (strcmp(name, "parallel") == 0) {
		return SIG_GEN_PARALLEL;
	} else if (strcmp(name, "pci_bus") == 0) {
		return SIG_GEN_PCI_BUS;
	} else if (strcmp(name, "pci_bus_idsel") == 0) {
		return SIG_GEN_PCI_BUS_IDSEL;
	} else if (strcmp(name, "pci_bus_main") == 0) {
		return SIG_GEN_PCI_BUS_MAIN;
	} else if (strcmp(name, "power_board") == 0) {
		return SIG_GEN_POWER_BOARD;
	} else if (strcmp(name, "power_board_at") == 0) {
		return SIG_GEN_POWER_BOARD_AT;
	} else if (strcmp(name, "power_device") == 0) {
		return SIG_GEN_POWER_DEVICE;
	} else if (strcmp(name, "ps2") == 0) {
		return SIG_GEN_PS2;
	} else if (strcmp(name, "ps2_main") == 0) {
		return SIG_GEN_PS2_MAIN;
	} else if (strcmp(name, "scsi_bus") == 0) {
		return SIG_GEN_SCSI_BUS;
	} else if (strcmp(name, "std_logic") == 0) {
		return SIG_GEN_STD_LOGIC;
	} else if (strcmp(name, "std_ulogic") == 0) {
		return SIG_GEN_STD_LOGIC;
	} else if (strcmp(name, "serial") == 0) {
		return SIG_GEN_SERIAL;
	} else if (strcmp(name, "shugart_bus") == 0) {
		return SIG_GEN_SHUGART_BUS;
	} else if (strcmp(name, "sound") == 0) {
		return SIG_GEN_SOUND;
	} else if (strcmp(name, "vga") == 0) {
		return SIG_GEN_VGA;
	} else if (strcmp(name, "video") == 0) {
		return SIG_GEN_VIDEO;
	} else if (strcmp(name, "usb_bus") == 0) {
		return SIG_GEN_USB_BUS;
	} else if (strcmp(name, "usb_bus_main") == 0) {
		return SIG_GEN_USB_BUS_MAIN;
	}


	faum_log(FAUM_LOG_WARNING, "glue-vhdl", __func__, 
		"Could not determine signal type '%s'.\n", name);
	return SIG_GEN_MANAGE;
}

static void
glue_vhdl_get_arg(
	const struct glue_vhdl *s,
	const char *arg,
	union fauhdli_value *param
)
{
	unsigned int i;
	for (i = 0; i < s->num_params; i++) {
		if (strcmp(arg, s->params[i].name) == 0) {
			/* found */
			*param = s->params[i].arg;
			return;
		}
	}

	faum_log(FAUM_LOG_FATAL, "glue-vhdl", __func__,
		"Argument %s not found, aborting.\n", arg);
	assert(0);
}

/** flatten a VHDL constant unconstraint string array to a c-string
 *  @param s_ptr base pointer to temporary.
 *  @param s_len length of the string.
 *  @return flattened c-string.
 */
static char *
glue_vhdl_flatten_string(union fauhdli_value s_ptr, size_t s_len)
{
	static char buf[16384];
	size_t i;
	union fauhdli_value *walk = (union fauhdli_value *)s_ptr.pointer;

	assert(s_len < sizeof(buf));
	
	for (i = 0; i < s_len; i++, walk++) {
		buf[i] = (char)(walk->univ_int);
	}
	buf[i] = '\0';
	return buf;
}

static void
glue_vhdl_send_string(
	union fauhdli_value dest_driver,
	union fauhdli_value s_array,
	union fauhdli_value s_lb,
	union fauhdli_value s_ub
)
{
	size_t sz; 
	const char *msg;
	unsigned int sig_id;
	void *_sig;
	struct sig_string *sig;
	
	sz = (size_t)(s_ub.univ_int - s_lb.univ_int + 1);
	msg = glue_vhdl_flatten_string(s_array, sz);
	sig_id = fauhdli_get_sig_id_driver(dest_driver.pointer);
	/* FIXME don't abuse system_sig_get */
	_sig = system_sig_get(sig_id);

	assert(_sig != NULL);
	sig = (struct sig_string *)_sig;


#if 0 /* debug code */
	faum_log(FAUM_LOG_DEBUG, "glue-vhdl", __func__,
		"sending %s to signal %d\n", msg, sig_id);
#endif /* debug code */
	sig_string_set(sig, dest_driver.pointer, msg);
}

static void
glue_vhdl_terminate(void)
{
	sim_exit();
}

static void
glue_vhdl_toggle_debug(void)
{
	loglevel ^= 1;
}

static void
glue_vhdl_shorcut_drv_out(
	union fauhdli_value s_driver,
	union fauhdli_value comp_array,
	union fauhdli_value comp_lb,
	union fauhdli_value comp_ub,
	union fauhdli_value comp_port_array,
	union fauhdli_value comp_port_lb,
	union fauhdli_value comp_port_ub
)
{
	size_t sz;
	int c_id;
	unsigned int sig_id;
	char *s;

	sz = (size_t)(comp_ub.univ_int - comp_lb.univ_int + 1);
	s = glue_vhdl_flatten_string(comp_array, sz);

	c_id = system_comp_lookup(s);
	assert(c_id >= 0);

	sz = (size_t)(comp_port_ub.univ_int - comp_port_lb.univ_int + 1);
	s = glue_vhdl_flatten_string(comp_port_array, sz);

	sig_id = fauhdli_get_sig_id_driver(s_driver.pointer);
	system_comp_connect(c_id, s, sig_id);
}

/* FIXME move to simsetup.c? */
const char *
glue_vhdl_simsetup_get_closest(char *path, const char *key)
{
	const char *ret;
	char *delim;

	while(1) {
		ret = simsetup_get(path, key);
		if (ret != NULL) {
			return ret;
		}
		
		delim = strrchr(path, '/');
		if (delim == NULL) {
			return delim;
		}
		*delim = '\0';
	}

	return ret;
}


static unsigned int
glue_vhdl_page_cache_get(struct glue_vhdl *s, const char *page_name)
{
	unsigned int i;

	for (i = 0; ; i++) {
		if (i == s->page_cache_count) {
			/* Add new page. */
			assert(i < sizeof(s->page_cache) / sizeof(s->page_cache[0]));
			s->page_cache[i].page_id = system_page_create(page_name);
			s->page_cache[i].page_name = page_name;
			s->page_cache_count++;
			break;
		}
		if (strcmp(s->page_cache[i].page_name, page_name) == 0) {
			/* Page found. */
			break;
		}
	}

	return s->page_cache[i].page_id;
}

static unsigned int
glue_vhdl_node_cache_get(struct glue_vhdl *s, const char *host)
{
	unsigned int i;
	unsigned int node_id;

	for (i = 0; i < s->node_cache_count; i++) {
		if (strcmp(s->node_cache[i].host, host) == 0) {
			return s->node_cache[i].node_id;
		}
	}

	/* lookup new host, add new entry */
	node_id = dist_node_lookup(host);

	assert(s->node_cache_count 
		< sizeof(s->node_cache) / sizeof(s->node_cache[0]));
	s->node_cache[s->node_cache_count].host = host;
	s->node_cache[s->node_cache_count].node_id = node_id;
	s->node_cache_count++;

	return node_id;
}



static unsigned int
glue_vhdl_get_page(struct glue_vhdl *s, const char *name)
{
	static char buf[16348];
	size_t i;
	const char *page_name;
	unsigned int ret;

	/* FIXME convert ':' to '/' in name, cf. simsetup.l 
	 */
	assert(strlen(name) < sizeof(buf) - 1);
	for (i = 0; i < strlen(name); i++) {
		if (name[i] == ':') {
			buf[i] = '/';
		} else {
			buf[i] = name[i];
		}
	}
	buf[i] = '\0';
	/* end FIXME */

	page_name = glue_vhdl_simsetup_get_closest(buf, "page");
	if (page_name == NULL) {
		/* no page specified in simsetup, use plain "/" with
		   a page name of "default"
		 */
		ret = glue_vhdl_page_cache_get(s, "default");
		return ret;
	}

	ret = glue_vhdl_page_cache_get(s, page_name);
	return ret;
}

static unsigned int
glue_vhdl_get_node(struct glue_vhdl *s, const char *name)
{
	static char buf[16348];
	size_t i;
	const char *page_name;
	unsigned int ret;

	/* FIXME convert ':' to '/' in name, cf. simsetup.l 
	 */
	assert(strlen(name) < sizeof(buf) - 1);
	for (i = 0; i < strlen(name); i++) {
		if (name[i] == ':') {
			buf[i] = '/';
		} else {
			buf[i] = name[i];
		}
	}
	buf[i] = '\0';
	/* end FIXME */

	page_name = glue_vhdl_simsetup_get_closest(buf, "host");
	if (page_name == NULL) {
		/* no host specified in simsetup, use plain "/" with
		   a host entry of "default"
		 */
		ret = glue_vhdl_node_cache_get(s, "default");
		return ret;
	}

	ret = glue_vhdl_node_cache_get(s, page_name);
	return ret;
}

static void
glue_vhdl_simsetup_set(unsigned int comp_id, const char *comp_name)
{
	static char buf[16348];
	size_t i;
	struct structure_component *sc;
	struct structure_keyval *key_val;

	/* FIXME convert ':' to '/' in name, cf. simsetup.l (see above)
	 */
	assert(strlen(comp_name) < sizeof(buf) - 1);
	for (i = 0; i < strlen(comp_name); i++) {
		if (comp_name[i] == ':') {
			buf[i] = '/';
		} else {
			buf[i] = comp_name[i];
		}
	}
	buf[i] = '\0';
	/* end FIXME */

	sc = simsetup_for_component(buf);
	if (sc == NULL) {
		return;
	}

	
	list_Foreach(&sc->simsetups, key_val) {
		/* FIXME hm... should page be handled specially? */
		if (strcmp(key_val->name, "page") == 0) {
			continue;
		}
		/* FIXME type string? */
		system_comp_generic_set(comp_id, 
					"string",
					key_val->name, 
					key_val->value);
	}
}

static void
glue_vhdl_proc_set(
	void *_cpssp,
	const char *proc,
	const char *param, 
	union fauhdli_value value
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(cpssp->num_params < (sizeof(cpssp->params) 
					/ sizeof(cpssp->params[0])));
	cpssp->params[cpssp->num_params].name = param;
	cpssp->params[cpssp->num_params].arg = value;
	cpssp->num_params++;
}


static void
glue_vhdl_proc_call(void *_cpssp, const char *proc)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	if (strcmp(proc, "send_string") == 0) {
		union fauhdli_value dest_driver;
		union fauhdli_value s_array;
		union fauhdli_value s_lb;
		union fauhdli_value s_ub;

		glue_vhdl_get_arg(cpssp, "dest__driver__", &dest_driver);
		glue_vhdl_get_arg(cpssp, "s", &s_array);
		glue_vhdl_get_arg(cpssp, "s_lb_0", &s_lb);
		glue_vhdl_get_arg(cpssp, "s_rb_0", &s_ub);

		glue_vhdl_send_string(dest_driver, s_array, s_lb, s_ub);

	} else if (strcmp(proc, "terminate") == 0) {
		glue_vhdl_terminate();

	} else if (strcmp(proc, "debug") == 0) {
		glue_vhdl_toggle_debug();

	} else if ((strcmp(proc, "shortcut_bool_out") == 0)
		|| (strcmp(proc, "shortcut_int_out")  == 0)) {
		union fauhdli_value s_driver;
		union fauhdli_value comp_array;
		union fauhdli_value comp_lb;
		union fauhdli_value comp_ub;
		union fauhdli_value comp_port_array;
		union fauhdli_value comp_port_lb;
		union fauhdli_value comp_port_ub;

		glue_vhdl_get_arg(cpssp, "s__driver__", &s_driver);
		glue_vhdl_get_arg(cpssp, "comp", &comp_array);
		glue_vhdl_get_arg(cpssp, "comp_lb_0", &comp_lb);
		glue_vhdl_get_arg(cpssp, "comp_rb_0", &comp_ub);
		glue_vhdl_get_arg(cpssp, "comp_port", &comp_port_array);
		glue_vhdl_get_arg(cpssp, "comp_port_lb_0", &comp_port_lb);
		glue_vhdl_get_arg(cpssp, "comp_port_rb_0", &comp_port_ub);

		glue_vhdl_shorcut_drv_out(
				s_driver,
				comp_array,
				comp_lb,
				comp_ub,
				comp_port_array,
				comp_port_lb,
				comp_port_ub);
	} else {
		/* not implemented */
		faum_log(FAUM_LOG_FATAL, "glue-vhdl", __func__,
			"Trying to call unimplemented foreign procedure "
			"%s.\n", proc);
		assert(0);
	}

	cpssp->num_params = 0;
}


static void
glue_vhdl_set(
	void *_cpssp, 
	unsigned int sig_id, 
	union fauhdli_value data,
	void *drv
)
{
	void *_sig;
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;

	assert(sig_id < GLUE_VHDL_MAX_SIGS);

	/* FIXME don't abuse system_sig_get */
	_sig = system_sig_get(sig_id);
	assert(_sig != NULL);
	
	switch (cpssp->signals[sig_id].type) {
	case SIG_GEN_BOOLEAN:
		sig_boolean_set(
			(struct sig_boolean *)_sig, 
			drv, 
			(unsigned int)data.univ_int);
		break;

	case SIG_GEN_INTEGER:
		sig_integer_set(
			(struct sig_integer *)_sig,
			drv,
			(int)data.univ_int);
		break;

	case SIG_GEN_STD_LOGIC:
		sig_std_logic_set(
			(struct sig_std_logic *)_sig,
			drv,
			(unsigned int)data.univ_int);
		break;

	case SIG_GEN_BOOLEAN_OR:
	case SIG_GEN_CS:
	case SIG_GEN_FAULT:
	case SIG_GEN_I2C_BUS:
	case SIG_GEN_ICC_BUS:
	case SIG_GEN_MANAGE:
	case SIG_GEN_MATCH:
	case SIG_GEN_STRING:
	case SIG_GEN_SERIAL:
	case SIG_GEN_VIDEO:
	case SIG_GEN_OPT_RGB:
	case SIG_GEN_USB_BUS:
	case SIG_GEN_USB_BUS_MAIN:
	case SIG_GEN_ETH:
	case SIG_GEN_FLOPPY:
	case SIG_GEN_MEM_BUS:
	case SIG_GEN_MEM_BUS_MAIN:
	case SIG_GEN_PCI_BUS:
	case SIG_GEN_PCI_BUS_IDSEL:
	case SIG_GEN_PCI_BUS_MAIN:
	case SIG_GEN_PARALLEL:
	case SIG_GEN_ISA_BUS:
	case SIG_GEN_ISA_BUS_DMA:
	case SIG_GEN_ISA_BUS_MAIN:
	case SIG_GEN_IDE_BUS:
	case SIG_GEN_HOST_BUS:
	case SIG_GEN_HOST_BUS_MAIN:
	case SIG_GEN_KEYBOARD:
	case SIG_GEN_AGP_BUS:
	case SIG_GEN_AGP_BUS_MAIN:
	case SIG_GEN_PS2:
	case SIG_GEN_PS2_MAIN:
	case SIG_GEN_SCSI_BUS:
	case SIG_GEN_SHUGART_BUS:
	case SIG_GEN_SOUND:
	case SIG_GEN_TELEPHONE:
	case SIG_GEN_VGA:
	case SIG_GEN_POWER_BOARD:
	case SIG_GEN_POWER_BOARD_AT:
	case SIG_GEN_POWER_DEVICE:
	case SIG_GEN_DIO48:
	case SIG_GEN_DIO24:
	case SIG_GEN_MAGNETO_OPTICAL:
		/* TODO */
		assert(0);
		break;
	}
}

static void
glue_vhdl_connect_out(
	void *_cpssp, 
	unsigned int sig_id,
	union fauhdli_value init,
	void *drv
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	void *_sig;
	assert(sig_id < GLUE_VHDL_MAX_SIGS);
	_sig = system_sig_get(sig_id);

	switch (cpssp->signals[sig_id].type) {
	case SIG_GEN_BOOLEAN:
		sig_boolean_connect_out((struct sig_boolean *)_sig,
					drv,
					(unsigned int)init.univ_int);
		break;

	case SIG_GEN_INTEGER:
		sig_integer_connect_out((struct sig_integer *)_sig,
					drv,
					(int)init.univ_int);
		break;

	case SIG_GEN_STRING:
		/* don't need to connect_out to string */
		break;

	case SIG_GEN_STD_LOGIC:
		sig_std_logic_connect_out((struct sig_std_logic *)_sig,
					drv,
					(unsigned int)init.univ_int);
		break;

	case SIG_GEN_BOOLEAN_OR:
	case SIG_GEN_CS:
	case SIG_GEN_FAULT:
	case SIG_GEN_FLOPPY:
	case SIG_GEN_I2C_BUS:
	case SIG_GEN_ICC_BUS:
	case SIG_GEN_MANAGE:
	case SIG_GEN_MATCH:
	case SIG_GEN_SERIAL:
	case SIG_GEN_VIDEO:
	case SIG_GEN_OPT_RGB:
	case SIG_GEN_USB_BUS:
	case SIG_GEN_USB_BUS_MAIN:
	case SIG_GEN_ETH:
	case SIG_GEN_MEM_BUS:
	case SIG_GEN_MEM_BUS_MAIN:
	case SIG_GEN_PCI_BUS:
	case SIG_GEN_PCI_BUS_IDSEL:
	case SIG_GEN_PCI_BUS_MAIN:
	case SIG_GEN_PARALLEL:
	case SIG_GEN_ISA_BUS:
	case SIG_GEN_ISA_BUS_DMA:
	case SIG_GEN_ISA_BUS_MAIN:
	case SIG_GEN_IDE_BUS:
	case SIG_GEN_HOST_BUS:
	case SIG_GEN_HOST_BUS_MAIN:
	case SIG_GEN_KEYBOARD:
	case SIG_GEN_AGP_BUS:
	case SIG_GEN_AGP_BUS_MAIN:
	case SIG_GEN_PS2:
	case SIG_GEN_PS2_MAIN:
	case SIG_GEN_SCSI_BUS:
	case SIG_GEN_SHUGART_BUS:
	case SIG_GEN_SOUND:
	case SIG_GEN_TELEPHONE:
	case SIG_GEN_VGA:
	case SIG_GEN_POWER_BOARD:
	case SIG_GEN_POWER_BOARD_AT:
	case SIG_GEN_POWER_DEVICE:
	case SIG_GEN_DIO48:
	case SIG_GEN_DIO24:
	case SIG_GEN_MAGNETO_OPTICAL:

		assert(0);
		break;
	}
}

static void
glue_vhdl_connect_in(
	void *_cpssp,
	unsigned int sig_id,
	void *drv
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	void *_sig;

	assert(sig_id < GLUE_VHDL_MAX_SIGS);

	_sig = system_sig_get(sig_id);
	assert(_sig != NULL);

	cpssp->signals[sig_id]._drv = drv;
	
	switch (cpssp->signals[sig_id].type) {
	case SIG_GEN_BOOLEAN: {
		static const struct sig_boolean_funcs f = {
			.set = glue_vhdl_data_get_boolean
		};

		sig_boolean_connect_in(
			(struct sig_boolean *)_sig,
			&cpssp->signals[sig_id],
			&f);
		break;
	    }
	case SIG_GEN_INTEGER: {
		static const struct sig_integer_funcs f = {
			.set = glue_vhdl_data_get_integer
		};

		sig_integer_connect_in(
			(struct sig_integer *)_sig,
			&cpssp->signals[sig_id],
			&f);
		break;
	    }

	case SIG_GEN_STD_LOGIC: {
		static const struct sig_std_logic_funcs f = {
			.set = glue_vhdl_data_get_std_logic
		};

		sig_std_logic_connect_in(
			(struct sig_std_logic *)_sig,
			&cpssp->signals[sig_id],
			&f);
		break;
	    }
	case SIG_GEN_BOOLEAN_OR:
	case SIG_GEN_CS:
	case SIG_GEN_FAULT:
	case SIG_GEN_FLOPPY:
	case SIG_GEN_I2C_BUS:
	case SIG_GEN_ICC_BUS:
	case SIG_GEN_MANAGE:
	case SIG_GEN_MATCH:
	case SIG_GEN_STRING:
	case SIG_GEN_SERIAL:
	case SIG_GEN_VIDEO:
	case SIG_GEN_OPT_RGB:
	case SIG_GEN_USB_BUS:
	case SIG_GEN_USB_BUS_MAIN:
	case SIG_GEN_ETH:
	case SIG_GEN_MEM_BUS:
	case SIG_GEN_MEM_BUS_MAIN:
	case SIG_GEN_PCI_BUS:
	case SIG_GEN_PCI_BUS_IDSEL:
	case SIG_GEN_PCI_BUS_MAIN:
	case SIG_GEN_PARALLEL:
	case SIG_GEN_ISA_BUS:
	case SIG_GEN_ISA_BUS_DMA:
	case SIG_GEN_ISA_BUS_MAIN:
	case SIG_GEN_IDE_BUS:
	case SIG_GEN_HOST_BUS:
	case SIG_GEN_HOST_BUS_MAIN:
	case SIG_GEN_KEYBOARD:
	case SIG_GEN_AGP_BUS:
	case SIG_GEN_AGP_BUS_MAIN:
	case SIG_GEN_PS2:
	case SIG_GEN_PS2_MAIN:
	case SIG_GEN_SCSI_BUS:
	case SIG_GEN_SHUGART_BUS:
	case SIG_GEN_SOUND:
	case SIG_GEN_TELEPHONE:
	case SIG_GEN_VGA:
	case SIG_GEN_POWER_BOARD:
	case SIG_GEN_POWER_BOARD_AT:
	case SIG_GEN_POWER_DEVICE:
	case SIG_GEN_DIO48:
	case SIG_GEN_DIO24:
	case SIG_GEN_MAGNETO_OPTICAL:
		/* TODO */
		assert(0);
		break;
	}
}

static unsigned int
glue_vhdl_create_signal(
	void *_cpssp, 
	const char *type,
	const char *name
)
{
	unsigned int sig_id;
	enum sig_gen_type t;
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;

	t = glue_vhdl_determine_type(type);
	/* FIXME need to special case cstring, due to conflicting type
	 *       string */
	/* FIXME need to special case std_(u)logic */
	switch (t) {
	case SIG_GEN_STRING:
		sig_id = system_sig_create("string", name);
		break;

	case SIG_GEN_STD_LOGIC:
		sig_id = system_sig_create("std_logic", name);
		break;

	default:
		sig_id = system_sig_create(type, name);
		break;
	}
	assert(sig_id < GLUE_VHDL_MAX_SIGS);
	assert(! cpssp->signals[sig_id].used);

	cpssp->signals[sig_id].type = t;
	cpssp->signals[sig_id].self = cpssp;
	cpssp->signals[sig_id]._drv = NULL;
	cpssp->signals[sig_id].used = true;

	return sig_id;
}

static unsigned int
glue_vhdl_comp_create(
	void *_cpssp,
	const char *type, 
	const char *name
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl*)_cpssp;
	int comp_id;
	unsigned int node_id;
	unsigned int page_id;
	int manage_id;

	page_id = glue_vhdl_get_page(cpssp, name);
	node_id = glue_vhdl_get_node(cpssp, name);

#if 0
	faum_log(FAUM_LOG_DEBUG, "glue-vhdl", __func__,
		"creating component %s of type %s\n", name, type);
#endif
	comp_id = system_comp_create(type, name, node_id, page_id);

	/* handle simulation.setup specifiers for component. */
	glue_vhdl_simsetup_set(comp_id, name);

	assert(0 <= comp_id
	    && comp_id < sizeof(cpssp->comp_in_use) 
	    			/ sizeof(cpssp->comp_in_use[0]));

	cpssp->comp_in_use[comp_id] = true;

	manage_id = system_sig_create("manage", name);
	assert(0 <= manage_id);

	system_comp_port_connect(comp_id, "manage", manage_id);

	return comp_id;
}

static unsigned int
glue_vhdl_arch_create(
	void *_cpssp, 
	const char *type, 
	const char *name
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	int arch_id;
	unsigned int node_id;
	unsigned int page_id;
	int manage_id;

	page_id = glue_vhdl_get_page(cpssp, name);
	node_id = glue_vhdl_get_node(cpssp, name);

	arch_id = system_arch_create(type, name, node_id, page_id);

	assert(0 <= arch_id
	    && arch_id < sizeof(cpssp->arch_in_use) 
	    		/ sizeof(cpssp->arch_in_use[0]));

	cpssp->arch_in_use[arch_id] = true;

	manage_id = system_sig_create("manage", name);
	assert(0 <= manage_id);

	system_comp_port_connect(arch_id, "manage", manage_id);

	return arch_id;
}

static void
glue_vhdl_comp_init(void *_cpssp, unsigned int comp)
{
	system_comp_init(comp);
}

static void
glue_vhdl_arch_init(void *_cpssp, unsigned int arch)
{
	system_arch_init(arch);
}


static void
glue_vhdl_comp_port_connect(
	void *_cpssp,
	unsigned int comp,
	const char *port,
	unsigned int sig
)
{
	system_comp_port_connect(comp, port, sig);
}

static void
glue_vhdl_arch_port_connect(
	void *_cpssp,
	unsigned int arch,
	const char *port,
	unsigned int sig
)
{
	system_arch_port_connect(arch, port, sig);
}


static void
glue_vhdl_comp_generic_nonarray_set(
	void *_cpssp,
	unsigned int comp,
	const char *generic,
	union fauhdli_value val,
	const char *type
)
{
	char buf[30];
	int ret;

	/* TODO float generics, composites */
	ret = snprintf(buf, sizeof(buf), "%" PRIi64, val.univ_int);
	assert(ret < sizeof(buf));

	system_comp_generic_set(comp, type, generic, buf);
}

static void
glue_vhdl_arch_generic_nonarray_set(
	void *_cpssp,
	unsigned int arch,
	const char *generic,
	union fauhdli_value val,
	const char *type
)
{
	char buf[30];
	int ret;

	/* TODO float generics, composites */
	ret = snprintf(buf, sizeof(buf), "%" PRIi64, val.univ_int);
	assert(ret < sizeof(buf));

	system_arch_generic_set(arch, type, generic, buf);
}


static void
glue_vhdl_comp_generic_array_set(
	void *_cpssp,
	unsigned int comp,
	const char *generic,
	const char *element_type,
	union fauhdli_value base,
	universal_integer array_size
)
{
	if (strcmp(element_type, "character") == 0) {
		const char *value;

		assert(0 <= array_size);
		
		value = glue_vhdl_flatten_string(base, array_size);

#if 0 /* debug output ... */
		faum_log(FAUM_LOG_DEBUG, "glue-vhdl", __func__,
			"setting generic %s of comp %d to %s\n",
			generic, comp, value);
#endif
		system_comp_generic_set(comp, "string", generic, value);
	} else {
		/* TODO ... */
		assert(0);
	}
}

static void
glue_vhdl_arch_generic_array_set(
	void *_cpssp,
	unsigned int arch,
	const char *generic,
	const char *element_type,
	union fauhdli_value base,
	universal_integer array_size
)
{
	if (strcmp(element_type, "character") == 0) {
		const char *value;

		assert(0 <= array_size);
		
		value = glue_vhdl_flatten_string(base, array_size);

#if 1 /* debug output ... */
		faum_log(FAUM_LOG_DEBUG, "glue-vhdl", __func__,
			"setting generic %s of arch %d to %s\n",
			generic, arch, value);
#endif

		system_arch_generic_set(arch, "string", generic, value);
	} else {
		/* TODO ... */
		assert(0);
	}
}



void *
glue_vhdl_create(struct glue_vhdl_cb *callbacks)
{
	struct glue_vhdl *cpssp;

	/* FIXME */
	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	/* set callbacks */
	assert(callbacks != NULL);
	callbacks->comp_create = glue_vhdl_comp_create;
	callbacks->arch_create = glue_vhdl_arch_create;
	callbacks->signal_create = glue_vhdl_create_signal;
	callbacks->arch_init = glue_vhdl_arch_init;
	callbacks->drv_set = glue_vhdl_set;
	callbacks->proc_set = glue_vhdl_proc_set;
	callbacks->proc_call = glue_vhdl_proc_call;
	callbacks->connect_out = glue_vhdl_connect_out;
	callbacks->connect_in = glue_vhdl_connect_in;
	callbacks->comp_init = glue_vhdl_comp_init;
	callbacks->comp_port_connect = glue_vhdl_comp_port_connect;
	callbacks->arch_port_connect = glue_vhdl_arch_port_connect;
	callbacks->comp_generic_nonarray_set =
				glue_vhdl_comp_generic_nonarray_set;
	callbacks->arch_generic_nonarray_set =
				glue_vhdl_arch_generic_nonarray_set;
	callbacks->comp_generic_array_set = glue_vhdl_comp_generic_array_set;
	callbacks->arch_generic_array_set = glue_vhdl_arch_generic_array_set;

	return cpssp;
}

void
glue_vhdl_init(void *_cpssp, void *fauhdli)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;

	memset(cpssp->signals, 0, sizeof(cpssp->signals));
	memset(cpssp->comp_in_use, 0, sizeof(cpssp->comp_in_use));
	memset(cpssp->arch_in_use, 0, sizeof(cpssp->arch_in_use));

	cpssp->interpreter = NULL;
	cpssp->num_params = 0;
	cpssp->page_cache_count = 0;
	cpssp->node_cache_count = 0;
	cpssp->fauhdli = fauhdli;
}

void
glue_vhdl_destroy(void *_cpssp)
{
	unsigned int i;
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;

	/* tear down components */
	for (i = 0; i < GLUE_VHDL_MAX_COMP; i++) {
		if (cpssp->comp_in_use[i]) {
			system_comp_exit(i);
		}
	}

	for (i = 0; i < GLUE_VHDL_MAX_COMP; i++) {
		if (cpssp->comp_in_use[i]) {
			/* Should destroy "manage" signal, too! FIXME */
			system_comp_destroy(i);
		}
	}

	for (i = 0; i < GLUE_VHDL_MAX_ARCH; i++) {
		if (cpssp->arch_in_use[i]) {
			/* Should destroy "manage" signal, too! FIXME */
			system_arch_destroy(i);
		}
	}

	/* tear down signals */
	for (i = 0; i < GLUE_VHDL_MAX_SIGS; i++) {
		if (cpssp->signals[i].used) {
			system_sig_destroy(i);	
		}
	}
	/* tear down pages */
	for (i = 0; i < cpssp->page_cache_count; i++) {
		system_page_destroy(cpssp->page_cache[i].page_id);
	}
	/* tear down nodes */
	for (i = 0; i < cpssp->node_cache_count; i++) {
		dist_node_destroy(cpssp->node_cache[i].node_id);
	}

	free(cpssp);
}
