/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "amperemeter.h"

#define COMP_(x)	amperemeter_ ## x

#define FREQ	128

struct cpssp {
	struct sig_std_logic *port_output;
	unsigned int state_current;
	struct sig_integer *port_current;

	int current;
	int count;

	unsigned long long time;
};

static void
COMP_(input_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_current = SIG_mA(val);

	sig_std_logic_set(cpssp->port_output, cpssp, val);
}

static void
COMP_(event)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->current += cpssp->state_current;
	cpssp->count++;

	if (cpssp->count == FREQ) {
		sig_integer_set(cpssp->port_current, cpssp,
				cpssp->current / cpssp->count);

		cpssp->current = 0;
		cpssp->count = 0;
	}

	cpssp->time += TIME_HZ / FREQ;
	time_call_at(cpssp->time, COMP_(event), cpssp);
}

void *
COMP_(create)(
	const char *name,
        struct sig_manage *port_manage,
	struct sig_std_logic *port_input,
	struct sig_std_logic *port_output,
	struct sig_integer *port_current
)
{
	static const struct sig_std_logic_funcs input_funcs = {
		.std_logic_set = COMP_(input_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->current = 0;
	cpssp->count = 0;

	/* Out */
	cpssp->port_output = port_output;
	sig_std_logic_connect_out(port_output, cpssp, SIG_STD_LOGIC_0);

	cpssp->port_current = port_current;

	/* In */
	sig_std_logic_connect_in(port_input, cpssp, &input_funcs);

	/* Time */
	cpssp->time = 0;
	time_call_at(cpssp->time, COMP_(event), cpssp);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
