/*
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "glue-gui-gtk.h"
#include "glue-gui-gtk-switch.h"
#include "umutil.h"
#include "glue.h"

#include "elevator_gui_gtk.h"

#define COMP "elevator"

struct cpssp {
	const char *comp_name;
	const char *comp_type;

	GtkWidget *widget_power_switch;
	struct sig_boolean *port_mech_power_switch;

	GdkPixbuf *pb_cabin;
	GdkPixbuf *pb_door_l;
	GdkPixbuf *pb_door_r;

	struct elevator_led {
		GtkWidget *digit;
		GdkPixbuf *on;
		GdkPixbuf *off;
	} leds[16];

	struct elevator_button {
		struct cpssp *parent;
		struct sig_boolean *sig;
	} buttons[7];

	struct elevator_door {
		int pos;
		int offset;
		struct cpssp *parent;
		GdkPixbuf *pixbuf;
		GtkWidget *door;
	} doors[3];

	int cabin_pos;
};

/* helper functions */

static void
elevator_update_doors(struct cpssp * cpssp)
{
	int i;

	for (i = 0; i < 3; i++) {
		gdk_pixbuf_composite(cpssp->pb_cabin, cpssp->doors[i].pixbuf, 0, 0, 49, 121,
				0.0, -1 * (cpssp->doors[i].offset + cpssp->cabin_pos),
				1.0, 1.0, GDK_INTERP_NEAREST, 255);

		gdk_pixbuf_composite(cpssp->pb_door_l, cpssp->doors[i].pixbuf, 0, 0, 25 - cpssp->doors[i].pos, 121,
				0.0, 0.0, 1.0, 1.0, GDK_INTERP_NEAREST, 255);
		gdk_pixbuf_composite(cpssp->pb_door_r, cpssp->doors[i].pixbuf,
				24 + cpssp->doors[i].pos, 0, 25 - cpssp->doors[i].pos, 121,
				0.0, 0.0, 1.0, 1.0, GDK_INTERP_NEAREST, 255);
		gtk_image_set_from_pixbuf(GTK_IMAGE(cpssp->doors[i].door), cpssp->doors[i].pixbuf);
	}
	gui_gtk_flush();
}

/*
 * Simulator Callbacks
 */

static void
elevator_mech_power_switch_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	gui_gtk_switch_set(GUI_GTK_SWITCH(cpssp->widget_power_switch),
			(gboolean) val);
	gui_gtk_flush();
}

static void
elevator_led_set(void *_cpssp, unsigned int val)
{
	struct elevator_led *cpssp = (struct elevator_led *) _cpssp;

	if (val) {
		gtk_image_set_from_pixbuf(GTK_IMAGE(cpssp->digit), cpssp->on);
	} else {
		gtk_image_set_from_pixbuf(GTK_IMAGE(cpssp->digit), cpssp->off);
	}
	gui_gtk_flush();
}

static void
elevator_door_set(void *_cpssp, int val)
{
	int new_pos;
	struct elevator_door * cpssp = (struct elevator_door *) _cpssp;

	new_pos = val / 4;
	if (new_pos != cpssp->pos) {
		cpssp->pos = new_pos;
		elevator_update_doors(cpssp->parent);
	}
}

static void
elevator_cabin_set(void *_cpssp, int val)
{
	int new_pos;
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	new_pos = (val * 338) / 200;
	if (new_pos != cpssp->cabin_pos) {
		cpssp->cabin_pos = new_pos;
		elevator_update_doors(cpssp);
	}
}

/*
 * GUI Callbacks
 */

static void
elevator_button_pressed_event(GtkButton *button, gpointer user_data)
{
	struct elevator_button * cpssp = (struct elevator_button *) user_data;

	sig_boolean_set(cpssp->sig, cpssp->parent, 1);
}

static void
elevator_button_released_event(GtkButton *button, gpointer user_data)
{
	struct elevator_button * cpssp = (struct elevator_button *) user_data;

	sig_boolean_set(cpssp->sig, cpssp->parent, 0);
}

static void
elevator_mech_power_switched_on_event(GtkWidget *w, gpointer _cpssp)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	sig_boolean_set(cpssp->port_mech_power_switch, cpssp, 1);
}

static void
elevator_mech_power_switched_off_event(GtkWidget *w, gpointer _cpssp)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	sig_boolean_set(cpssp->port_mech_power_switch, cpssp, 0);
}

/*
 * GUI interface
 */
void *
elevator_gui_gtk_create(
	unsigned int page,
	const char *name,
	struct sig_manage *port_manage,
	struct sig_boolean *port_mech_power_switch,
	struct sig_dio48_conn *port_dio48,
	struct sig_boolean *port_opt_led_a,
	struct sig_boolean *port_opt_led_b,
	struct sig_boolean *port_opt_led_c,
	struct sig_boolean *port_opt_led_d,
	struct sig_boolean *port_opt_led_e,
	struct sig_boolean *port_opt_led_f,
	struct sig_boolean *port_opt_led_g,
	struct sig_boolean *port_opt_led_arrow_down,
	struct sig_boolean *port_opt_led_arrow_up,
	struct sig_boolean *port_opt_led_panel_0,
	struct sig_boolean *port_opt_led_panel_1,
	struct sig_boolean *port_opt_led_panel_2,
	struct sig_boolean *port_opt_led_0_up,
	struct sig_boolean *port_opt_led_1_down,
	struct sig_boolean *port_opt_led_1_up,
	struct sig_boolean *port_opt_led_2_down,
	struct sig_boolean *port_mech_button_panel_0,
	struct sig_boolean *port_mech_button_panel_1,
	struct sig_boolean *port_mech_button_panel_2,
	struct sig_boolean *port_mech_button_0_up,
	struct sig_boolean *port_mech_button_1_down,
	struct sig_boolean *port_mech_button_1_up,
	struct sig_boolean *port_mech_button_2_down,
	struct sig_integer *port_mech_door_0,
	struct sig_integer *port_mech_door_1,
	struct sig_integer *port_mech_door_2,
	struct sig_integer *port_mech_cabin
)
{
	static struct sig_boolean_funcs led_funcs = {
		.set = elevator_led_set
	};
	static struct sig_integer_funcs door_funcs = {
		.set = elevator_door_set
	};
	static struct sig_integer_funcs cabin_funcs = {
		.set = elevator_cabin_set
	};
	static struct sig_boolean_funcs mech_power_switch_funcs = {
		.set = elevator_mech_power_switch_set,
	};
	GtkButton *b_elevator_2down;
	GtkButton *b_elevator_1up;
	GtkButton *b_elevator_1down;
	GtkButton *b_elevator_0up;
	GtkButton *b_panel_2;
	GtkButton *b_panel_1;
	GtkButton *b_panel_0;
	GdkPixbuf *background;
	GdkPixbuf *wall;
	GdkPixbuf *elevator;
	GdkPixbuf *elevator_up_off;
	GdkPixbuf *elevator_up_on;
	GdkPixbuf *elevator_down_off;
	GdkPixbuf *elevator_down_on;
	GdkPixbuf *digit_h_off;
	GdkPixbuf *digit_v_off;
	GdkPixbuf *digit_h_on;
	GdkPixbuf *digit_v_on;
	GdkPixbuf *arrow_up_off;
	GdkPixbuf *arrow_up_on;
	GdkPixbuf *arrow_down_off;
	GdkPixbuf *arrow_down_on;

	GtkWidget *i_back;
	GtkWidget *i_wall_2;
	GtkWidget *i_wall_1;
	GtkWidget *i_wall_0;
	GtkWidget *i_elevator_2;
	GtkWidget *i_elevator_1;
	GtkWidget *i_elevator_0;

	GtkWidget *vbox;
	GtkFixed *fixed;
	GtkWidget *hbox;

	const char *path;
	struct cpssp *cpssp;
	int i;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->comp_name = name;
	cpssp->comp_type = COMP;

	vbox = gtk_vbox_new(FALSE, 0);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);

	cpssp->widget_power_switch = gui_gtk_switch_new();
	g_signal_connect(G_OBJECT(cpssp->widget_power_switch), "switched-on",
			G_CALLBACK(elevator_mech_power_switched_on_event),
			cpssp);
	g_signal_connect(G_OBJECT(cpssp->widget_power_switch), "switched-off",
			G_CALLBACK(elevator_mech_power_switched_off_event),
			cpssp);
	gtk_widget_show(cpssp->widget_power_switch);
	gtk_box_pack_start(GTK_BOX(hbox), cpssp->widget_power_switch, FALSE, FALSE, 2);

	gtk_widget_show_all(hbox);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);

	fixed = GTK_FIXED(gtk_fixed_new());

	/**************
	 * background *
	 **************/

	path = buildpath(PNGDIR, "aufzug_schwarz.png");
	background = gdk_pixbuf_new_from_file_at_scale(path, 400, 500, FALSE, NULL);

	i_back = gtk_image_new_from_pixbuf(background);

	gtk_widget_show_all(i_back);
	gtk_fixed_put(fixed, i_back, 0, 0);

  	/*********
	 * walls *
	 *********/

	path = buildpath(PNGDIR, "aufzug_gelber_hintergrund.png");
	wall = gdk_pixbuf_new_from_file_at_scale(path, 256, 162, FALSE, NULL);

	i_wall_2 = gtk_image_new_from_pixbuf(wall);
	i_wall_1 = gtk_image_new_from_pixbuf(wall);
	i_wall_0 = gtk_image_new_from_pixbuf(wall);

	gtk_widget_show_all(i_wall_2);
	gtk_fixed_put(fixed, i_wall_2, 0, 0);
	gtk_widget_show_all(i_wall_1);
	gtk_fixed_put(fixed, i_wall_1, 0, 169);
	gtk_widget_show_all(i_wall_0);
	gtk_fixed_put(fixed, i_wall_0, 0, 338);

  	/************************
	 * entrance of elevator *
	 ************************/

	path = buildpath(PNGDIR, "aufzug_aussen.png");
	elevator = gdk_pixbuf_new_from_file(path, NULL);

	i_elevator_2 = gtk_image_new_from_pixbuf(elevator);
	i_elevator_1 = gtk_image_new_from_pixbuf(elevator);
	i_elevator_0 = gtk_image_new_from_pixbuf(elevator);

	gtk_widget_show_all(i_elevator_2);
	gtk_fixed_put(fixed, i_elevator_2, 89, 16);
	gtk_widget_show_all(i_elevator_1);
	gtk_fixed_put(fixed, i_elevator_1, 89, 185);
	gtk_widget_show_all(i_elevator_0);
	gtk_fixed_put(fixed, i_elevator_0, 89, 354);

	/*************************
	 * elevator call buttons *
	 *************************/

	/* create elevator buttons */
	b_elevator_2down = GTK_BUTTON(gtk_button_new());
	gtk_widget_set_size_request(GTK_WIDGET(b_elevator_2down), 28, 28);
  	gtk_container_set_border_width(GTK_CONTAINER(b_elevator_2down), 2);
	GTK_WIDGET_UNSET_FLAGS(b_elevator_2down, GTK_CAN_FOCUS);

	b_elevator_1up = GTK_BUTTON(gtk_button_new());
	gtk_widget_set_size_request(GTK_WIDGET(b_elevator_1up), 28, 28);
  	gtk_container_set_border_width(GTK_CONTAINER(b_elevator_1up), 2);
	GTK_WIDGET_UNSET_FLAGS(b_elevator_1up, GTK_CAN_FOCUS);

	b_elevator_1down = GTK_BUTTON(gtk_button_new());
	gtk_widget_set_size_request(GTK_WIDGET(b_elevator_1down), 28, 28);
  	gtk_container_set_border_width(GTK_CONTAINER(b_elevator_1down), 2);
	GTK_WIDGET_UNSET_FLAGS(b_elevator_1down, GTK_CAN_FOCUS);

	b_elevator_0up = GTK_BUTTON(gtk_button_new());
	gtk_widget_set_size_request(GTK_WIDGET(b_elevator_0up), 28, 28);
  	gtk_container_set_border_width(GTK_CONTAINER(b_elevator_0up), 2);
	GTK_WIDGET_UNSET_FLAGS(b_elevator_0up, GTK_CAN_FOCUS);

	/* loading elevator button elements */
	path = buildpath(PNGDIR, "aufzug_button_rauf_aus.png");
	elevator_up_off = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "aufzug_button_rauf_an.png");
	elevator_up_on = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "aufzug_button_runter_aus.png");
	elevator_down_off = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "aufzug_button_runter_an.png");
	elevator_down_on = gdk_pixbuf_new_from_file(path, NULL);

	/* elevator button images */
	cpssp->leds[12].digit = gtk_image_new_from_pixbuf(elevator_up_off);
	cpssp->leds[12].off = elevator_up_off;
	cpssp->leds[12].on = elevator_up_on;

	cpssp->leds[13].digit = gtk_image_new_from_pixbuf(elevator_down_off);
	cpssp->leds[13].off = elevator_down_off;
	cpssp->leds[13].on = elevator_down_on;

	cpssp->leds[14].digit = gtk_image_new_from_pixbuf(elevator_up_off);
	cpssp->leds[14].off = elevator_up_off;
	cpssp->leds[14].on = elevator_up_on;

	cpssp->leds[15].digit = gtk_image_new_from_pixbuf(elevator_down_off);
	cpssp->leds[15].off = elevator_down_off;
	cpssp->leds[15].on = elevator_down_on;

	/* connect elevator buttons */
	gtk_container_add(GTK_CONTAINER(b_elevator_0up), cpssp->leds[12].digit);
	g_signal_connect(GTK_WIDGET(b_elevator_0up), "pressed", G_CALLBACK(elevator_button_pressed_event), (gpointer *)&cpssp->buttons[3]);
	g_signal_connect(GTK_WIDGET(b_elevator_0up), "released", G_CALLBACK(elevator_button_released_event), (gpointer *)&cpssp->buttons[3]);

	gtk_container_add(GTK_CONTAINER(b_elevator_1down), cpssp->leds[13].digit);
	g_signal_connect(GTK_WIDGET(b_elevator_1down), "pressed", G_CALLBACK(elevator_button_pressed_event), (gpointer *)&cpssp->buttons[4]);
	g_signal_connect(GTK_WIDGET(b_elevator_1down), "released", G_CALLBACK(elevator_button_released_event), (gpointer *)&cpssp->buttons[4]);

	gtk_container_add(GTK_CONTAINER(b_elevator_1up), cpssp->leds[14].digit);
	g_signal_connect(GTK_WIDGET(b_elevator_1up), "pressed", G_CALLBACK(elevator_button_pressed_event), (gpointer *)&cpssp->buttons[5]);
	g_signal_connect(GTK_WIDGET(b_elevator_1up), "released", G_CALLBACK(elevator_button_released_event), (gpointer *)&cpssp->buttons[5]);

	gtk_container_add(GTK_CONTAINER(b_elevator_2down), cpssp->leds[15].digit);
	g_signal_connect(GTK_WIDGET(b_elevator_2down), "pressed", G_CALLBACK(elevator_button_pressed_event), (gpointer *)&cpssp->buttons[6]);
	g_signal_connect(GTK_WIDGET(b_elevator_2down), "released", G_CALLBACK(elevator_button_released_event), (gpointer *)&cpssp->buttons[6]);

	/* placing door button elements */
	gtk_widget_show_all(GTK_WIDGET(b_elevator_2down));
	gtk_fixed_put(fixed, GTK_WIDGET(b_elevator_2down), 170, 84);
	gtk_widget_show_all(GTK_WIDGET(b_elevator_1up));
	gtk_fixed_put(fixed, GTK_WIDGET(b_elevator_1up), 170, 232);
	gtk_widget_show_all(GTK_WIDGET(b_elevator_1down));
	gtk_fixed_put(fixed, GTK_WIDGET(b_elevator_1down), 170, 258);
	gtk_widget_show_all(GTK_WIDGET(b_elevator_0up));
	gtk_fixed_put(fixed, GTK_WIDGET(b_elevator_0up), 170, 416);

	/************************************
	 * external panel 7 segment display *
	 ************************************/

	/* loading panel 7 segment display elements */
	path = buildpath(PNGDIR, "digital_horizontal_aus.png");
	digit_h_off = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "digital_vertikal_aus.png");
	digit_v_off = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "digital_horizontal_an.png");
	digit_h_on = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "digital_vertikal_an.png");
	digit_v_on = gdk_pixbuf_new_from_file(path, NULL);

	/* panel 7 segment display images */
	cpssp->leds[0].digit = gtk_image_new_from_pixbuf(digit_h_off);
	cpssp->leds[0].off = digit_h_off;
	cpssp->leds[0].on = digit_h_on;

 	cpssp->leds[1].digit = gtk_image_new_from_pixbuf(digit_v_off);
	cpssp->leds[1].off = digit_v_off;
	cpssp->leds[1].on = digit_v_on;

 	cpssp->leds[2].digit = gtk_image_new_from_pixbuf(digit_v_off);
	cpssp->leds[2].off = digit_v_off;
	cpssp->leds[2].on = digit_v_on;

 	cpssp->leds[3].digit = gtk_image_new_from_pixbuf(digit_h_off);
	cpssp->leds[3].off = digit_h_off;
	cpssp->leds[3].on = digit_h_on;

 	cpssp->leds[4].digit = gtk_image_new_from_pixbuf(digit_v_off);
	cpssp->leds[4].off = digit_v_off;
	cpssp->leds[4].on = digit_v_on;

 	cpssp->leds[5].digit = gtk_image_new_from_pixbuf(digit_v_off);
	cpssp->leds[5].off = digit_v_off;
	cpssp->leds[5].on = digit_v_on;

	cpssp->leds[6].digit = gtk_image_new_from_pixbuf(digit_h_off);
	cpssp->leds[6].off = digit_h_off;
	cpssp->leds[6].on = digit_h_on;

	/* 7 segment displays */
	gtk_widget_show_all(cpssp->leds[0].digit);
	gtk_fixed_put(fixed, cpssp->leds[0].digit, 310, 26);
	gtk_widget_show_all(cpssp->leds[1].digit);
	gtk_fixed_put(fixed, cpssp->leds[1].digit, 341, 34);
	gtk_widget_show_all(cpssp->leds[2].digit);
	gtk_fixed_put(fixed, cpssp->leds[2].digit, 341, 72);
	gtk_widget_show_all(cpssp->leds[3].digit);
	gtk_fixed_put(fixed, cpssp->leds[3].digit, 310, 102);
	gtk_widget_show_all(cpssp->leds[4].digit);
	gtk_fixed_put(fixed, cpssp->leds[4].digit, 302, 72);
	gtk_widget_show_all(cpssp->leds[5].digit);
	gtk_fixed_put(fixed, cpssp->leds[5].digit, 302, 34);
	gtk_widget_show_all(cpssp->leds[6].digit);
	gtk_fixed_put(fixed, cpssp->leds[6].digit, 310, 64);

	/*************************
	 * external panel arrows *
	 *************************/

	/* loading panel arrow image elements */
	path = buildpath(PNGDIR, "aufzug_rauf_aus.png");
	arrow_up_off = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "aufzug_runter_aus.png");
	arrow_down_off = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "aufzug_rauf_an.png");
	arrow_up_on = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "aufzug_runter_an.png");
	arrow_down_on = gdk_pixbuf_new_from_file(path, NULL);

 	/* panel arrows images */
 	cpssp->leds[7].digit = gtk_image_new_from_pixbuf(arrow_down_off);
	cpssp->leds[7].off = arrow_down_off;
	cpssp->leds[7].on = arrow_down_on;

 	cpssp->leds[8].digit = gtk_image_new_from_pixbuf(arrow_up_off);
 	cpssp->leds[8].off = arrow_up_off;
	cpssp->leds[8].on = arrow_up_on;

  	/* up/down arrows */
	gtk_widget_show_all(cpssp->leds[7].digit);
	gtk_fixed_put(fixed, cpssp->leds[7].digit, 292, 198);
	gtk_widget_show_all(cpssp->leds[8].digit);
	gtk_fixed_put(fixed, cpssp->leds[8].digit, 292, 140);

	/***************************
	 * inner panel of elevator *
	 ***************************/

	/* create panel buttons */
	b_panel_0 = GTK_BUTTON(gtk_button_new());
	gtk_widget_set_size_request(GTK_WIDGET(b_panel_0), 64, 64);
  	gtk_container_set_border_width(GTK_CONTAINER(b_panel_0), 2);
	GTK_WIDGET_UNSET_FLAGS(b_panel_0, GTK_CAN_FOCUS);

	b_panel_1 = GTK_BUTTON(gtk_button_new());
	gtk_widget_set_size_request(GTK_WIDGET(b_panel_1), 64, 64);
  	gtk_container_set_border_width(GTK_CONTAINER(b_panel_1), 2);
	GTK_WIDGET_UNSET_FLAGS(b_panel_1, GTK_CAN_FOCUS);

	b_panel_2 = GTK_BUTTON(gtk_button_new());
	gtk_widget_set_size_request(GTK_WIDGET(b_panel_2), 64, 64);
  	gtk_container_set_border_width(GTK_CONTAINER(b_panel_2), 2);
	GTK_WIDGET_UNSET_FLAGS(b_panel_2, GTK_CAN_FOCUS);

	/* loading panel button elements */
	path = buildpath(PNGDIR, "aufzug_stockwerk_e_an.png");
	cpssp->leds[9].on = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "aufzug_stockwerk_e_aus.png");
	cpssp->leds[9].off = gdk_pixbuf_new_from_file(path, NULL);

	path = buildpath(PNGDIR, "aufzug_stockwerk_1_an.png");
	cpssp->leds[10].on = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "aufzug_stockwerk_1_aus.png");
	cpssp->leds[10].off = gdk_pixbuf_new_from_file(path, NULL);

	path = buildpath(PNGDIR, "aufzug_stockwerk_2_aus.png");
	cpssp->leds[11].off = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "aufzug_stockwerk_2_an.png");
	cpssp->leds[11].on = gdk_pixbuf_new_from_file(path, NULL);

	/* panel button images */
	cpssp->leds[9].digit = gtk_image_new_from_pixbuf(cpssp->leds[9].off);
	cpssp->leds[10].digit = gtk_image_new_from_pixbuf(cpssp->leds[10].off);
	cpssp->leds[11].digit = gtk_image_new_from_pixbuf(cpssp->leds[11].off);

	/* connect panel buttons */
	gtk_container_add(GTK_CONTAINER(b_panel_0), cpssp->leds[9].digit);
	g_signal_connect(GTK_WIDGET(b_panel_0), "pressed", G_CALLBACK(elevator_button_pressed_event), (gpointer *)&cpssp->buttons[0]);
	g_signal_connect(GTK_WIDGET(b_panel_0), "released", G_CALLBACK(elevator_button_released_event), (gpointer *)&cpssp->buttons[0]);

	gtk_container_add(GTK_CONTAINER(b_panel_1), cpssp->leds[10].digit);
	g_signal_connect(GTK_WIDGET(b_panel_1), "pressed", G_CALLBACK(elevator_button_pressed_event), (gpointer *)&cpssp->buttons[1]);
	g_signal_connect(GTK_WIDGET(b_panel_1), "released", G_CALLBACK(elevator_button_released_event), (gpointer *)&cpssp->buttons[1]);

	gtk_container_add(GTK_CONTAINER(b_panel_2), cpssp->leds[11].digit);
	g_signal_connect(GTK_WIDGET(b_panel_2), "pressed", G_CALLBACK(elevator_button_pressed_event), (gpointer *)&cpssp->buttons[2]);
	g_signal_connect(GTK_WIDGET(b_panel_2), "released", G_CALLBACK(elevator_button_released_event), (gpointer *)&cpssp->buttons[2]);

  	/* inner buttons */
	gtk_widget_show_all(GTK_WIDGET(b_panel_0));
	gtk_fixed_put(fixed, GTK_WIDGET(b_panel_0), 296, 418);
	gtk_widget_show_all(GTK_WIDGET(b_panel_1));
	gtk_fixed_put(fixed, GTK_WIDGET(b_panel_1), 296, 354);
	gtk_widget_show_all(GTK_WIDGET(b_panel_2));
	gtk_fixed_put(fixed, GTK_WIDGET(b_panel_2), 296, 290);

	/*******************
	 * doors and cabin *
	 *******************/

	/* loading elevator cabin image */
	path = buildpath(PNGDIR, "aufzug_hintergrund.png");
	cpssp->pb_cabin = gdk_pixbuf_new_from_file(path, NULL);

	/* loading door images */
	path = buildpath(PNGDIR, "aufzug_tuer_links.png");
	cpssp->pb_door_l = gdk_pixbuf_new_from_file(path, NULL);
	path = buildpath(PNGDIR, "aufzug_tuer_rechts.png");
	cpssp->pb_door_r = gdk_pixbuf_new_from_file(path, NULL);

	path = buildpath(PNGDIR, "aufzug_schwarz.png");
	cpssp->doors[2].pixbuf = gdk_pixbuf_new_from_file_at_scale(path, 49, 121, FALSE, NULL);
	cpssp->doors[2].offset = 2;

	gdk_pixbuf_composite(cpssp->pb_door_l, cpssp->doors[2].pixbuf, 0, 0, 25, 121,
			0.0, 0.0, 1.0, 1.0, GDK_INTERP_NEAREST, 255);
	gdk_pixbuf_composite(cpssp->pb_door_r, cpssp->doors[2].pixbuf, 24, 0, 25, 121,
			0.0, 0.0, 1.0, 1.0, GDK_INTERP_NEAREST, 255);

	path = buildpath(PNGDIR, "aufzug_schwarz.png");
	cpssp->doors[1].pixbuf = gdk_pixbuf_new_from_file_at_scale(path, 49, 121, FALSE, NULL);
	cpssp->doors[1].offset = 171;

	gdk_pixbuf_composite(cpssp->pb_door_l, cpssp->doors[1].pixbuf, 0, 0, 25, 121,
			0.0, 0.0, 1.0, 1.0, GDK_INTERP_NEAREST, 255);
	gdk_pixbuf_composite(cpssp->pb_door_r, cpssp->doors[1].pixbuf, 24, 0, 25, 121,
			0.0, 0.0, 1.0, 1.0, GDK_INTERP_NEAREST, 255);

	path = buildpath(PNGDIR, "aufzug_schwarz.png");
	cpssp->doors[0].pixbuf = gdk_pixbuf_new_from_file_at_scale(path, 49, 121, FALSE, NULL);
	cpssp->doors[0].offset = 340;

	gdk_pixbuf_composite(cpssp->pb_door_l, cpssp->doors[0].pixbuf, 0, 0, 25, 121,
			0.0, 0.0, 1.0, 1.0, GDK_INTERP_NEAREST, 255);
	gdk_pixbuf_composite(cpssp->pb_door_r, cpssp->doors[0].pixbuf, 24, 0, 25, 121,
			0.0, 0.0, 1.0, 1.0, GDK_INTERP_NEAREST, 255);

	/* elevator door images */
	cpssp->doors[2].door = gtk_image_new_from_pixbuf(cpssp->doors[2].pixbuf);
	cpssp->doors[1].door = gtk_image_new_from_pixbuf(cpssp->doors[1].pixbuf);
	cpssp->doors[0].door = gtk_image_new_from_pixbuf(cpssp->doors[0].pixbuf);

 	/* placing elevator door images */
	gtk_widget_show_all(cpssp->doors[0].door);
	gtk_fixed_put(fixed, cpssp->doors[0].door, 102, 365);

	gtk_widget_show_all(cpssp->doors[1].door);
	gtk_fixed_put(fixed, cpssp->doors[1].door, 102, 196);

	gtk_widget_show_all(cpssp->doors[2].door);
	gtk_fixed_put(fixed, cpssp->doors[2].door, 102, 27);

	gtk_widget_show_all(GTK_WIDGET(fixed));
	gtk_container_add(GTK_CONTAINER(vbox), GTK_WIDGET(fixed));

	gtk_widget_show_all(vbox);

	gui_gtk_comp_add(page, COMP, name,
			GTK_WIDGET(vbox), TRUE, TRUE, NULL);

	cpssp->buttons[0].sig = port_mech_button_panel_0;
	cpssp->buttons[1].sig = port_mech_button_panel_1;
	cpssp->buttons[2].sig = port_mech_button_panel_2;
	cpssp->buttons[3].sig = port_mech_button_0_up;
	cpssp->buttons[4].sig = port_mech_button_1_down;
	cpssp->buttons[5].sig = port_mech_button_1_up;
	cpssp->buttons[6].sig = port_mech_button_2_down;

	for (i = 0; i < 7; i++) {
		cpssp->buttons[i].parent = cpssp;
		sig_boolean_connect_out(cpssp->buttons[i].sig, cpssp, 0);
	}

	sig_boolean_connect_in(port_opt_led_a, &cpssp->leds[0], &led_funcs);
	sig_boolean_connect_in(port_opt_led_b, &cpssp->leds[1], &led_funcs);
	sig_boolean_connect_in(port_opt_led_c, &cpssp->leds[2], &led_funcs);
	sig_boolean_connect_in(port_opt_led_d, &cpssp->leds[3], &led_funcs);
	sig_boolean_connect_in(port_opt_led_e, &cpssp->leds[4], &led_funcs);
	sig_boolean_connect_in(port_opt_led_f, &cpssp->leds[5], &led_funcs);
	sig_boolean_connect_in(port_opt_led_g, &cpssp->leds[6], &led_funcs);
	sig_boolean_connect_in(port_opt_led_arrow_down, &cpssp->leds[7], &led_funcs);
	sig_boolean_connect_in(port_opt_led_arrow_up, &cpssp->leds[8], &led_funcs);
	sig_boolean_connect_in(port_opt_led_panel_0, &cpssp->leds[9], &led_funcs);
	sig_boolean_connect_in(port_opt_led_panel_1, &cpssp->leds[10], &led_funcs);
	sig_boolean_connect_in(port_opt_led_panel_2, &cpssp->leds[11], &led_funcs);
	sig_boolean_connect_in(port_opt_led_0_up, &cpssp->leds[12], &led_funcs);
	sig_boolean_connect_in(port_opt_led_1_down, &cpssp->leds[13], &led_funcs);
	sig_boolean_connect_in(port_opt_led_1_up, &cpssp->leds[14], &led_funcs);
	sig_boolean_connect_in(port_opt_led_2_down, &cpssp->leds[15], &led_funcs);

	for (i = 0; i < 3; i++) {
		cpssp->doors[i].parent = cpssp;
		cpssp->doors[i].pos = 0;
	}
	sig_integer_connect_in(port_mech_door_0, &cpssp->doors[0], &door_funcs);
	sig_integer_connect_in(port_mech_door_1, &cpssp->doors[1], &door_funcs);
	sig_integer_connect_in(port_mech_door_2, &cpssp->doors[2], &door_funcs);

	cpssp->cabin_pos = 0;
	sig_integer_connect_in(port_mech_cabin, cpssp, &cabin_funcs);

	cpssp->port_mech_power_switch = port_mech_power_switch;
	sig_boolean_connect_out(port_mech_power_switch, cpssp, 0);
	sig_boolean_connect_in(port_mech_power_switch, cpssp,
			&mech_power_switch_funcs);

	return cpssp;
}

void
elevator_gui_gtk_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
elevator_gui_gtk_suspend(void *_cpssp, FILE *fComp)
{
	fprintf(stdout, "elevator_gui_gtk_suspend not yet implemented\n");
}

void
elevator_gui_gtk_resume(void *_cpssp, FILE *fComp)
{
	fprintf(stdout, "elevator_gui_gtk_resume not yet implemented\n");
}
