/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>

#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include "config.h"

#include "glue-gui-gtk.h"
#include "glue.h"

#include "lcd_s401m16kr_gui.h"

#ifdef HAVE_GTK
#include "lcd_s401m16kr_gui_gtk.h"
#endif

#define OFFX	5
#define OFFY	5
#define LX	20
#define LY	20
#define LSPACE	16
#define DX	(LX + LSPACE)

#define COMP_(x) lcd_s401m16kr_gui_gtk_ ## x

struct cpssp {
	int state_seg[32];

	GtkWidget *area;
};

static void
COMP_(hor)(
	GtkWidget *widget,
	int state,
	gint x,
	gint y
)
{
	GdkGC *gc = state
			? widget->style->fg_gc[widget->state]
			: widget->style->bg_gc[widget->state];

	gdk_draw_line(widget->window, gc,
			x + 4, y - 2, x + LX - 4, y - 2);
	gdk_draw_line(widget->window, gc,
			x + 3, y - 1, x + LX - 3, y - 1);
	gdk_draw_line(widget->window, gc,
			x + 2, y, x + LX - 2, y);
	gdk_draw_line(widget->window, gc,
			x + 3, y + 1, x + LX - 3, y + 1);
	gdk_draw_line(widget->window, gc,
			x + 4, y + 2, x + LX - 4, y + 2);
}

static void
COMP_(vert)(
	GtkWidget *widget,
	int state,
	gint x,
	gint y
)
{
	GdkGC *gc = state
			? widget->style->fg_gc[widget->state]
			: widget->style->bg_gc[widget->state];

	gdk_draw_line(widget->window, gc,
			x - 2, y + 4, x - 2, y + LY - 4);
	gdk_draw_line(widget->window, gc,
			x - 1, y + 3, x - 1, y + LY - 3);
	gdk_draw_line(widget->window, gc,
			x, y + 2, x, y + LY - 2);
	gdk_draw_line(widget->window, gc,
			x + 1, y + 3, x + 1, y + LY - 3);
	gdk_draw_line(widget->window, gc,
			x + 2, y + 4, x + 2, y + LY - 4);
}

static void
COMP_(dot)(
	GtkWidget *widget,
	int state,
	gint x,
	gint y
)
{
	GdkGC *gc = state
			? widget->style->fg_gc[widget->state]
			: widget->style->bg_gc[widget->state];

	gdk_draw_line(widget->window, gc,
			x - 2, y - 2, x + 2, y - 2);
	gdk_draw_line(widget->window, gc,
			x - 2, y, x + 2, y);
	gdk_draw_line(widget->window, gc,
			x - 2, y + 2, x + 2, y + 2);
}

static void
COMP_(colon)(
	GtkWidget *widget,
	int state,
	gint x,
	gint y
)
{
	COMP_(dot)(widget, state, x, y);
	COMP_(dot)(widget, state, x, y + LY);
}

static const struct {
	void (*func)(GtkWidget *widget, int state, gint x, gint y);
	gint x;
	gint y;
} COMP_(table)[32] = {
	/* 1D */  { COMP_(hor), 0 * DX + 0, LY * 2 },
	/* 1E */  { COMP_(vert), 0 * DX + 0, LY },
	/* 1G */  { COMP_(hor), 0 * DX + 0, LY },
	/* 1F */  { COMP_(vert), 0 * DX + 0, 0 },
	/* DP1 */ { COMP_(dot), 0 * DX + LX + LSPACE / 2, LY * 2 },
	/* 1C */  { COMP_(vert), 0 * DX + LX, LY },
	/* 1B */  { COMP_(vert), 0 * DX + LX, 0 },
	/* 1A */  { COMP_(hor), 0 * DX + 0, 0 },

	/* 2D */  { COMP_(hor), 1 * DX + 0, LY * 2 },
	/* 2E */  { COMP_(vert), 1 * DX + 0, LY },
	/* 2G */  { COMP_(hor), 1 * DX + 0, LY },
	/* 2F */  { COMP_(vert), 1 * DX + 0, 0 },
	/* DP2 */ { COMP_(dot), 1 * DX + LX + LSPACE / 2, LY * 2 },
	/* 2C */  { COMP_(vert), 1 * DX + LX, LY },
	/* 2B */  { COMP_(vert), 1 * DX + LX, 0 },
	/* 2A */  { COMP_(hor), 1 * DX + 0, 0 },

	/* 3D */  { COMP_(hor), 2 * DX + 0, LY * 2 },
	/* 3E */  { COMP_(vert), 2 * DX + 0, LY },
	/* 3G */  { COMP_(hor), 2 * DX + 0, LY },
	/* 3F */  { COMP_(vert), 2 * DX + 0, 0 },
	/* DP3 */ { COMP_(dot), 2 * DX + LX + LSPACE / 2, LY * 2 },
	/* 3C */  { COMP_(vert), 2 * DX + LX, LY },
	/* 3B */  { COMP_(vert), 2 * DX + LX, 0 },
	/* 3A */  { COMP_(hor), 2 * DX + 0, 0 },

	/* 4D */  { COMP_(hor), 3 * DX + 0, LY * 2 },
	/* 4E */  { COMP_(vert), 3 * DX + 0, LY },
	/* 4G */  { COMP_(hor), 3 * DX + 0, LY },
	/* 4F */  { COMP_(vert), 3 * DX + 0, 0 },
	/* COL */ { COMP_(colon), 1 * DX + LX + LSPACE / 2, LY / 2 },
	/* 4C */  { COMP_(vert), 3 * DX + LX, LY },
	/* 4B */  { COMP_(vert), 3 * DX + LX, 0 },
	/* 4A */  { COMP_(hor), 3 * DX + 0, 0 },
};

static int
COMP_(expose)(
	GtkWidget *widget,
	GdkEventExpose *event,
	gpointer _cpssp
)
{
	struct cpssp *cpssp = _cpssp;
	int i;

	if (0 < event->count)
		return FALSE;

	gdk_window_clear_area(cpssp->area->window,
			0, 0,
			cpssp->area->allocation.width,
			cpssp->area->allocation.height);

	for (i = 0; i < 32; i++) {
		(*COMP_(table)[i].func)(cpssp->area,
				cpssp->state_seg[i],
				OFFX + COMP_(table)[i].x,
				OFFY + COMP_(table)[i].y);
	}
	gui_gtk_flush();

	return FALSE;
}

static void
COMP_(segN_set)(struct cpssp *cpssp, unsigned int n, uint8_t col)
{
	cpssp->state_seg[n] = col ? 0 : 1;

	(*COMP_(table)[n].func)(cpssp->area,
			cpssp->state_seg[n],
			OFFX + COMP_(table)[n].x,
			OFFY + COMP_(table)[n].y);
	gui_gtk_flush();
}

#define FUNC(n) \
static void \
COMP_(seg ## n ## _set)( \
	void *_cpssp, \
	unsigned int x, unsigned int y, \
	uint8_t r, uint8_t g, uint8_t b \
) \
{ \
	struct cpssp *cpssp = _cpssp; \
	\
	COMP_(segN_set)(cpssp, n, r); \
}
FUNC(0) FUNC(1) FUNC(2) FUNC(3)
FUNC(4) FUNC(5) FUNC(6) FUNC(7)
FUNC(8) FUNC(9) FUNC(10) FUNC(11)
FUNC(12) FUNC(13) FUNC(14) FUNC(15)
FUNC(16) FUNC(17) FUNC(18) FUNC(19)
FUNC(20) FUNC(21) FUNC(22) FUNC(23)
FUNC(24) FUNC(25) FUNC(26) FUNC(27)
FUNC(28) FUNC(29) FUNC(30) FUNC(31)
#undef FUNC

void *
COMP_(create)(
	unsigned int page,
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_com0,
	struct sig_std_logic *port_com1,
	struct sig_std_logic *port_com2,
	struct sig_std_logic *port_com3,
	struct sig_std_logic *port_D1_E1_G1_F1,
	struct sig_std_logic *port_DP1_C1_B1_A1,
	struct sig_std_logic *port_D2_E2_G2_F2,
	struct sig_std_logic *port_DP2_C2_B2_A2,
	struct sig_std_logic *port_D3_E3_G3_F3,
	struct sig_std_logic *port_DP3_C3_B3_A3,
	struct sig_std_logic *port_D4_E4_G4_F4,
	struct sig_std_logic *port_COL_C4_B4_A4,
	struct sig_opt_rgb *port_D1,
	struct sig_opt_rgb *port_E1,
	struct sig_opt_rgb *port_G1,
	struct sig_opt_rgb *port_F1,
	struct sig_opt_rgb *port_DP1,
	struct sig_opt_rgb *port_C1,
	struct sig_opt_rgb *port_B1,
	struct sig_opt_rgb *port_A1,
	struct sig_opt_rgb *port_D2,
	struct sig_opt_rgb *port_E2,
	struct sig_opt_rgb *port_G2,
	struct sig_opt_rgb *port_F2,
	struct sig_opt_rgb *port_DP2,
	struct sig_opt_rgb *port_C2,
	struct sig_opt_rgb *port_B2,
	struct sig_opt_rgb *port_A2,
	struct sig_opt_rgb *port_D3,
	struct sig_opt_rgb *port_E3,
	struct sig_opt_rgb *port_G3,
	struct sig_opt_rgb *port_F3,
	struct sig_opt_rgb *port_DP3,
	struct sig_opt_rgb *port_C3,
	struct sig_opt_rgb *port_B3,
	struct sig_opt_rgb *port_A3,
	struct sig_opt_rgb *port_D4,
	struct sig_opt_rgb *port_E4,
	struct sig_opt_rgb *port_G4,
	struct sig_opt_rgb *port_F4,
	struct sig_opt_rgb *port_COL,
	struct sig_opt_rgb *port_C4,
	struct sig_opt_rgb *port_B4,
	struct sig_opt_rgb *port_A4
)
{
#define FUNC(n) \
	static const struct sig_opt_rgb_funcs seg ## n ## _funcs = { \
		.pixel_set = COMP_(seg ## n ## _set), \
	}
	FUNC(0); FUNC(1); FUNC(2); FUNC(3);
	FUNC(4); FUNC(5); FUNC(6); FUNC(7);
	FUNC(8); FUNC(9); FUNC(10); FUNC(11);
	FUNC(12); FUNC(13); FUNC(14); FUNC(15);
	FUNC(16); FUNC(17); FUNC(18); FUNC(19);
	FUNC(20); FUNC(21); FUNC(22); FUNC(23);
	FUNC(24); FUNC(25); FUNC(26); FUNC(27);
	FUNC(28); FUNC(29); FUNC(30); FUNC(31);
#undef FUNC
	struct cpssp *cpssp;
	int i;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	for (i = 0; i < 32; i++) {
		cpssp->state_seg[i] = 0;
	}

	sig_opt_rgb_connect(port_D1, cpssp, &seg0_funcs);
	sig_opt_rgb_connect(port_E1, cpssp, &seg1_funcs);
	sig_opt_rgb_connect(port_G1, cpssp, &seg2_funcs);
	sig_opt_rgb_connect(port_F1, cpssp, &seg3_funcs);
	sig_opt_rgb_connect(port_DP1, cpssp, &seg4_funcs);
	sig_opt_rgb_connect(port_C1, cpssp, &seg5_funcs);
	sig_opt_rgb_connect(port_B1, cpssp, &seg6_funcs);
	sig_opt_rgb_connect(port_A1, cpssp, &seg7_funcs);

	sig_opt_rgb_connect(port_D2, cpssp, &seg8_funcs);
	sig_opt_rgb_connect(port_E2, cpssp, &seg9_funcs);
	sig_opt_rgb_connect(port_G2, cpssp, &seg10_funcs);
	sig_opt_rgb_connect(port_F2, cpssp, &seg11_funcs);
	sig_opt_rgb_connect(port_DP2, cpssp, &seg12_funcs);
	sig_opt_rgb_connect(port_C2, cpssp, &seg13_funcs);
	sig_opt_rgb_connect(port_B2, cpssp, &seg14_funcs);
	sig_opt_rgb_connect(port_A2, cpssp, &seg15_funcs);

	sig_opt_rgb_connect(port_D3, cpssp, &seg16_funcs);
	sig_opt_rgb_connect(port_E3, cpssp, &seg17_funcs);
	sig_opt_rgb_connect(port_G3, cpssp, &seg18_funcs);
	sig_opt_rgb_connect(port_F3, cpssp, &seg19_funcs);
	sig_opt_rgb_connect(port_DP3, cpssp, &seg20_funcs);
	sig_opt_rgb_connect(port_C3, cpssp, &seg21_funcs);
	sig_opt_rgb_connect(port_B3, cpssp, &seg22_funcs);
	sig_opt_rgb_connect(port_A3, cpssp, &seg23_funcs);

	sig_opt_rgb_connect(port_D4, cpssp, &seg24_funcs);
	sig_opt_rgb_connect(port_E4, cpssp, &seg25_funcs);
	sig_opt_rgb_connect(port_G4, cpssp, &seg26_funcs);
	sig_opt_rgb_connect(port_F4, cpssp, &seg27_funcs);
	sig_opt_rgb_connect(port_COL, cpssp, &seg28_funcs);
	sig_opt_rgb_connect(port_C4, cpssp, &seg29_funcs);
	sig_opt_rgb_connect(port_B4, cpssp, &seg30_funcs);
	sig_opt_rgb_connect(port_A4, cpssp, &seg31_funcs);

	cpssp->area = gtk_drawing_area_new();
	gtk_widget_set_size_request(cpssp->area, 200, 50);
	g_signal_connect(G_OBJECT(cpssp->area), "expose_event",
			G_CALLBACK(COMP_(expose)), cpssp);
	gtk_widget_show(cpssp->area);

	gui_gtk_comp_add(page, "LCD", name, cpssp->area, TRUE, TRUE, NULL);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
