/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "jit-comp.h"

static struct jit_expr *jit_expr_free_first = NULL;
static struct jit_expr *jit_expr_free_last = NULL;

static struct jit_label *jit_label_free_first = NULL;
static struct jit_label *jit_label_free_last = NULL;

static struct jit_stmt *jit_stmt_free_first = NULL;
static struct jit_stmt *jit_stmt_free_last = NULL;


static struct jit_expr *
jit__expr_alloc(void)
{
	struct jit_expr *expr;

	if (jit_expr_free_first) {
		expr = jit_expr_free_first;

		/* Remove from free-list. */
		if (expr->prev) {
			assert(0);
		} else {
			jit_expr_free_first = expr->next;
		}
		if (expr->next) {
			expr->next->prev = expr->prev;
		} else {
			jit_expr_free_last = expr->prev;
		}
	} else {
		/* Allocate new expr. */
		expr = malloc(sizeof(*expr));
		assert(expr);
	}

	return expr;
}

static void
jit__expr_free(struct jit_expr *expr)
{
	/* Add to free-list. */
	expr->prev = NULL;
	expr->next = jit_expr_free_first;

	if (expr->prev) {
		assert(0);
	} else {
		jit_expr_free_first = expr;
	}
	if (expr->next) {
		expr->next->prev = expr;
	} else {
		jit_expr_free_last = expr;
	}
}

static struct jit_label *
jit__label_alloc(void)
{
	struct jit_label *label;

	if (jit_label_free_first) {
		label = jit_label_free_first;

		/* Remove from free-list. */
		if (label->prev) {
			assert(0);
		} else {
			jit_label_free_first = label->next;
		}
		if (label->next) {
			label->next->prev = label->prev;
		} else {
			jit_label_free_last = label->prev;
		}
	} else {
		/* Allocate new label. */
		label = malloc(sizeof(*label));
		assert(label);
	}

	return label;
}

static void
jit__label_free(struct jit_label *label)
{
	/* Add to free-list. */
	label->prev = NULL;
	label->next = jit_label_free_first;

	if (label->prev) {
		assert(0);
	} else {
		jit_label_free_first = label;
	}
	if (label->next) {
		label->next->prev = label;
	} else {
		jit_label_free_last = label;
	}
}

static struct jit_stmt *
jit__stmt_alloc(void)
{
	struct jit_stmt *stmt;

	if (jit_stmt_free_first) {
		stmt = jit_stmt_free_first;

		/* Remove from free-list. */
		if (stmt->prev) {
			assert(0);
		} else {
			jit_stmt_free_first = stmt->next;
		}
		if (stmt->next) {
			stmt->next->prev = stmt->prev;
		} else {
			jit_stmt_free_last = stmt->prev;
		}
	} else {
		/* Allocate new stmt. */
		stmt = malloc(sizeof(*stmt));
		assert(stmt);
	}

	return stmt;
}

static void
jit__stmt_free(struct jit_stmt *stmt)
{
	/* Add to free-list. */
	stmt->prev = NULL;
	stmt->next = jit_stmt_free_first;

	if (stmt->prev) {
		assert(0);
	} else {
		jit_stmt_free_first = stmt;
	}
	if (stmt->next) {
		stmt->next->prev = stmt;
	} else {
		jit_stmt_free_last = stmt;
	}
}

#if defined(__x86_64__) && ! defined(CONFIG_JIT_BC)
#include "jit-x86_64.c"
#else
#include "jit-bc.c"
#endif

struct jit_expr *
jit_param_alloc(struct jit_func *func, int sign, unsigned long size)
{
	struct jit_expr *param;

	param = jit__expr_alloc();

	param->type = JIT_PARAM;
	param->sign = sign;
	param->size = size;

	param->offset = -1; /* Will be set later. */

	param->prev = func->param_last;
	param->next = NULL;
	if (param->prev) {
		param->prev->next = param;
	} else {
		func->param_first = param;
	}
	func->param_last = param;

	return param;
}

struct jit_expr *
jit_local_alloc(struct jit_func *func, int sign, unsigned long size)
{
	struct jit_expr *local;

	local = jit__expr_alloc();

	local->type = JIT_LOCAL;
	local->sign = sign;
	local->size = size;

	local->offset = -1; /* Will be set later. */

	local->prev = func->local_last;
	local->next = NULL;
	if (local->prev) {
		local->prev->next = local;
	} else {
		func->local_first = local;
	}
	func->local_last = local;

	return local;
}

struct jit_label *
jit_label_alloc(struct jit_func *func)
{
	static int id = 0;
	struct jit_label *label;
	int i;

	label = jit__label_alloc();

	label->id = id++;
	for (i = 0; i < sizeof(label->jmp) / sizeof(label->jmp[0]); i++) {
		label->jmp[i] = NULL;
	}
	label->label = NULL;

	label->prev = func->label_last;
	label->next = NULL;
	if (label->prev) {
		label->prev->next = label;
	} else {
		func->label_first = label;
	}
	if (label->next) {
		assert(0);
	} else {
		func->label_last = label;
	}

	return label;
}

static void
jit__expr_append(struct jit_func *func, struct jit_expr *expr)
{
	expr->prev = func->expr_last;
	expr->next = NULL;
	if (expr->prev) {
		expr->prev->next = expr;
	} else {
		func->expr_first = expr;
	}
	if (expr->next) {
		assert(0);
	} else {
		func->expr_last = expr;
	}

}

struct jit_expr *
jit_sconst(struct jit_func *func, signed long val)
{
	struct jit_expr *expr;

	expr = jit__expr_alloc();
	jit__expr_append(func, expr);

	expr->type = JIT_CONST;

	expr->sign = -1;
	expr->size = sizeof(expr->val);

	expr->val = (unsigned long) val;

	return expr;
}

struct jit_expr *
jit_uconst(struct jit_func *func, unsigned long val)
{
	struct jit_expr *expr;

	expr = jit__expr_alloc();
	jit__expr_append(func, expr);

	expr->type = JIT_CONST;

	expr->sign = 1;
	expr->size = sizeof(expr->val);

	expr->val = val;

	return expr;
}

struct jit_expr *
jit__global(struct jit_func *func, int sign, unsigned long size, void *addr)
{
	struct jit_expr *expr;

	expr = jit__expr_alloc();
	jit__expr_append(func, expr);

	expr->type = JIT_GLOBAL;

	expr->sign = sign;
	expr->size = size;

	expr->addr = addr;

	return expr;
}

struct jit_expr *
jit_star(struct jit_func *func, int sign, unsigned long size, struct jit_expr *addr)
{
	struct jit_expr *expr;

	expr = jit__expr_alloc();
	jit__expr_append(func, expr);

	expr->type = JIT_STAR;

	expr->sign = sign;
	expr->size = size;

	expr->op0 = addr;

	return expr;
}

struct jit_expr *
jit_addr(struct jit_func *func, struct jit_expr *op)
{
	struct jit_expr *expr;

	if (op->type == JIT_STAR) {
		/*
		 * &*addr
		 * ->
		 * addr
		 */
		expr = op->op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_ADDR;

		expr->sign = 0;
		expr->size = 0;

		expr->op0 = op;
	}

	return expr;
}

struct jit_expr *
jit_conv(struct jit_func *func, int sign, unsigned long size, struct jit_expr *op)
{
	struct jit_expr *expr;

	if (op->size == 8) {
		/*
		 * (uint64_t) op
		 * ->
		 * op
		 */
		expr = op;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONV;

		expr->sign = sign;
		expr->size = size;
		expr->op0 = op;
	}

	return expr;
}

struct jit_expr *
jit_not(struct jit_func *func, struct jit_expr *op)
{
	struct jit_expr *expr;

	if (op->type == JIT_CONST) {
		/*
		 * ! const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = ! op->val;

	} else if (op->type == JIT_NOT) {
		/*
		 * ! ! op
		 * ->
		 * op
		 */
		expr = op->op0;

	} else if (op->type == JIT_EQ) {
		/*
		 * ! (op->op0 == op->op1)
		 * ->
		 * op->op0 != op->op1
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_NEQ;
		expr->op0 = op->op0;
		expr->op1 = op->op1;

	} else if (op->type == JIT_NEQ) {
		/*
		 * ! (op->op0 != op->op1)
		 * ->
		 * op->op0 == op->op1
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_EQ;
		expr->op0 = op->op0;
		expr->op1 = op->op1;

	} else if (op->type == JIT_B) {
		/*
		 * ! (op->op0 < op->op1)
		 * ->
		 * op->op0 >= op->op1
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_AE;
		expr->op0 = op->op0;
		expr->op1 = op->op1;

	} else if (op->type == JIT_AE) {
		/*
		 * ! (op->op0 >= op->op1)
		 * ->
		 * op->op0 < op->op1
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_B;
		expr->op0 = op->op0;
		expr->op1 = op->op1;

	} else {
		expr = jit__expr_alloc();

		expr->type = JIT_NOT;
		expr->op0 = op;
	}

	return expr;
}

struct jit_expr *
jit_inv(struct jit_func *func, struct jit_expr *op)
{
	struct jit_expr *expr;

	if (op->type == JIT_CONST) {
		/*
		 * ~const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = ~op->val;

	} else if (op->type == JIT_NOT) {
		/*
		 * ~ ~ op
		 * ->
		 * op
		 */
		expr = op->op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_INV;
		expr->op0 = op;
	}

	return expr;
}

struct jit_expr *
jit_neg(struct jit_func *func, struct jit_expr *op)
{
	struct jit_expr *expr;

	if (op->type == JIT_CONST) {
		/*
		 * -const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = -op->val;

	} else if (op->type == JIT_NOT) {
		/*
		 * - - op
		 * ->
		 * op
		 */
		expr = op->op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_NEG;
		expr->op0 = op;
	}

	return expr;
}

struct jit_expr *
jit_add(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;
	struct jit_expr *expr2;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const + const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val + op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 + op1
		 * ->
		 * op1
		 */
		expr = op1;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 + 0
		 * ->
		 * op0
		 */
		expr = op0;

	} else if (op0->type == JIT_ADD
		&& op0->op1->type == JIT_CONST
		&& op1->type == JIT_CONST) {
		/*
		 * (op00 + const) + const
		 * ->
		 * op00 + const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);
		expr2 = jit__expr_alloc();
		jit__expr_append(func, expr2);

		expr2->type = JIT_CONST;
		expr2->val = op0->op1->val + op1->val;

		expr->type = JIT_ADD;
		expr->op0 = op0->op0;
		expr->op1 = expr2;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_ADD;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_sub(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const - const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val - op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 - op1;
		 * ->
		 * -op1
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_NEG;
		expr->op0 = op1;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 - 0
		 * ->
		 * op0
		 */
		expr = op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_SUB;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_mul(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;
	unsigned int i;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const * const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val * op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 * op1
		 * ->
		 * 0
		 */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 * 0
		 * ->
		 * 0
		 */
		expr = op1;

	} else {
		for (i = 0; ; i++) {
			if (i == 64) {
				expr = jit__expr_alloc();
				jit__expr_append(func, expr);

				expr->type = JIT_MUL;
				expr->op0 = op0;
				expr->op1 = op1;
				break;

			} else if (op0->type == JIT_CONST
				&& op0->val == 1L << i) {
				/*
				 * (1 << i) * op1
				 * ->
				 * op1 << i
				 */
				expr = jit_lsl(func, op1,
						jit_uconst(func, i));
				break;

			} else if (op1->type == JIT_CONST
				&& op1->val == 1L << i) {
				/*
				 * op0 * (1 << i)
				 * ->
				 * op0 << i
				 */
				expr = jit_lsl(func, op0,
						jit_uconst(func, i));
				break;
			}
		}
	}

	return expr;
}

struct jit_expr *
jit_div(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const / const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val / op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 / op1
		 * ->
		 * 0
		 */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 / 0
		 * ->
		 * ?!?
		 */
		assert(0); /* FIXME */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == 1) {
		/*
		 * op0 / 1
		 * ->
		 * op0
		 */
		expr = op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_DIV;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_rem(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const % const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val % op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 % op1
		 * ->
		 * 0
		 */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 % 0
		 * ->
		 * ?!?
		 */
		assert(0); /* FIXME */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == 1) {
		/*
		 * op0 % 1
		 * ->
		 * 0
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = 0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_REM;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_and(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const & const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val & op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 & op1
		 * ->
		 * 0
		 */
		expr = op0;

	} else if (op0->type == JIT_CONST
		&& op0->val == ~(uint64_t) 0) {
		/*
		 * 0b111..111 & op1
		 * ->
		 * op1
		 */
		expr = op1;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 & 0
		 * ->
		 * 0
		 */
		expr = op1;

	} else if (op1->type == JIT_CONST
		&& op1->val == ~(uint64_t) 0) {
		/*
		 * op0 & 0b111..111
		 * ->
		 * op0
		 */
		expr = op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_AND;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_or(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const | const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val | op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 | op1
		 * ->
		 * op1
		 */
		expr = op1;

	} else if (op0->type == JIT_CONST
		&& op0->val == ~(uint64_t) 0) {
		/*
		 * 0b111..111 | op1
		 * ->
		 * 0b111..111
		 */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 | 0
		 * ->
		 * op0
		 */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == ~(uint64_t) 0) {
		/*
		 * op0 | 0b111..111
		 * ->
		 * 0b111..111
		 */
		expr = op1;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_OR;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_xor(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const ^ const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val ^ op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 ^ op1
		 * ->
		 * op1
		 */
		expr = op1;

	} else if (op0->type == JIT_CONST
		&& op0->val == ~(uint64_t) 0) {
		/*
		 * 0b111..111 ^ op1
		 * ->
		 * ~op1
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_INV;
		expr->op0 = op1;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 ^ 0
		 * ->
		 * op0
		 */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == ~(uint64_t) 0) {
		/*
		 * op0 ^ 0b111..111
		 * ->
		 * ~op0
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_INV;
		expr->op0 = op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_XOR;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_lsl(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const << const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val << op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 << op1
		 * ->
		 * 0
		 */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 << 0
		 * ->
		 * op0
		 */
		expr = op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_LSL;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_asr(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const >> const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = (int64_t) op0->val >> op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 >> op1
		 * ->
		 * 0
		 */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 >> 0
		 * ->
		 * op0
		 */
		expr = op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_ASR;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_lsr(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const >> const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val >> op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 >> op1
		 * ->
		 * 0
		 */
		expr = op0;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 >> 0
		 * ->
		 * op0
		 */
		expr = op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_LSR;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_eq(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const == const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val == op1->val;

	} else if (op0->type == JIT_CONST
		&& op0->val == 0) {
		/*
		 * 0 == op1
		 * ->
		 * ! op1
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_NOT;
		expr->op0 = op1;

	} else if (op1->type == JIT_CONST
		&& op1->val == 0) {
		/*
		 * op0 == 0
		 * ->
		 * ! op0
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_NOT;
		expr->op0 = op0;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_EQ;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_neq(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const != const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val != op1->val;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_NEQ;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_b(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const < const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val < op1->val;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_B;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

struct jit_expr *
jit_ae(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1)
{
	struct jit_expr *expr;

	if (op0->type == JIT_CONST
	 && op1->type == JIT_CONST) {
		/*
		 * const >= const
		 * ->
		 * const
		 */
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_CONST;
		expr->val = op0->val >= op1->val;

	} else {
		expr = jit__expr_alloc();
		jit__expr_append(func, expr);

		expr->type = JIT_AE;
		expr->op0 = op0;
		expr->op1 = op1;
	}

	return expr;
}

static void
jit__stmt_append(struct jit_func *func, struct jit_stmt *stmt)
{
	stmt->prev = func->stmt_last;
	stmt->next = NULL;
	if (stmt->prev) {
		stmt->prev->next = stmt;
	} else {
		func->stmt_first = stmt;
	}
	if (stmt->next) {
		assert(0);
	} else {
		func->stmt_last = stmt;
	}
}

void
jit_assign(struct jit_func *func, struct jit_expr *dst, struct jit_expr *src)
{
	struct jit_stmt *stmt;

	stmt = jit__stmt_alloc();

	stmt->type = JIT_STMT_ASSIGN;
	stmt->assign.dst = dst;
	stmt->assign.src = src;

	jit__stmt_append(func, stmt);
}

void
jit_goto(struct jit_func *func, struct jit_label *label)
{
	struct jit_stmt *stmt;

	stmt = jit__stmt_alloc();

	stmt->type = JIT_STMT_GOTO;
	stmt->goto_label = label;

	jit__stmt_append(func, stmt);
}

void
jit_ifgoto(struct jit_func *func, struct jit_expr *cond, struct jit_label *label)
{
	struct jit_stmt *stmt;

	stmt = jit__stmt_alloc();

	stmt->type = JIT_STMT_IFGOTO;
	stmt->ifgoto.cond = cond;
	stmt->ifgoto.label = label;

	jit__stmt_append(func, stmt);
}

void
jit_switchgoto(
	struct jit_func *func,
	struct jit_expr *expr,
	struct jit_label *def,
	...
)
{
	va_list arglist;
	struct jit_stmt *stmt;
	unsigned int n;

	stmt = jit__stmt_alloc();

	stmt->type = JIT_STMT_SWITCHGOTO;
	stmt->switchgoto.expr = expr;
	stmt->switchgoto.def = def;
	va_start(arglist, def);
	for (n = 0; ; n++) {
		assert(n < sizeof(stmt->switchgoto.val) / sizeof(stmt->switchgoto.val[0]));
		stmt->switchgoto.val[n] = va_arg(arglist, int);
		stmt->switchgoto.lab[n] = va_arg(arglist, struct jit_label *);

		if (! stmt->switchgoto.lab[n]) {
			break;
		}
	}
	va_end(arglist);

	jit__stmt_append(func, stmt);
}

void
jit_label(struct jit_func *func, struct jit_label *label)
{
	struct jit_stmt *stmt;

	stmt = jit__stmt_alloc();

	stmt->type = JIT_STMT_LABEL;
	stmt->label = label;

	jit__stmt_append(func, stmt);
}

void
jit_call(
	struct jit_func *func,
	struct jit_expr *dst,
	void *ptr,
	...
)
{
	struct jit_stmt *stmt;
	va_list arglist;
	unsigned int n;
	struct jit_expr *arg;

	stmt = jit__stmt_alloc();

	stmt->type = JIT_STMT_CALL;
	stmt->call.dst = dst;
	stmt->call.func = ptr;
	va_start(arglist, ptr);
	for (n = 0; ; n++) {
		arg = va_arg(arglist, struct jit_expr *);
		if (! arg) {
			break;
		}
		assert(n < sizeof(stmt->call.arg) / sizeof(stmt->call.arg[0]));
		stmt->call.arg[n] = arg;
	}
	va_end(arglist);
	for (; n < sizeof(stmt->call.arg) / sizeof(stmt->call.arg[0]); n++) {
		stmt->call.arg[n] = NULL;
	}

	jit__stmt_append(func, stmt);
}

void
jit_brk(struct jit_func *func)
{
	struct jit_stmt *stmt;

	stmt = jit__stmt_alloc();

	stmt->type = JIT_STMT_BRK;
	
	jit__stmt_append(func, stmt);
}

struct jit_func *
jit_func_alloc(void)
{
	static struct jit_func func;

	func.param_first = NULL;
	func.param_last = NULL;

	func.local_first = NULL;
	func.local_last = NULL;

	func.expr_first = NULL;
	func.expr_last = NULL;

	func.label_first = NULL;
	func.label_last = NULL;

	func.stmt_first = NULL;
	func.stmt_last = NULL;

	return &func;
}

void *
jit_compile(struct jit_func *func)
{
	struct jit_stmt *stmt;
	struct jit_label *label;
	struct jit_expr *expr;
	struct jit_expr *local;
	struct jit_expr *param;

#if 0
	jit_func_dump(func);
#endif

	jit__compile(func);

#if 0
	jit__dump(func);
#endif

	for (stmt = func->stmt_first; stmt; ) {
		struct jit_stmt *old;

		old = stmt;
		stmt = stmt->next;

		if (old->prev) {
			assert(0);
		} else {
			func->stmt_first = old->next;
		}
		if (old->next) {
			old->next->prev = old->prev;
		} else {
			func->stmt_last = old->prev;
		}

		jit__stmt_free(old);
	}

	for (label = func->label_first; label; ) {
		struct jit_label *old;

		old = label;
		label = label->next;

		if (old->prev) {
			assert(0);
		} else {
			func->label_first = old->next;
		}
		if (old->next) {
			old->next->prev = old->prev;
		} else {
			func->label_last = old->prev;
		}

		jit__label_free(old);
	}

	for (expr = func->expr_first; expr; ) {
		struct jit_expr *old;

		old = expr;
		expr = expr->next;

		if (old->prev) {
			assert(0);
		} else {
			func->expr_first = old->next;
		}
		if (old->next) {
			old->next->prev = old->prev;
		} else {
			func->expr_last = old->prev;
		}

		jit__expr_free(old);
	}

	for (local = func->local_first; local; ) {
		struct jit_expr *old;

		old = local;
		local = local->next;

		if (old->prev) {
			assert(0);
		} else {
			func->local_first = old->next;
		}
		if (old->next) {
			old->next->prev = old->prev;
		} else {
			func->local_last = old->next;
		}

		jit__expr_free(old);
	}

	for (param = func->param_first; param; ) {
		struct jit_expr *old;

		old = param;
		param = param->next;

		if (old->prev) {
			assert(0);
		} else {
			func->param_first = old->next;
		}
		if (old->next) {
			old->next->prev = old->prev;
		} else {
			func->param_last = old->next;
		}

		jit__expr_free(old);
	}

	return func->head;
}

static void
jit_type_dump(int sign, unsigned long size)
{
	switch (sign * size) {
	case 0:
		fprintf(stderr, "void *");
		break;
	case -1:
		fprintf(stderr, "int8_t");
		break;
	case 1:
		fprintf(stderr, "uint8_t");
		break;
	case -2:
		fprintf(stderr, "int16_t");
		break;
	case 2:
		fprintf(stderr, "uint16_t");
		break;
	case -4:
		fprintf(stderr, "int32_t");
		break;
	case 4:
		fprintf(stderr, "uint32_t");
		break;
	case -8:
		fprintf(stderr, "int64_t");
		break;
	case 8:
		fprintf(stderr, "uint64_t");
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
}

static void
jit_expr_dump(struct jit_expr *expr)
{
	switch (expr->type) {
	case JIT_CONST:
		fprintf(stderr, "0x%lx", expr->val);
		break;
	case JIT_GLOBAL:
		fprintf(stderr, "global");
		break;
	case JIT_PARAM:
		fprintf(stderr, "param");
		break;
	case JIT_LOCAL:
		fprintf(stderr, "local");
		break;
	case JIT_STAR:
		fprintf(stderr, "*(");
		jit_type_dump(expr->sign, expr->size);
		fprintf(stderr, ") ");
		fprintf(stderr, "(");
		jit_expr_dump(expr->op0);
		fprintf(stderr, ")");
		break;
	case JIT_CONV:
		fprintf(stderr, "(...) ");
		jit_expr_dump(expr->op0);
		break;
	case JIT_NOT:
		fprintf(stderr, "! ");
		jit_expr_dump(expr->op0);
		break;
	case JIT_INV:
		fprintf(stderr, "~");
		jit_expr_dump(expr->op0);
		break;
	case JIT_NEG:
		fprintf(stderr, "-");
		jit_expr_dump(expr->op0);
		break;
	case JIT_ADD:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " + ");
		jit_expr_dump(expr->op1);
		break;
	case JIT_SUB:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " - ");
		jit_expr_dump(expr->op1);
		break;
	case JIT_AND:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " & ");
		jit_expr_dump(expr->op1);
		break;
	case JIT_OR:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " | ");
		jit_expr_dump(expr->op1);
		break;
	case JIT_XOR:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " ^ ");
		jit_expr_dump(expr->op1);
		break;
	case JIT_LSL:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " << ");
		jit_expr_dump(expr->op1);
		break;
	case JIT_LSR:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " >> ");
		jit_expr_dump(expr->op1);
		break;
	case JIT_EQ:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " == ");
		jit_expr_dump(expr->op1);
		break;
	case JIT_NEQ:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " != ");
		jit_expr_dump(expr->op1);
		break;
	case JIT_B:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " < ");
		jit_expr_dump(expr->op1);
		break;
	case JIT_AE:
		jit_expr_dump(expr->op0);
		fprintf(stderr, " >= ");
		jit_expr_dump(expr->op1);
		break;
	default:
		fprintf(stderr, "...");
		break;
	}
}

static void
jit_stmt_dump(struct jit_stmt *stmt)
{
	int i;

	switch (stmt->type) {
	case JIT_STMT_ASSIGN:
		fprintf(stderr, "\t");
		jit_expr_dump(stmt->assign.dst);
		fprintf(stderr, " = ");
		jit_expr_dump(stmt->assign.src);
		fprintf(stderr, ";\n");
		break;
	case JIT_STMT_GOTO:
		fprintf(stderr, "\t");
		fprintf(stderr, "goto ");
		fprintf(stderr, "%d", stmt->goto_label->id);
		fprintf(stderr, ";\n");
		break;
	case JIT_STMT_IFGOTO:
		fprintf(stderr, "\t");
		fprintf(stderr, "if (");
		jit_expr_dump(stmt->ifgoto.cond);
		fprintf(stderr, ") goto ");
		fprintf(stderr, "%d", stmt->ifgoto.label->id);
		fprintf(stderr, ";\n");
		break;
	case JIT_STMT_SWITCHGOTO:
		fprintf(stderr, "\t");
		fprintf(stderr, "switch (");
		jit_expr_dump(stmt->switchgoto.expr);
		fprintf(stderr, ") {");
		fprintf(stderr, "\n");
		for (i = 0; stmt->switchgoto.lab[i]; i++) {
			fprintf(stderr, "\t");
			fprintf(stderr, "case %d:", stmt->switchgoto.val[i]);
			fprintf(stderr, " ");
			fprintf(stderr, "goto ");
			fprintf(stderr, "%d", stmt->switchgoto.lab[i]->id);
			fprintf(stderr, ";\n");
		}
		fprintf(stderr, "\t");
		fprintf(stderr, "goto ");
		fprintf(stderr, "%d", stmt->switchgoto.def->id);
		fprintf(stderr, ";\n");
		break;
	case JIT_STMT_LABEL:
		fprintf(stderr, "%d:", stmt->label->id);
		fprintf(stderr, "\n");
		break;
	case JIT_STMT_CALL:
		fprintf(stderr, "\t");
		if (stmt->call.dst) {
			jit_expr_dump(stmt->call.dst);
			fprintf(stderr, " = ");
		}
		fprintf(stderr, "%p(", stmt->call.func);
		for (i = 0; stmt->call.arg[i]; i++) {
			if (i != 0) {
				fprintf(stderr, ", ");
			}
			jit_expr_dump(stmt->call.arg[i]);
		}
		fprintf(stderr, ");\n");
		break;
	case JIT_STMT_BRK:
		fprintf(stderr, "*** brk ***\n");
		break;
	}
}

void
jit_dump(struct jit_func *func)
{
	struct jit_expr *param;
	struct jit_expr *local;
	struct jit_stmt *stmt;

	fprintf(stderr, "void\n");
	fprintf(stderr, "func(");
	for (param = func->param_first; param; param = param->next) {
		jit_type_dump(param->sign, param->size);
		if (param->next) {
			fprintf(stderr, ", ");
		}
	}
	fprintf(stderr, ")\n");
	fprintf(stderr, "{\n");
	for (local = func->local_first; local; local = local->next) {
		jit_type_dump(local->sign, local->size);
		fprintf(stderr, ";\n");
	}
	for (stmt = func->stmt_first; stmt; stmt = stmt->next) {
		jit_stmt_dump(stmt);
	}
	fprintf(stderr, "}\n");

	fprintf(stderr, "\n");
}
