/*
 * Copyright (C) 2001-2004 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <ferite.h>
#include "aphex.h"

FeriteThreadGroup *ferite_create_thread_group()
{
    FeriteThreadGroup *ptr = NULL;

    FE_ENTER_FUNCTION;
#ifdef THREAD_SAFE
    ptr = fmalloc( sizeof( FeriteThreadGroup ) );
    ptr->lock = aphex_mutex_create();
    ptr->thread_list = ferite_create_stack( NULL, 1 );
#endif
    FE_LEAVE_FUNCTION( ptr );
}

void ferite_thread_group_destroy( FeriteScript *script, FeriteThreadGroup *group )
{
    FE_ENTER_FUNCTION;
#ifdef THREAD_SAFE
    if( group != NULL )
    {
        ferite_thread_group_wait( script, group ); /* we should atleast make sure they have all stopped
                                                    * before we delete */
        ferite_delete_stack( NULL, group->thread_list );
        aphex_mutex_destroy( group->lock );
        ffree( group );
    }
#endif
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_thread_group_attach( FeriteScript *script, FeriteThreadGroup *group, FeriteThread *thread )
{
    int i = 0;

    FE_ENTER_FUNCTION;
#ifdef THREAD_SAFE
    if( group != NULL && thread != NULL )
    {
        aphex_mutex_lock( group->lock );

    /* search the thread list for a gap */
        for( ; i <= group->thread_list->stack_ptr; i++ )
        {
            if( group->thread_list->stack[i] == NULL )
            {
                group->thread_list->stack[i] = thread;
                aphex_mutex_unlock( group->lock );
                FE_LEAVE_FUNCTION(NOWT);
            }
        }

    /* we didn't find a spot, put the thread at the end of the list */
        ferite_stack_push( group->thread_list, thread );
        aphex_mutex_unlock( group->lock );
    }
#endif
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_thread_group_dettach( FeriteScript *script, FeriteThreadGroup *group, FeriteThread *thread )
{
    int i = 0;

    FE_ENTER_FUNCTION;
#ifdef THREAD_SAFE
    if( group != NULL && thread != NULL )
    {
        aphex_mutex_lock( group->lock );
    /* search the thread list, remove the thread by setting it to NULL */
        for( ; i <= group->thread_list->stack_ptr; i++ )
        {
            if( group->thread_list->stack[i] == thread )
            {
                group->thread_list->stack[i] = NULL;
                break;
            }
        }
        aphex_mutex_unlock( group->lock );
    }
#endif
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_thread_group_wait( FeriteScript *script, FeriteThreadGroup *group )
{
    int i = 0, keep_going = FE_TRUE;

    FE_ENTER_FUNCTION;
#ifdef THREAD_SAFE
   /* we have to sit in a loop, check to see if group->thread_list is empty
  * if it is, return, if not, block until it is */
    if( group != NULL )
    {
        while( FE_TRUE )
        {
            keep_going = FE_FALSE;
            aphex_mutex_lock( group->lock );
            for( i = 1; i <= group->thread_list->stack_ptr; i++ )
            {
                if( group->thread_list->stack[i] != NULL )
                {
                    keep_going = FE_TRUE;
                    break; /* we dont need to keep looking if we have at least one thread */
                }
            }
            aphex_mutex_unlock( group->lock );
            if( keep_going == FE_FALSE )
              break;
            aphex_thread_sleep( NULL, 100 );
        }
    }
#endif
    FE_LEAVE_FUNCTION(NOWT);
}
