package fix;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.Socket;
import java.net.URL;
import java.net.UnknownHostException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.net.ssl.KeyManager;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;

/**
 * A Client for HTTP/HTTPS Connections.
 * 
 * With this class you can send and receive HTTP-MIME-messages
 * 
 * $Date: 2010-11-03 16:58:25 +0100 (Mi, 03. Nov 2010) $
 * $Revision: 33 $
 * $Author: roth $
 * 
 * @author Dominik Greibl
 * 
 * Copyright (C) 2008 Dominik Greibl
 * Copyright (C) 2009 Sebastian Roth
 * 
 * All Rights Reserved
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

public class HttpClient {

	private Socket socket = null;
	private String fexHost = "";
	private int fexPort = 0;
	private Proxy proxy = null;
	private String proxyHost = "";
	private int proxyPort = 0;
	private String protocol = "";
	public OutputStream out = null;
	private ErrorWindow er = null;
	private static boolean error = false;
	private static SSLSocketFactory sslSocketFactory = null;
	private boolean debug = false;
	public boolean isConnected = false;
	public boolean isProxyfied = false;
	public boolean isSecureConn = false;
	private ArrayList<Map<String,String>> requestProperties = null;
	private Map<String, String> responseProperties = null;

	/**
	 * The standard constructor for this class.
	 * 
	 * @param e
	 *            The instance of ErrorWindow the errors are displayed with.
	 * @param d
	 *            The debug value, if true the debugging is enabled.
	 */
	public HttpClient(ErrorWindow e, boolean d) {
		debug = d;
		er = e;
		requestProperties = new ArrayList<Map<String,String>>();
		responseProperties = new HashMap<String, String>();
		System.setProperty("java.net.useSystemProxies", "true");
	}

	/**
	 * Establishes a connection to the given host URL by using
	 * <code>httpConnect()</code> or <code>httpsConnect()</code>
	 * 
	 * @param host
	 *            URL to connect to
	 */
	public void connect(String host) {
		String[] parsed = parseURL(host);
		protocol = parsed[0];
		fexHost = parsed[1];
		fexPort = Integer.parseInt(parsed[2]);
		try {
			List<Proxy> l = ProxySelector.getDefault().select(new URI(host));
			if (!l.isEmpty()) {
				proxy = l.get(0);
				if (proxy.type() == Proxy.Type.DIRECT) {
					proxy = null;
					l.clear();
					l = null;
					isProxyfied = false;
				}
				else {
					isProxyfied = true;
				}
			}

		} catch (URISyntaxException e1) {
			e1.printStackTrace();
		}

		if (isProxyfied) {
			proxyHost = ((InetSocketAddress)proxy.address()).getHostName();
			proxyPort = ((InetSocketAddress)proxy.address()).getPort();
		}
		if (protocol.startsWith("https")) {
			isSecureConn = true;
			httpsConnect(fexHost, fexPort);
		}
		else if (protocol.startsWith("http")) {
			isSecureConn= false;
			httpConnect();
		}
		else {
			er.setMess("Wrong protocol: Please use HTTP or HTTPS.", null);
			return;
		}
		/** Creates the OutputStream for the connection * */
		try {
			out = socket.getOutputStream();
		} catch (IOException e) {
			er.setMess("Unable to open output stream to server: " + e.getMessage(), e);
		}
	}

	/**
	 * Establishes a HTTP-Connection using a Socket
	 * 
	 * @param host
	 *            The hostname to connect to
	 */
	public void httpConnect() {
		String host = "";
		int port = 0;
		if (!isProxyfied) {
			host = fexHost;
			port = fexPort;
		}
		else {
			host = proxyHost;
			port = proxyPort;
		}
		try {
			socket = new Socket();
			socket.connect(new InetSocketAddress(host, port));
		} catch (UnknownHostException e) {
			er.setMess("Host-Server is unknown. " + e.getMessage(), e);
		} catch (IOException e) {
			er.setMess("Could not connect to server '" + host + ":"
					 + port + "': " + e.getMessage(), e);
		}
		isConnected = true;
	}

	/**
	 * Establishes a HTTPS-Connection using a Socket.
	 * 
	 * @param host
	 *            The host name to connect to
	 */
	public void httpsConnect(String host, int port) {
		if (!isProxyfied) {
			try {
				socket = getSocketFactory().createSocket();
				socket.connect(new InetSocketAddress(host, port));
			} catch (UnknownHostException e) {
				er.setMess("Host-Server is unknown. " + e.getMessage(), e);
			} catch (IOException e) {
				er.setMess("Could not connect to server '" + host + ":"
						 + port + "': " + e.getMessage(), e);
			}
		}
		else {
			Socket proxySocket = null;
			try {
				proxySocket = new Socket();
				proxySocket.connect(new InetSocketAddress(proxyHost, proxyPort));
			} catch (UnknownHostException e) {
				er.setMess("Host-Server is unknown. " + e.getMessage(), e);
			} catch (IOException e) {
				er.setMess("Could not connect to server '" + host + ":"
						 + port + "': " + e.getMessage(), e);
			}
			try {
				out = proxySocket.getOutputStream();
			} catch (IOException e) {
				er.setMess("Unable to open output stream to proxy server: " + e.getMessage(), e);
			}
			String proxyConnect = "CONNECT " + fexHost + ":" + fexPort + " HTTP/1.0\r\n";
			proxyConnect += "User-Agent: F*IX\r\n\r\n";
			send(proxyConnect);
			String resp = "";
			try {
				resp = receive(new BufferedReader(new InputStreamReader
						(proxySocket.getInputStream())));
			} catch (IOException e) {
				er.setMess("Unable to read from proxy server: "	+ e.getMessage(), e);
			}
			if (Pattern.compile("HTTP\\/[0-9]\\.[0-9]\\s*200\\s*.*").matcher(resp).find()) {
				try {
					socket = getSocketFactory().createSocket(proxySocket, fexHost, fexPort, true);
					//socket.connect(new InetSocketAddress(fexHost, fexPort));
				} catch (IOException e) {
					er.setMess("Couldn't create tunneling SSL socket throught proxy server: "
							+e.getMessage(), e);
				}
			}
			else {
				// TODO better message?!? include server response??
				er.setMess("Unable to tunnel SSL connection through proxy server.", null);
			}
		}
		isConnected = true;
	}


	public long addRequestProperty(String key, String value) {
		long length = key.length() + ": ".length() + value.length() + "\r\n".length();
		Map<String,String> header = new HashMap<String, String>();
		header.put(key, value);
		requestProperties.add(header);
		return length;
	}

	public long prependRequestProperty(String key, String value) {
		long length = key.length() + ": ".length() + value.length() + "\r\n".length();
		Map<String,String> header = new HashMap<String, String>();
		header.put(key, value);
		requestProperties.add(0, header);
		return length;
	}

	public String getResponseProperty(String key) {
		return responseProperties.get(key);
	}

	public String getRequestProperties() {
		String reqProps = "";
		Iterator<Map<String,String>> it = requestProperties.iterator();
		while (it.hasNext()) {
			for (Map.Entry<String,String> header : it.next().entrySet()) {
				reqProps = reqProps + header.getKey() + ": " + header.getValue() + "\r\n";
			}
		}
		requestProperties.clear();
		try {
			return new String(reqProps.getBytes("UTF-8"));
		} catch (UnsupportedEncodingException e) {
			er.setMess("HTTP POST request could not be converted to UTF-8.", e);
			return null;
		}
	}

	/**
	 * Sends data to the server using a socket stream.
	 * 
	 * @param data
	 *            string message to send
	 */
	public void send(String data) {
		try {
			out.write(data.getBytes());
			if (debug)
				System.out.print(data);
		} catch (Exception e) {
			e.printStackTrace();
			if (!error) {
				er.setMess("Error while sending to the server: " + e.getMessage(), e);
				error = true;
			}
		}
	}

	/**
	 * Sends data to the server using a socket stream.
	 * 
	 * @param data
	 *            byte array to send
	 */
	public void send(byte[] data) {
		try {
			out.write(data);
/*			if (debug)
				for (byte b : data) {
					System.out.print((char) b);
				}*/
		} catch (IOException e) {
			if (!error) {
				er.setMess("Error while sending to the server: " + e.getMessage(), e);
				error = true;
			}
		}
	}

	/**
	 * Sends data to the server using a socket stream.
	 * 
	 * @param data
	 *            byte array to send
	 * @param offset
	 *            starting from this offset
	 * @param data
	 *            up to this length
	 */
	public void send(byte[] data, int offset, int length) {
		try {
			out.write(data, offset, length);
/*			if (debug)
				for (byte b : data) {
					System.out.print((char) b);
				}*/
		} catch (IOException e) {
			if (!error) {
				er.setMess("Error while sending to the server: " + e.getMessage(), e);
				error = true;
			}
		}
	}
	
	/**
	 * Receives and parses the response from the server
	 * 
	 * @return response from server
	 */
	public String receive(BufferedReader from_server) {
		boolean isHttpRespHeader = true;
		String s = "";
		String httpRespHeader = "";
		try {
			while (((s = from_server.readLine()) != null) && s.length() > 0) {
				if (isHttpRespHeader) {
					if (s.startsWith("HTTP")) {
						httpRespHeader = s;
						isHttpRespHeader = false;
					}
				}
				else {
					String key = s.split(":[\\s\\t]+")[0];
					String value = s.split(":[\\s\\t]+")[1];
					responseProperties.put(key.toUpperCase(), value);
				}
				if (debug) {
					System.err.println("\t" + s);
				}
			}
		} catch (IOException e) {
			if (e.getMessage().equals("Connection reset")) { // &&
//				e.getClass().equals(new java.net.SocketException())) {
					System.out.println("Communication terminated. Server has reset the connection.");
			}
			er.setMess("Can not read from Server: " + s, e);
			return s;
		}
		return httpRespHeader;
	}

	/**
	 * Parses the given URL address and returns an array existing of: [0]:
	 * Protocol (http, https) [1]: Host-Address [2]: the Host-Port as a String
	 * [3]: the Requested File.
	 * 
	 * If the protocol is not http oder https a {@link IllegalArgumentException}
	 * will be thrown
	 * 
	 * @param address
	 *            the URL address to be parsed
	 * @return String[] as described above
	 * 
	 */
	public String[] parseURL(String address) {
		String[] response = new String[4];
		try {
			URL url;
			url = new URL(address);

			response[0] = url.getProtocol();
			response[1] = url.getHost();
			response[2] = "" + url.getPort();
			response[3] = url.getFile();

			if (!response[0].equals("http") && !response[0].equals("https")) {
				throw new IllegalArgumentException("Protocol must be http or https.");
			}
			if (Integer.parseInt(response[2]) == -1)
				if (response[0].equals("https"))
					response[2] = "443";
				else
					response[2] = "80";
		} catch (MalformedURLException e) {
			er.setMess("Malformed Server URL: " + address, e);
			// System.out.println(address);
		}
		return response;
	}

	/**
	 * Returns the remote port number either proxified or not. 
	 * 
	 * @return int
	 */
	public int getPort() {
		return fexPort;
	}

	/**
	 * Returns remote address either proxified or not.
	 * 
	 * @return String
	 */
	public String getHost() {
		return fexHost;
	}

	/**
	 * Returns the correct URL only if connecting through a normal
	 * HTTP proxy server. Otherwise an empty string.
	 * 
	 * @return String
	 */
	public String getURLifProxified() {
		if (isProxyfied && !isSecureConn) {
			return "http://" + getHost() + ":" + getPort();
		}
		return "";
	}
	/**
	 * Gets the Socket for use in other classes
	 * 
	 * @return socket
	 */
	public Socket getSocket() {
		return socket;
	}

	/**
	 * Returns a SSL Factory instance that accepts all server certificates. use:
	 * 
	 * <pre>
	 * SSLSocket sock = (SSLSocket) getSocketFactory.createSocket(host, port);
	 * </pre>
	 * 
	 * @return An SSL-specific socket factory.
	 */
	public final SSLSocketFactory getSocketFactory() {
		if (sslSocketFactory == null) {
			try {
				TrustManager[] tm = new TrustManager[] { new NaiveTrustManager() };
				SSLContext context = SSLContext.getInstance("SSL");
				context.init(new KeyManager[0], tm, new SecureRandom());

				sslSocketFactory = (SSLSocketFactory)context.getSocketFactory();

			} catch (KeyManagementException e) {
				er.setMess("No SSL algorithm support: ", e);
			} catch (NoSuchAlgorithmException e) {
				er.setMess("Exception when setting up the Naive key management.", e);
			}
		}
		return sslSocketFactory;
	}

}