/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file audio_pulse.c
 * \brief Pulseaudio audio plugin
 * Based upon the pulseaudio examples
 */

#include <ffgtk.h>
#include <preferences.h>
#include <string.h>

#include <pulse/pulseaudio.h>
#include <pulse/simple.h>
#include <pulse/error.h>


struct sPulsePipes {
	pa_simple *psSimpleIn;
	pa_simple *psSimpleOut;
};

/** predefined backup values */
static gint nPulseChannels = 2;
static gint nPulseSampleRate = 8000;
static gint nPulseBitsPerSample = 16;

struct sPulseDeviceList {
	gchar nInitialized;
	gchar anName[ 512 ];
	gint nIndex;
	gchar anDescription[ 256 ];
};

/** Get and set selected pulseaudio output device */
CREATE_STRING_PREFS( pulse, SelectedOutputDevice, "/plugins/pulseaudio/output" );
/** Get and set selected pulseaudio input device */
CREATE_STRING_PREFS( pulse, SelectedInputDevice, "/plugins/pulseaudio/input" );

/** This is where we'll store the input device list */
struct sPulseDeviceList asInputDeviceList[ 16 ];
/** This is where we'll store the output device list */
struct sPulseDeviceList asOutputDeviceList[ 16 ];

/**
 * \brief This callback gets called when our context changes state.
 * \param psContext pulseaudio context
 * \param pUserData pa ready flag
 */
static void pulseStateCb( pa_context *psContext, void *pUserData ) {
	pa_context_state_t nState;
	int *pnPulseReady = pUserData;

	nState = pa_context_get_state( psContext );

	switch ( nState ) {
		/* There are just here for reference */
		case PA_CONTEXT_UNCONNECTED:
		case PA_CONTEXT_CONNECTING:
		case PA_CONTEXT_AUTHORIZING:
		case PA_CONTEXT_SETTING_NAME:
		default:
			break;
		case PA_CONTEXT_FAILED:
		case PA_CONTEXT_TERMINATED:
			*pnPulseReady = 2;
			break;
		case PA_CONTEXT_READY:
			*pnPulseReady = 1;
			break;
	}
}

/**
 * \brief mainloop will call this function when it's ready to tell us about a sink.
 * \param psContext pulseaudio context
 * \param psSinkInfo sink information
 * \param nEol end-of-list
 * \param pUserData pointer to device list
 */
static void pulseSinkListCb( pa_context *psContext, const pa_sink_info *psSinkInfo, int nEol, void *pUserData ) {
	struct sPulseDeviceList *asDeviceList = pUserData;
	int nIndex = 0;

	/* If eol is set to a positive number, you're at the end of the list */
	if ( nEol > 0 ) {
		return;
	}

	for ( nIndex = 0; nIndex < 16; nIndex++) {
		if ( !asDeviceList[ nIndex ].nInitialized) {
			strncpy( asDeviceList[ nIndex ].anName, psSinkInfo -> name, 511 );
			strncpy( asDeviceList[ nIndex ].anDescription, psSinkInfo -> description, 255 );
			asDeviceList[ nIndex ].nIndex = psSinkInfo -> index;
			asDeviceList[ nIndex ].nInitialized = 1;
			break;
		}
	}
}

/**
 * \brief See above.  This callback is pretty much identical to the previous
 * \param psContext pulseaudio context
 * \param psSourceInfo source information
 * \param nEol end-of-list
 * \param pUserData pointer to device list
 */
static void pulseSourceListCb( pa_context *psContext, const pa_source_info *psSourceInfo, int nEol, void *pUserData ) {
	struct sPulseDeviceList *asDeviceList = pUserData;
	int nIndex = 0;

	if ( nEol > 0 ) {
		return;
	}

	for ( nIndex = 0; nIndex < 16; nIndex++ ) {
		if ( !asDeviceList[ nIndex ].nInitialized ) {
			strncpy( asDeviceList[ nIndex ].anName, psSourceInfo -> name, 511 );
			strncpy( asDeviceList[ nIndex ].anDescription, psSourceInfo -> description, 255 );
			asDeviceList[ nIndex ].nIndex = psSourceInfo -> index;
			asDeviceList[ nIndex ].nInitialized = 1;
			break;
		}
	}
}

/**
 * \brief Get device list for input and output devices
 * \param asInput pointer input device list
 * \param asOutput pointer output device list
 * \return error code
 */
static int pulseGetDeviceList( struct sPulseDeviceList *asInput, struct sPulseDeviceList *asOutput ) {
	/* Define our pulse audio loop and connection variables */
	pa_mainloop *psMainLoop;
	pa_mainloop_api *psMainApi;
	pa_operation *psOperation = NULL;
	pa_context *psContext;
	/* We'll need these state variables to keep track of our requests */
	int nState = 0;
	int nReady = 0;

	/* Initialize our device lists */
	memset( asInput, 0, sizeof( struct sPulseDeviceList ) * 16 );
	memset( asOutput, 0, sizeof( struct sPulseDeviceList ) * 16 );

	/* Create a mainloop API and connection to the default server */
	psMainLoop = pa_mainloop_new();
	psMainApi = pa_mainloop_get_api( psMainLoop );
	psContext = pa_context_new( psMainApi, "test" );

	/* This function connects to the pulse server */
	pa_context_connect( psContext, NULL, 0, NULL );

	/**
	 * This function defines a callback so the server will tell us it's state.
	 * Our callback will wait for the state to be ready.  The callback will
	 * modify the variable to 1 so we know when we have a connection and it's
	 * ready.
	 * If there's an error, the callback will set pa_ready to 2
	 */
	pa_context_set_state_callback( psContext, pulseStateCb, &nReady );

	/**
	 * Now we'll enter into an infinite loop until we get the data we receive
	 * or if there's an error
	 */
	for ( ; ; ) {
		/**
		 * We can't do anything until PA is ready, so just iterate the mainloop
		 * and continue
		 */
		if ( nReady == 0 ) {
			pa_mainloop_iterate( psMainLoop, 1, NULL );
			continue;
		}

		/* We couldn't get a connection to the server, so exit out */
		if ( nReady == 2 ) {
			pa_context_disconnect( psContext );
			pa_context_unref( psContext );
			pa_mainloop_free( psMainLoop );
			return -1;
		}

		/**
		 * At this point, we're connected to the server and ready to make
		 * requests
		 */
		switch ( nState ) {
			/* State 0: we haven't done anything yet */
			case 0:
				psOperation = pa_context_get_sink_info_list( psContext, pulseSinkListCb, asOutput );

				/* Update state for next iteration through the loop */
				nState++;
				break;
			case 1:
				if ( pa_operation_get_state( psOperation ) == PA_OPERATION_DONE ) {
					pa_operation_unref( psOperation );

					psOperation = pa_context_get_source_info_list( psContext, pulseSourceListCb, asInput );

					/* Update the state so we know what to do next */
					nState++;
				}
				break;
			case 2:
				if ( pa_operation_get_state( psOperation ) == PA_OPERATION_DONE ) {
					pa_operation_unref( psOperation );
					pa_context_disconnect( psContext );
					pa_context_unref( psContext );
					pa_mainloop_free( psMainLoop );
					return 0;
				}
				break;
			default:
				fprintf( stderr, "in state %d\n", nState );
				return -1;
		}

		pa_mainloop_iterate( psMainLoop, 1, NULL );
	}
}

/**
 * \brief Detect pulseaudio devices
 * \return error code, negative values on error otherwise success
 */
static int pulseAudioDetectDevices( void ) {
	int nFoundIn = 0;
	int nFoundOut = 0;
	int nIndex;

	if ( pulseGetDeviceList( asInputDeviceList, asOutputDeviceList ) < 0 ) {
		fprintf( stderr, "failed to get device list\n" );
		return -1;
	}

	for ( nIndex = 0; nIndex < 16; nIndex++ ) {
		if ( !asOutputDeviceList[ nIndex ].nInitialized ) {
			break;
		}
		nFoundOut++;
	}

	for ( nIndex = 0; nIndex < 16; nIndex++ ) {
		if ( !asInputDeviceList[ nIndex].nInitialized ) {
			break;
		}
		nFoundIn++;
	}

	return !( nFoundIn && nFoundOut );
}

/**
 * \brief Initialize audio device
 * \param nChannels number of channels
 * \param nSampleRate sample rate
 * \param nBitsPerSample number of bits per samplerate
 * \return TRUE on success, otherwise error
 */
static int pulseAudioInit( unsigned char nChannels, unsigned short nSampleRate, unsigned char nBitsPerSample ) {
	/* TODO: Check if configuration is valid and usable */
	nPulseChannels = nChannels;
	nPulseSampleRate = nSampleRate;
	nPulseBitsPerSample = nBitsPerSample;

	return 0;
}

/**
 * \brief Open new playback and record pipes
 * \return pipe pointer or NULL on error
 */
static void *pulseAudioOpen( void ) {
	pa_sample_spec sSampleSpec = {
		.format = PA_SAMPLE_S16LE,
	};
	int nError;
	pa_buffer_attr sBuffer = {
		.fragsize = 320,
		.maxlength = -1,
		.minreq = -1,
		.prebuf = -1,
		.tlength = -1,
	};
	struct sPulsePipes *psPipes = malloc( sizeof( struct sPulsePipes ) );

	if ( psPipes == NULL ) {
		return NULL;
	}

	sSampleSpec.rate = nPulseSampleRate;
	sSampleSpec.channels = nPulseChannels;
	if ( nPulseBitsPerSample == 2 ) {
		sSampleSpec.format = PA_SAMPLE_S16LE;
	}

	psPipes -> psSimpleOut = pa_simple_new( NULL, "ffgtk", PA_STREAM_PLAYBACK, pulseGetSelectedOutputDevice( getActiveProfile() ), "phone", &sSampleSpec, NULL, NULL, &nError );
	if ( psPipes -> psSimpleOut == NULL ) {
		Debug( KERN_DEBUG, "Failed: %s\n", pa_strerror( nError ) );
		free( psPipes );
		return NULL;
	}

	psPipes -> psSimpleIn = pa_simple_new( NULL, "ffgtk", PA_STREAM_RECORD, pulseGetSelectedInputDevice( getActiveProfile() ), "phone", &sSampleSpec, NULL, &sBuffer, &nError );
	if ( psPipes -> psSimpleIn == NULL ) {
		Debug( KERN_DEBUG, "Failed: %s\n", pa_strerror( nError ) );
		pa_simple_free( psPipes -> psSimpleOut );
		free( psPipes );
		return NULL;
	}

	return psPipes;
}

/**
 * \brief Close audio pipelines
 * \param pPriv pointer to private input/output pipes
 * \param bForce force quit
 * \return error code
 */
static int pulseAudioClose( void *pPriv, gboolean bForce ) {
	struct sPulsePipes *psPipes = pPriv;
	int nError;

	if ( psPipes == NULL ) {
		return -EINVAL;
	}

	if ( psPipes -> psSimpleOut != NULL ) {
		if ( pa_simple_drain( psPipes -> psSimpleOut, &nError ) < 0 ) {
			Debug( KERN_DEBUG, "Failed: %s\n", pa_strerror( nError ) );
		}

		pa_simple_free( psPipes -> psSimpleOut );
		psPipes -> psSimpleOut = NULL;
	}

	if ( psPipes -> psSimpleIn != NULL ) {
		pa_simple_free( psPipes -> psSimpleIn );
		psPipes -> psSimpleIn = NULL;
	}

	free( psPipes );

	return 0;
}

/**
 * \brief Write data to audio device
 * \param pPriv pointer to private pipes
 * \param pnBuf audio data pointer
 * \param nLen length of buffer
 * \return error code
 */
static int pulseAudioWrite( void *pPriv, unsigned char *pnBuf, unsigned int nLen ) {
	struct sPulsePipes *psPipes = pPriv;
	int nError;

	if ( psPipes == NULL || psPipes -> psSimpleOut == NULL ) {
		return -1;
	}

	if ( pa_simple_write( psPipes -> psSimpleOut, pnBuf, ( size_t ) nLen, &nError ) < 0 ) {
		Debug( KERN_DEBUG, "Failed: %s\n", pa_strerror( nError ) );
	}

	return 0;
}

/**
 * \brief Read data from audio device
 * \param pPriv pointer to private pipes
 * \param pnBuf audio data pointer
 * \param nLen maximal length of buffer
 * \return error code
 */
static int pulseAudioRead( void *pPriv, unsigned char *pnBuf, unsigned int nLen ) {
	struct sPulsePipes *psPipes = pPriv;
	int nRet = 0;
	int nError;

	if ( psPipes == NULL || psPipes -> psSimpleIn == NULL ) {
		return 0;
	}

	nRet = pa_simple_read( psPipes -> psSimpleIn, pnBuf, nLen, &nError );
	if ( nRet < 0 ) {
		Debug( KERN_DEBUG, "Failed: %s\n", pa_strerror( nError ) );
		nLen = 0;
	}

	return nLen;
}

/**
 * \brief Deinit audio interface
 * \return 0
 */
static int pulseAudioDeinit( void ) {
	return 0;
}

/**
 * \brief Display pulse preferences window
 */
static void pulsePreferences( void ) {
	GtkWidget *psDialog;
	GtkWidget *psComboBoxOut;
	GtkWidget *psComboBoxIn;
	GtkWidget *psLabelIn;
	GtkWidget *psLabelOut;
	GtkWidget *psBox;
	int nIndex = 0;
	int nCount = 0;
	struct sProfile *psProfile = getActiveProfile();

	pulseAudioDetectDevices();

	psDialog = gtk_dialog_new_with_buttons( _( "PulseAudio Preferences" ), NULL, 0, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL );
	psComboBoxOut = gtk_combo_box_new_text();
	psComboBoxIn = gtk_combo_box_new_text();
	psLabelIn = gtk_label_new( "" );
	psLabelOut = gtk_label_new( "" );

	/* Set output device */
	gtk_label_set_markup( GTK_LABEL( psLabelOut ), _( "<b>Select output device:</b>" ) );
	psBox = gtk_dialog_get_content_area( GTK_DIALOG( psDialog ) );
	gtk_box_pack_start( GTK_BOX( psBox ), psLabelOut, FALSE, FALSE, 0 );

	nIndex = 0;

	for ( nCount = 0; nCount < 16; nCount++ ) {
		if ( !asOutputDeviceList[ nCount ].nInitialized ) {
			break;
		}

		gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBoxOut ), asOutputDeviceList[ nCount ].anDescription );
		if ( pulseGetSelectedOutputDevice( psProfile ) != NULL ) {
			if ( !strcmp( asOutputDeviceList[ nCount ].anName, pulseGetSelectedOutputDevice( psProfile ) ) ) {
				gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBoxOut ), nIndex );
			}
		} else {
			gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBoxOut ), 0 );
		}
		nIndex++;
	}

	if ( gtk_combo_box_get_active( GTK_COMBO_BOX( psComboBoxOut ) ) < 0 ) {
		gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBoxOut ), 0 );
	}

	gtk_box_pack_start( GTK_BOX( psBox ), psComboBoxOut, FALSE, TRUE, 5 );

	/* Set input device */
	gtk_label_set_markup( GTK_LABEL( psLabelIn ), _( "<b>Select input device:</b>" ) );
	gtk_box_pack_start( GTK_BOX( psBox ), psLabelIn, FALSE, TRUE, 0 );

	nIndex = 0;

	for ( nCount = 0; nCount < 16; nCount++ ) {
		if ( !asInputDeviceList[ nCount ].nInitialized ) {
			break;
		}

		gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBoxIn ), asInputDeviceList[ nCount ].anDescription );
		if ( pulseGetSelectedInputDevice( psProfile ) != NULL ) {
			if ( !strcmp( asInputDeviceList[ nCount ].anName, pulseGetSelectedInputDevice( psProfile ) ) ) {
				gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBoxIn ), nIndex );
			}
		} else {
			gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBoxIn ), 0 );
		}
		nIndex++;
	}

	if ( gtk_combo_box_get_active( GTK_COMBO_BOX( psComboBoxIn ) ) < 0 ) {
		gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBoxIn ), 0 );
	}

	gtk_box_pack_start( GTK_BOX( psBox ), psComboBoxIn, FALSE, TRUE, 5 );

	gtk_widget_set_size_request( psDialog, 300, 220 );
	gtk_widget_show( GTK_WIDGET( psLabelOut ) );
	gtk_widget_show( GTK_WIDGET( psLabelIn ) );
	gtk_widget_show( GTK_WIDGET( psComboBoxOut ) );
	gtk_widget_show( GTK_WIDGET( psComboBoxIn ) );
	gtk_dialog_run( GTK_DIALOG( psDialog ) );

	prefsAddNone( getActiveProfile(), "/plugins/pulseaudio" );

	nIndex = 0;
	for ( nCount = 0; nCount < 16; nCount++ ) {
		if ( !asOutputDeviceList[ nCount ].nInitialized ) {
			break;
		}

		if ( gtk_combo_box_get_active( GTK_COMBO_BOX( psComboBoxOut ) ) == nIndex ) {
			pulseSetSelectedOutputDevice( psProfile, asOutputDeviceList[ nCount ].anName );
			break;
		}

		nIndex++;
	}

	nIndex = 0;
	for ( nCount = 0; nCount < 16; nCount++ ) {
		if ( !asInputDeviceList[ nCount ].nInitialized ) {
			break;
		}

		if ( gtk_combo_box_get_active( GTK_COMBO_BOX( psComboBoxIn ) ) == nIndex ) {
			pulseSetSelectedInputDevice( psProfile, asInputDeviceList[ nCount ].anName );
			break;
		}

		nIndex++;
	}

	gtk_widget_destroy( psDialog );
	SavePreferences( getActiveProfile() );

	// TODO: Reload driver
}

/** audio definition */
struct sAudio sPulse = {
	pulseAudioInit,
	pulseAudioOpen,
	pulseAudioWrite,
	pulseAudioRead,
	pulseAudioClose,
	pulseAudioDeinit,
	pulsePreferences
};

MODULE_INIT( PLUGIN_TYPE_AUDIO, _( "PulseAudio" ), &sPulse, NULL, pulseAudioDetectDevices );
