/*---[ firestarter.c ]------------------------------------------------
 * Copyright (C) 2000-2002 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * The main application file
 *--------------------------------------------------------------------*/

#include <config.h>
#include <gnome.h>
#include <sys/stat.h>
#include <stdio.h>
#include <netdb.h>
#include <sys/socket.h>
#include <errno.h>
#include <popt.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "globals.h"
#include "firestarter.h"
#include "gui.h"
#include "menus.h"
#include "util.h"
#include "logread.h"
#include "wizard.h"
#include "tray.h"
#include "preferences.h"
#include "scriptwriter.h"

#define RETURNED_EXT_FAILED 2
#define RETURNED_INT_FAILED 3

FirestarterApp Firestarter;

static gint save_session (GnomeClient       *client,
                          gint               phase,
                          GnomeSaveStyle     save_style,
                          gint               is_shutdown,
                          GnomeInteractStyle interact_style,
                          gint               is_fast,
                          gpointer           client_data);

static void session_die (GnomeClient        *client,
                         gpointer            client_data);

static void appbar_update (FirewallStatus status);
static void unlock (void);
static gboolean is_locked (void);
static FirewallStatus current_status;

const static gchar *
iptables_path (void) {
	static gchar *path = NULL;

	if (path == NULL) {
		if (access("/sbin/iptables", R_OK) == 0)
			path = g_strdup ("/sbin/iptables");
		else
			path = g_strdup ("`which iptables`");
	}
	
	return path;
}

/* [ stop_firewall ]
 * Flushes, zeroes and sets all policies to accept
 */
void
stop_firewall (void)
{
	gint retval;
	const gchar *ipt;
	gchar *error;
	gchar *command = NULL;

	ipt = iptables_path ();

	command = g_strconcat (ipt, " -F;",
	                       ipt, " -X;",
	                       ipt, " -Z;",
	                       ipt, " -P INPUT ACCEPT;",
	                       ipt, " -P FORWARD ACCEPT;",
	                       ipt, " -P OUTPUT ACCEPT;", NULL);
	retval = system (command);

	if (retval == 0) {
		g_print (_("Firewall stopped, network traffic is now flowing freely\n"));
		unlock ();

		if (!CONSOLE)
			update_status (STATUS_STOPPED);
	} else {
		retval = WEXITSTATUS (retval);
	
		error = g_strdup (_("Failed to stop the firewall\n\n"));
		if (!CONSOLE)
			error = g_strconcat ("<span weight=\"bold\" size=\"larger\">", error, "</span>", NULL);

		error = g_strconcat (error,
			_("There was an undetermined error when trying to stop your firewall."), NULL);

		if (CONSOLE)
			fprintf (stderr, "%s", error);
		else
			error_dialog (error);

		g_free (error);
	}

	g_free (command);
}

/* [ start_firewall ]
 * Executes the firewall script
 */
void
start_firewall (void)
{
	gint retval;
	gchar *error;
	gchar *command = g_strjoin (NULL, "/bin/sh ",
	                            FIRESTARTER_RULES_DIR "/firestarter/firewall.sh",
	                            NULL);

	retval = system (command);

	if (retval == 0) {
		g_print (_("Firewall started\n"));

		if (!CONSOLE)
			update_status (STATUS_RUNNING);
	} else {
		retval = WEXITSTATUS (retval);
	
		error = g_strdup (_("Failed to start the firewall\n\n"));
		if (!CONSOLE)
			error = g_strconcat ("<span weight=\"bold\" size=\"larger\">", error, "</span>", NULL);

		if (retval == RETURNED_EXT_FAILED) {
			error = g_strconcat (error, g_strdup_printf (_(
				"The device %s is not ready.\n\n"), 
				preferences_get_string (PREFS_FW_EXT_IF)), NULL);
		} else if (retval == RETURNED_INT_FAILED) {
			error = g_strconcat (error, g_strdup_printf (_(
				"The device %s is not ready.\n\n"), 
				preferences_get_string (PREFS_FW_EXT_IF)), NULL);
		}

		error = g_strconcat (error, _(
			"Please check your network device settings and make sure your\n"
			"Internet connection is active."), NULL);
		
		if (CONSOLE)
			fprintf (stderr, "%s", error);
		else {
			error_dialog (error);
			update_status (STATUS_STOPPED);
		}

		g_free (error);
	}

	g_free (command);
}

/* [ halt_firewall ]
 * Flushes and sets all policies to deny
 */
void
halt_firewall (void)
{
	gint retval;
	const gchar *ipt;
	gchar *error;
	gchar *command;

	ipt = iptables_path ();
	
	command = g_strconcat (ipt, " -P INPUT DROP;",
	                       ipt, " -P FORWARD DROP;",
	                       ipt, " -P OUTPUT DROP;",
	                       ipt, " -F;",
	                       ipt, " -X;",
	                       ipt, " -Z;", NULL);
	retval = system (command);

	if (retval == 0) {
		g_print (_("All traffic halted\n"));
		
		if (!CONSOLE)
			update_status (STATUS_HALTED);
	} else {
		retval = WEXITSTATUS (retval);
	
		error = g_strdup (_("Failed to enter halted firewall mode\n\n"));
		if (!CONSOLE)
			error = g_strconcat ("<span weight=\"bold\" size=\"larger\">", error, "</span>", NULL);

		error = g_strconcat (error,
			_("There was an undetermined error when trying to halt the network traffic."), NULL);

		if (CONSOLE)
			fprintf (stderr, "%s", error);
		else
			error_dialog (error);
	
		g_free (error);
	}

	g_free (command);
}

/* [ exit_firestarter ]
 * Quit firestater
 */
void
exit_firestarter (void)
{
	if (preferences_get_bool (PREFS_STOP_ON_EXIT))
		stop_firewall ();

	gtk_main_quit ();
}

/* [ save_session ]
 * Saves the current session for later revival
 */
static gint
save_session (GnomeClient       *client,
              gint               phase,
              GnomeSaveStyle     ave_style,
              gint               is_shutdown,
              GnomeInteractStyle interact_style,
              gint               is_fast,
              gpointer           client_data)
{
	gchar **argv = g_new0 (gchar*, 4);
	guint argc = 1;

	argv[0] = client_data;
	gnome_client_set_clone_command (client, argc, argv);
	gnome_client_set_restart_command (client, argc, argv);

	return TRUE;
}

/* [ session_die ]
 * Gracefully end the session
 */
static void
session_die (GnomeClient *client, gpointer client_data)
{
	exit_firestarter ();
}

void
update_status (FirewallStatus status)
{
	if (status != STATUS_HIT)
		appbar_update (status);
	tray_update (status);	

	current_status = status;
}

FirewallStatus
get_current_status (void)
{
	return current_status;
}

/* [ appbar_update ]
 * Updates the application status bar with the status of the firewall,
 */
static void
appbar_update (FirewallStatus status)
{
	if (status == STATUS_STOPPED) {
		gnome_appbar_set_default (GNOME_APPBAR (appbar), _("Firewall stopped"));
	}
	else if (status == STATUS_RUNNING) {
		gnome_appbar_set_default (GNOME_APPBAR (appbar), _("Firewall running"));
	}
	else if (status == STATUS_HALTED) {
		gnome_appbar_set_default (GNOME_APPBAR (appbar), _("All traffic halted"));
	}
}

static const gchar *
get_lock_file_path (void)
{
	static gchar *path = NULL;

	if (path == NULL) {
		DIR *d;

		if ((d = opendir ("/var/lock/subsys")) != NULL) {
			closedir (d);
			path = g_strdup ("/var/lock/subsys/firestarter");
		} else if ((d = opendir ("/var/lock")) != NULL) {
			closedir (d);
			path = g_strdup ("/var/lock/firestarter");
		} else {
			perror ("Not able to determine a lock file");
		}
	}

	return path;
}

static void
unlock (void)
{
	remove (get_lock_file_path ());
}

static gboolean
is_locked (void)
{
	FILE *f = fopen (get_lock_file_path (), "r");

	if (f != NULL) {
		fclose (f);
		return TRUE;
	} else
		return FALSE;
}

static int
check_status_timeout (gpointer data)
{
	FirewallStatus state = get_current_status ();

	if (state == STATUS_HIT)
		return TRUE;

	if (is_locked ()) {
		if (state != STATUS_HALTED)
			update_status (STATUS_RUNNING);
	} else
		update_status (STATUS_STOPPED);

	return TRUE;
}

static void
show_help (void)
{
	gchar *help = g_strdup (_(
		"Firestarter " VERSION "\n"
		"\n"
		" -s, --start            Start the firewall\n"
		" -p, --stop             Stop the firewall\n"
		"     --halt             Halt all outgoing and incoming traffic (panic mode)\n"
		"     --generate-scripts Generate firewall scripts from current configuration\n"
		" -v, --version          Prints Firestarter's version number\n"
		" -h, --help             You're looking at it\n"
	));

	fprintf (stderr, "%s", help);
	g_free (help);
}

static gboolean
is_root (void)
{
	if (getuid () != 0) {
		gchar *error = g_strdup (_("Insufficient privileges\n\n"));
		if (!CONSOLE)
			error = g_strconcat ("<span weight=\"bold\" size=\"larger\">", error, "</span>", NULL);
		error = g_strconcat (error, _("You must have root user privileges to use Firestarter.\n"), NULL);

		if (CONSOLE)
			fprintf (stderr, "%s", error);
		else
			error_dialog (error);

		g_free (error);
		return FALSE;
	}

	return TRUE;
}

/* [ main ]
 * The main function, this is where it all begins and ends
 */
int
main (int argc, char* argv[])
{
	GnomeClient *client;
	gint i;
	gboolean modal_wizard;

	/* Text domain and codeset */	
	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	/* Parse arguments */
	for (i = 0; i < argc; i++) {
		const char * arg = argv[i];

		if (!strcmp (arg, "-s") || !strcmp(arg, "--start")) {
			CONSOLE = TRUE;
			if (is_root ())
				start_firewall ();
			return 0;
		} else if (!strcmp (arg, "-p") || !strcmp(arg, "--stop")) {
			CONSOLE = TRUE;
			if (is_root ())
				stop_firewall ();
			return 0;
		} else if (!strcmp(arg, "--halt")) {
			CONSOLE = TRUE;
			if (is_root ())
				halt_firewall ();
			return 0;
		} else if (!strcmp(arg, "--generate-scripts")) {
			gnome_init ("firestarter", VERSION, argc, argv);

			CONSOLE = TRUE;
			if (is_root ())
				write_script ();
			return 0;
		} else if (!strcmp (arg, "-v") || !strcmp(arg, "--version")) {
			printf ("Firestarter %s\n", VERSION);
			return 0;
		} else if (!strcmp (arg, "-h") || !strcmp (arg, "--help") || !strcmp(arg, "-help")) {
			show_help ();
			return 0;	
		}
	}

	gnome_init ("firestarter", VERSION, argc, argv);

	/* Set up the session managment */
	client = gnome_master_client ();
	g_signal_connect (G_OBJECT (client), "save_yourself",
			  G_CALLBACK (save_session), argv[0]);
	g_signal_connect (G_OBJECT (client), "die",
			  G_CALLBACK (session_die), NULL);


	/* Check that the user is root */
	if (!is_root ())
		return 1;

	/* Check that the user is running a 2.4 or 2.5 series kernel */
	if (!detect_netfilter ()) {
		error_dialog (_("Firestarter requires a Linux 2.4 or later kernel to function.\n\n"
				"A client for your system might be available at http://firestarter.sf.net"));
		exit (1);
	}

	/* Check that a valid gconf schema is installed */
	preferences_check_schema();

	/* If we're starting Firestarter for the first time or the script is missing, create modal wizard */
	modal_wizard = (preferences_get_bool (PREFS_FIRST_RUN) || !script_exists ());

	/* Creating the GUI */
	gui_construct (!modal_wizard);

	/* Initialize the system log file polling function */
	open_logfile ((gchar *)get_system_log_path ());


	if (modal_wizard)
		run_wizard ();
	else {
		if (preferences_get_bool (PREFS_START_ON_STARTUP))
			start_firewall ();

	}

	/* Attach a timeout that updates the GUI with the true status of the firewall */
	gtk_timeout_add (5000, check_status_timeout, NULL);

	gtk_main ();

	return 0;
}
