/*---[ preferences.c ]------------------------------------------------
 * Copyright (C) 2000-2002 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * The preferences notebook window
 *--------------------------------------------------------------------*/
#include <stdio.h>
#include <errno.h>
#include <gconf/gconf-client.h>

#include "globals.h"
#include "firestarter.h"
#include "preferences.h"
#include "wizard.h"
#include "wizard-choices.h"
#include "scriptwriter.h"

enum 
{
	GENERAL_PAGE,
	FILTERS_PAGE,
	EXTERNAL_PAGE,
	SHARING_PAGE,
	SERVICES_PAGE,
	ICMP_PAGE,
	TOS_PAGE,
	ADVANCED_PAGE,
};

static GtkWidget *pbox = NULL;
static GConfClient *client = NULL;
static gboolean prefs_init = FALSE;

static GtkWidget *sections_clist;

Wizard *wizard;

/* [ preferences_init ]
 * Initialize the GConfClient
 */
void
preferences_init (void)
{
	if (prefs_init)
		return;

	client = gconf_client_get_default ();
	gconf_client_add_dir (client, "/apps/firestarter", GCONF_CLIENT_PRELOAD_NONE, NULL);

	prefs_init = TRUE;
}

/* [ preferences_check_schema ]
 * Check that a Firestarter gconf schema has been installed
 * The schema is not optional
 */
void
preferences_check_schema (void)
{
	gchar *string;

	if (!prefs_init)
		preferences_init ();    

	string = preferences_get_string (PREFS_SYSLOG_FILE);
        if (string == NULL) {
		gchar msg[] = N_("A proper configuration for Firestarter was not found. "
        	                 "If you are running Firestarter from the directory you "
	                         "built it in, run 'make install-data-local' to install a "
	                         "configuration, or simply 'make install' to install the "
	                         "whole program.\n\n"
	                         "Firestarter will now close.");
		GtkWidget *dialog;

		dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
        	                                 GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, msg);
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		exit(1);
	}
}

gboolean
preferences_get_bool (const gchar *gconf_key)
{
	if (!prefs_init)
		preferences_init ();

	return gconf_client_get_bool (client, gconf_key, NULL);
}

gchar *
preferences_get_string (const gchar *gconf_key)
{
	gchar *str;

	if (!prefs_init)
		preferences_init ();

	str = gconf_client_get_string (client, gconf_key, NULL);

	return str;
}

void
preferences_set_bool (const gchar *gconf_key, gboolean data)
{
	g_return_if_fail (gconf_key);

	if (!prefs_init)
		preferences_init ();

	gconf_client_set_bool (client, gconf_key, data, NULL);
}

void
preferences_set_string (const gchar *gconf_key, const gchar *data)
{
	g_return_if_fail (gconf_key || data);

	if (!prefs_init)
		preferences_init ();

	gconf_client_set_string (client, gconf_key, data, NULL);
}

/* [ pref_button_toggled ]
 * Callback when button changes, update GConf key
 */
static void
pref_button_toggled (GtkToggleButton *button, const gchar *gconf_key)
{
	preferences_set_bool (gconf_key, gtk_toggle_button_get_active (button));	
}

/* [ bind_toggle_to_key  ]
 * Associcate an toggle button with an GConf key
 */
void
bind_toggle_to_key (GtkToggleButton *button, const gchar *gconf_key)
{
	gboolean b;

	b = preferences_get_bool (gconf_key);
	gtk_toggle_button_set_active (button, b);

	/* Button changes, update GConf key */
	g_signal_connect (G_OBJECT (button), "toggled",
			  G_CALLBACK (pref_button_toggled), (gpointer)gconf_key);
}


/* [ bind_entry_to_key  ]
 * Associcate an text entry with an GConf key and keep them in sync
 */
void
bind_entry_to_key (GtkEntry *entry, const gchar *gconf_key)
{
/*	gchar *text;

	text = preferences_get_string (gconf_key);
	gtk_entry_set_text (entry, text);

	g_signal_connect (G_OBJECT (button), "toggled",
			  G_CALLBACK (pref_button_toggled), (gpointer)gconf_key);
			  
	gconf_client_notify_add (client, gconf_key,
	                         gconf_notify_toggle, button,
	                         NULL, NULL);			  
*/
}

/* [ save_preferences ]
 * Save the preferences to the gnome registry
 */
static void
save_preferences (GnomePropertyBox *property_box, gint page_num, gint num)
{
	/* On the last run the page is -1 */
	if (page_num == -1) {
		FirewallStatus state;

		save_choices (wizard); /* This saves the wizard imported widgets states */
		write_script ();

		state = get_current_status ();
		if (state == STATUS_RUNNING || state == STATUS_HIT)		
			start_firewall ();
	}
}

/* [ create_general_page ]
 * Create the general property page
 */
static GtkWidget*
create_general_page (GtkWidget *pbox)
{
	GtkWidget *table;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *startup;
	GtkWidget *stopatexit;
	GtkWidget *resolve_names;
	GtkWidget *tray_mode;

	table = gtk_table_new (1, 1, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 0);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	gtk_table_attach (GTK_TABLE (table), vbox, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_(
		"The general program settings let you configure the overall behavior\n"
		"of the program.\n"));
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

	startup = gtk_check_button_new_with_label (_(
		"Start firewall on program startup"));
	gtk_tooltips_set_tip (Firestarter.ttips, startup, _(
		"Check this and the firewall will be started when the "
		"program loads."), "");

	bind_toggle_to_key (GTK_TOGGLE_BUTTON (startup), PREFS_START_ON_STARTUP);
	gtk_box_pack_start (GTK_BOX (vbox), startup, FALSE, FALSE, 0);

	stopatexit = gtk_check_button_new_with_label (_(
		"Stop firewall on program exit"));
	gtk_tooltips_set_tip (Firestarter.ttips, stopatexit, _(
		"Check this and the firewall will be stopped when you exit "
		"the program."), "");
	bind_toggle_to_key (GTK_TOGGLE_BUTTON (stopatexit), PREFS_STOP_ON_EXIT);
	gtk_box_pack_start (GTK_BOX (vbox), stopatexit, FALSE, FALSE, 0);

	resolve_names = gtk_check_button_new_with_label (_("Autoresolve IP numbers to hostnames for hits"));
	gtk_tooltips_set_tip (Firestarter.ttips, resolve_names, _(
		"Check this and instead of the IP number you will see a hostname"
		" in the Hits view. This option uses some network bandwidth."), "");
	bind_toggle_to_key (GTK_TOGGLE_BUTTON (resolve_names), PREFS_AUTORESOLVE);
	gtk_box_pack_start (GTK_BOX (vbox), resolve_names, FALSE, FALSE, 0);

	tray_mode = gtk_check_button_new_with_label (_("System tray operation mode"));
	gtk_tooltips_set_tip (Firestarter.ttips, tray_mode, _(
		"When closing the program window in tray mode the program is hid "
		"instead of exited and can be restored by clicking the Firestarter system tray icon."), "");
	bind_toggle_to_key (GTK_TOGGLE_BUTTON (tray_mode), PREFS_TRAY_MODE);
	gtk_box_pack_start (GTK_BOX (vbox), tray_mode, FALSE, FALSE, 0);


	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	return table;
}

/* [ create_filters_page ]
 * Create the general property page
 */
static GtkWidget*
create_filters_page (GtkWidget *pbox)
{
	GtkWidget *table;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *redundant_filter;
	GtkWidget *broadcast_filter;

	table = gtk_table_new (1, 1, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 0);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	gtk_table_attach (GTK_TABLE (table), vbox, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_(
		"The filtering settings let you weed out excessive information\n"
		"from the Hits view.\n"));
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

	redundant_filter = gtk_check_button_new_with_label (_(
		"Filter out redundant hits"));
	gtk_tooltips_set_tip (Firestarter.ttips, redundant_filter, _(
		"Check this and the Hits view will not show identical "
		"hits that follow each other."), "");
	bind_toggle_to_key (GTK_TOGGLE_BUTTON (redundant_filter), PREFS_FILTER_REDUNDANT);
	gtk_box_pack_start (GTK_BOX (vbox), redundant_filter, FALSE, FALSE, 0);

	broadcast_filter = gtk_check_button_new_with_label (_(
		"Filter out hits not meant for your machine"));
	gtk_tooltips_set_tip (Firestarter.ttips, broadcast_filter, _(
		"Check this and the hits list will not show hits "
		"that have a destination field other than your IP."), "");
	bind_toggle_to_key (GTK_TOGGLE_BUTTON (broadcast_filter), PREFS_FILTER_BROADCAST);
	gtk_box_pack_start (GTK_BOX (vbox), broadcast_filter, FALSE, FALSE, 0);


	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	return (table);
}

static GtkWidget*
create_advanced_page (GtkWidget *pbox)
{
	GtkWidget *frame;
	GtkWidget *table;
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *selectdeny;
	GtkWidget *selectreject;	
	GtkWidget *experiments;

	table = gtk_table_new (3, 1, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 0);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_table_attach (GTK_TABLE (table), vbox, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_(
		"The advanced settings let the experienced user fine tune\n"
		"the program operation.\n"));
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

	experiments = gtk_check_button_new_with_label (_("Enable Experimental Options"));
	gtk_tooltips_set_tip (Firestarter.ttips, experiments, _(
		"If you enable this, Firestarter can use some experimental patches for filtering. "
		"Do not enable this option until you have read the manual for the prerequisites."), "");
	gtk_box_pack_start (GTK_BOX (vbox), experiments, FALSE, FALSE, 0);

	frame = gtk_frame_new (_("Preferred packet rejection method"));
	gtk_table_attach (GTK_TABLE (table), frame, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER (frame), vbox);

	selectdeny = gtk_radio_button_new_with_label (NULL, _("Deny"));
	gtk_tooltips_set_tip (Firestarter.ttips, selectdeny, _(
		"Silently ignore the rejected packets."), "");
	bind_toggle_to_key (GTK_TOGGLE_BUTTON (selectdeny), PREFS_FW_DENY_PACKETS);
	gtk_box_pack_start (GTK_BOX (vbox), selectdeny, FALSE, FALSE, 0);

	selectreject = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (selectdeny),
		_("Reject"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (selectreject),
				      !preferences_get_bool (PREFS_FW_DENY_PACKETS));	
	gtk_tooltips_set_tip (Firestarter.ttips, selectreject, _(
		"Reply with a notice of rejection to the sender."), "");
	gtk_box_pack_start (GTK_BOX (vbox), selectreject, FALSE, FALSE, 0);

	g_signal_connect_object (G_OBJECT (selectdeny), "toggled",
		G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (selectreject), "toggled",
		G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	

	return table;
}

static void
prefs_select_row_cb (GtkCList *clist,
                     gint row,
                     gint column,
                     GdkEventButton *event,
                     GtkWidget *pbox)
{
        gtk_notebook_set_page (GTK_NOTEBOOK (
		GNOME_PROPERTY_BOX (pbox)->notebook), row);
}

static int
prefs_add_section_page (gchar     *name,
                        GtkWidget *widget,
			GtkWidget *pbox)
{
	gchar *text[2] = {name, NULL};
	gtk_notebook_append_page (GTK_NOTEBOOK (GNOME_PROPERTY_BOX (pbox)->notebook),
				  widget, gtk_label_new (name));
	return gtk_clist_append (GTK_CLIST (sections_clist), text);
}

static void
preferences_show_help (GnomePropertyBox *property_box, gint page_num)
{
	gnome_url_show ("http://firestarter.sourceforge.net/manual/preferences.php", NULL);
}

static gboolean
preferences_cleanup (void)
{
	pbox = NULL;
	return FALSE;
}

/* [ preferences_cb ]
 * Create the preferences property_box
 */
void
preferences_cb (GtkWidget *widget, gpointer data)
{
	GtkWidget *hbox;

	if (pbox != NULL) {
		g_assert (GTK_WIDGET_REALIZED (pbox));
		gdk_window_show (pbox->window);
		gdk_window_raise (pbox->window);

		return;
	}

	printf ("GENERATING\n");

	pbox = gnome_property_box_new ();
	gtk_window_set_title (GTK_WINDOW (pbox), _("Firestarter: Preferences"));

	hbox = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);

	sections_clist = gtk_clist_new (1);
	gtk_clist_set_selection_mode (GTK_CLIST (sections_clist),
				      GTK_SELECTION_BROWSE);

	g_signal_connect (G_OBJECT (sections_clist), "select_row",
			  G_CALLBACK (prefs_select_row_cb), pbox);

	prefs_add_section_page (_("General"), create_general_page (pbox), pbox);
	prefs_add_section_page (_("Filters"), create_filters_page (pbox), pbox);
	
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (GNOME_PROPERTY_BOX (pbox)->notebook), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (GNOME_PROPERTY_BOX (pbox)->notebook), FALSE);

	gtk_widget_ref (GNOME_PROPERTY_BOX (pbox)->notebook);

	gtk_container_remove (GTK_CONTAINER (GNOME_DIALOG (pbox)->vbox),
			      GNOME_PROPERTY_BOX (pbox)->notebook);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG(pbox)->vbox),
			    hbox, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (hbox),
			    sections_clist,
			    TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (hbox),
			    GNOME_PROPERTY_BOX (pbox)->notebook,
			    TRUE, TRUE, 0);

	/* Import pages from wizard */
	wizard = g_new (Wizard, 1);
	prefs_add_section_page (_("External Device"), create_device_page (wizard), pbox);
	prefs_add_section_page (_("Connection Sharing"), create_masq_page (wizard), pbox);
	prefs_add_section_page (_("Services"), create_services_page (wizard), pbox);
	prefs_add_section_page (_("ICMP Filtering"), create_icmp_page (wizard), pbox);
	prefs_add_section_page (_("Type of Service"), create_tos_page (wizard), pbox);

	/* Widgets imported from wizard that can affect the script must be registered
	   with the property box */

	g_signal_connect_object (G_OBJECT (wizard->extdevice), "changed", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->pppcheck), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->dhcpcheck), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	


	g_signal_connect_object (G_OBJECT (wizard->masq), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->auto_int_ip), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->intdevice), "changed", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->intrange), "changed", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);

	g_signal_connect_object (G_OBJECT (wizard->services), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->ftp), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->ssh), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->telnet), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->smtp), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->dns), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->finger), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->www), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->sslweb), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->pop), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->imap), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->ident), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->nntp), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->ntp), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->samba), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->ipsec), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->routed), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->nfs), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->x), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->dhcp), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->upnp), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->pptp), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);
	
	g_signal_connect_object (G_OBJECT (wizard->icmp), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->icmp_echo), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->icmp_traceroute), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->icmp_tracert), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->icmp_unreach), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->icmp_timestamp), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->icmp_masking), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->icmp_redir), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->icmp_quench), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	

	g_signal_connect_object (G_OBJECT (wizard->tos), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->tos_client), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->tos_server), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->tos_X), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->tos_option_throughput), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->tos_option_reliability), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);	
	g_signal_connect_object (G_OBJECT (wizard->tos_option_delay), "toggled", G_CALLBACK (gnome_property_box_changed), G_OBJECT (pbox), G_CONNECT_SWAPPED);
	prefs_add_section_page (_("Advanced"), create_advanced_page (pbox), pbox);

	gtk_clist_columns_autosize (GTK_CLIST (sections_clist) );

	gtk_widget_unref (GNOME_PROPERTY_BOX (pbox)->notebook);

	g_signal_connect (G_OBJECT (pbox), "apply",
	                  G_CALLBACK (save_preferences), NULL);

	g_signal_connect (G_OBJECT (pbox), "help",
	                  G_CALLBACK (preferences_show_help), NULL);

	g_signal_connect (G_OBJECT (pbox), "destroy",
			  G_CALLBACK (preferences_cleanup), &pbox);

	load_choices (wizard);

	gtk_widget_show_all (pbox);
}
