/*---[ scriptwriter.c ]-----------------------------------------------
 * Copyright (C) 2000-2002 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Creates the firewall script, based on wizard selections.
 *--------------------------------------------------------------------*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>

#include "firestarter.h"
#include "globals.h"
#include "wizard.h"
#include "util.h"
#include "scriptwriter.h"
#include "netfilter-script.h"
#include "preferences.h"

static gchar *FIRESTARTER_HOOK = "PATH=$PATH:/sbin:/usr/sbin:/bin:/usr/bin\nsh " FIRESTARTER_RULES_DIR "/firestarter/firewall.sh\n";

/* [ script_exists ]
 * Return true if script has been generated
 */
gboolean
script_exists (void)
{
	struct stat statd;
	stat (FIRESTARTER_RULES_DIR "/firestarter/firewall.sh", &statd);
	return (statd.st_size != 0);
}

static gboolean
file_exists (gchar *path)
{
	gboolean exists;
	FILE *f = fopen (path, "r");

	exists = (f != NULL);

	if (exists)
		fclose (f);

	return exists;
}

static gboolean
dhclient_is_running (void)
{
	gboolean exists;
	gchar *path = g_strconcat ("/var/run/dhclient-",
				preferences_get_string (PREFS_FW_EXT_IF),
				".pid", NULL);

	exists = file_exists (path);
	g_free (path);

	return exists;
}

static gboolean
dhcpcd_is_running (void)
{
	gboolean exists;
	gchar *path;
	
	if (file_exists ("/etc/slackware-version")) {
		path = g_strconcat ("/etc/dhcpc/dhcpcd-",
			 preferences_get_string (PREFS_FW_EXT_IF),
			 ".pid", NULL);
	} else {
		path = g_strconcat ("/var/run/dhcpcd-",
			 preferences_get_string (PREFS_FW_EXT_IF),
			 ".pid", NULL);
	}

	exists = file_exists (path);
	g_free (path);

	return exists;
}

gboolean
dhcp_client_enabled (void)
{
	return (dhclient_is_running () || dhcpcd_is_running ());
}

static void
append_hook_to_script (FILE *f)
{
	gchar buf[512];
	GList *list = NULL;
	GList *p;

	while (fgets (buf, 512, f) != NULL) {
		if (strstr (buf, FIRESTARTER_HOOK))
			return;
		else
			list = g_list_append (list, g_strdup (buf));
	}

	rewind (f);
	fprintf (f, FIRESTARTER_HOOK);

	p = list;
	while (p != NULL) {
		fprintf (f, p->data);
		p = p->next;
	}
}

static void
remove_hook (gchar *path)
{
	FILE *f;
	gchar buf[512];
	GList *list = NULL;
	GList *link = NULL;
	gint pos = 0;

	f = fopen (path, "r");

	if (f == NULL) {
/*		perror (g_strconcat ("Could not remove firestarter hook in ", path, NULL)); */
		return;
	}

	while (fgets (buf, 512, f) != NULL) {
		list = g_list_append (list, g_strdup (buf));
		if (strstr (buf, FIRESTARTER_HOOK))
			link = g_list_nth (list, pos);
			
		pos++;
	}

	fclose (f);

	list = g_list_remove_link (list, link);

	f = fopen (path, "w");

	if (f == NULL) {
/*		perror (g_strconcat ("Could not remove firestarter hook in ", path, NULL)); */
		return;
	}

	link = list;
	while (link != NULL) {
		fprintf (f, link->data);
		link = link->next;
	}

	fclose (f);
}

static void
add_hook (gchar *path)
{
	FILE *f;

	printf ("Adding Firestarter startup hook to %s\n", path);

	if (file_exists (path)) {
		f = fopen (path, "r+");

		if (f == NULL) {
			perror ("Could not append firestarter DHCP hook");
			return;
		}

		append_hook_to_script (f);
		fclose (f);

	} else {
		f = fopen (path, "w");

		if (f == NULL) {
			perror ("Could not write firestarter DHCP hook");
			return;
		}

		fprintf (f, FIRESTARTER_HOOK);
		fclose (f);
	}
}


void
write_ppp_hook (void)
{
	gchar *path = "/etc/ppp/ip-up.local";

	if (!file_exists ("/etc/ppp")) {
		printf ("No ppp detected on system. Not adding starting hook\n");
		return;
	}

	add_hook (path);
}

void
remove_ppp_hook (void)
{
	gchar *path = "/etc/ppp/ip-up.local";

	if (!file_exists ("/etc/ppp/ip-up.local")) {
		return;
	}

	remove_hook (path);
}

void
write_dhcp_hook (void)
{
	/* Red Hat 8+, some Mandrake 9 configurations use dhclient */
	if (dhclient_is_running ()) {
		gchar *path = g_strdup ("/etc/dhclient-exit-hooks");

		add_hook (path);
		g_free (path);

	/* Slackware uses DHCPCD, but it's path is different */
	} else if (dhcpcd_is_running () && file_exists ("/etc/slackware-version")) {
		gchar *path = g_strconcat ("/etc/dhcpc/dhcpcd-",
					   preferences_get_string (PREFS_FW_EXT_IF),
					   ".exe", NULL);

		add_hook (path);
		g_free (path);

	/* Most other distributions use DHCPCD */
	} else if (dhcpcd_is_running ()) {
		gchar *path = g_strconcat ("/etc/dhcpcd/dhcpcd-",
					   preferences_get_string (PREFS_FW_EXT_IF),
					   ".exe", NULL);

		add_hook (path);
		g_free (path);
	} else {
		perror (_("No DHCP client configuration found\n\n"
			  "The firewall will not be loaded automatically on a lease renewal. "
			  "Please make sure the external device is configured properly "
			  "or disable the DHCP option if you are using static settings.\n"));
	}
}

void
remove_dhcp_hook (void)
{
	/* Red Hat 8+, some Mandrake 9 configurations use dhclient */
	if (dhclient_is_running ()) {
		gchar *path = g_strdup ("/etc/dhclient-exit-hooks");

		remove_hook (path);
		g_free (path);

	/* Most other distributions use DHCPCD */
	} else if (dhcpcd_is_running ()) {
		gchar *path = g_strconcat ("/etc/dhcpcd/dhcpcd-",
					   preferences_get_string (PREFS_FW_EXT_IF),
					   ".exe", NULL);

		remove_hook (path);
		g_free (path);
	}
}

/* [ check_file ]
 * Check that file exists, if not, create
 */
static void
check_file (const gchar *path)
{
	FILE *file = NULL;

	if ((fopen (path, "r") == NULL) && (errno == ENOENT)) {
	        if ((file = fopen (path, "w")) != NULL) {
			chmod (path, 00700);
			fclose (file);
        	}
	}
}

/* [ create_rules_files ]
 * Create the empty modrules and user scripts, unless already exists.
 */
static void
create_rules_files (void)
{
	check_file (get_file_path (RULETYPE_TRUSTED_HOST));
	check_file (get_file_path (RULETYPE_BLOCKED_HOST));
	check_file (get_file_path (RULETYPE_FORWARD));
	check_file (get_file_path (RULETYPE_OPEN_PORT));
	check_file (get_file_path (RULETYPE_STEALTHED_PORT));
	check_file (get_file_path (RULETYPE_BLOCKED_PORT));
	check_file (FIRESTARTER_RULES_DIR "/firestarter/user-pre");
	check_file (FIRESTARTER_RULES_DIR "/firestarter/user-post");
}

/* [ write_script ]
 * Creates the shell script, called when the finish button is pressed
 */
void
write_script (void)
{
	gboolean script_existed = script_exists(); /* Script existed before we write it*/
	/* Creating the dir for storing scripts if it doesn't exist */
	mkdir (FIRESTARTER_RULES_DIR "/firestarter", 00700);

	write_netfilter_script ();

	create_rules_files ();

	/* Start firewall on ppp interface up */
	if (preferences_get_bool (PREFS_FW_EXT_PPP))
		write_ppp_hook ();
	else
		remove_ppp_hook ();

	/* Start firewall on DCHP lease renewal */
	if (preferences_get_bool (PREFS_FW_EXT_DHCP))
		write_dhcp_hook ();
	else
		remove_dhcp_hook ();

	if (preferences_get_bool (PREFS_FIRST_RUN) || !script_existed) {
		/* Mark that the wizard has been run at least once */
		preferences_set_bool (PREFS_FIRST_RUN, FALSE);

		/* Show the main interface */
		gtk_widget_show_all (Firestarter.window);

		/* Finally, start up the newly created firewall */
                start_firewall ();
	}
}
