##Module: pog

##    Fonty Python Copyright  (C)  2006 Donn.C.Ingle
##
##    Contact: donn.ingle@gmail.com - I hope this email lasts.
##
##    This program is free software; you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation; either version 2 of the License, or
##     ( at your option )  any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program; if not, write to the Free Software
##    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##

from basicfontlist import *
from pathcontrol import * 
import fontybugs

class Pog ( BasicFontList ):
    """Controller.
    It represents an entire Pog. 
    Contains a list of pafs to ttf files in a given pog.
    Supply the pog name.
    Must call genList () if you want actual font items in the list."""
    def __init__ ( self, name, progressCallback = None):
        BasicFontList.__init__ ( self )
        ##private props
        self.__progressCallback = progressCallback
        self.__pc = PathControl (  ) # My own path control.
        
        ## Public properties:
        self.name = name
        self.__installed = "dirty" #am I installed?
        #self.empty = None #do I have fonts in me?
        self.paf = self.__pc.appPath (  )  + self.name + ".pog"
        self.badpog = False #To be used mainly to draw icons and ask user to purge.
        
    def label ( self ):
        return str ( self.name )
        
    def __doProgressCallback ( self, message ):
        if self.__progressCallback: self.__progressCallback ( message )
        else: print message
    
    def __openfile ( self ):
        """Open my pog file. Raise PogInvalid error or return a file handle"""
        ## NB: NO error is raised if an empty file is opened and read...
        ## If there is some chronic hard drive problem, I assume Python will quit anyway...
        f = open ( self.paf, "r" )
        ## Let's see what kind of line 1 we have
        line1 = f.readline ( ) [:-1]
        self.__installed = "dirty" # unsure as to the status
        if line1.upper (  )  == "INSTALLED": self.__installed = "yes"
        if line1.upper (  )  == "NOT INSTALLED": self.__installed = "no"
        if self.__installed == "dirty":
            ## We have a bad pog.
            self.__renameBadPog ( )
            raise fontybugs.PogInvalid ( self.paf )
        ## At this point, we have a valid pog file:
        ## It has a valid line 1
        ## It may or may not have paf lines below it.
        return f
        
        
    def isInstalled ( self ):
        """Will ONLY raise a PogInvalid error. Any other will abort app."""
        if self.__installed == "yes": return True
        if self.__installed == "no": return False
        ## Else it == "dirty" and:
        ## We must open the file to discover the status:
        ## Will raise an error, so don't handle it, let it propogate upwards.
        f = self.__openfile ( ) #sets __installed flag
        f.close ( )
        if self.__installed == "yes": return True
        if self.__installed == "no": return False        
        
    def __renameBadPog ( self ):
        ## This is a bad pog, My plan is to rename it out of the .pog namespace
        newpaf =  self.paf [:-4] + ".badpog" #kick out the .pog and append .badpog
        #print "Invalid Pog : \"%s\"\nRenaming it to \"%s\"" % ( self.paf, newpaf )
        os.rename ( self.paf, newpaf )
        self.paf = newpaf
        
    ####
    ## generate the list
    ## (FontItem class is in basicfontlist.py)
    def genList ( self ):
        """Access the disk. Build the object up. All attribs and the list of fonts."""
        f = self.__openfile ( ) #sets install flag, raises PogInvalid error.
        self.clear ( )
        for paf in f: #This continues from line 2 onwards ...
            paf = paf[:-1] #Strip the damn \n from the file
            fi = FontItem ( paf, ticked =  False )  #Make an item, unticked
            self.append ( fi ) 
        f.close (  )      

    ####
    ## Purge method - remove fonts in the pog that are not on disk.
    ##
    ## Raises
    ##          PogEmpty
    ##          PogInstalled
    def purge ( self ):
        ## can't purge an empty pog
        if len ( self ) == 0:
            raise fontybugs.PogEmpty # RAISED :: PogEmpty
        ## can't purge an installed pog
        if self.__installed == "yes":
            raise fontybugs.PogInstalled # RAISED :: PogInstalled
        else:
            ## Let's build a new list of all the bad font items.
            badfonts = []
            for i in self:
                try: #prevent weird errors on path test...
                    if not os.path.exists ( i.paf ) :
                        badfonts.append ( i ) 
                except:
                    pass # it's a bad through and through! It'll be axed too.
            ## Now go thru this list and remove the bad items.
            for bi in badfonts:
                self.remove ( bi ) 
            self.write ( ) 

    ####
    ## Install the fonts in myself to the user's fonts folder.
    ## NOTE:
    ## Even if ONLY ONE font out of a gigazillion in the pog actually installs, the POG = INSTALLED.
    ## If we have a font that cannot be sourced, flag BADPOG
    ##
    ## Raises:
    ##          PogEmpty
    ##          PogAllFontsFailedToInstall
    ##          PogSomeFontsDidNotInstall
    def install ( self ):
        ## We start thinking all is rosey:
        self.__installed = "yes"
        self.__badpog = False
        ## Now we make sure ...
        if len ( self ) == 0: 
            self.__installed = "no"
            raise fontybugs.PogEmpty ( self.name ) # RAISED :: PogEmpty
        ## Now we go through the guts of the pog, font by font:
        bugs = 0
        for fi in self:
            linkDestination = os.path.join ( self.__pc.userFontPath ( ), fi.name  ) 
            #Link it if it ain't already there.
            if not os.path.exists ( linkDestination ) :  
                if os.path.exists ( fi.paf ):
                    os.symlink ( fi.paf, linkDestination )  #Should do the trick.
                    self.__doProgressCallback ( "installed : %s" % fi.paf )
                else:
                    self.__doProgressCallback ( "Could not find %s" % fi.paf )
                    bugs += 1
        if bugs == len ( self ): # There was 100% failure to install fonts.
            ## We flag ourselves as NOT INSTALLED
            self.__installed = "no"
            self.__badpog = True # For sure!
            self.write (  )
            raise fontybugs.PogAllFontsFailedToInstall ( self.name ) # RAISED :: PogAllFontsFailedToInstall
        elif bugs > 0: 
            ## Some fonts did get installed, but not all. so, we are INSTALLED
            ## We are, however, a badpog
            self.__badpog = True
            self.write (  )
            raise fontybugs.PogSomeFontsDidNotInstall ( self.name ) # RAISED :: PogSomeFontsDidNotInstall
        self.write (  )

    ####
    ## Uninstall the fonts.
    ## NOTE:
    ## If any font is NOT removed POG = INSTALLED
    ## Any links that are not found = just ignore: They could have been removed by another pog, or this
    ## could have been a bad pog anyway.
    ## NO BAD POG flag EVER.
    ##
    ## Raises:
    ##          PogEmpty
    ##          PogLinksRemain
    ##          PogNotInstalled
    def uninstall ( self ):
        if len ( self ) == 0: raise fontybugs.PogEmpty # RAISED :: PogEmpty
        bugs = 0
        if self.__installed == "yes":
            for fi in self:
                link = os.path.join ( self.__pc.userFontPath ( ) , fi.name ) #Point at the link in .fonts folder.
                ## Step one - look for the actual file (link)
                if os.path.exists ( link ):
                    try:
                        os.unlink ( link ) 
                        self.__doProgressCallback ( "removed : %s" % fi.paf )
                    except: # e.g. Permission denied [ err 13 ]
                        ## Only bugs that imply that the file is THERE but CANNOT BE REMOVED
                        ## are classified as bugs. We are making a sweeping assumption here.
                        self.__doProgressCallback ( "Could not remove %s" % link )
                        bugs += 1
            ## Okay, we are currently INSTALLED, so what is the result of the loop?
            if bugs > 0:
                ## We still have fonts in the pog that could NOT be removed, ergo we stay INSTALLED
                raise fontybugs.PogLinksRemain ( self.name )  # RAISED :: PogLinksRemain
            else:
                ## Okay - there were no problems, so we are now done.
                self.__installed = "no"
                self.write ( ) #save to disk
        else:
            ## self.__installed says we are not installed:
            raise fontybugs.PogNotInstalled ( self.name ) # RAISED :: PogNotInstalled
            
    ####
    ## Write a pog to disk.
    def write ( self ) :
        try:
            f = open ( self.paf, "w" ) 
            i = "not installed\n"
            if self.__installed == "yes":
                i = "installed\n"
            f.write ( i ) 
            #Now write the font pafs
            for i in self:
                f.write ( i.paf + "\n" ) 
            f.close (  ) 
        except:
            raise fontybugs.PogWriteError ( self.paf )

    ####
    ## Delete my pogfile, then clean myself up, ready to be destroyed.
    def delete ( self ):
        try:
            os.unlink ( self.paf )
        except:
            raise fontybugs.PogCannotDelete ( self.paf )
        self.clear ( )
        self.__installed = "no"
    
