/* FIGARO'S PASSWORD MANAGER (FPM)
 * Copyright (C) 2000 John Conneely
 * 
 * FPM is open source / free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * FPM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * passfile.c -- Routines to save and load XML files with FPM data.
 */

#include <gtk/gtk.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <string.h>

#include "fpm.h"
#include "passfile.h"
#include "fpm_crypt.h"
#include "support.h"


/* Variables for parsing tokens out of a string. */
static char *_s_token;
static char *_c_token_pos;
static int _token_end;



static void
passfile_update_key()
/* This routine goes through the entire password list and will decrypt the
 * passwords with an old key and encrypt it again with a new key.  This is
 * needed in two situations: First, if the user changes a password.  Secondly,
 * (and much more common) if the salt changes.  In practice, this routine gets
 * called whenever we save a file.
 */
{

  fpm_data* data;
  gchar plaintext[FPM_PASSWORD_LEN+1] = {0};
  GList *list;
  int ix;



//   printf("XXX passfile_update_key being called\n");
  if (old_context!=new_context)
  {
//     printf("XXX run through the list\n");
    list=g_list_first(glb_pass_list);
    while(list!=NULL)
    {
      data = list->data;
      ix = strlen(data->password) / 2;
      fpm_decrypt_field(old_context, plaintext,
	 data->password, ix);
//       printf("XXX passwd = %s\n", plaintext);
      fpm_encrypt_field(new_context, data->password,
	 plaintext, FPM_PASSWORD_LEN);
      list = g_list_next(list);
    }

    memset(plaintext, 0, FPM_PASSWORD_LEN);
    bcopy(new_context, old_context , blowfish_contextsize());
    old_salt = new_salt;
  }
//   printf("XXX passfile_update_key exit\n");
}

static void
move_backup(gchar* file_name)
/* In case of bugs, etc. I want to save old password files.  If you would
 * like to prevent this, you can set FPM_BACKUP_NUM to 0.  This is
 * HIGHLY UNRECOMMENDED while we are in beta!
 */
{
  gchar file1[512];
  gchar file2[512];
  gint i;


  for(i=glb_fpm_ini->number_backup_files;i>0;i--)
  {
    g_snprintf(file2, 512, "%s.%02d", file_name, i);
    if (i>1)
      g_snprintf(file1, 512, "%s.%02d", file_name, i-1);
    else
      g_snprintf(file1, 512, "%s", file_name);
  
    rename(file1, file2);

  }
}

static void
passfile_upd_attr(xmlNodePtr node, char *cond, char** data_ptr)
{
  
  if (!strcmp((char *)node->name, cond))
  {
    g_free(*data_ptr);
    if(node->xmlChildrenNode != NULL)
      *data_ptr=g_strdup((char *)node->xmlChildrenNode->content);
    else
      *data_ptr=g_strdup("");
  }
}

static void new_leaf(xmlDocPtr doc, xmlNodePtr item, const xmlChar *name, xmlChar *text)
{
  xmlChar *tmp;

  tmp = xmlEncodeEntitiesReentrant(doc, text);
  xmlNewChild(item, NULL, name, tmp);
  free(tmp);
}
  
static void new_leaf_encrypt
	(xmlDocPtr doc, xmlNodePtr item, const xmlChar *name, char *text)
{
  char *cipher_text;
  // printf("XXX calling new_leaf_encrypt: %s\n", text);
  cipher_text=fpm_encrypt_field_var(new_context, text);
  new_leaf(doc, item, name, (xmlChar *)cipher_text);
  g_free(cipher_text);
}

void
passfile_save(gchar* file_name)
{
  xmlDocPtr doc;
  xmlNodePtr nodelist, item;
  fpm_data* data;
  fpm_launcher* launcher;
  FILE * file;
  gchar* vstring_cipher;
  gchar* vstring_plain;
  gchar* cmd;
  gchar* tmp;
  GList * list;
  gint i;
  gchar *dia_str;

  if (glb_fpm_ini->create_backup) 
    move_backup(file_name);
  passfile_update_key();

  vstring_cipher=g_malloc0(17);
  vstring_plain=g_malloc0(9);

  doc = xmlNewDoc((xmlChar *)"1.0");
  doc->xmlRootNode = xmlNewDocNode(doc, NULL, (xmlChar *)"FPM", NULL);
  xmlSetProp(doc->xmlRootNode, (xmlChar *)"full_version", (xmlChar *)FULL_VERSION);
  xmlSetProp(doc->xmlRootNode, (xmlChar *)"min_version", (xmlChar *)MIN_VERSION);
  xmlSetProp(doc->xmlRootNode, (xmlChar *)"display_version", (xmlChar *)DISPLAY_VERSION);
  item = xmlNewChild(doc->xmlRootNode, NULL, (xmlChar *)"KeyInfo", NULL);
  xmlSetProp(item, (xmlChar *)"salt", (xmlChar *)new_salt);
  strncpy(vstring_plain, "FIGARO", 8);
  fpm_encrypt_field(new_context, vstring_cipher, vstring_plain, 8);
  xmlSetProp(item, (xmlChar *)"vstring", (xmlChar *)vstring_cipher);

  fpm_decrypt_field(new_context, vstring_plain, vstring_cipher, 8);

  nodelist = xmlNewChild(doc->xmlRootNode, NULL, (xmlChar *)"LauncherList", NULL);
  list = g_list_first(glb_launcher_list);
  
  while (list!=NULL)
  {
    launcher=list->data;
    item=xmlNewChild(nodelist, NULL, (xmlChar *)"LauncerItem", NULL);
    new_leaf(doc, item, (xmlChar *)"title", (xmlChar *)launcher->title);
    new_leaf(doc, item, (xmlChar *)"cmdline", (xmlChar *)launcher->cmdline);
    tmp=g_strdup_printf("%d", launcher->copy_user);
    new_leaf(doc, item, (xmlChar *)"copy_user", (xmlChar *)tmp);
    g_free(tmp);
    tmp=g_strdup_printf("%d", launcher->copy_password);
    new_leaf(doc, item, (xmlChar *)"copy_password", (xmlChar *)tmp);
    g_free(tmp);

    list=g_list_next(list);
  }

  nodelist = xmlNewChild(doc->xmlRootNode, NULL, (xmlChar *)"PasswordList", NULL);
  list = g_list_first(glb_pass_list);
  i=0;
  while (list!=NULL)
  {
    data = list->data;
    item = xmlNewChild(nodelist, NULL, (xmlChar *)"PasswordItem", NULL);
    new_leaf_encrypt(doc, item, (xmlChar *)"title", data->title);
    new_leaf_encrypt(doc, item, (xmlChar *)"user", data->user);
    new_leaf_encrypt(doc, item, (xmlChar *)"url", data->arg);
    new_leaf(doc, item, (xmlChar *)"password", (xmlChar *)data->password);
    new_leaf_encrypt(doc, item, (xmlChar *)"notes", data->notes);
    new_leaf_encrypt(doc, item, (xmlChar *)"category", data->category);
    new_leaf_encrypt(doc, item, (xmlChar *)"launcher", data->launcher);
  
    if (data->default_list) xmlNewChild(item, NULL, (xmlChar *)"default", NULL);

    list=g_list_next(list);
    i++;
  }
  
  file=fopen(file_name, "w");
  xmlDocDump(file, doc);
  fclose(file);
  cmd = g_strdup_printf("chmod 0600 %s", file_name);
  fpm_execute_shell(cmd);
  g_free(cmd);

  dia_str = g_strdup_printf(_("Saved %d password(s)."), i);
  fpm_statusbar_push(dia_str);

  printf("Saved %d password(s).\n", i);
  glb_dirty = FALSE;
  g_free(vstring_plain);
  g_free(vstring_cipher);
}

void
passfile_export(gchar* file_name)
{
  xmlDocPtr doc;
  xmlNodePtr nodelist, item;
  fpm_data* data;
  fpm_launcher* launcher;
  FILE * file;
  FILE * fp;
  gchar* vstring_cipher;
  gchar* vstring_plain;
  gchar* cmd;
  gchar* tmp;
  GList * list;
  gint i;
  gchar plaintext[FPM_PASSWORD_LEN+1] = {0};
  gchar *new_file_name;
  gchar *new_file_name2;
  gchar *dia_str;


  new_file_name = g_malloc0(strlen(file_name) + 10);
  new_file_name2 = g_malloc0(strlen(file_name) + 15);
  sprintf(new_file_name, "%s.export", file_name);
  sprintf(new_file_name2, "%s.export.asc", file_name);
  dia_str = g_malloc0(strlen(new_file_name) + strlen(new_file_name2) + 200);

  printf("Export file %s\n", new_file_name);

  // move_backup(file_name);
  passfile_update_key();

  vstring_cipher=g_malloc0(17);
  vstring_plain=g_malloc0(9);

  doc = xmlNewDoc((xmlChar *)"1.0");
  doc->xmlRootNode = xmlNewDocNode(doc, NULL, (xmlChar *)"FPM", NULL);
  xmlSetProp(doc->xmlRootNode, (xmlChar *)"full_version", (xmlChar *)FULL_VERSION);
  xmlSetProp(doc->xmlRootNode, (xmlChar *)"min_version", (xmlChar *)MIN_VERSION);
  xmlSetProp(doc->xmlRootNode, (xmlChar *)"display_version", (xmlChar *)DISPLAY_VERSION);
  item = xmlNewChild(doc->xmlRootNode, NULL, (xmlChar *)"KeyInfo", NULL);
  xmlSetProp(item, (xmlChar *)"salt", (xmlChar *)new_salt);
  strncpy(vstring_plain, "FIGARO", 8);
  fpm_encrypt_field(new_context, vstring_cipher, vstring_plain, 8);
  xmlSetProp(item, (xmlChar *)"vstring", (xmlChar *)vstring_cipher);

  fpm_decrypt_field(new_context, vstring_plain, vstring_cipher, 8);

  nodelist = xmlNewChild(doc->xmlRootNode, NULL, (xmlChar *)"LauncherList", NULL);
  list = g_list_first(glb_launcher_list);
  
  while (list!=NULL)
  {
    launcher=list->data;
    item=xmlNewChild(nodelist, NULL, (xmlChar *)"LauncerItem", NULL);
    new_leaf(doc, item, (xmlChar *)"title", (xmlChar *)launcher->title);
    new_leaf(doc, item, (xmlChar *)"cmdline", (xmlChar *)launcher->cmdline);
    tmp=g_strdup_printf("%d", launcher->copy_user);
    new_leaf(doc, item, (xmlChar *)"copy_user", (xmlChar *)tmp);
    g_free(tmp);
    tmp=g_strdup_printf("%d", launcher->copy_password);
    new_leaf(doc, item, (xmlChar *)"copy_password", (xmlChar *)tmp);
    g_free(tmp);

    list=g_list_next(list);
  }
  
  nodelist = xmlNewChild(doc->xmlRootNode, NULL, (xmlChar *)"PasswordList", NULL);
  list = g_list_first(glb_pass_list);
  i=0;
  if ( ( fp = fopen(new_file_name2, "w")) == NULL )
  {
  	perror(new_file_name2);
	exit(1);
  }
  while (list!=NULL)
  {
    data = list->data;
    item = xmlNewChild(nodelist, NULL, (xmlChar *)"PasswordItem", NULL);
    new_leaf(doc, item, (xmlChar *)"title", (xmlChar *)data->title);
    new_leaf(doc, item, (xmlChar *)"user", (xmlChar *)data->user);
    new_leaf(doc, item, (xmlChar *)"url", (xmlChar *)data->arg);
    fpm_decrypt_field(old_context, plaintext, data->password, FPM_PASSWORD_LEN);
    new_leaf(doc, item, (xmlChar *)"password", (xmlChar *)plaintext);
    new_leaf(doc, item, (xmlChar *)"notes", (xmlChar *)data->notes);
    new_leaf(doc, item, (xmlChar *)"category", (xmlChar *)data->category);
    new_leaf(doc, item, (xmlChar *)"launcher", (xmlChar *)data->launcher);
    fprintf(fp, "title=%s\nuser=%s\npasswd=%s\nurl=%s\nnotes=%s\n\n",  data->title, data->user, plaintext, data->arg, data->notes);
  
    if (data->default_list) xmlNewChild(item, NULL, (xmlChar *)"default", NULL);

    list=g_list_next(list);
    i++;
  }
  
  fclose(fp);
  file=fopen(new_file_name, "w");
  xmlDocDump(file, doc);
  fclose(file);
  cmd = g_strdup_printf("chmod 0600 %s", new_file_name);
  fpm_execute_shell(cmd);
  g_free(cmd);
  cmd = g_strdup_printf("chmod 0600 %s", new_file_name2);
 fpm_execute_shell(cmd);
  g_free(cmd);

  printf("Saved %d passwords.\n", i);
  glb_dirty = FALSE;
  g_free(vstring_plain);
  g_free(vstring_cipher);
  sprintf(dia_str, _("Exported clear text passwords to\n %s and\n %s"), new_file_name, new_file_name2);
  fpm_message(GTK_WINDOW(glb_win_app), dia_str, GTK_MESSAGE_INFO);
  g_free(new_file_name);
  g_free(new_file_name2);
  g_free(dia_str);
  /* Zero out plain text */
  memset(plaintext, 0, FPM_PASSWORD_LEN);

}


gint
passfile_load(gchar* file_name)
{
  xmlDocPtr doc;
  xmlNodePtr list, item, attr;
  fpm_data* data;
  fpm_launcher* launcher;
  gint i;
  //GtkStatusbar *entry;
  gchar *dia_str;


  /* Start from scratch */
  if(glb_pass_list != NULL) fpm_clear_list();

  LIBXML_TEST_VERSION
  xmlKeepBlanksDefault(0);
  doc=xmlParseFile(file_name);
  if (doc==NULL)
  {
    /* If we can't read the doc, then assume we are running for first time.*/
    old_salt=get_new_salt();
    new_salt=old_salt;
    glb_pass_list=NULL;
    glb_dirty=TRUE;
    fpm_init_launchers();
    
    return(-1);
  }


  /* Check if document is one of ours */
  g_return_val_if_fail(!xmlStrcmp(doc->xmlRootNode->name, (xmlChar *)"FPM"), -2);

  file_version = (char *)xmlGetProp(doc->xmlRootNode, (xmlChar *)"min_version");
  if (strcmp(file_version, FULL_VERSION) > 0)
  {
    g_error("Sorry, the password file cannot be read because it uses a future file format.  Please download the latest version of FPM and try again.\n");
    gtk_main_quit(); 
  }

  file_version = (char *)xmlGetProp(doc->xmlRootNode, (xmlChar *)"full_version");
 
/* Current versions of FPM encrypt all fields.  Fields other than the 
 * password are decrypted when the program reads the password file, 
 * and the password is decrypted as it is needed (to try to prevent it
 * from showing up in coredumps, etc.)  The global variable  glb_need_decrypt
 * is set by the passfile loading routines to specify that the passfile
 * was created with a version of FPM that requires this decryption.  
 * This allows backward compatibility with previous versions of FPM which
 * only encrypted passwords.
 */
 

  glb_need_decrypt = (strcmp(file_version, ENCRYPT_VERSION) >= 0);

  list=doc->xmlRootNode->xmlChildrenNode;
  old_salt = (char *)xmlGetProp(list, (xmlChar *)"salt");
  vstring = (char *)xmlGetProp(list, (xmlChar *)"vstring");
  new_salt = get_new_salt();
  glb_using_defaults=FALSE;
  
  list=list->next;

  if (list==NULL || ( strcmp((char *)list->name, "PasswordList") && strcmp((char *)list->name, "LauncherList") ))
  {
    g_error("Invalid password file.");
    gtk_main_quit();
  }

  if(!strcmp((char *)list->name, "LauncherList"))
  {
    printf("Loading launchers...\n");
    glb_launcher_list=NULL;
    item=list->xmlChildrenNode;
    while(item!=NULL)
    {
      launcher=g_malloc0(sizeof(fpm_launcher));
      launcher->title=g_strdup("");
      launcher->cmdline=g_strdup("");
      launcher->copy_user=0;
      launcher->copy_password=0;
      attr=item->xmlChildrenNode;
      while(attr!=NULL)
      {
	if(!strcmp((char *)attr->name, "title") && attr->xmlChildrenNode && attr->xmlChildrenNode->content)
	{
	  g_free(launcher->title);
	  launcher->title=g_strdup((char *)attr->xmlChildrenNode->content);
	}
	if(!strcmp((char *)attr->name, "cmdline") && attr->xmlChildrenNode && attr->xmlChildrenNode->content)
	{
	  g_free(launcher->cmdline);
	  launcher->cmdline=g_strdup((char *)attr->xmlChildrenNode->content);
	}
	if(!strcmp((char *)attr->name, "copy_user"))
	{
	  if(!strcmp((char *)attr->xmlChildrenNode->content, "1")) launcher->copy_user=1;
	  if(!strcmp((char *)attr->xmlChildrenNode->content, "2")) launcher->copy_user=2;
	}
	if(!strcmp((char *)attr->name, "copy_password"))
	{
	  if(!strcmp((char *)attr->xmlChildrenNode->content, "1")) launcher->copy_password=1;
	  if(!strcmp((char *)attr->xmlChildrenNode->content, "2")) launcher->copy_password=2;
	}
      attr=attr->next;
      }
      glb_launcher_list=g_list_append(glb_launcher_list, launcher);

      item=item->next;
    }
    fpm_create_launcher_string_list();

    /* Incurement top-level list from launcher list to password list. */
    list=list->next; 
  }
  else
  {
    fpm_init_launchers();
  }

  if (list==NULL || strcmp((char *)list->name, "PasswordList"))
  {
    g_error("Invalid password file.");
    gtk_main_quit();
  }


  i=0;
  
  item=list->xmlChildrenNode;
  while(item!=NULL)
  {

    /* Start with blank data record */
    data = g_malloc0(sizeof(fpm_data));
    data->title=g_strdup("");
    data->arg=g_strdup("");
    data->user=g_strdup("");
    data->notes=g_strdup("");
    data->category=g_strdup("");
    data->launcher=g_strdup("");
    data->default_list=0;
   
    /* Update data record with each type of attribute */
    attr=item->xmlChildrenNode;
    while(attr!=NULL)
    {
      passfile_upd_attr(attr, "title", &data->title);
      passfile_upd_attr(attr, "url", &data->arg);
//      passfile_upd_attr(attr, "arg", &data->arg);
      passfile_upd_attr(attr, "user", &data->user);
      passfile_upd_attr(attr, "notes", &data->notes);
      passfile_upd_attr(attr, "category", &data->category);
      passfile_upd_attr(attr, "launcher", &data->launcher);
      if(!strcmp((char *)attr->name, "default"))
      {
        data->default_list=1;
	glb_using_defaults=TRUE;
      }
 
      if(!strcmp((char *)attr->name, "password"))
        strncpy(data->password, (char *)attr->xmlChildrenNode->content, FPM_PASSWORD_LEN*2);

      attr=attr->next;
    }

    /* Insert item into GList */

    glb_pass_list=g_list_append(glb_pass_list, data);

    item=item->next;
    i++;
  }
  printf("Loaded %d password(s).\n", i);

  dia_str = g_strdup_printf(_("Loaded %d password(s)."), i);
  fpm_statusbar_push(dia_str);

  glb_dirty = FALSE;

  return(0);

}   

static void
set_token(char *tmps)
{
	_s_token = strdup(tmps);
	_c_token_pos = _s_token;
	_token_end = 0;
}

static void
free_token(void)
{
	free(_s_token);
}

static char *
get_token(char *delim)
{
	char *cp;
	char *scp;

	if ( _token_end )
	{
		return(NULL);
	}

	cp = _c_token_pos;
	scp = _c_token_pos;

	while (*cp)
	{
		if ( index(delim, *cp) != NULL )
		{
			*cp = '\0';
			cp++;
			_c_token_pos = cp;
			if ( strlen(scp) == 0 )
			{
				return(NULL);
			}
			return(scp);
		}
		cp++;
	}
	_token_end = 1;
	return(NULL);
}

void
passfile_import(gchar *fname)
{

	fpm_data *new_data;
	gchar plaintext[FPM_PASSWORD_LEN+1];
	gchar *dia_str;
	int row = 0;
	
	FILE *fp;
	char tmps[2048];
	char *title;
	char *userid;
	char *passwd;
	char *url;
	char *category;
	int no_data = 0;
	int num_added = 0;

    if ( ( fp = fopen(fname, "r")) != NULL )
    {
    	while( fgets(tmps, 2048, fp) != NULL )
	{

		row++;
		no_data = 0;
		set_token(tmps);
		if ( (title = get_token("\t\n")) == NULL )
		{
			title = "";
			no_data++;
		}
		if ( (userid = get_token("\t\n")) == NULL )
		{
			userid = "";
			no_data++;
		}
		if ( (passwd = get_token("\t\n")) == NULL )
		{
			passwd = "";
			no_data++;
		}
		if ( (url = get_token("\t\n")) == NULL )
		{
			url = "";
			no_data++;
		}
		if ( (category = get_token("\t\n")) == NULL )
		{
			category = "";
			no_data++;
		}

		/* Want at least 2 fields */
		if ( no_data > 3 )
		{
			/* Debugging info

			printf("%3d) (%d) title = [%s] ", row, no_data, title);
			printf("userid = [%s] ", userid);
			printf("passwd = [%s] ", passwd);
			printf("url = [%s] ", url);
			printf("category = [%s]\n", category);
			END of debugging info */

			printf("Not enough data on line %d\n", row);
			free_token();
			continue;
		}

		num_added++;
		/* Create new password entry */
		new_data = g_malloc0(sizeof(fpm_data));
		strncpy(plaintext, "", FPM_PASSWORD_LEN);
		strcpy(plaintext, passwd);
		fpm_encrypt_field(  old_context, new_data->password,
			plaintext, FPM_PASSWORD_LEN);
		/* Zero out plain text */
		memset(plaintext, 0, FPM_PASSWORD_LEN);
		new_data->title=g_strdup(title);
		new_data->arg=g_strdup(url);
		new_data->user=g_strdup(userid);
		new_data->notes=g_strdup("");
		new_data->category=g_strdup(category);
		new_data->launcher=g_strdup("");
		glb_pass_list = g_list_append(glb_pass_list, new_data);
		free_token();	/* Free up space for token parse */

	}
	fclose(fp);
     }
     dia_str = g_strdup_printf(_("Imported %d password(s)"), num_added);
     fpm_message(GTK_WINDOW(glb_win_app), dia_str, GTK_MESSAGE_INFO);
     g_free(dia_str);
     printf("Imported %d entries\n", num_added);
     passfile_save(glb_filename);
     fpm_clist_populate_cat_list();
     fpm_check_view(FPM_ALL_CAT_MSG);
}
