type token =
  | FOR_SPEC of (Cabs.cabsloc * string list * Logic_ptree.spec)
  | SPEC of (Lexing.position * string)
  | DECL of (Logic_ptree.decl list)
  | CODE_ANNOT of (Logic_ptree.code_annot * Cabs.cabsloc)
  | LOOP_ANNOT of (Logic_ptree.code_annot list * Cabs.cabsloc)
  | ATTRIBUTE_ANNOT of (string * Cabs.cabsloc)
  | IDENT of (string)
  | CST_CHAR of (int64 list * Cabs.cabsloc)
  | CST_WCHAR of (int64 list * Cabs.cabsloc)
  | CST_INT of (string * Cabs.cabsloc)
  | CST_FLOAT of (string * Cabs.cabsloc)
  | NAMED_TYPE of (string)
  | CST_STRING of (int64 list * Cabs.cabsloc)
  | CST_WSTRING of (int64 list * Cabs.cabsloc)
  | EOF
  | BOOL of (Cabs.cabsloc)
  | CHAR of (Cabs.cabsloc)
  | INT of (Cabs.cabsloc)
  | DOUBLE of (Cabs.cabsloc)
  | FLOAT of (Cabs.cabsloc)
  | VOID of (Cabs.cabsloc)
  | INT64 of (Cabs.cabsloc)
  | INT32 of (Cabs.cabsloc)
  | ENUM of (Cabs.cabsloc)
  | STRUCT of (Cabs.cabsloc)
  | TYPEDEF of (Cabs.cabsloc)
  | UNION of (Cabs.cabsloc)
  | SIGNED of (Cabs.cabsloc)
  | UNSIGNED of (Cabs.cabsloc)
  | LONG of (Cabs.cabsloc)
  | SHORT of (Cabs.cabsloc)
  | VOLATILE of (Cabs.cabsloc)
  | EXTERN of (Cabs.cabsloc)
  | STATIC of (Cabs.cabsloc)
  | CONST of (Cabs.cabsloc)
  | RESTRICT of (Cabs.cabsloc)
  | AUTO of (Cabs.cabsloc)
  | REGISTER of (Cabs.cabsloc)
  | THREAD of (Cabs.cabsloc)
  | SIZEOF of (Cabs.cabsloc)
  | ALIGNOF of (Cabs.cabsloc)
  | EQ
  | PLUS_EQ
  | MINUS_EQ
  | STAR_EQ
  | SLASH_EQ
  | PERCENT_EQ
  | AND_EQ
  | PIPE_EQ
  | CIRC_EQ
  | INF_INF_EQ
  | SUP_SUP_EQ
  | ARROW
  | DOT
  | EQ_EQ
  | EXCLAM_EQ
  | INF
  | SUP
  | INF_EQ
  | SUP_EQ
  | PLUS of (Cabs.cabsloc)
  | MINUS of (Cabs.cabsloc)
  | STAR of (Cabs.cabsloc)
  | SLASH
  | PERCENT
  | TILDE of (Cabs.cabsloc)
  | AND of (Cabs.cabsloc)
  | PIPE
  | CIRC
  | EXCLAM of (Cabs.cabsloc)
  | AND_AND of (Cabs.cabsloc)
  | PIPE_PIPE
  | INF_INF
  | SUP_SUP
  | PLUS_PLUS of (Cabs.cabsloc)
  | MINUS_MINUS of (Cabs.cabsloc)
  | RPAREN
  | LPAREN of (Cabs.cabsloc)
  | RBRACE of (Cabs.cabsloc)
  | LBRACE of (Cabs.cabsloc)
  | LBRACKET
  | RBRACKET
  | COLON
  | SEMICOLON of (Cabs.cabsloc)
  | COMMA
  | ELLIPSIS
  | QUEST
  | BREAK of (Cabs.cabsloc)
  | CONTINUE of (Cabs.cabsloc)
  | GOTO of (Cabs.cabsloc)
  | RETURN of (Cabs.cabsloc)
  | SWITCH of (Cabs.cabsloc)
  | CASE of (Cabs.cabsloc)
  | DEFAULT of (Cabs.cabsloc)
  | WHILE of (Cabs.cabsloc)
  | DO of (Cabs.cabsloc)
  | FOR of (Cabs.cabsloc)
  | IF of (Cabs.cabsloc)
  | TRY of (Cabs.cabsloc)
  | EXCEPT of (Cabs.cabsloc)
  | FINALLY of (Cabs.cabsloc)
  | ELSE
  | ATTRIBUTE of (Cabs.cabsloc)
  | INLINE of (Cabs.cabsloc)
  | ASM of (Cabs.cabsloc)
  | TYPEOF of (Cabs.cabsloc)
  | FUNCTION__ of (Cabs.cabsloc)
  | PRETTY_FUNCTION__ of (Cabs.cabsloc)
  | LABEL__
  | BUILTIN_VA_ARG of (Cabs.cabsloc)
  | ATTRIBUTE_USED of (Cabs.cabsloc)
  | BUILTIN_VA_LIST
  | BLOCKATTRIBUTE
  | BUILTIN_TYPES_COMPAT of (Cabs.cabsloc)
  | BUILTIN_OFFSETOF of (Cabs.cabsloc)
  | DECLSPEC of (Cabs.cabsloc)
  | MSASM of (string * Cabs.cabsloc)
  | MSATTR of (string * Cabs.cabsloc)
  | PRAGMA_LINE of (string * Cabs.cabsloc)
  | PRAGMA of (Cabs.cabsloc)
  | PRAGMA_EOL
  | AT_TRANSFORM of (Cabs.cabsloc)
  | AT_TRANSFORMEXPR of (Cabs.cabsloc)
  | AT_SPECIFIER of (Cabs.cabsloc)
  | AT_EXPR of (Cabs.cabsloc)
  | AT_NAME
  | LGHOST
  | RGHOST

open Parsing;;
# 48 "cil/src/frontc/cparser.mly"

open Cabs
open Cabshelper
let parse_error msg = Errorloc.parse_error msg

(*
** Expression building
*)
let smooth_expression lst =
  match lst with
    [] -> { expr_loc = cabslu; expr_node = NOTHING }
  | [expr] -> expr
  | _ ->
      let beg_loc = fst (List.hd lst).expr_loc in
      let end_loc = snd (Cilutil.list_last lst).expr_loc in
      { expr_loc = (beg_loc,end_loc); expr_node = COMMA (lst) }

(* To be called only inside a grammar rule. *)
let make_expr e =
  { expr_loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos ();
    expr_node = e }

let currentFunctionName = ref "<outside any function>"

(* Go through all the parameter names and mark them as identifiers *)
let rec findProto = function
    PROTO (d, args, _) when isJUSTBASE d ->
      List.iter (fun (_, (an, _, _, _)) -> !Lexerhack.add_identifier an) args
  | PROTO (d, _, _) -> findProto d
  | PARENTYPE (_, d, _) -> findProto d
  | PTR (_, d) -> findProto d
  | ARRAY (d, _, _) -> findProto d
  | _ -> raise Parsing.Parse_error

and isJUSTBASE = function
    JUSTBASE -> true
  | PARENTYPE (_, d, _) -> isJUSTBASE d
  | _ -> false

let announceFunctionName ((n, decl, _, _):name) =
  !Lexerhack.add_identifier n;
  (* Start a context that includes the parameter names and the whole body.
   * Will pop when we finish parsing the function body *)
  !Lexerhack.push_context ();
  (try findProto decl
   with Parsing.Parse_error ->
     parse_error "Cannot find the prototype in a function definition");
  currentFunctionName := n


let check_funspec_abrupt_clauses fname (spec,_) =
  List.iter
    (fun bhv -> List.iter
       (function
            (Cil_types.Normal | Cil_types.Exits),_ -> ()
          | (Cil_types.Breaks | Cil_types.Continues |
                 Cil_types.Returns), {Logic_ptree.lexpr_loc = (loc,_)} ->
              Cil.error_loc (loc.Lexing.pos_fname, loc.Lexing.pos_lnum)
                "Specification of function %s can only contain ensures or \
                 exits post-conditions" fname;
              raise Parsing.Parse_error
       )
       bhv.Cil_types.b_post_cond)
    spec.Cil_types.spec_behavior

let applyPointer (ptspecs: attribute list list) (dt: decl_type)
       : decl_type =
  (* Outer specification first *)
  let rec loop = function
      [] -> dt
    | attrs :: rest -> PTR(attrs, loop rest)
  in
  loop ptspecs

let doDeclaration logic_spec (loc: cabsloc) (specs: spec_elem list) (nl: init_name list)  =
  if isTypedef specs then begin
    (* Tell the lexer about the new type names *)
    List.iter (fun ((n, _, _, _), _) -> !Lexerhack.add_type n) nl;
    TYPEDEF ((specs, List.map (fun (n, _) -> n) nl), loc)
  end else
    if nl = [] then
      ONLYTYPEDEF (specs, loc)
    else begin
      (* Tell the lexer about the new variable names *)
      List.iter (fun ((n, _, _, _), _) -> !Lexerhack.add_identifier n) nl;
      !Lexerhack.push_context ();
      List.iter
        (fun ((_,t,_,_),_) ->
           try findProto t with Parsing.Parse_error -> ())
        nl;
      let logic_spec =
        match logic_spec with
            None -> None
          | Some ls ->
	      Cabshelper.continue_annot loc
		(fun () ->
                   let spec = Logic_lexer.spec ls in
                   let name =
                     match nl with
                         [ (n,_,_,_),_ ] -> n
                       | _ -> "unknown function"
                   in
                   check_funspec_abrupt_clauses name spec;
                   Some spec)
		(fun () -> None)
		"Skipping annotation"
      in
      !Lexerhack.pop_context ();
      DECDEF (logic_spec, (specs, nl), loc)
    end


let doFunctionDef spec (loc: cabsloc)
                  (lend: cabsloc)
                  (specs: spec_elem list)
                  (n: name)
                  (b: block) : definition =
  let fname = (specs, n) in
  let name = match n with (n,_,_,_) -> n in
  Extlib.may_map ~dft:() (check_funspec_abrupt_clauses name) spec;
  FUNDEF (spec, fname, b, loc, lend)

let doOldParDecl (names: string list)
                 ((pardefs: name_group list), (isva: bool))
    : single_name list * bool =
  let findOneName n =
    (* Search in pardefs for the definition for this parameter *)
    let rec loopGroups = function
        [] -> ([SpecType Tint], (n, JUSTBASE, [], cabslu))
      | (specs, names) :: restgroups ->
          let rec loopNames = function
              [] -> loopGroups restgroups
            | ((n',_, _, _) as sn) :: _ when n' = n -> (specs, sn)
            | _ :: restnames -> loopNames restnames
          in
          loopNames names
    in
    loopGroups pardefs
  in
  let args = List.map findOneName names in
  (args, isva)

let checkConnective (s : string) : unit =
begin
  (* checking this means I could possibly have more connectives, with *)
  (* different meaning *)
  if (s <> "to") then parse_error "transformer connective must be 'to'";
end

let int64_to_char value =
  if (Int64.compare value (Int64.of_int 255) > 0) ||
    (Int64.compare value Int64.zero < 0) then
    parse_error (Printf.sprintf "integral literal 0x%Lx too big" value)
  else
    Char.chr (Int64.to_int value)

(* takes a not-nul-terminated list, and converts it to a string. *)
let rec intlist_to_string (str: int64 list):string =
  match str with
    [] -> ""  (* add nul-termination *)
  | value::rest ->
      let this_char = int64_to_char value in
      (String.make 1 this_char) ^ (intlist_to_string rest)

let fst3 (result, _, _) = result
let snd3 (_, result, _) = result
let trd3 (_, _, result) = result

let fourth4 (_,_,_,result) = result

(*
   transform:  __builtin_offsetof(type, member)
   into     :  (size_t) (&(type * ) 0)->member
 *)

let transformOffsetOf (speclist, dtype) member =
  let mk_expr e = { expr_loc = member.expr_loc; expr_node = e } in
  let rec addPointer = function
    | JUSTBASE ->
	PTR([], JUSTBASE)
    | PARENTYPE (attrs1, dtype, attrs2) ->
	PARENTYPE (attrs1, addPointer dtype, attrs2)
    | ARRAY (dtype, attrs, expr) ->
	ARRAY (addPointer dtype, attrs, expr)
    | PTR (attrs, dtype) ->
	PTR (attrs, addPointer dtype)
    | PROTO (dtype, names, variadic) ->
	PROTO (addPointer dtype, names, variadic)
  in
  let nullType = (speclist, addPointer dtype) in
  let nullExpr = mk_expr (CONSTANT (CONST_INT "0")) in
  let castExpr = mk_expr (CAST (nullType, SINGLE_INIT nullExpr)) in

  let rec replaceBase e =
    let node = match e.expr_node with
      | VARIABLE field ->
	  MEMBEROFPTR (castExpr, field)
      | MEMBEROF (base, field) ->
	  MEMBEROF (replaceBase base, field)
      | INDEX (base, index) ->
	  INDEX (replaceBase base, index)
      | _ ->
	  parse_error "malformed offset expression in __builtin_offsetof"
    in { e with expr_node = node }
  in
  let memberExpr = replaceBase member in
  let addrExpr = { memberExpr with expr_node = UNARY (ADDROF, memberExpr)} in
  (* slight cheat: hard-coded assumption that size_t == unsigned int *)
  let sizeofType = [SpecType Tunsigned], JUSTBASE in
  { addrExpr with expr_node = CAST (sizeofType, SINGLE_INIT addrExpr)}

let no_ghost_stmt s = {stmt_ghost = false ; stmt_node = s}

let no_ghost = List.map no_ghost_stmt

let in_ghost =
  let ghost_me = object
    inherit Cabsvisit.nopCabsVisitor
    method vstmt s =
      s.stmt_ghost <- true;
      Cil.DoChildren
  end
  in
  List.map
    (fun s -> ignore (Cabsvisit.visitCabsStatement ghost_me s); s)

let in_block l =
  match l with
      [] -> no_ghost_stmt (NOP cabslu)
    | [s] -> s
    | _::_ ->
        no_ghost_stmt (BLOCK ({ blabels = []; battrs = []; bstmts = l},
                              get_statementloc (List.hd l),
                              get_statementloc (Cilutil.list_last l)))

# 369 "cil/src/frontc/cparser.ml"
let yytransl_const = [|
    0 (* EOF *);
  297 (* EQ *);
  298 (* PLUS_EQ *);
  299 (* MINUS_EQ *);
  300 (* STAR_EQ *);
  301 (* SLASH_EQ *);
  302 (* PERCENT_EQ *);
  303 (* AND_EQ *);
  304 (* PIPE_EQ *);
  305 (* CIRC_EQ *);
  306 (* INF_INF_EQ *);
  307 (* SUP_SUP_EQ *);
  308 (* ARROW *);
  309 (* DOT *);
  310 (* EQ_EQ *);
  311 (* EXCLAM_EQ *);
  312 (* INF *);
  313 (* SUP *);
  314 (* INF_EQ *);
  315 (* SUP_EQ *);
  319 (* SLASH *);
  320 (* PERCENT *);
  323 (* PIPE *);
  324 (* CIRC *);
  327 (* PIPE_PIPE *);
  328 (* INF_INF *);
  329 (* SUP_SUP *);
  332 (* RPAREN *);
  336 (* LBRACKET *);
  337 (* RBRACKET *);
  338 (* COLON *);
  340 (* COMMA *);
  341 (* ELLIPSIS *);
  342 (* QUEST *);
  357 (* ELSE *);
  364 (* LABEL__ *);
  367 (* BUILTIN_VA_LIST *);
  368 (* BLOCKATTRIBUTE *);
  376 (* PRAGMA_EOL *);
  381 (* AT_NAME *);
  382 (* LGHOST *);
  383 (* RGHOST *);
    0|]

let yytransl_block = [|
  257 (* FOR_SPEC *);
  258 (* SPEC *);
  259 (* DECL *);
  260 (* CODE_ANNOT *);
  261 (* LOOP_ANNOT *);
  262 (* ATTRIBUTE_ANNOT *);
  263 (* IDENT *);
  264 (* CST_CHAR *);
  265 (* CST_WCHAR *);
  266 (* CST_INT *);
  267 (* CST_FLOAT *);
  268 (* NAMED_TYPE *);
  269 (* CST_STRING *);
  270 (* CST_WSTRING *);
  271 (* BOOL *);
  272 (* CHAR *);
  273 (* INT *);
  274 (* DOUBLE *);
  275 (* FLOAT *);
  276 (* VOID *);
  277 (* INT64 *);
  278 (* INT32 *);
  279 (* ENUM *);
  280 (* STRUCT *);
  281 (* TYPEDEF *);
  282 (* UNION *);
  283 (* SIGNED *);
  284 (* UNSIGNED *);
  285 (* LONG *);
  286 (* SHORT *);
  287 (* VOLATILE *);
  288 (* EXTERN *);
  289 (* STATIC *);
  290 (* CONST *);
  291 (* RESTRICT *);
  292 (* AUTO *);
  293 (* REGISTER *);
  294 (* THREAD *);
  295 (* SIZEOF *);
  296 (* ALIGNOF *);
  316 (* PLUS *);
  317 (* MINUS *);
  318 (* STAR *);
  321 (* TILDE *);
  322 (* AND *);
  325 (* EXCLAM *);
  326 (* AND_AND *);
  330 (* PLUS_PLUS *);
  331 (* MINUS_MINUS *);
  333 (* LPAREN *);
  334 (* RBRACE *);
  335 (* LBRACE *);
  339 (* SEMICOLON *);
  343 (* BREAK *);
  344 (* CONTINUE *);
  345 (* GOTO *);
  346 (* RETURN *);
  347 (* SWITCH *);
  348 (* CASE *);
  349 (* DEFAULT *);
  350 (* WHILE *);
  351 (* DO *);
  352 (* FOR *);
  353 (* IF *);
  354 (* TRY *);
  355 (* EXCEPT *);
  356 (* FINALLY *);
  358 (* ATTRIBUTE *);
  359 (* INLINE *);
  360 (* ASM *);
  361 (* TYPEOF *);
  362 (* FUNCTION__ *);
  363 (* PRETTY_FUNCTION__ *);
  365 (* BUILTIN_VA_ARG *);
  366 (* ATTRIBUTE_USED *);
  369 (* BUILTIN_TYPES_COMPAT *);
  370 (* BUILTIN_OFFSETOF *);
  371 (* DECLSPEC *);
  372 (* MSASM *);
  373 (* MSATTR *);
  374 (* PRAGMA_LINE *);
  375 (* PRAGMA *);
  377 (* AT_TRANSFORM *);
  378 (* AT_TRANSFORMEXPR *);
  379 (* AT_SPECIFIER *);
  380 (* AT_EXPR *);
    0|]

let yylhs = "\255\255\
\001\000\002\000\003\000\003\000\003\000\003\000\050\000\052\000\
\052\000\052\000\004\000\004\000\004\000\004\000\004\000\004\000\
\004\000\004\000\004\000\004\000\004\000\004\000\055\000\055\000\
\056\000\056\000\057\000\057\000\058\000\058\000\058\000\058\000\
\058\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
\059\000\059\000\059\000\059\000\060\000\060\000\060\000\062\000\
\062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
\062\000\062\000\062\000\062\000\062\000\063\000\063\000\064\000\
\064\000\064\000\064\000\065\000\065\000\065\000\066\000\066\000\
\066\000\067\000\067\000\067\000\067\000\067\000\068\000\068\000\
\068\000\069\000\069\000\070\000\070\000\071\000\071\000\072\000\
\072\000\073\000\073\000\074\000\074\000\075\000\075\000\075\000\
\075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
\075\000\010\000\008\000\008\000\008\000\008\000\008\000\008\000\
\009\000\076\000\017\000\017\000\018\000\018\000\018\000\077\000\
\077\000\077\000\012\000\012\000\020\000\020\000\061\000\061\000\
\019\000\019\000\019\000\078\000\078\000\021\000\021\000\021\000\
\022\000\022\000\079\000\015\000\015\000\011\000\011\000\013\000\
\013\000\013\000\080\000\080\000\014\000\014\000\016\000\016\000\
\034\000\034\000\081\000\082\000\082\000\035\000\035\000\035\000\
\035\000\035\000\084\000\084\000\086\000\086\000\036\000\036\000\
\037\000\037\000\087\000\087\000\085\000\085\000\085\000\085\000\
\085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
\085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
\085\000\085\000\085\000\085\000\085\000\085\000\088\000\088\000\
\093\000\094\000\089\000\089\000\030\000\030\000\030\000\030\000\
\040\000\040\000\039\000\039\000\023\000\023\000\023\000\023\000\
\023\000\023\000\023\000\023\000\023\000\023\000\095\000\095\000\
\096\000\096\000\024\000\024\000\024\000\024\000\024\000\024\000\
\024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
\024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
\024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
\025\000\025\000\025\000\025\000\025\000\025\000\043\000\043\000\
\042\000\042\000\042\000\029\000\029\000\029\000\028\000\028\000\
\041\000\099\000\099\000\099\000\044\000\044\000\044\000\044\000\
\101\000\102\000\102\000\103\000\103\000\103\000\027\000\027\000\
\027\000\027\000\026\000\104\000\104\000\038\000\038\000\054\000\
\054\000\054\000\105\000\105\000\105\000\048\000\049\000\049\000\
\033\000\033\000\047\000\047\000\045\000\045\000\045\000\045\000\
\046\000\046\000\031\000\031\000\032\000\032\000\032\000\032\000\
\032\000\051\000\051\000\051\000\051\000\005\000\005\000\006\000\
\006\000\006\000\097\000\097\000\097\000\097\000\090\000\090\000\
\100\000\100\000\100\000\100\000\100\000\107\000\107\000\098\000\
\098\000\053\000\053\000\053\000\053\000\109\000\109\000\109\000\
\109\000\109\000\109\000\109\000\109\000\110\000\110\000\111\000\
\111\000\112\000\112\000\112\000\113\000\113\000\114\000\114\000\
\114\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
\116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
\116\000\116\000\116\000\117\000\118\000\118\000\118\000\118\000\
\119\000\119\000\119\000\120\000\120\000\120\000\121\000\121\000\
\121\000\121\000\121\000\122\000\122\000\122\000\123\000\123\000\
\124\000\124\000\125\000\125\000\126\000\126\000\127\000\127\000\
\128\000\128\000\108\000\129\000\129\000\129\000\130\000\130\000\
\083\000\083\000\106\000\106\000\007\000\007\000\007\000\091\000\
\091\000\092\000\092\000\131\000\131\000\133\000\133\000\134\000\
\134\000\132\000\132\000\135\000\135\000\136\000\136\000\137\000\
\137\000\000\000\000\000"

let yylen = "\002\000\
\002\000\001\000\000\000\002\000\003\000\002\000\000\000\002\000\
\002\000\001\000\001\000\001\000\001\000\003\000\005\000\005\000\
\001\000\006\000\004\000\008\000\008\000\003\000\001\000\001\000\
\001\000\004\000\000\000\001\000\001\000\001\000\001\000\003\000\
\004\000\001\000\002\000\004\000\006\000\006\000\006\000\003\000\
\003\000\002\000\002\000\006\000\001\000\003\000\002\000\001\000\
\002\000\002\000\002\000\004\000\002\000\004\000\002\000\002\000\
\002\000\002\000\002\000\002\000\002\000\001\000\004\000\001\000\
\003\000\003\000\003\000\001\000\003\000\003\000\001\000\003\000\
\003\000\001\000\003\000\003\000\003\000\003\000\001\000\003\000\
\003\000\001\000\003\000\001\000\003\000\001\000\003\000\001\000\
\003\000\001\000\003\000\001\000\005\000\001\000\003\000\003\000\
\003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
\003\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\002\000\001\000\002\000\002\000\001\000\
\001\000\001\000\001\000\003\000\001\000\003\000\000\000\001\000\
\003\000\002\000\001\000\001\000\000\000\003\000\004\000\005\000\
\000\000\001\000\002\000\000\000\001\000\000\000\001\000\001\000\
\003\000\003\000\000\000\001\000\003\000\003\000\003\000\003\000\
\005\000\003\000\001\000\000\000\002\000\001\000\003\000\003\000\
\003\000\003\000\000\000\001\000\002\000\004\000\000\000\004\000\
\001\000\003\000\001\000\002\000\001\000\002\000\002\000\002\000\
\001\000\003\000\005\000\003\000\004\000\006\000\009\000\004\000\
\004\000\006\000\002\000\002\000\003\000\002\000\002\000\003\000\
\004\000\007\000\001\000\005\000\004\000\003\000\000\000\001\000\
\001\000\001\000\002\000\001\000\003\000\002\000\004\000\003\000\
\001\000\003\000\001\000\003\000\002\000\002\000\002\000\002\000\
\002\000\002\000\002\000\002\000\002\000\004\000\000\000\001\000\
\000\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
\001\000\001\000\001\000\001\000\001\000\002\000\003\000\005\000\
\004\000\006\000\005\000\002\000\005\000\004\000\006\000\005\000\
\002\000\006\000\005\000\007\000\006\000\001\000\004\000\004\000\
\000\000\003\000\002\000\004\000\002\000\003\000\001\000\003\000\
\001\000\004\000\002\000\001\000\003\000\003\000\001\000\003\000\
\003\000\002\000\004\000\003\000\001\000\004\000\004\000\004\000\
\001\000\000\000\002\000\000\000\002\000\003\000\002\000\002\000\
\001\000\003\000\002\000\005\000\003\000\001\000\003\000\000\000\
\004\000\004\000\001\000\003\000\001\000\003\000\000\000\001\000\
\002\000\001\000\003\000\001\000\004\000\003\000\004\000\004\000\
\001\000\000\000\003\000\002\000\002\000\002\000\004\000\005\000\
\003\000\001\000\001\000\001\000\001\000\000\000\002\000\000\000\
\002\000\005\000\004\000\002\000\001\000\001\000\000\000\002\000\
\001\000\001\000\001\000\001\000\001\000\004\000\002\000\001\000\
\002\000\002\000\003\000\004\000\001\000\001\000\001\000\003\000\
\001\000\001\000\003\000\003\000\003\000\001\000\001\000\001\000\
\002\000\001\000\002\000\003\000\001\000\001\000\001\000\003\000\
\001\000\001\000\002\000\003\000\002\000\003\000\003\000\004\000\
\001\000\002\000\004\000\002\000\004\000\002\000\002\000\002\000\
\002\000\002\000\002\000\001\000\001\000\003\000\003\000\003\000\
\001\000\003\000\003\000\001\000\003\000\003\000\001\000\003\000\
\003\000\003\000\003\000\001\000\003\000\003\000\001\000\003\000\
\001\000\003\000\001\000\003\000\001\000\003\000\001\000\003\000\
\001\000\005\000\001\000\001\000\003\000\003\000\000\000\001\000\
\003\000\003\000\003\000\003\000\000\000\002\000\002\000\001\000\
\002\000\000\000\003\000\000\000\001\000\001\000\003\000\005\000\
\005\000\000\000\003\000\000\000\003\000\000\000\002\000\001\000\
\003\000\002\000\002\000"

let yydefred = "\000\000\
\000\000\000\000\000\000\000\000\011\000\069\001\000\000\254\000\
\229\000\228\000\231\000\235\000\234\000\227\000\233\000\000\000\
\000\000\000\000\000\000\236\000\237\000\232\000\230\000\067\001\
\000\000\000\000\066\001\068\001\000\000\000\000\078\001\000\000\
\000\000\000\000\000\000\000\000\000\000\077\001\093\001\000\000\
\000\000\000\000\000\000\000\000\194\001\000\000\002\000\000\000\
\000\000\000\000\012\000\013\000\000\000\000\000\000\000\017\000\
\000\000\195\001\000\000\000\000\000\000\000\000\000\000\000\000\
\023\000\024\000\000\000\000\000\000\000\000\000\025\000\000\000\
\000\000\000\000\000\000\000\000\000\000\224\000\213\000\000\000\
\000\000\000\000\120\000\121\000\122\000\000\000\000\000\115\000\
\214\000\215\000\216\000\217\000\006\000\000\000\219\000\000\000\
\000\000\000\000\076\001\000\000\000\000\000\000\098\001\097\001\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\090\001\113\001\000\000\000\000\103\001\000\000\
\114\001\000\000\132\001\133\001\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\163\001\000\000\
\000\000\000\000\010\000\000\000\000\000\000\000\001\000\004\000\
\000\000\206\000\062\001\000\000\000\000\000\000\048\001\000\000\
\226\000\218\000\007\000\155\000\060\001\000\000\000\000\220\000\
\221\000\000\000\208\000\000\000\059\001\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\012\001\000\000\000\000\
\087\001\000\000\000\000\000\000\000\000\089\001\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\014\000\116\000\000\000\
\000\000\000\000\029\000\109\000\110\000\107\000\108\000\117\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\030\000\111\000\000\000\031\000\000\000\000\000\000\000\034\000\
\000\000\062\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\094\000\106\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\122\001\000\000\
\124\001\126\001\127\001\128\001\131\001\129\001\130\001\000\000\
\000\000\000\000\115\001\000\000\091\001\094\001\095\001\110\001\
\000\000\000\000\109\001\117\001\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\008\000\009\000\005\000\
\085\001\084\001\082\001\083\001\000\000\081\001\000\000\000\000\
\205\000\000\000\000\000\000\000\021\001\035\001\000\000\000\000\
\000\000\022\000\065\001\000\000\207\000\000\000\019\000\000\000\
\000\000\031\001\032\001\000\000\000\000\000\000\027\001\063\001\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\003\001\000\000\000\000\000\000\000\000\000\000\000\000\
\241\000\005\001\000\000\000\000\000\000\246\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\168\001\000\000\075\001\
\000\000\000\000\051\000\000\000\053\000\055\000\056\000\057\000\
\060\000\058\000\059\000\061\000\000\000\049\000\050\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\255\000\119\000\118\000\049\001\000\000\000\001\000\000\000\000\
\042\000\043\000\000\000\000\000\035\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\170\001\000\000\000\000\169\001\099\001\
\101\001\100\001\000\000\000\000\112\001\096\001\116\001\092\001\
\105\001\000\000\118\001\119\001\000\000\134\001\135\001\136\001\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\222\000\046\001\071\001\210\000\000\000\123\000\
\212\000\000\000\000\000\000\000\000\000\017\001\000\000\000\000\
\154\000\000\000\000\000\000\000\000\000\000\000\039\001\000\000\
\000\000\034\001\000\000\000\000\000\000\000\000\029\001\000\000\
\016\001\014\001\013\001\251\000\086\001\026\000\000\000\000\000\
\000\000\006\001\011\001\002\001\000\000\000\000\000\000\240\000\
\243\000\000\000\245\000\248\000\000\000\015\000\172\001\171\001\
\016\000\000\000\000\000\000\000\150\000\000\000\000\000\149\000\
\000\000\032\000\000\000\000\000\000\000\000\000\000\000\041\000\
\040\000\000\000\141\000\000\000\000\000\000\000\095\000\096\000\
\097\000\098\000\099\000\100\000\101\000\102\000\103\000\104\000\
\105\000\065\000\066\000\067\000\064\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\143\000\000\000\166\001\
\165\001\000\000\000\000\108\001\120\001\000\000\000\000\000\000\
\000\000\000\000\000\000\131\000\000\000\128\000\000\000\000\000\
\000\000\000\000\000\000\037\001\000\000\000\000\000\000\000\000\
\000\000\000\000\073\001\000\000\000\000\000\000\157\000\000\000\
\000\000\000\000\000\000\164\000\064\001\045\001\000\000\000\000\
\018\000\000\000\000\000\025\001\051\001\000\000\000\000\030\001\
\250\000\253\000\000\000\000\000\008\001\004\001\242\000\247\000\
\000\000\000\000\000\000\146\000\145\000\000\000\063\000\000\000\
\000\000\000\000\033\000\000\000\036\000\152\000\151\000\000\000\
\162\001\000\000\000\000\000\000\000\000\000\000\132\000\000\000\
\139\000\124\000\130\000\022\001\000\000\000\000\148\000\018\001\
\023\001\000\000\000\000\024\001\168\000\170\000\165\000\000\000\
\153\000\000\000\000\000\000\000\202\000\000\000\173\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\195\000\000\000\000\000\177\000\000\000\000\000\000\000\
\000\000\200\000\201\000\000\000\000\000\054\001\000\000\000\000\
\000\000\252\000\010\001\000\000\000\000\000\000\045\000\000\000\
\093\000\000\000\000\000\138\000\134\000\000\000\000\000\126\000\
\129\000\036\001\020\001\000\000\000\000\000\000\000\000\000\000\
\000\000\171\000\000\000\174\000\175\000\190\000\191\000\000\000\
\000\000\188\000\000\000\000\000\000\000\000\000\187\000\000\000\
\000\000\000\000\000\000\000\000\176\000\159\000\161\000\000\000\
\160\000\000\000\000\000\000\000\044\001\041\001\042\001\053\001\
\056\001\055\001\044\000\037\000\038\000\000\000\039\000\047\000\
\020\000\021\000\135\000\000\000\019\001\074\001\166\000\198\000\
\000\000\172\000\000\000\192\000\189\000\000\000\180\000\000\000\
\000\000\000\000\000\000\000\000\174\001\175\001\000\000\000\000\
\000\000\000\000\000\000\000\000\046\000\136\000\193\000\185\000\
\000\000\000\000\000\000\197\000\114\000\000\000\000\000\184\000\
\080\001\181\000\000\000\000\000\204\000\000\000\000\000\179\000\
\196\000\177\001\000\000\000\000\000\000\203\000\000\000\186\000\
\000\000\000\000\000\000\182\001\000\000\000\000\182\000\000\000\
\000\000\000\000\179\001\000\000\000\000\194\000\000\000\189\001\
\000\000\183\001\000\000\000\000\000\000\187\001\000\000\000\000\
\183\000\000\000\191\001\185\001\184\001\000\000\193\001"

let yydgoto = "\003\000\
\045\000\046\000\047\000\048\000\053\001\230\001\244\002\225\000\
\226\000\129\001\069\002\084\002\187\002\228\000\036\002\149\001\
\087\000\229\000\085\002\086\002\087\002\213\002\185\000\050\000\
\186\000\147\000\171\000\174\000\175\000\051\000\052\000\053\000\
\231\000\189\002\106\002\065\001\234\001\168\000\148\000\149\000\
\108\001\094\001\095\001\244\001\245\001\246\001\199\002\151\000\
\096\001\054\000\055\000\142\000\187\000\241\001\117\000\061\001\
\083\001\232\000\233\000\208\002\089\002\234\000\235\000\236\000\
\237\000\238\000\239\000\240\000\241\000\242\000\243\000\244\000\
\245\000\246\000\247\000\046\003\088\000\152\002\090\002\160\002\
\158\000\237\001\099\000\107\002\226\002\227\002\229\002\193\002\
\054\003\032\003\047\003\060\003\194\002\195\002\089\000\154\000\
\057\000\073\000\096\002\055\001\064\000\172\000\079\001\062\001\
\112\002\201\000\074\000\249\000\119\000\120\000\018\001\019\001\
\020\001\121\000\122\000\123\000\124\000\125\000\126\000\127\000\
\128\000\129\000\130\000\131\000\132\000\133\000\134\000\135\000\
\250\000\111\001\066\003\075\003\067\003\068\003\069\003\086\003\
\091\003"

let yysindex = "\247\000\
\063\011\063\011\000\000\056\017\000\000\000\000\041\255\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\030\255\
\058\255\053\018\109\000\000\000\000\000\000\000\000\000\000\000\
\136\016\053\018\000\000\000\000\053\018\053\018\000\000\063\011\
\112\255\053\018\146\255\148\255\181\255\000\000\000\000\087\023\
\009\255\236\255\016\000\086\002\000\000\113\001\000\000\063\011\
\048\000\086\018\000\000\000\000\044\255\115\000\053\018\000\000\
\053\018\000\000\047\000\053\018\120\000\044\255\137\000\166\017\
\000\000\000\000\147\000\091\000\181\255\099\000\000\000\106\000\
\107\255\191\255\216\013\119\000\040\000\000\000\000\000\216\013\
\123\000\093\000\000\000\000\000\000\000\230\014\063\255\000\000\
\000\000\000\000\000\000\000\000\000\000\129\000\000\000\063\255\
\206\015\102\029\000\000\000\000\145\000\000\000\000\000\000\000\
\201\019\016\020\095\011\095\011\095\011\095\011\095\011\095\011\
\095\011\158\000\000\000\000\000\152\000\001\255\000\000\192\001\
\000\000\011\255\000\000\000\000\123\001\201\000\175\000\021\002\
\058\001\205\000\230\000\220\000\232\000\174\255\000\000\193\014\
\035\020\048\001\000\000\086\002\086\002\063\011\000\000\000\000\
\198\003\000\000\000\000\240\000\244\000\032\001\000\000\101\255\
\000\000\000\000\000\000\000\000\000\000\226\000\254\000\000\000\
\000\000\137\000\000\000\001\001\000\000\003\001\012\001\022\001\
\166\017\039\001\024\001\055\001\092\001\000\000\056\001\129\000\
\000\000\135\001\147\000\147\000\066\001\000\000\069\001\216\013\
\079\000\080\001\216\013\216\013\216\013\082\001\087\001\216\013\
\216\013\091\001\053\018\063\011\048\000\000\000\000\000\165\029\
\117\001\127\001\000\000\000\000\000\000\000\000\000\000\000\000\
\106\020\125\020\035\020\035\020\035\020\035\020\035\020\035\020\
\180\255\196\020\196\020\014\010\112\001\133\001\138\001\148\001\
\000\000\000\000\131\001\000\000\121\255\039\001\151\001\000\000\
\040\001\000\000\152\003\175\001\064\001\065\001\140\002\141\001\
\168\001\161\001\178\001\172\001\013\000\000\000\000\000\006\255\
\166\001\180\001\253\001\005\002\013\002\014\010\000\000\014\010\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\205\001\
\031\002\031\029\000\000\184\001\000\000\000\000\000\000\000\000\
\046\001\063\255\000\000\000\000\131\255\131\255\095\011\095\011\
\095\011\095\011\095\011\095\011\095\011\095\011\095\011\095\011\
\095\011\095\011\095\011\095\011\095\011\095\011\095\011\095\011\
\095\011\095\011\222\001\234\001\247\001\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\039\001\000\000\198\003\039\001\
\000\000\229\018\198\003\163\001\000\000\000\000\251\001\180\255\
\215\001\000\000\000\000\006\002\000\000\137\000\000\000\053\018\
\008\002\000\000\000\000\000\000\119\255\169\016\000\000\000\000\
\035\020\118\000\011\002\019\002\032\002\056\001\056\001\147\000\
\216\013\000\000\035\020\216\013\015\002\041\002\061\002\101\255\
\000\000\000\000\070\002\089\002\216\013\000\000\112\002\122\002\
\216\013\120\000\124\002\032\001\064\255\000\000\128\002\000\000\
\125\002\014\010\000\000\014\010\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\014\010\000\000\000\000\221\255\
\134\002\136\002\151\002\169\002\035\020\053\018\053\018\214\002\
\000\000\000\000\000\000\000\000\167\002\000\000\131\255\131\255\
\000\000\000\000\173\012\244\012\000\000\035\020\035\020\035\020\
\035\020\035\020\035\020\035\020\035\020\035\020\035\020\035\020\
\035\020\035\020\035\020\035\020\035\020\035\020\035\020\035\020\
\035\020\035\020\035\020\035\020\035\020\035\020\035\020\035\020\
\035\020\035\020\173\012\000\000\228\029\228\029\000\000\000\000\
\000\000\000\000\177\002\179\002\000\000\000\000\000\000\000\000\
\000\000\046\001\000\000\000\000\175\002\000\000\000\000\000\000\
\123\001\123\001\201\000\201\000\175\000\175\000\175\000\175\000\
\021\002\021\002\058\001\205\000\230\000\220\000\232\000\183\002\
\250\002\003\003\000\000\000\000\000\000\000\000\022\016\000\000\
\000\000\039\001\137\000\243\000\190\002\000\000\224\001\166\017\
\000\000\187\002\193\002\181\255\020\255\053\018\000\000\048\255\
\195\002\000\000\129\001\152\008\116\011\202\002\000\000\024\001\
\000\000\000\000\000\000\000\000\000\000\000\000\205\002\223\002\
\056\001\000\000\000\000\000\000\035\020\017\255\216\013\000\000\
\000\000\242\002\000\000\000\000\243\002\000\000\000\000\000\000\
\000\000\251\002\252\002\253\002\000\000\173\012\173\012\000\000\
\049\019\000\000\247\002\248\002\002\003\004\003\129\001\000\000\
\000\000\006\003\000\000\005\003\045\000\254\002\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\175\001\175\001\064\001\
\064\001\065\001\065\001\065\001\065\001\140\002\140\002\141\001\
\168\001\161\001\178\001\172\001\009\003\000\000\010\003\000\000\
\000\000\008\003\008\003\000\000\000\000\095\011\016\003\017\003\
\000\000\131\255\035\020\000\000\015\003\000\000\063\003\025\003\
\030\003\229\018\046\003\000\000\047\003\198\003\173\012\043\003\
\198\003\063\255\000\000\055\003\226\000\180\255\000\000\020\255\
\020\255\056\003\150\009\000\000\000\000\000\000\049\003\052\003\
\000\000\067\003\039\001\000\000\000\000\166\017\173\012\000\000\
\000\000\000\000\066\003\198\003\000\000\000\000\000\000\000\000\
\008\003\008\003\008\003\000\000\000\000\022\016\000\000\053\018\
\053\018\131\255\000\000\039\001\000\000\000\000\000\000\035\020\
\000\000\063\011\035\020\011\000\004\255\022\016\000\000\229\018\
\000\000\000\000\000\000\000\000\053\018\173\012\000\000\000\000\
\000\000\112\003\072\003\000\000\000\000\000\000\000\000\022\003\
\000\000\006\003\147\006\149\005\000\000\000\000\000\000\071\003\
\090\003\046\000\206\011\078\003\035\020\110\003\078\003\044\255\
\153\000\000\000\122\003\020\255\000\000\020\255\124\003\020\255\
\121\000\000\000\000\000\048\255\199\017\000\000\131\003\132\003\
\128\003\000\000\000\000\136\003\134\003\139\003\000\000\049\255\
\000\000\140\003\141\003\000\000\000\000\011\000\035\020\000\000\
\000\000\000\000\000\000\173\012\198\003\020\255\187\000\147\006\
\142\003\000\000\204\010\000\000\000\000\000\000\000\000\173\012\
\144\003\000\000\148\003\007\013\147\006\198\000\000\000\147\006\
\120\001\153\000\153\000\146\003\000\000\000\000\000\000\250\000\
\000\000\078\003\147\006\158\003\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\230\003\000\000\000\000\
\000\000\000\000\000\000\169\003\000\000\000\000\000\000\000\000\
\250\000\000\000\168\003\000\000\000\000\221\255\000\000\147\006\
\035\020\160\003\078\003\044\255\000\000\000\000\239\003\147\006\
\250\000\147\006\165\003\133\010\000\000\000\000\000\000\000\000\
\180\003\147\006\044\255\000\000\000\000\239\003\189\003\000\000\
\000\000\000\000\078\003\190\003\000\000\173\012\147\006\000\000\
\000\000\000\000\192\003\200\003\194\003\000\000\199\003\000\000\
\022\004\212\003\213\003\000\000\063\255\216\003\000\000\173\012\
\221\003\192\003\000\000\192\003\226\003\000\000\234\003\000\000\
\232\003\000\000\078\013\147\006\239\003\000\000\240\003\244\003\
\000\000\237\003\000\000\000\000\000\000\239\003\000\000"

let yyrindex = "\000\000\
\029\000\031\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\076\000\000\000\000\000\000\000\000\000\000\000\000\000\
\087\000\076\000\000\000\000\000\076\000\076\000\000\000\033\000\
\000\000\076\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\033\000\
\080\000\023\001\000\000\000\000\000\000\000\000\076\000\000\000\
\076\000\000\000\000\000\076\000\080\000\000\000\023\017\250\003\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\006\007\
\000\000\167\000\008\004\118\007\000\000\000\000\000\000\008\004\
\230\007\000\000\000\000\000\000\000\000\000\000\074\014\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\109\022\009\023\187\022\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\241\005\
\000\000\217\007\000\000\000\000\218\014\191\017\064\029\241\003\
\038\016\225\011\058\018\137\008\020\007\111\002\000\000\087\004\
\000\000\000\000\000\000\000\000\000\000\033\000\000\000\000\000\
\183\000\000\000\000\000\007\004\000\000\237\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\151\004\000\000\000\000\
\000\000\023\017\000\000\000\000\000\000\015\004\045\255\000\000\
\000\000\213\000\018\004\000\000\253\255\000\000\017\004\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\008\004\
\080\000\000\000\008\004\008\004\008\004\086\008\000\000\008\004\
\008\004\000\000\000\000\034\255\080\000\000\000\000\000\020\004\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\024\021\174\000\000\000\000\000\
\111\021\000\000\154\023\223\023\174\024\235\001\123\026\035\027\
\147\027\003\028\115\028\038\012\112\012\000\000\000\000\000\000\
\027\004\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\112\006\249\006\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\219\000\000\000\165\008\080\000\
\000\000\000\000\179\000\149\001\000\000\000\000\000\000\000\000\
\022\005\000\000\000\000\000\000\000\000\000\000\000\000\078\255\
\000\000\000\000\000\000\067\001\019\004\000\000\000\000\000\000\
\000\000\026\004\000\000\000\000\000\000\017\004\017\004\000\000\
\008\004\000\000\000\000\008\004\164\001\024\004\000\000\000\000\
\000\000\000\000\000\000\000\000\008\004\000\000\000\000\000\000\
\008\004\080\000\000\000\170\001\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\030\004\
\249\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\019\004\000\000\000\000\000\000\
\000\000\000\000\033\004\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\040\004\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\105\007\000\000\000\000\000\000\000\000\000\000\000\000\
\073\015\048\017\244\029\028\030\061\030\096\030\129\030\164\030\
\187\030\210\030\142\016\097\013\114\018\241\009\132\007\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\045\004\000\000\
\000\000\080\000\023\017\021\012\000\000\000\000\130\001\250\003\
\000\000\000\000\041\004\000\000\030\009\080\255\000\000\080\000\
\086\255\000\000\055\000\019\000\116\000\000\000\000\000\018\004\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\017\004\000\000\000\000\000\000\000\000\080\000\008\004\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\159\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\036\024\105\024\243\024\
\056\025\125\025\191\025\001\026\067\026\179\026\235\026\091\027\
\203\027\059\028\171\028\227\028\000\000\000\000\000\000\000\000\
\000\000\073\008\221\013\000\000\000\000\000\000\000\000\000\000\
\034\031\000\000\000\000\000\000\047\004\000\000\125\019\000\000\
\000\000\000\000\000\000\000\000\000\000\149\013\050\004\000\000\
\130\011\000\000\000\000\000\000\151\004\000\000\000\000\023\003\
\151\003\000\000\122\001\000\000\000\000\000\000\043\004\000\000\
\000\000\000\000\192\000\000\000\000\000\250\003\050\004\000\000\
\000\000\000\000\000\000\177\001\000\000\000\000\000\000\000\000\
\198\021\029\022\000\000\000\000\000\000\045\004\000\000\000\000\
\000\000\000\000\000\000\238\000\000\000\000\000\000\000\000\000\
\000\000\034\255\000\000\153\018\000\000\045\004\000\000\000\000\
\000\000\000\000\000\000\000\000\080\255\000\000\000\000\000\000\
\000\000\097\012\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\222\000\181\001\181\001\000\000\086\015\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\051\004\000\000\000\000\023\003\000\000\023\003\000\000\023\003\
\000\000\000\000\000\000\080\000\078\255\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\153\018\000\000\000\000\
\000\000\000\000\000\000\000\000\222\003\023\003\000\000\181\001\
\000\000\000\000\181\001\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\181\001\000\000\000\000\181\001\
\000\000\051\004\051\004\000\000\000\000\000\000\000\000\023\004\
\000\000\000\000\181\001\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\020\006\000\000\000\000\000\000\000\000\000\000\000\000\181\001\
\000\000\145\002\000\000\000\000\000\000\000\000\000\000\181\001\
\020\006\181\001\000\000\052\004\000\000\000\000\000\000\000\000\
\000\000\181\001\000\000\000\000\000\000\065\255\057\004\000\000\
\000\000\000\000\000\000\000\000\000\000\052\004\181\001\000\000\
\000\000\000\000\225\000\000\000\000\000\000\000\000\000\000\000\
\000\000\058\004\061\000\000\000\000\000\000\000\000\000\062\004\
\000\000\225\000\000\000\228\255\000\000\000\000\000\000\000\000\
\064\004\000\000\000\000\181\001\000\000\000\000\000\000\000\000\
\000\000\065\004\000\000\000\000\000\000\000\000\000\000"

let yygindex = "\000\000\
\000\000\140\005\231\255\007\005\012\255\170\003\041\000\000\000\
\241\255\007\000\050\254\211\254\074\255\112\253\000\000\192\002\
\000\000\000\000\000\000\000\000\157\253\188\002\150\000\000\000\
\050\000\000\000\141\255\066\004\136\255\216\255\068\001\150\005\
\077\255\208\255\110\254\091\003\092\003\111\255\000\000\207\255\
\213\255\000\000\167\003\041\005\000\000\000\000\159\255\095\255\
\235\255\103\255\000\000\168\002\020\000\038\254\240\255\034\000\
\196\254\000\000\000\000\000\000\209\253\220\001\221\001\150\001\
\176\001\105\002\183\001\021\004\025\004\028\004\029\004\031\004\
\000\000\053\003\178\003\135\253\185\255\000\000\000\000\077\003\
\000\000\000\000\195\255\187\253\179\253\062\254\139\254\000\000\
\000\000\165\002\153\002\000\000\000\000\000\000\048\004\000\000\
\158\255\103\000\000\000\002\255\215\254\056\254\209\003\000\000\
\007\003\026\005\000\000\015\000\000\000\147\255\188\004\013\004\
\000\000\000\000\000\000\000\000\218\000\189\002\196\002\036\003\
\194\002\171\004\172\004\175\004\178\004\174\004\000\000\229\254\
\114\255\000\000\148\002\000\000\000\000\143\002\000\000\000\000\
\129\002"

let yytablesize = 8312
let yytable = "\071\000\
\071\000\063\001\071\000\140\000\157\000\150\000\093\000\177\000\
\076\001\086\000\017\001\164\000\225\001\165\000\216\001\199\000\
\068\001\150\000\232\001\109\002\056\000\056\000\144\000\104\002\
\116\000\255\001\000\002\152\000\003\000\192\002\003\000\100\002\
\003\000\007\000\167\002\237\002\065\000\130\001\240\002\152\000\
\131\001\066\000\108\002\155\000\065\001\198\000\054\001\110\002\
\212\002\072\000\076\000\056\000\081\000\073\001\118\000\011\001\
\071\000\110\001\086\001\087\001\071\000\231\001\021\001\022\001\
\065\000\071\000\221\001\056\000\076\001\066\000\226\001\130\001\
\075\001\130\001\187\001\083\000\188\001\040\001\145\000\040\001\
\202\000\180\001\116\000\012\001\214\002\064\001\204\002\136\000\
\215\002\181\001\023\001\116\000\116\000\116\000\116\000\116\000\
\116\000\116\000\091\001\140\000\140\000\006\003\216\002\227\000\
\016\001\034\003\181\000\065\000\067\000\145\000\190\000\003\000\
\066\000\065\000\212\002\194\000\048\001\063\000\066\000\077\000\
\013\001\082\000\156\000\065\001\007\003\065\000\074\001\008\001\
\148\001\191\000\066\000\068\000\140\001\083\000\138\001\071\000\
\075\000\065\000\043\003\015\002\176\001\093\001\066\000\044\001\
\069\000\105\002\176\001\181\001\077\001\018\003\049\000\049\000\
\015\003\061\000\070\000\056\000\040\001\139\001\040\001\068\000\
\040\001\056\000\061\003\017\001\064\001\108\002\108\002\078\000\
\084\000\085\000\107\001\132\001\069\000\200\002\078\000\078\000\
\182\000\059\001\078\000\078\000\239\001\049\000\070\000\078\000\
\116\000\180\000\065\000\130\001\094\000\130\001\018\002\066\000\
\019\002\049\000\218\002\243\001\123\002\049\000\130\001\153\000\
\124\001\020\002\232\001\118\002\078\000\132\001\078\000\132\001\
\141\001\078\000\028\002\029\002\054\001\170\000\168\002\056\000\
\054\001\054\001\248\001\090\003\035\002\038\002\096\000\001\002\
\097\000\070\000\084\000\085\000\090\003\097\002\063\001\070\000\
\231\001\090\001\255\002\197\000\098\001\099\001\100\001\095\002\
\188\001\103\001\104\001\070\000\041\001\231\001\230\000\001\000\
\002\000\108\002\116\000\108\002\070\002\108\002\115\002\070\000\
\117\002\098\000\194\001\042\001\071\000\071\000\222\001\116\000\
\116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
\116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
\116\000\116\000\116\000\108\002\007\000\049\000\007\000\220\001\
\007\000\049\000\049\000\049\000\068\000\197\001\072\002\073\002\
\021\002\246\002\140\002\247\002\155\002\249\002\065\000\235\001\
\022\002\069\000\145\002\066\000\065\000\228\002\195\001\196\001\
\164\000\066\000\137\000\072\001\071\000\070\001\170\000\082\002\
\224\001\132\001\070\001\132\001\002\001\003\001\004\001\005\001\
\006\001\007\001\015\001\223\000\132\001\188\001\188\001\071\000\
\015\001\093\002\223\000\178\001\017\001\052\003\047\001\249\001\
\106\001\049\000\083\002\047\001\138\000\223\000\072\001\132\002\
\133\002\003\002\179\001\065\000\072\001\072\001\072\001\063\003\
\066\000\158\002\217\002\232\002\162\002\145\000\003\000\141\000\
\143\000\230\000\159\000\065\000\070\001\250\001\189\000\023\003\
\066\000\079\003\026\003\162\000\173\000\142\002\071\000\071\000\
\022\002\054\001\146\000\070\001\054\001\035\003\070\001\203\002\
\181\001\223\000\002\002\027\002\145\000\004\002\181\001\166\000\
\054\001\054\001\054\001\230\000\223\000\230\000\010\002\223\000\
\223\000\173\000\013\002\223\000\047\001\223\000\223\000\223\000\
\091\001\092\001\040\003\223\000\070\000\116\000\116\000\176\000\
\159\002\223\000\048\003\193\000\050\003\088\001\103\002\178\000\
\032\002\033\002\088\001\070\001\056\003\145\000\091\002\242\002\
\179\000\070\001\243\002\080\000\054\001\070\001\070\001\070\001\
\159\002\064\003\070\001\057\001\111\002\188\000\047\001\070\001\
\223\000\192\000\163\000\047\001\047\001\200\000\071\000\141\000\
\141\000\076\001\068\000\223\000\205\002\206\002\250\002\251\002\
\252\002\070\000\093\001\047\001\070\001\240\001\089\003\069\000\
\047\001\047\001\252\000\170\000\010\001\224\001\047\001\219\002\
\014\003\070\000\076\001\070\001\061\001\188\001\070\001\009\001\
\070\001\198\001\199\001\200\001\070\001\088\001\029\001\030\001\
\049\001\050\001\047\001\054\001\235\002\047\001\054\001\070\001\
\088\002\050\001\070\001\070\001\027\001\028\001\070\001\230\000\
\233\001\230\000\070\001\124\002\047\001\016\003\037\001\047\001\
\223\002\050\001\230\000\094\002\051\001\052\001\225\000\024\003\
\031\000\054\001\025\003\230\000\230\000\225\000\039\001\031\000\
\033\001\047\001\225\000\088\001\047\001\013\003\047\001\047\001\
\033\001\038\001\047\001\007\000\180\001\040\001\047\001\070\001\
\007\000\019\003\180\001\070\001\014\001\130\001\045\001\101\000\
\126\002\015\001\047\001\061\001\047\001\047\001\116\000\211\000\
\211\000\071\000\188\002\056\001\255\000\001\001\057\001\091\002\
\058\001\048\001\188\001\188\001\103\000\064\001\048\001\104\000\
\066\001\047\001\163\002\069\001\225\000\235\001\070\001\047\001\
\033\000\149\002\191\002\143\001\144\001\077\001\071\001\033\000\
\224\001\072\001\225\000\225\000\145\000\037\000\225\000\038\000\
\225\000\225\000\225\000\078\001\037\000\070\002\038\000\035\001\
\036\001\145\001\146\001\148\002\147\001\071\000\141\001\148\001\
\210\002\071\000\054\001\164\001\165\001\170\000\190\002\070\002\
\114\002\072\001\080\001\240\001\081\001\071\000\049\001\241\002\
\166\001\167\001\114\000\082\001\224\001\085\001\052\001\048\001\
\088\001\070\002\048\001\225\000\072\001\033\003\052\001\089\001\
\111\002\211\002\225\002\225\002\224\001\097\001\224\001\050\001\
\101\001\233\002\051\001\052\001\102\001\056\000\031\000\088\002\
\049\001\105\001\072\001\207\002\115\001\117\001\033\003\118\001\
\119\001\120\001\121\001\122\001\123\001\126\001\127\001\088\002\
\024\001\025\001\026\001\238\002\133\001\072\001\033\003\048\001\
\112\001\050\001\172\001\173\001\051\001\052\001\014\001\158\000\
\031\000\101\000\113\001\015\001\083\000\072\001\137\001\225\002\
\072\001\134\001\225\002\072\001\072\001\072\001\135\001\199\000\
\199\000\199\000\027\003\028\003\225\002\012\003\103\000\225\002\
\136\001\104\000\142\001\072\001\175\001\049\001\033\000\072\001\
\072\001\174\001\225\002\044\003\161\001\162\001\163\001\227\001\
\074\000\177\001\228\001\037\000\176\001\038\000\009\001\009\001\
\158\000\182\001\057\003\053\003\211\000\211\000\050\001\183\001\
\197\000\051\001\052\001\070\001\070\001\031\000\184\001\225\002\
\033\000\074\000\229\001\170\000\074\000\074\000\185\001\225\002\
\074\000\225\002\199\000\199\000\199\000\037\000\186\001\038\000\
\189\001\225\002\029\003\030\003\114\000\230\000\230\000\041\003\
\074\000\074\000\074\000\074\000\074\000\074\000\225\002\049\000\
\190\001\084\000\085\000\217\001\074\000\074\000\074\000\192\001\
\074\000\074\000\240\001\046\001\047\001\077\003\074\000\218\001\
\074\000\054\002\055\002\074\000\074\000\074\000\074\000\074\000\
\074\000\106\001\219\001\225\002\144\000\033\000\236\001\229\001\
\233\001\144\000\144\000\144\000\031\001\032\001\033\001\034\001\
\074\000\238\001\037\000\242\001\038\000\056\002\057\002\004\000\
\252\001\088\003\240\001\006\000\059\000\074\000\253\001\074\000\
\005\002\008\000\062\002\063\002\009\000\010\000\011\000\012\000\
\013\000\014\000\015\000\254\001\016\000\017\000\018\000\019\000\
\020\000\021\000\022\000\023\000\024\000\060\000\026\000\027\000\
\028\000\029\000\030\000\031\000\006\002\050\002\051\002\052\002\
\053\002\053\002\053\002\053\002\053\002\053\002\053\002\053\002\
\053\002\053\002\053\002\053\002\053\002\053\002\053\002\007\002\
\178\000\178\000\178\000\008\002\178\000\178\000\178\000\178\000\
\178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
\178\000\178\000\178\000\178\000\178\000\178\000\009\002\178\000\
\178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
\178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
\178\000\197\000\161\001\033\000\034\000\011\002\036\000\161\001\
\161\001\161\001\161\001\168\001\169\001\170\001\171\001\012\002\
\037\000\014\002\038\000\016\002\178\000\178\000\178\000\017\002\
\043\000\178\000\178\000\024\002\139\000\178\000\178\000\201\001\
\202\001\023\002\178\000\178\000\030\002\178\000\178\000\178\000\
\203\001\204\001\025\002\178\000\209\001\210\001\161\001\178\000\
\178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
\178\000\178\000\178\000\031\002\026\002\135\002\178\000\178\000\
\178\000\178\000\178\000\178\000\074\002\178\000\075\002\077\002\
\079\002\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
\078\002\080\002\098\002\178\000\178\000\101\002\178\000\178\000\
\058\002\059\002\060\002\061\002\102\002\113\002\163\000\163\000\
\163\000\119\002\121\002\163\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\122\002\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\127\002\
\128\002\079\000\205\001\206\001\207\001\208\001\129\002\130\002\
\131\002\090\000\136\002\137\002\091\000\092\000\143\002\139\002\
\141\002\095\000\163\000\163\000\163\000\138\002\134\002\163\000\
\163\000\022\002\144\002\163\000\163\000\181\001\146\002\147\002\
\163\000\163\000\150\002\163\000\163\000\163\000\160\000\151\002\
\161\000\163\000\153\002\154\002\053\002\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
\163\000\156\002\157\002\161\002\163\000\163\000\163\000\163\000\
\163\000\163\000\164\002\163\000\196\002\169\002\197\002\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\198\002\202\002\
\220\002\163\000\163\000\221\002\222\002\163\000\163\000\163\000\
\163\000\230\002\236\002\163\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\231\002\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\239\002\
\150\001\151\001\152\001\153\001\154\001\155\001\156\001\157\001\
\158\001\159\001\160\001\049\001\245\002\248\002\000\003\001\003\
\002\003\004\003\163\000\163\000\163\000\003\003\005\003\163\000\
\163\000\009\003\010\003\163\000\163\000\070\001\031\003\017\003\
\163\000\163\000\020\003\163\000\050\001\163\000\021\003\051\001\
\052\001\163\000\036\003\031\000\037\003\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
\163\000\038\003\039\003\045\003\163\000\163\000\163\000\163\000\
\163\000\163\000\051\003\163\000\042\003\055\003\070\001\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\059\003\065\003\
\062\003\163\000\163\000\070\003\071\003\163\000\079\001\079\001\
\079\001\072\003\079\001\079\001\073\003\079\001\079\001\079\001\
\079\001\079\001\079\001\079\001\079\001\074\003\148\001\148\001\
\076\003\070\001\078\003\033\000\070\001\080\003\083\003\070\001\
\070\001\070\001\148\001\148\001\148\001\084\003\148\001\148\001\
\037\000\085\003\038\000\092\003\148\001\079\001\079\001\093\003\
\094\003\148\001\148\001\148\001\148\001\026\001\148\001\039\002\
\040\002\041\002\042\002\043\002\044\002\045\002\046\002\047\002\
\048\002\049\002\079\001\079\001\079\001\001\001\007\000\079\001\
\079\001\209\000\038\001\079\001\079\001\028\001\027\000\167\001\
\079\001\079\001\058\001\079\001\162\000\079\001\164\001\028\000\
\148\001\079\001\007\001\007\000\140\000\079\001\079\001\079\001\
\079\001\079\001\079\001\079\001\079\001\079\001\079\001\079\001\
\079\001\142\000\127\000\169\000\125\000\043\001\079\001\173\001\
\079\001\079\001\147\000\079\001\178\001\186\001\142\000\079\001\
\079\001\142\000\079\001\190\001\192\001\058\000\043\001\008\003\
\099\002\011\003\079\001\251\001\079\001\162\000\167\000\167\000\
\167\000\062\000\167\000\167\000\167\000\167\000\167\000\167\000\
\167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
\167\000\167\000\167\000\167\000\125\002\167\000\167\000\167\000\
\167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
\167\000\167\000\167\000\167\000\167\000\167\000\167\000\165\002\
\060\001\166\002\064\002\201\002\209\002\049\003\058\003\065\002\
\120\002\084\001\253\002\066\002\193\001\067\002\076\002\211\001\
\068\002\212\001\167\000\167\000\167\000\213\001\215\001\167\000\
\167\000\214\001\082\003\167\000\167\000\081\003\095\003\000\000\
\167\000\167\000\000\000\167\000\167\000\167\000\000\000\000\000\
\000\000\167\000\000\000\000\000\000\000\167\000\167\000\167\000\
\167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
\167\000\000\000\000\000\000\000\167\000\167\000\167\000\167\000\
\167\000\167\000\000\000\167\000\000\000\000\000\167\000\167\000\
\167\000\167\000\167\000\167\000\167\000\167\000\000\000\000\000\
\000\000\167\000\167\000\000\000\167\000\156\000\156\000\156\000\
\000\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
\156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
\156\000\156\000\156\000\000\000\156\000\156\000\156\000\156\000\
\156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
\156\000\156\000\156\000\156\000\156\000\156\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\156\000\156\000\156\000\000\000\000\000\156\000\156\000\
\000\000\000\000\156\000\156\000\000\000\000\000\000\000\156\000\
\156\000\000\000\156\000\156\000\156\000\000\000\000\000\000\000\
\156\000\000\000\000\000\000\000\156\000\156\000\156\000\156\000\
\156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
\000\000\000\000\000\000\156\000\156\000\156\000\156\000\156\000\
\156\000\000\000\156\000\000\000\000\000\000\000\156\000\156\000\
\156\000\156\000\156\000\156\000\156\000\000\000\000\000\000\000\
\156\000\156\000\000\000\156\000\170\002\171\002\224\002\000\000\
\104\002\173\002\006\000\081\002\204\000\205\000\206\000\207\000\
\174\002\083\000\208\000\009\000\010\000\011\000\012\000\013\000\
\014\000\015\000\000\000\016\000\017\000\018\000\019\000\020\000\
\021\000\022\000\023\000\024\000\060\000\026\000\027\000\028\000\
\029\000\030\000\031\000\209\000\210\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\211\000\212\000\213\000\000\000\000\000\214\000\215\000\000\000\
\000\000\216\000\217\000\000\000\000\000\000\000\218\000\219\000\
\000\000\220\000\000\000\156\000\000\000\000\000\000\000\175\002\
\000\000\000\000\000\000\176\002\177\002\178\002\179\002\180\002\
\181\002\182\002\000\000\000\000\000\000\183\002\184\002\000\000\
\000\000\000\000\033\000\034\000\185\002\036\000\084\000\085\000\
\000\000\221\000\000\000\000\000\000\000\222\000\223\000\037\000\
\186\002\038\000\000\000\000\000\000\000\000\000\000\000\043\000\
\224\000\000\000\105\002\079\001\079\001\079\001\000\000\079\001\
\079\001\000\000\079\001\079\001\079\001\079\001\079\001\079\001\
\079\001\079\001\000\000\000\000\111\001\111\001\111\001\111\001\
\111\001\111\001\111\001\111\001\111\001\111\001\111\001\111\001\
\111\001\000\000\111\001\111\001\111\001\000\000\111\001\111\001\
\111\001\111\001\079\001\079\001\111\001\000\000\000\000\000\000\
\111\001\111\001\111\001\111\001\111\001\000\000\111\001\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\079\001\
\079\001\079\001\000\000\000\000\079\001\079\001\000\000\000\000\
\079\001\079\001\000\000\000\000\000\000\079\001\079\001\000\000\
\079\001\000\000\079\001\000\000\000\000\000\000\079\001\000\000\
\111\001\000\000\079\001\079\001\079\001\079\001\079\001\079\001\
\079\001\079\001\079\001\079\001\079\001\079\001\000\000\000\000\
\000\000\000\000\000\000\079\001\104\001\079\001\079\001\000\000\
\079\001\000\000\000\000\000\000\079\001\079\001\000\000\079\001\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\079\001\
\000\000\079\001\170\002\171\002\224\002\000\000\104\002\173\002\
\000\000\081\002\204\000\205\000\206\000\207\000\066\000\083\000\
\208\000\000\000\000\000\104\001\104\001\104\001\104\001\104\001\
\104\001\104\001\104\001\104\001\104\001\104\001\104\001\104\001\
\000\000\104\001\104\001\104\001\000\000\104\001\104\001\104\001\
\104\001\209\000\210\000\104\001\000\000\000\000\000\000\104\001\
\104\001\104\001\104\001\104\001\000\000\104\001\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\211\000\212\000\
\213\000\000\000\000\000\214\000\215\000\000\000\000\000\216\000\
\217\000\104\001\104\001\000\000\218\000\219\000\000\000\220\000\
\000\000\156\000\000\000\000\000\000\000\175\002\000\000\104\001\
\000\000\176\002\177\002\178\002\179\002\180\002\181\002\182\002\
\000\000\000\000\000\000\183\002\184\002\000\000\000\000\000\000\
\000\000\000\000\185\002\000\000\084\000\085\000\000\000\221\000\
\000\000\000\000\000\000\222\000\223\000\249\000\186\002\000\000\
\000\000\000\000\000\000\249\000\249\000\000\000\224\000\000\000\
\105\002\249\000\000\000\000\000\249\000\249\000\249\000\249\000\
\249\000\249\000\249\000\000\000\249\000\249\000\249\000\249\000\
\249\000\249\000\249\000\249\000\249\000\249\000\249\000\249\000\
\249\000\249\000\249\000\249\000\106\001\106\001\106\001\106\001\
\106\001\106\001\106\001\106\001\106\001\106\001\106\001\106\001\
\106\001\000\000\106\001\106\001\106\001\000\000\106\001\106\001\
\106\001\106\001\000\000\249\000\106\001\000\000\000\000\000\000\
\106\001\106\001\106\001\106\001\106\001\000\000\106\001\000\000\
\000\000\249\000\249\000\000\000\000\000\249\000\000\000\249\000\
\249\000\249\000\159\001\000\000\000\000\000\000\000\000\159\001\
\000\000\000\000\000\000\000\000\159\001\159\001\159\001\159\001\
\000\000\159\001\000\000\249\000\249\000\000\000\249\000\000\000\
\106\001\000\000\000\000\000\000\000\000\238\000\000\000\000\000\
\249\000\000\000\249\000\238\000\238\000\000\000\000\000\000\000\
\249\000\238\000\249\000\000\000\238\000\238\000\238\000\238\000\
\238\000\238\000\238\000\159\001\238\000\238\000\238\000\238\000\
\238\000\238\000\238\000\238\000\238\000\238\000\238\000\238\000\
\238\000\238\000\238\000\238\000\107\001\107\001\107\001\107\001\
\107\001\107\001\107\001\107\001\107\001\107\001\107\001\107\001\
\107\001\000\000\107\001\107\001\107\001\000\000\107\001\107\001\
\107\001\107\001\000\000\238\000\107\001\000\000\000\000\000\000\
\107\001\107\001\107\001\107\001\107\001\000\000\107\001\000\000\
\000\000\238\000\238\000\000\000\000\000\238\000\000\000\238\000\
\238\000\238\000\160\001\000\000\000\000\000\000\000\000\160\001\
\000\000\000\000\000\000\000\000\160\001\160\001\160\001\160\001\
\000\000\160\001\000\000\238\000\238\000\000\000\238\000\000\000\
\107\001\000\000\000\000\000\000\000\000\244\000\000\000\000\000\
\238\000\000\000\238\000\244\000\244\000\000\000\000\000\000\000\
\238\000\244\000\238\000\000\000\244\000\244\000\244\000\244\000\
\244\000\244\000\244\000\160\001\244\000\244\000\244\000\244\000\
\244\000\244\000\244\000\244\000\244\000\244\000\244\000\244\000\
\244\000\244\000\244\000\244\000\000\000\000\000\121\001\121\001\
\121\001\121\001\121\001\121\001\121\001\121\001\121\001\121\001\
\121\001\000\000\121\001\121\001\121\001\000\000\121\001\121\001\
\121\001\121\001\000\000\244\000\121\001\000\000\000\000\000\000\
\000\000\121\001\121\001\121\001\121\001\000\000\121\001\000\000\
\000\000\244\000\244\000\000\000\000\000\244\000\000\000\244\000\
\244\000\244\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\244\000\244\000\000\000\244\000\000\000\
\121\001\000\000\000\000\000\000\000\000\239\000\000\000\000\000\
\244\000\000\000\244\000\239\000\239\000\000\000\000\000\000\000\
\244\000\239\000\244\000\000\000\239\000\239\000\239\000\239\000\
\239\000\239\000\239\000\000\000\239\000\239\000\239\000\239\000\
\239\000\239\000\239\000\239\000\239\000\239\000\239\000\239\000\
\239\000\239\000\239\000\239\000\000\000\000\000\123\001\123\001\
\123\001\123\001\123\001\123\001\123\001\123\001\123\001\123\001\
\123\001\000\000\123\001\123\001\123\001\000\000\123\001\123\001\
\123\001\123\001\000\000\239\000\123\001\000\000\000\000\000\000\
\000\000\123\001\123\001\123\001\123\001\049\001\123\001\000\000\
\000\000\239\000\239\000\000\000\070\001\239\000\000\000\239\000\
\239\000\239\000\000\000\070\001\070\001\070\001\070\001\070\001\
\070\001\070\001\070\001\000\000\000\000\000\000\050\001\000\000\
\000\000\051\001\052\001\239\000\239\000\031\000\239\000\000\000\
\123\001\000\000\000\000\000\000\000\000\070\001\000\000\000\000\
\239\000\000\000\239\000\070\001\070\001\070\001\157\001\157\001\
\239\000\000\000\239\000\000\000\157\001\000\000\000\000\000\000\
\000\000\157\001\157\001\157\001\157\001\000\000\157\001\000\000\
\070\001\070\001\070\001\000\000\116\002\070\001\070\001\228\001\
\000\000\070\001\070\001\000\000\000\000\000\000\070\001\070\001\
\070\001\070\001\000\000\070\001\070\001\070\001\070\001\070\001\
\070\001\000\000\000\000\000\000\000\000\033\000\000\000\229\001\
\157\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\037\000\000\000\038\000\000\000\070\001\070\001\
\000\000\070\001\000\000\000\000\000\000\070\001\070\001\000\000\
\000\000\000\000\000\000\000\000\000\000\163\000\163\000\163\000\
\070\001\070\001\163\000\163\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\000\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\163\000\163\000\163\000\000\000\000\000\163\000\163\000\
\000\000\000\000\163\000\163\000\000\000\000\000\000\000\163\000\
\163\000\000\000\163\000\163\000\163\000\000\000\000\000\000\000\
\163\000\000\000\000\000\000\000\163\000\163\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
\000\000\000\000\000\000\163\000\163\000\163\000\163\000\163\000\
\163\000\000\000\163\000\000\000\000\000\000\000\163\000\163\000\
\163\000\163\000\163\000\163\000\163\000\170\002\171\002\172\002\
\163\000\163\000\173\002\006\000\081\002\204\000\205\000\206\000\
\207\000\174\002\083\000\208\000\009\000\010\000\011\000\012\000\
\013\000\014\000\015\000\000\000\016\000\017\000\018\000\019\000\
\020\000\021\000\022\000\023\000\024\000\060\000\026\000\027\000\
\028\000\029\000\030\000\031\000\209\000\210\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\211\000\212\000\213\000\000\000\000\000\214\000\215\000\
\000\000\000\000\216\000\217\000\000\000\000\000\000\000\218\000\
\219\000\000\000\220\000\000\000\156\000\000\000\000\000\000\000\
\175\002\000\000\000\000\000\000\176\002\177\002\178\002\179\002\
\180\002\181\002\182\002\000\000\000\000\000\000\183\002\184\002\
\000\000\000\000\000\000\033\000\034\000\185\002\036\000\084\000\
\085\000\000\000\221\000\000\000\000\000\000\000\222\000\223\000\
\037\000\186\002\038\000\039\000\040\000\128\001\000\000\000\000\
\043\000\224\000\000\000\006\000\203\000\204\000\205\000\206\000\
\207\000\008\000\083\000\208\000\009\000\010\000\011\000\012\000\
\013\000\014\000\015\000\000\000\016\000\017\000\018\000\019\000\
\020\000\021\000\022\000\023\000\024\000\060\000\026\000\027\000\
\028\000\029\000\030\000\031\000\209\000\210\000\158\001\158\001\
\000\000\000\000\000\000\000\000\158\001\000\000\000\000\000\000\
\000\000\158\001\158\001\158\001\158\001\000\000\158\001\000\000\
\000\000\211\000\212\000\213\000\000\000\000\000\214\000\215\000\
\000\000\000\000\216\000\217\000\000\000\000\000\000\000\218\000\
\219\000\000\000\220\000\000\000\156\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\158\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\033\000\034\000\000\000\036\000\084\000\
\085\000\000\000\221\000\000\000\000\000\000\000\222\000\223\000\
\037\000\000\000\038\000\000\000\034\002\000\000\195\000\000\000\
\043\000\224\000\006\000\203\000\204\000\205\000\206\000\207\000\
\008\000\083\000\208\000\009\000\010\000\011\000\012\000\013\000\
\014\000\015\000\000\000\016\000\017\000\018\000\019\000\020\000\
\021\000\022\000\023\000\024\000\060\000\026\000\027\000\028\000\
\029\000\030\000\031\000\209\000\210\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\211\000\212\000\213\000\000\000\000\000\214\000\215\000\000\000\
\000\000\216\000\217\000\170\002\171\002\224\002\218\000\219\000\
\173\002\220\000\081\002\204\000\205\000\206\000\207\000\066\000\
\083\000\208\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\033\000\034\000\000\000\036\000\084\000\085\000\
\000\000\221\000\209\000\210\000\000\000\222\000\223\000\037\000\
\000\000\038\000\000\000\000\000\000\000\000\000\000\000\043\000\
\224\000\000\000\000\000\000\000\000\000\000\000\000\000\211\000\
\212\000\213\000\000\000\000\000\214\000\215\000\000\000\000\000\
\216\000\217\000\000\000\000\000\000\000\218\000\219\000\000\000\
\220\000\000\000\156\000\000\000\000\000\000\000\175\002\000\000\
\000\000\000\000\176\002\177\002\178\002\179\002\180\002\181\002\
\182\002\000\000\000\000\000\000\183\002\184\002\000\000\000\000\
\000\000\000\000\000\000\185\002\000\000\084\000\085\000\000\000\
\221\000\000\000\000\000\000\000\222\000\223\000\000\000\186\002\
\004\000\005\000\000\000\000\000\006\000\007\000\000\000\224\000\
\000\000\000\000\008\000\000\000\000\000\009\000\010\000\011\000\
\012\000\013\000\014\000\015\000\000\000\016\000\017\000\018\000\
\019\000\020\000\021\000\022\000\023\000\024\000\025\000\026\000\
\027\000\028\000\029\000\030\000\031\000\100\000\000\000\000\000\
\101\000\000\000\102\000\083\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\049\001\000\000\000\000\000\000\103\000\000\000\000\000\
\104\000\070\001\000\000\000\000\000\000\105\000\106\000\000\000\
\070\001\070\001\070\001\070\001\070\001\000\000\070\001\070\001\
\000\000\032\000\050\001\000\000\000\000\051\001\052\001\000\000\
\000\000\031\000\107\000\108\000\109\000\000\000\000\000\110\000\
\111\000\000\000\070\001\112\000\033\000\034\000\035\000\036\000\
\070\001\070\001\000\000\113\000\000\000\000\000\000\000\000\000\
\000\000\037\000\000\000\038\000\039\000\040\000\000\000\041\000\
\042\000\043\000\000\000\114\000\044\000\070\001\070\001\070\001\
\116\002\000\000\070\001\070\001\000\000\000\000\070\001\070\001\
\084\000\085\000\000\000\070\001\070\001\034\002\070\001\000\000\
\000\000\000\000\070\001\000\000\203\000\204\000\205\000\206\000\
\207\000\033\000\083\000\208\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\037\000\000\000\
\038\000\000\000\000\000\070\001\070\001\000\000\070\001\000\000\
\000\000\000\000\070\001\070\001\209\000\210\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\070\001\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\211\000\212\000\213\000\000\000\000\000\214\000\215\000\
\000\000\000\000\216\000\217\000\070\001\000\000\000\000\218\000\
\219\000\000\000\220\000\070\001\070\001\070\001\070\001\070\001\
\234\002\070\001\070\001\153\001\153\001\000\000\153\001\153\001\
\000\000\000\000\000\000\090\000\153\001\000\000\000\000\000\000\
\000\000\153\001\153\001\153\001\153\001\000\000\153\001\084\000\
\085\000\000\000\221\000\070\001\070\001\000\000\222\000\223\000\
\000\000\000\000\000\000\000\000\090\000\000\000\000\000\090\000\
\090\000\224\000\000\000\090\000\000\000\000\000\000\000\000\000\
\070\001\070\001\070\001\000\000\000\000\070\001\070\001\000\000\
\153\001\070\001\070\001\000\000\000\000\000\000\070\001\070\001\
\071\001\070\001\000\000\000\000\000\000\070\001\000\000\071\001\
\071\001\071\001\071\001\071\001\090\000\071\001\071\001\000\000\
\000\000\090\000\000\000\090\000\000\000\092\000\090\000\090\000\
\090\000\090\000\090\000\090\000\000\000\000\000\070\001\070\001\
\000\000\070\001\000\000\000\000\000\000\070\001\070\001\071\001\
\071\001\000\000\000\000\090\000\000\000\000\000\092\000\000\000\
\070\001\092\000\092\000\000\000\000\000\092\000\000\000\000\000\
\090\000\000\000\090\000\000\000\071\001\071\001\071\001\000\000\
\000\000\071\001\071\001\000\000\000\000\071\001\071\001\000\000\
\000\000\000\000\071\001\071\001\034\002\071\001\000\000\000\000\
\000\000\071\001\000\000\203\000\204\000\205\000\206\000\207\000\
\000\000\083\000\208\000\092\000\000\000\092\000\000\000\000\000\
\092\000\092\000\092\000\092\000\092\000\000\000\000\000\000\000\
\000\000\000\000\071\001\071\001\000\000\071\001\000\000\000\000\
\000\000\071\001\071\001\209\000\210\000\092\000\000\000\000\000\
\000\000\000\000\000\000\000\000\071\001\000\000\000\000\000\000\
\000\000\000\000\092\000\000\000\092\000\000\000\000\000\000\000\
\211\000\212\000\213\000\000\000\000\000\214\000\215\000\000\000\
\000\000\216\000\217\000\037\002\000\000\000\000\218\000\219\000\
\000\000\220\000\203\000\204\000\205\000\206\000\207\000\000\000\
\083\000\208\000\000\000\000\000\000\000\000\000\022\003\000\000\
\000\000\000\000\000\000\000\000\000\000\203\000\204\000\205\000\
\206\000\207\000\000\000\083\000\208\000\000\000\084\000\085\000\
\000\000\221\000\209\000\210\000\000\000\222\000\223\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\224\000\000\000\000\000\000\000\000\000\209\000\210\000\211\000\
\212\000\213\000\000\000\000\000\214\000\215\000\000\000\000\000\
\216\000\217\000\000\000\000\000\000\000\218\000\219\000\000\000\
\220\000\000\000\211\000\212\000\213\000\000\000\000\000\214\000\
\215\000\000\000\000\000\216\000\217\000\087\003\000\000\000\000\
\218\000\219\000\000\000\220\000\203\000\204\000\205\000\206\000\
\207\000\000\000\083\000\208\000\000\000\084\000\085\000\000\000\
\221\000\000\000\000\000\000\000\222\000\223\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\224\000\
\084\000\085\000\000\000\221\000\209\000\210\000\000\000\222\000\
\223\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\224\000\000\000\000\000\000\000\000\000\000\000\
\000\000\211\000\212\000\213\000\000\000\000\000\214\000\215\000\
\000\000\000\000\216\000\217\000\070\001\000\000\000\000\218\000\
\219\000\000\000\220\000\070\001\070\001\070\001\070\001\070\001\
\000\000\070\001\070\001\154\001\154\001\000\000\154\001\154\001\
\000\000\000\000\000\000\000\000\154\001\000\000\000\000\000\000\
\000\000\154\001\154\001\154\001\154\001\000\000\154\001\084\000\
\085\000\000\000\221\000\070\001\070\001\000\000\222\000\223\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\224\000\000\000\000\000\000\000\000\000\000\000\000\000\
\070\001\070\001\070\001\000\000\000\000\070\001\070\001\183\000\
\154\001\070\001\070\001\000\000\000\000\006\000\070\001\070\001\
\000\000\070\001\000\000\008\000\000\000\000\000\009\000\010\000\
\011\000\012\000\013\000\014\000\015\000\000\000\016\000\017\000\
\018\000\019\000\020\000\021\000\022\000\023\000\024\000\060\000\
\026\000\027\000\028\000\029\000\030\000\031\000\070\001\070\001\
\000\000\070\001\000\000\000\000\000\000\070\001\070\001\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\070\001\000\000\125\001\125\001\125\001\125\001\125\001\125\001\
\125\001\125\001\125\001\125\001\125\001\000\000\125\001\125\001\
\125\001\000\000\125\001\125\001\125\001\125\001\000\000\000\000\
\125\001\000\000\184\000\000\000\000\000\125\001\125\001\125\001\
\125\001\000\000\125\001\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\033\000\034\000\000\000\
\036\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\037\000\113\000\038\000\039\000\040\000\113\000\
\113\000\000\000\043\000\113\000\125\001\113\000\000\000\000\000\
\113\000\113\000\113\000\113\000\113\000\113\000\113\000\000\000\
\113\000\113\000\113\000\113\000\113\000\113\000\113\000\113\000\
\113\000\113\000\113\000\113\000\113\000\113\000\113\000\113\000\
\000\000\000\000\113\000\113\000\113\000\113\000\113\000\113\000\
\113\000\113\000\113\000\113\000\113\000\113\000\113\000\113\000\
\113\000\113\000\113\000\113\000\113\000\113\000\113\000\113\000\
\113\000\113\000\000\000\113\000\113\000\113\000\000\000\113\000\
\113\000\113\000\113\000\113\000\113\000\113\000\113\000\113\000\
\113\000\113\000\113\000\113\000\113\000\113\000\113\000\113\000\
\000\000\000\000\000\000\000\000\000\000\000\000\113\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\113\000\
\113\000\000\000\113\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\113\000\000\000\113\000\000\000\
\000\000\113\000\004\000\005\000\113\000\000\000\006\000\007\000\
\000\000\000\000\000\000\000\000\008\000\000\000\000\000\009\000\
\010\000\011\000\012\000\013\000\014\000\015\000\000\000\016\000\
\017\000\018\000\019\000\020\000\021\000\022\000\023\000\024\000\
\025\000\026\000\027\000\028\000\029\000\030\000\031\000\195\000\
\000\000\000\000\000\000\006\000\000\000\000\000\000\000\000\000\
\000\000\008\000\000\000\000\000\009\000\010\000\011\000\012\000\
\013\000\014\000\015\000\000\000\016\000\017\000\018\000\019\000\
\020\000\021\000\022\000\023\000\024\000\060\000\026\000\027\000\
\028\000\029\000\030\000\031\000\000\000\000\000\000\000\137\001\
\137\001\137\001\137\001\137\001\137\001\137\001\137\001\000\000\
\000\000\000\000\000\000\137\001\137\001\137\001\000\000\137\001\
\137\001\137\001\137\001\000\000\000\000\137\001\033\000\034\000\
\035\000\036\000\137\001\137\001\137\001\137\001\000\000\137\001\
\000\000\000\000\000\000\037\000\196\000\038\000\039\000\040\000\
\000\000\041\000\042\000\043\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\033\000\034\000\000\000\036\000\000\000\
\000\000\137\001\000\000\000\000\000\000\000\000\000\000\000\000\
\037\000\000\000\038\000\254\000\254\000\000\000\000\000\000\000\
\043\000\254\000\000\000\000\000\254\000\254\000\254\000\254\000\
\254\000\254\000\254\000\000\000\254\000\254\000\254\000\254\000\
\254\000\254\000\254\000\254\000\254\000\254\000\254\000\254\000\
\254\000\254\000\254\000\254\000\000\000\000\000\138\001\138\001\
\138\001\138\001\138\001\138\001\138\001\138\001\000\000\000\000\
\000\000\000\000\138\001\138\001\138\001\000\000\138\001\138\001\
\138\001\138\001\000\000\254\000\138\001\000\000\000\000\000\000\
\000\000\138\001\138\001\138\001\138\001\000\000\138\001\000\000\
\000\000\000\000\254\000\000\000\000\000\000\000\000\000\024\000\
\254\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\254\000\254\000\000\000\254\000\000\000\
\138\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\254\000\000\000\254\000\000\000\000\000\000\000\000\000\000\000\
\254\000\000\000\254\000\006\000\203\000\204\000\205\000\206\000\
\207\000\008\000\083\000\208\000\009\000\010\000\011\000\012\000\
\013\000\014\000\015\000\000\000\016\000\017\000\018\000\019\000\
\020\000\021\000\022\000\023\000\024\000\060\000\026\000\027\000\
\028\000\029\000\030\000\031\000\209\000\210\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\211\000\212\000\213\000\000\000\000\000\214\000\215\000\
\000\000\000\000\216\000\217\000\000\000\000\000\000\000\218\000\
\219\000\000\000\220\000\000\000\081\002\204\000\205\000\206\000\
\207\000\066\000\083\000\208\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\033\000\034\000\000\000\036\000\084\000\
\085\000\000\000\221\000\000\000\209\000\210\000\222\000\223\000\
\037\000\000\000\038\000\000\000\000\000\000\000\000\000\000\000\
\043\000\224\000\082\002\000\000\000\000\000\000\000\000\000\000\
\000\000\211\000\212\000\213\000\000\000\000\000\214\000\215\000\
\000\000\000\000\216\000\217\000\000\000\000\000\000\000\218\000\
\219\000\000\000\220\000\000\000\223\001\083\002\000\000\151\001\
\151\001\151\001\000\000\151\001\151\001\000\000\000\000\000\000\
\000\000\151\001\000\000\000\000\000\000\000\000\151\001\151\001\
\151\001\151\001\000\000\151\001\000\000\000\000\000\000\084\000\
\085\000\000\000\221\000\000\000\000\000\000\000\222\000\223\000\
\000\000\000\000\000\000\000\000\000\000\006\000\000\000\000\000\
\000\000\224\000\070\000\008\000\083\000\000\000\009\000\010\000\
\011\000\012\000\013\000\014\000\015\000\151\001\016\000\017\000\
\018\000\019\000\020\000\021\000\022\000\023\000\024\000\060\000\
\026\000\027\000\028\000\029\000\030\000\031\000\006\000\000\000\
\000\000\000\000\000\000\000\000\008\000\000\000\000\000\009\000\
\010\000\011\000\012\000\013\000\014\000\015\000\000\000\016\000\
\017\000\018\000\019\000\020\000\021\000\022\000\023\000\024\000\
\060\000\026\000\027\000\028\000\029\000\030\000\031\000\152\001\
\152\001\152\001\000\000\152\001\152\001\000\000\000\000\000\000\
\000\000\152\001\000\000\000\000\000\000\000\000\152\001\152\001\
\152\001\152\001\000\000\152\001\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\033\000\034\000\000\000\
\036\000\084\000\085\000\000\000\000\000\169\000\000\000\000\000\
\000\000\000\000\037\000\000\000\038\000\247\001\000\000\000\000\
\000\000\000\000\043\000\000\000\000\000\152\001\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\033\000\034\000\
\000\000\036\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\037\000\025\001\038\000\000\000\000\000\
\000\000\000\000\025\001\043\000\000\000\025\001\025\001\025\001\
\025\001\025\001\025\001\025\001\000\000\025\001\025\001\025\001\
\025\001\025\001\025\001\025\001\025\001\025\001\025\001\025\001\
\025\001\025\001\025\001\025\001\025\001\006\000\059\000\000\000\
\000\000\000\000\000\000\008\000\000\000\000\000\009\000\010\000\
\011\000\012\000\013\000\014\000\015\000\000\000\016\000\017\000\
\018\000\019\000\020\000\021\000\022\000\023\000\024\000\060\000\
\026\000\027\000\028\000\029\000\030\000\031\000\000\000\000\000\
\000\000\000\000\025\001\025\001\000\000\139\001\139\001\139\001\
\139\001\139\001\139\001\139\001\139\001\000\000\000\000\000\000\
\000\000\139\001\139\001\139\001\000\000\139\001\139\001\139\001\
\139\001\000\000\000\000\139\001\025\001\025\001\000\000\025\001\
\139\001\139\001\139\001\139\001\000\000\139\001\000\000\000\000\
\000\000\025\001\000\000\025\001\000\000\000\000\000\000\000\000\
\000\000\025\001\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\033\000\034\000\000\000\
\036\000\000\000\000\000\000\000\000\000\000\000\000\000\139\001\
\000\000\000\000\037\000\006\000\038\000\000\000\000\000\000\000\
\000\000\008\000\043\000\000\000\009\000\010\000\011\000\012\000\
\013\000\014\000\015\000\000\000\016\000\017\000\018\000\019\000\
\020\000\021\000\022\000\023\000\024\000\060\000\026\000\027\000\
\028\000\029\000\030\000\031\000\006\000\000\000\000\000\000\000\
\000\000\000\000\008\000\000\000\000\000\009\000\010\000\011\000\
\012\000\013\000\014\000\015\000\000\000\016\000\017\000\018\000\
\019\000\020\000\021\000\022\000\023\000\024\000\060\000\026\000\
\027\000\028\000\029\000\030\000\031\000\000\000\000\000\000\000\
\000\000\000\000\169\000\000\000\140\001\140\001\140\001\140\001\
\140\001\140\001\000\000\000\000\000\000\000\000\000\000\000\000\
\140\001\140\001\140\001\000\000\140\001\140\001\140\001\140\001\
\000\000\000\000\140\001\033\000\034\000\000\000\036\000\140\001\
\140\001\140\001\140\001\000\000\140\001\000\000\000\000\000\000\
\037\000\000\000\038\000\254\002\000\000\000\000\000\000\000\000\
\043\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\033\000\034\000\000\000\036\000\
\000\000\000\000\000\000\000\000\000\000\000\000\140\001\000\000\
\000\000\037\000\006\000\038\000\000\000\000\000\000\000\000\000\
\008\000\043\000\000\000\009\000\010\000\011\000\012\000\013\000\
\014\000\015\000\000\000\016\000\017\000\018\000\019\000\020\000\
\021\000\022\000\023\000\024\000\060\000\026\000\027\000\028\000\
\029\000\030\000\031\000\006\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\009\000\010\000\011\000\012\000\
\013\000\014\000\015\000\000\000\016\000\017\000\018\000\019\000\
\020\000\021\000\022\000\023\000\024\000\060\000\026\000\027\000\
\028\000\029\000\030\000\031\000\155\001\000\000\000\000\155\001\
\155\001\000\000\000\000\000\000\000\000\155\001\000\000\000\000\
\000\000\000\000\155\001\155\001\155\001\155\001\000\000\155\001\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\033\000\034\000\000\000\036\000\000\000\137\000\
\137\000\137\000\137\000\137\000\000\000\137\000\137\000\037\000\
\000\000\038\000\000\000\000\000\000\000\000\000\000\000\043\000\
\000\000\155\001\000\000\000\000\156\001\000\000\000\000\156\001\
\156\001\000\000\000\000\033\000\034\000\156\001\036\000\137\000\
\137\000\137\000\156\001\156\001\156\001\156\001\000\000\156\001\
\037\000\000\000\038\000\000\000\000\000\000\000\000\000\000\000\
\043\000\000\000\000\000\000\000\137\000\137\000\137\000\000\000\
\000\000\137\000\137\000\000\000\000\000\137\000\137\000\000\000\
\000\000\000\000\137\000\137\000\000\000\137\000\000\000\137\000\
\000\000\156\001\000\000\203\000\204\000\205\000\206\000\207\000\
\000\000\083\000\208\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\137\000\137\000\000\000\137\000\000\000\000\000\
\000\000\137\000\137\000\209\000\210\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\137\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\211\000\212\000\213\000\000\000\000\000\214\000\215\000\000\000\
\000\000\216\000\217\000\000\000\000\000\000\000\218\000\219\000\
\000\000\220\000\000\000\223\001\000\000\000\000\000\000\203\000\
\204\000\205\000\206\000\207\000\000\000\083\000\208\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\084\000\085\000\
\000\000\221\000\000\000\000\000\000\000\222\000\223\000\209\000\
\210\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\224\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\211\000\212\000\213\000\000\000\
\000\000\214\000\215\000\000\000\000\000\216\000\217\000\000\000\
\000\000\000\000\218\000\219\000\000\000\220\000\000\000\134\002\
\000\000\000\000\000\000\133\000\133\000\133\000\133\000\133\000\
\000\000\133\000\133\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\084\000\085\000\000\000\221\000\000\000\000\000\
\000\000\222\000\223\000\133\000\133\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\224\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\133\000\133\000\133\000\000\000\000\000\133\000\133\000\000\000\
\000\000\133\000\133\000\000\000\000\000\000\000\133\000\133\000\
\000\000\133\000\000\000\133\000\000\000\000\000\000\000\203\000\
\204\000\205\000\206\000\207\000\000\000\083\000\208\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\133\000\133\000\
\000\000\133\000\000\000\000\000\000\000\133\000\133\000\209\000\
\210\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\133\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\211\000\212\000\213\000\000\000\
\000\000\214\000\215\000\000\000\000\000\216\000\217\000\000\000\
\000\000\000\000\218\000\219\000\000\000\254\000\203\000\204\000\
\205\000\206\000\207\000\000\000\083\000\208\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\203\000\204\000\205\000\206\000\207\000\000\000\083\000\
\208\000\000\000\084\000\085\000\000\000\221\000\209\000\210\000\
\000\000\222\000\223\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\224\000\000\000\000\000\000\000\
\000\000\209\000\210\000\211\000\212\000\213\000\000\000\000\000\
\214\000\215\000\000\000\000\000\216\000\217\000\000\000\000\000\
\000\000\218\000\219\000\000\000\000\001\000\000\211\000\212\000\
\213\000\000\000\000\000\214\000\215\000\000\000\000\000\216\000\
\217\000\000\000\000\000\000\000\218\000\219\000\000\000\220\000\
\203\000\204\000\205\000\206\000\207\000\000\000\083\000\208\000\
\000\000\084\000\085\000\000\000\221\000\000\000\000\000\000\000\
\222\000\223\000\000\000\203\000\204\000\205\000\206\000\207\000\
\000\000\083\000\208\000\224\000\084\000\085\000\000\000\221\000\
\209\000\210\000\000\000\222\000\223\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\224\000\000\000\
\000\000\000\000\000\000\209\000\210\000\211\000\212\000\213\000\
\000\000\000\000\214\000\215\000\000\000\000\000\216\000\217\000\
\000\000\000\000\000\000\218\000\219\000\000\000\114\001\000\000\
\211\000\212\000\213\000\000\000\000\000\214\000\215\000\000\000\
\000\000\216\000\217\000\000\000\000\000\000\000\218\000\219\000\
\000\000\116\001\203\000\204\000\205\000\206\000\207\000\000\000\
\083\000\208\000\000\000\084\000\085\000\000\000\221\000\000\000\
\000\000\000\000\222\000\223\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\224\000\084\000\085\000\
\000\000\221\000\209\000\210\000\000\000\222\000\223\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\224\000\000\000\000\000\000\000\000\000\000\000\000\000\211\000\
\212\000\213\000\000\000\000\000\214\000\215\000\000\000\000\000\
\216\000\217\000\000\000\000\000\000\000\218\000\219\000\000\000\
\125\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\112\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\084\000\085\000\000\000\
\221\000\000\000\000\000\000\000\222\000\223\000\112\000\000\000\
\000\000\112\000\112\000\000\000\000\000\112\000\000\000\224\000\
\112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
\112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
\112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
\000\000\112\000\112\000\112\000\000\000\112\000\112\000\112\000\
\112\000\112\000\112\000\112\000\112\000\112\000\000\000\112\000\
\112\000\112\000\112\000\112\000\112\000\112\000\000\000\000\000\
\000\000\000\000\000\000\000\000\048\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\112\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\112\000\000\000\112\000\048\000\000\000\112\000\
\048\000\048\000\000\000\000\000\048\000\000\000\000\000\048\000\
\048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
\048\000\048\000\000\000\000\000\048\000\048\000\048\000\048\000\
\048\000\048\000\048\000\048\000\048\000\048\000\048\000\000\000\
\048\000\048\000\048\000\000\000\048\000\048\000\048\000\048\000\
\000\000\000\000\048\000\000\000\048\000\000\000\000\000\048\000\
\048\000\048\000\048\000\048\000\048\000\000\000\000\000\000\000\
\000\000\000\000\000\000\052\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\048\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\048\000\000\000\048\000\052\000\000\000\048\000\052\000\
\052\000\000\000\000\000\052\000\000\000\000\000\052\000\052\000\
\052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
\052\000\000\000\000\000\052\000\052\000\052\000\052\000\052\000\
\052\000\052\000\052\000\052\000\052\000\052\000\000\000\052\000\
\052\000\052\000\000\000\052\000\052\000\052\000\052\000\000\000\
\000\000\052\000\000\000\052\000\000\000\000\000\052\000\052\000\
\052\000\052\000\052\000\052\000\000\000\000\000\000\000\000\000\
\000\000\000\000\054\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\052\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\052\000\000\000\052\000\054\000\000\000\052\000\054\000\054\000\
\000\000\000\000\054\000\000\000\000\000\054\000\054\000\054\000\
\054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
\000\000\000\000\054\000\054\000\054\000\054\000\054\000\054\000\
\054\000\054\000\054\000\054\000\054\000\000\000\054\000\054\000\
\054\000\000\000\054\000\054\000\054\000\054\000\000\000\000\000\
\054\000\000\000\054\000\000\000\000\000\054\000\054\000\054\000\
\054\000\054\000\054\000\094\001\000\000\000\000\094\001\000\000\
\094\001\094\001\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\054\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\094\001\000\000\000\000\094\001\054\000\
\000\000\054\000\000\000\000\000\054\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\094\001\094\001\094\001\094\001\094\001\094\001\094\001\094\001\
\094\001\094\001\094\001\094\001\094\001\000\000\094\001\094\001\
\094\001\000\000\094\001\094\001\094\001\094\001\000\000\000\000\
\094\001\023\000\000\000\000\000\094\001\094\001\094\001\094\001\
\094\001\095\001\094\001\000\000\095\001\000\000\095\001\095\001\
\000\000\094\001\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\094\001\094\001\
\000\000\095\001\000\000\000\000\095\001\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\094\001\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\095\001\095\001\
\095\001\095\001\095\001\095\001\095\001\095\001\095\001\095\001\
\095\001\095\001\095\001\000\000\095\001\095\001\095\001\000\000\
\095\001\095\001\095\001\095\001\000\000\000\000\095\001\024\000\
\000\000\000\000\095\001\095\001\095\001\095\001\095\001\102\001\
\095\001\000\000\102\001\000\000\102\001\102\001\000\000\095\001\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\095\001\095\001\000\000\102\001\
\000\000\000\000\102\001\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\095\001\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\102\001\102\001\102\001\102\001\
\102\001\102\001\102\001\102\001\102\001\102\001\102\001\102\001\
\102\001\000\000\102\001\102\001\102\001\000\000\102\001\102\001\
\102\001\102\001\000\000\000\000\102\001\000\000\000\000\000\000\
\102\001\102\001\000\000\102\001\102\001\100\000\102\001\000\000\
\101\000\000\000\102\000\083\000\000\000\102\001\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\102\001\102\001\000\000\103\000\000\000\000\000\
\104\000\000\000\000\000\000\000\000\000\105\000\106\000\000\000\
\102\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\107\000\108\000\109\000\000\000\000\000\110\000\
\111\000\000\000\000\000\112\000\000\000\000\000\000\000\064\000\
\000\000\000\000\000\000\113\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\114\000\000\000\000\000\000\000\000\000\
\064\000\000\000\000\000\064\000\064\000\000\000\000\000\064\000\
\084\000\085\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\115\000\064\000\
\064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
\064\000\064\000\000\000\064\000\064\000\064\000\000\000\064\000\
\064\000\064\000\064\000\000\000\068\000\064\000\000\000\064\000\
\000\000\000\000\064\000\064\000\064\000\064\000\064\000\064\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\068\000\000\000\064\000\
\068\000\068\000\000\000\000\000\068\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\064\000\000\000\064\000\000\000\
\000\000\000\000\000\000\000\000\068\000\068\000\068\000\068\000\
\068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
\068\000\068\000\068\000\000\000\068\000\068\000\068\000\068\000\
\000\000\069\000\068\000\000\000\068\000\000\000\000\000\068\000\
\068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\069\000\000\000\068\000\069\000\069\000\000\000\
\000\000\069\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\068\000\000\000\068\000\000\000\000\000\000\000\000\000\
\000\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
\069\000\000\000\000\000\000\000\000\000\069\000\069\000\069\000\
\000\000\069\000\069\000\069\000\069\000\000\000\070\000\069\000\
\000\000\069\000\000\000\000\000\069\000\069\000\069\000\069\000\
\069\000\069\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\070\000\
\000\000\069\000\070\000\070\000\000\000\000\000\070\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\069\000\000\000\
\069\000\000\000\000\000\000\000\000\000\000\000\070\000\070\000\
\070\000\070\000\070\000\070\000\070\000\070\000\000\000\000\000\
\000\000\000\000\070\000\070\000\070\000\000\000\070\000\070\000\
\070\000\070\000\000\000\071\000\070\000\000\000\070\000\000\000\
\000\000\070\000\070\000\070\000\070\000\070\000\070\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\071\000\000\000\070\000\071\000\
\071\000\000\000\000\000\071\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\070\000\000\000\070\000\000\000\000\000\
\000\000\000\000\000\000\071\000\071\000\071\000\071\000\071\000\
\071\000\000\000\000\000\000\000\000\000\000\000\000\000\071\000\
\071\000\071\000\000\000\071\000\071\000\071\000\071\000\000\000\
\072\000\071\000\000\000\071\000\000\000\000\000\071\000\071\000\
\071\000\071\000\071\000\071\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\072\000\000\000\071\000\072\000\072\000\000\000\000\000\
\072\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\071\000\000\000\071\000\000\000\000\000\000\000\000\000\000\000\
\072\000\072\000\072\000\072\000\072\000\072\000\000\000\000\000\
\000\000\000\000\000\000\000\000\072\000\072\000\072\000\000\000\
\072\000\072\000\072\000\072\000\000\000\073\000\072\000\000\000\
\072\000\000\000\000\000\072\000\072\000\072\000\072\000\072\000\
\072\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\073\000\000\000\
\072\000\073\000\073\000\000\000\000\000\073\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\072\000\000\000\072\000\
\000\000\000\000\000\000\000\000\000\000\073\000\073\000\073\000\
\073\000\073\000\073\000\000\000\000\000\000\000\000\000\000\000\
\000\000\073\000\073\000\073\000\000\000\073\000\073\000\073\000\
\073\000\000\000\075\000\073\000\000\000\073\000\000\000\000\000\
\073\000\073\000\073\000\073\000\073\000\073\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\075\000\000\000\073\000\075\000\075\000\
\000\000\000\000\075\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\073\000\000\000\073\000\000\000\000\000\000\000\
\000\000\000\000\075\000\075\000\075\000\075\000\075\000\075\000\
\000\000\000\000\000\000\000\000\000\000\000\000\075\000\075\000\
\075\000\000\000\075\000\075\000\076\000\000\000\000\000\000\000\
\075\000\000\000\075\000\000\000\000\000\075\000\075\000\075\000\
\075\000\075\000\075\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\076\000\000\000\000\000\
\076\000\076\000\075\000\000\000\076\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\075\000\
\000\000\075\000\000\000\000\000\076\000\076\000\076\000\076\000\
\076\000\076\000\000\000\000\000\000\000\000\000\000\000\000\000\
\076\000\076\000\076\000\000\000\076\000\076\000\077\000\000\000\
\000\000\000\000\076\000\000\000\076\000\000\000\000\000\076\000\
\076\000\076\000\076\000\076\000\076\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\077\000\
\000\000\000\000\077\000\077\000\076\000\000\000\077\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\076\000\000\000\076\000\000\000\000\000\077\000\077\000\
\077\000\077\000\077\000\077\000\000\000\000\000\000\000\000\000\
\000\000\000\000\077\000\077\000\077\000\000\000\077\000\077\000\
\078\000\000\000\000\000\000\000\077\000\000\000\077\000\000\000\
\000\000\077\000\077\000\077\000\077\000\077\000\077\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\078\000\000\000\000\000\078\000\078\000\077\000\000\000\
\078\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\077\000\000\000\077\000\000\000\000\000\
\078\000\078\000\078\000\078\000\078\000\078\000\000\000\000\000\
\079\000\000\000\000\000\000\000\078\000\078\000\078\000\000\000\
\078\000\078\000\000\000\000\000\000\000\000\000\078\000\000\000\
\078\000\000\000\000\000\078\000\078\000\078\000\078\000\078\000\
\078\000\079\000\000\000\000\000\079\000\079\000\000\000\000\000\
\079\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\078\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\079\000\079\000\000\000\000\000\000\000\078\000\000\000\078\000\
\080\000\000\000\000\000\000\000\079\000\079\000\079\000\000\000\
\079\000\079\000\000\000\000\000\000\000\000\000\079\000\000\000\
\079\000\000\000\000\000\079\000\079\000\079\000\079\000\079\000\
\079\000\080\000\000\000\000\000\080\000\080\000\000\000\000\000\
\080\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\079\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\080\000\080\000\000\000\000\000\000\000\079\000\000\000\079\000\
\081\000\000\000\000\000\000\000\080\000\080\000\080\000\000\000\
\080\000\080\000\000\000\000\000\000\000\000\000\080\000\000\000\
\080\000\000\000\000\000\080\000\080\000\080\000\080\000\080\000\
\080\000\081\000\000\000\000\000\081\000\081\000\000\000\000\000\
\081\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\080\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\081\000\081\000\000\000\000\000\000\000\080\000\000\000\080\000\
\082\000\000\000\000\000\000\000\081\000\081\000\081\000\000\000\
\081\000\081\000\000\000\000\000\000\000\000\000\081\000\000\000\
\081\000\000\000\000\000\081\000\081\000\081\000\081\000\081\000\
\081\000\082\000\000\000\000\000\082\000\082\000\000\000\000\000\
\082\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\081\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\081\000\000\000\081\000\
\083\000\000\000\000\000\000\000\082\000\082\000\082\000\000\000\
\082\000\082\000\000\000\000\000\000\000\000\000\082\000\000\000\
\082\000\000\000\000\000\082\000\082\000\082\000\082\000\082\000\
\082\000\083\000\000\000\000\000\083\000\083\000\000\000\000\000\
\083\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\082\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\082\000\000\000\082\000\
\084\000\000\000\000\000\000\000\083\000\083\000\083\000\000\000\
\083\000\083\000\000\000\000\000\000\000\000\000\083\000\000\000\
\083\000\000\000\000\000\083\000\083\000\083\000\083\000\083\000\
\083\000\084\000\000\000\000\000\084\000\084\000\000\000\000\000\
\084\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\083\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\083\000\000\000\083\000\
\085\000\000\000\000\000\000\000\000\000\084\000\084\000\000\000\
\084\000\084\000\000\000\000\000\000\000\000\000\084\000\000\000\
\084\000\000\000\000\000\084\000\084\000\084\000\084\000\084\000\
\084\000\085\000\000\000\000\000\085\000\085\000\000\000\000\000\
\085\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\084\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\084\000\000\000\084\000\
\086\000\000\000\000\000\000\000\000\000\085\000\085\000\000\000\
\085\000\085\000\000\000\000\000\000\000\000\000\085\000\000\000\
\085\000\000\000\000\000\085\000\085\000\085\000\085\000\085\000\
\085\000\086\000\000\000\000\000\086\000\086\000\000\000\000\000\
\086\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\085\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\085\000\000\000\085\000\
\087\000\000\000\000\000\000\000\000\000\086\000\000\000\000\000\
\086\000\086\000\000\000\000\000\000\000\000\000\086\000\000\000\
\086\000\000\000\000\000\086\000\086\000\086\000\086\000\086\000\
\086\000\087\000\000\000\000\000\087\000\087\000\000\000\000\000\
\087\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\086\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\086\000\000\000\086\000\
\088\000\000\000\000\000\000\000\000\000\087\000\000\000\000\000\
\087\000\087\000\000\000\000\000\000\000\000\000\087\000\000\000\
\087\000\000\000\000\000\087\000\087\000\087\000\087\000\087\000\
\087\000\088\000\000\000\000\000\088\000\088\000\000\000\000\000\
\088\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\087\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\087\000\000\000\087\000\
\089\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\088\000\088\000\000\000\000\000\000\000\000\000\088\000\000\000\
\088\000\000\000\000\000\088\000\088\000\088\000\088\000\088\000\
\088\000\089\000\000\000\000\000\089\000\089\000\000\000\000\000\
\089\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\088\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\088\000\000\000\088\000\
\091\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\089\000\089\000\000\000\000\000\000\000\000\000\089\000\000\000\
\089\000\000\000\000\000\089\000\089\000\089\000\089\000\089\000\
\089\000\091\000\000\000\000\000\091\000\091\000\000\000\000\000\
\091\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\089\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\089\000\248\000\089\000\
\000\000\000\000\000\000\000\000\000\000\100\000\000\000\000\000\
\101\000\091\000\102\000\083\000\000\000\000\000\091\000\000\000\
\091\000\000\000\000\000\091\000\091\000\091\000\091\000\091\000\
\091\000\000\000\000\000\000\000\000\000\103\000\000\000\000\000\
\104\000\000\000\000\000\000\000\000\000\105\000\106\000\000\000\
\091\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\091\000\000\000\091\000\
\000\000\000\000\107\000\108\000\109\000\000\000\000\000\110\000\
\111\000\000\000\000\000\112\000\000\000\248\000\000\000\000\000\
\000\000\000\000\191\001\113\000\100\000\000\000\000\000\101\000\
\000\000\102\000\083\000\000\000\000\000\143\001\143\001\143\001\
\143\001\143\001\143\001\114\000\000\000\000\000\000\000\000\000\
\000\000\143\001\143\001\143\001\103\000\143\001\143\001\104\000\
\084\000\085\000\000\000\143\001\105\000\106\000\000\000\000\000\
\143\001\143\001\143\001\143\001\000\000\143\001\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\107\000\108\000\109\000\109\001\000\000\110\000\111\000\
\000\000\000\000\112\000\100\000\000\000\000\000\101\000\000\000\
\102\000\083\000\113\000\000\000\000\000\000\000\000\000\143\001\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\114\000\103\000\000\000\000\000\104\000\000\000\
\000\000\000\000\000\000\105\000\106\000\000\000\000\000\084\000\
\085\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\107\000\108\000\109\000\071\002\000\000\110\000\111\000\000\000\
\000\000\112\000\100\000\000\000\000\000\101\000\000\000\102\000\
\083\000\113\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\114\000\103\000\000\000\000\000\104\000\000\000\000\000\
\000\000\000\000\105\000\106\000\000\000\000\000\084\000\085\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\107\000\
\108\000\109\000\000\000\000\000\110\000\111\000\000\000\000\000\
\112\000\141\001\141\001\141\001\141\001\141\001\141\001\000\000\
\113\000\000\000\000\000\000\000\000\000\141\001\141\001\141\001\
\000\000\141\001\141\001\141\001\141\001\000\000\000\000\141\001\
\114\000\000\000\000\000\000\000\141\001\141\001\141\001\141\001\
\000\000\141\001\000\000\000\000\000\000\084\000\085\000\000\000\
\000\000\142\001\142\001\142\001\142\001\142\001\142\001\000\000\
\000\000\000\000\000\000\000\000\000\000\142\001\142\001\142\001\
\000\000\142\001\142\001\142\001\142\001\000\000\000\000\142\001\
\000\000\000\000\000\000\141\001\142\001\142\001\142\001\142\001\
\000\000\142\001\144\001\144\001\144\001\144\001\144\001\144\001\
\000\000\000\000\000\000\000\000\000\000\000\000\144\001\144\001\
\144\001\000\000\144\001\144\001\000\000\000\000\000\000\000\000\
\144\001\000\000\000\000\000\000\000\000\144\001\144\001\144\001\
\144\001\000\000\144\001\142\001\000\000\145\001\145\001\145\001\
\145\001\145\001\145\001\000\000\000\000\000\000\000\000\000\000\
\000\000\145\001\145\001\145\001\000\000\145\001\145\001\000\000\
\000\000\000\000\000\000\145\001\000\000\000\000\000\000\000\000\
\145\001\145\001\145\001\145\001\144\001\145\001\146\001\146\001\
\146\001\146\001\146\001\146\001\000\000\000\000\000\000\000\000\
\000\000\000\000\146\001\146\001\146\001\000\000\146\001\146\001\
\000\000\000\000\000\000\000\000\146\001\000\000\000\000\000\000\
\000\000\146\001\146\001\146\001\146\001\000\000\146\001\145\001\
\000\000\147\001\147\001\147\001\147\001\147\001\147\001\000\000\
\000\000\000\000\000\000\000\000\000\000\147\001\147\001\147\001\
\000\000\147\001\147\001\000\000\000\000\000\000\000\000\147\001\
\149\001\149\001\000\000\000\000\147\001\147\001\147\001\147\001\
\146\001\147\001\000\000\000\000\149\001\149\001\149\001\000\000\
\149\001\149\001\000\000\000\000\000\000\000\000\149\001\150\001\
\150\001\000\000\000\000\149\001\149\001\149\001\149\001\000\000\
\149\001\000\000\000\000\150\001\150\001\150\001\000\000\150\001\
\150\001\000\000\000\000\147\001\000\000\150\001\000\000\000\000\
\000\000\000\000\150\001\150\001\150\001\150\001\000\000\150\001\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\149\001\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\150\001\029\000\029\000\029\000\029\000\029\000\029\000\
\029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
\029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
\029\000\029\000\000\000\029\000\029\000\029\000\000\000\029\000\
\029\000\029\000\029\000\029\000\029\000\000\000\029\000\029\000\
\000\000\029\000\000\000\023\000\029\000\029\000\000\000\029\000"

let yycheck = "\016\000\
\017\000\155\000\019\000\044\000\053\000\049\000\032\000\069\000\
\170\000\025\000\120\000\061\000\058\001\062\000\042\001\087\000\
\162\000\061\000\060\001\238\001\001\000\002\000\048\000\004\001\
\040\000\086\001\087\001\049\000\000\000\107\002\000\000\232\001\
\000\000\000\001\104\002\180\002\007\001\220\000\183\002\061\000\
\220\000\012\001\237\001\000\001\000\001\086\000\145\000\000\001\
\148\002\016\000\017\000\032\000\019\000\169\000\040\000\117\000\
\073\000\200\000\179\000\180\000\077\000\060\001\052\001\053\001\
\007\001\082\000\055\001\048\000\230\000\012\001\059\001\254\000\
\170\000\000\001\254\000\013\001\000\001\000\001\062\001\000\001\
\096\000\076\001\098\000\083\001\081\001\000\001\134\002\079\001\
\085\001\084\001\080\001\107\000\108\000\109\000\110\000\111\000\
\112\000\113\000\082\001\140\000\141\000\053\001\150\002\097\000\
\120\000\250\002\073\000\007\001\079\001\062\001\077\000\078\001\
\012\001\007\001\214\002\082\000\142\000\077\001\012\001\017\000\
\120\001\019\000\079\001\079\001\076\001\007\001\170\000\113\000\
\080\001\080\000\012\001\102\001\230\000\013\001\014\001\152\000\
\079\001\007\001\027\003\076\001\076\001\185\000\012\001\137\000\
\115\001\126\001\082\001\084\001\170\000\227\002\001\000\002\000\
\222\002\004\000\125\001\136\000\079\001\229\000\079\001\102\001\
\083\001\142\000\051\003\017\001\079\001\104\002\105\002\018\000\
\106\001\107\001\196\000\220\000\115\001\118\002\025\000\026\000\
\074\000\077\001\029\000\030\000\070\001\032\000\125\001\034\000\
\200\000\079\001\007\001\114\001\077\001\116\001\114\001\012\001\
\116\001\044\000\157\002\077\001\001\002\048\000\125\001\050\000\
\217\000\125\001\244\001\245\001\055\000\254\000\057\000\000\001\
\230\000\060\000\134\001\135\001\055\001\064\000\105\002\196\000\
\059\001\060\001\078\001\085\003\147\001\148\001\077\001\088\001\
\077\001\125\001\106\001\107\001\094\003\228\001\128\001\125\001\
\231\001\184\000\197\002\086\000\187\000\188\000\189\000\228\001\
\013\001\192\000\193\000\125\001\071\001\244\001\097\000\001\000\
\002\000\188\002\010\001\190\002\179\001\192\002\243\001\125\001\
\245\001\077\001\018\001\086\001\021\001\022\001\056\001\023\001\
\024\001\025\001\026\001\027\001\028\001\029\001\030\001\031\001\
\032\001\033\001\034\001\035\001\036\001\037\001\038\001\039\001\
\040\001\041\001\042\001\222\002\000\001\136\000\000\001\053\001\
\000\001\140\000\141\000\142\000\102\001\023\001\181\001\182\001\
\076\001\188\002\031\002\190\002\090\002\192\002\007\001\064\001\
\084\001\115\001\078\002\012\001\007\001\171\002\021\001\022\001\
\106\001\012\001\079\001\041\001\077\001\007\001\169\000\053\001\
\058\001\114\001\012\001\116\001\107\000\108\000\109\000\110\000\
\111\000\112\000\078\001\000\001\125\001\106\001\107\001\096\001\
\084\001\227\001\007\001\071\001\194\001\036\003\007\001\081\001\
\195\000\196\000\080\001\012\001\077\001\007\001\076\001\022\002\
\023\002\091\001\086\001\007\001\082\001\083\001\084\001\054\003\
\012\001\094\002\152\002\062\001\097\002\062\001\078\001\044\000\
\000\000\220\000\000\001\007\001\062\001\000\001\079\001\237\002\
\012\001\072\003\240\002\077\001\007\001\081\001\143\001\144\001\
\084\001\228\001\083\001\077\001\231\001\251\002\080\001\124\002\
\076\001\062\001\089\001\133\001\062\001\092\001\082\001\007\001\
\243\001\244\001\245\001\254\000\062\001\000\001\101\001\076\001\
\077\001\007\001\105\001\080\001\077\001\082\001\083\001\084\001\
\082\001\083\001\024\003\077\001\125\001\181\001\182\001\077\001\
\095\002\083\001\032\003\079\001\034\003\007\001\236\001\077\001\
\143\001\144\001\012\001\125\001\042\003\062\001\226\001\031\001\
\079\001\007\001\034\001\079\001\031\002\007\001\012\001\076\001\
\119\002\055\003\012\001\080\001\240\001\079\001\007\001\084\001\
\125\001\079\001\083\001\012\001\125\001\077\001\223\001\140\000\
\141\000\115\002\102\001\125\001\136\002\137\002\094\001\095\001\
\096\001\125\001\006\002\007\001\062\001\072\001\084\003\115\001\
\012\001\007\001\082\001\078\001\077\001\223\001\012\001\158\002\
\221\002\125\001\140\002\077\001\000\001\013\001\080\001\082\001\
\062\001\024\001\025\001\026\001\062\001\079\001\072\001\073\001\
\006\001\076\001\077\001\094\002\179\002\080\001\097\002\077\001\
\223\001\084\001\076\001\077\001\060\001\061\001\080\001\114\001\
\078\001\116\001\084\001\005\002\077\001\083\001\066\001\080\001\
\170\002\031\001\125\001\033\001\034\001\035\001\000\001\082\001\
\038\001\124\002\085\001\134\001\135\001\007\001\067\001\038\001\
\076\001\077\001\012\001\125\001\080\001\220\002\076\001\077\001\
\084\001\068\001\080\001\078\001\076\001\070\001\084\001\125\001\
\083\001\232\002\082\001\125\001\007\001\236\002\007\001\010\001\
\007\002\012\001\077\001\079\001\125\001\080\001\078\002\083\001\
\084\001\082\002\107\002\084\001\105\000\106\000\083\001\115\002\
\041\001\007\001\106\001\107\001\031\001\108\001\012\001\034\001\
\083\001\125\001\098\002\083\001\062\001\102\002\084\001\125\001\
\102\001\083\002\107\002\052\001\053\001\115\002\083\001\102\001\
\090\002\076\001\076\001\077\001\062\001\115\001\080\001\117\001\
\082\001\083\001\084\001\084\001\115\001\036\003\117\001\054\001\
\055\001\074\001\075\001\082\002\077\001\134\002\140\002\080\001\
\146\002\138\002\221\002\060\001\061\001\232\001\107\002\054\003\
\000\001\000\001\076\001\238\001\041\001\150\002\006\001\184\002\
\072\001\073\001\093\001\084\001\134\002\007\001\076\001\077\001\
\079\001\072\003\080\001\125\001\000\001\248\002\084\001\083\001\
\196\002\147\002\171\002\172\002\150\002\078\001\152\002\031\001\
\079\001\178\002\034\001\035\001\078\001\146\002\038\001\134\002\
\006\001\079\001\041\001\138\002\209\000\210\000\017\003\211\000\
\212\000\213\000\214\000\215\000\216\000\218\000\219\000\150\002\
\062\001\063\001\064\001\181\002\077\001\041\001\033\003\125\001\
\076\001\031\001\054\001\055\001\034\001\035\001\007\001\078\001\
\038\001\010\001\076\001\012\001\013\001\076\001\076\001\224\002\
\079\001\077\001\227\002\082\001\083\001\084\001\077\001\094\001\
\095\001\096\001\099\001\100\001\237\002\215\002\031\001\240\002\
\077\001\034\001\076\001\079\001\068\001\006\001\102\001\083\001\
\084\001\066\001\251\002\028\003\062\001\063\001\064\001\077\001\
\006\001\070\001\080\001\115\001\067\001\117\001\083\001\084\001\
\127\001\084\001\043\003\036\003\083\001\084\001\031\001\076\001\
\107\002\034\001\035\001\083\001\084\001\038\001\010\001\024\003\
\102\001\031\001\104\001\118\002\034\001\035\001\010\001\032\003\
\038\001\034\003\094\001\095\001\096\001\115\001\010\001\117\001\
\076\001\042\003\242\002\243\002\093\001\136\002\137\002\025\003\
\054\001\055\001\056\001\057\001\058\001\059\001\055\003\146\002\
\010\001\106\001\107\001\078\001\066\001\067\001\068\001\120\001\
\070\001\071\001\157\002\140\000\141\000\069\003\076\001\078\001\
\078\001\164\001\165\001\081\001\082\001\083\001\084\001\085\001\
\086\001\172\002\076\001\084\003\076\001\102\001\112\001\104\001\
\078\001\081\001\082\001\083\001\056\001\057\001\058\001\059\001\
\102\001\076\001\115\001\076\001\117\001\166\001\167\001\002\001\
\078\001\083\003\197\002\006\001\007\001\115\001\076\001\117\001\
\082\001\012\001\172\001\173\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\076\001\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\084\001\161\001\162\001\163\001\
\164\001\165\001\166\001\167\001\168\001\169\001\170\001\171\001\
\172\001\173\001\174\001\175\001\176\001\177\001\178\001\083\001\
\000\001\001\001\002\001\078\001\004\001\005\001\006\001\007\001\
\008\001\009\001\010\001\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\078\001\023\001\
\024\001\025\001\026\001\027\001\028\001\029\001\030\001\031\001\
\032\001\033\001\034\001\035\001\036\001\037\001\038\001\039\001\
\040\001\036\003\076\001\102\001\103\001\078\001\105\001\081\001\
\082\001\083\001\084\001\056\001\057\001\058\001\059\001\078\001\
\115\001\078\001\117\001\076\001\060\001\061\001\062\001\083\001\
\123\001\065\001\066\001\076\001\127\001\069\001\070\001\027\001\
\028\001\084\001\074\001\075\001\007\001\077\001\078\001\079\001\
\029\001\030\001\076\001\083\001\035\001\036\001\120\001\087\001\
\088\001\089\001\090\001\091\001\092\001\093\001\094\001\095\001\
\096\001\097\001\098\001\077\001\076\001\025\002\102\001\103\001\
\104\001\105\001\106\001\107\001\076\001\109\001\076\001\081\001\
\007\001\113\001\114\001\115\001\116\001\117\001\118\001\119\001\
\082\001\007\001\077\001\123\001\124\001\083\001\126\001\127\001\
\168\001\169\001\170\001\171\001\084\001\083\001\000\001\001\001\
\002\001\080\001\078\001\005\001\006\001\007\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\078\001\023\001\024\001\025\001\
\026\001\027\001\028\001\029\001\030\001\031\001\032\001\033\001\
\034\001\035\001\036\001\037\001\038\001\039\001\040\001\078\001\
\078\001\018\000\031\001\032\001\033\001\034\001\076\001\076\001\
\076\001\026\000\084\001\084\001\029\000\030\000\081\001\076\001\
\076\001\034\000\060\001\061\001\062\001\084\001\079\001\065\001\
\066\001\084\001\082\001\069\001\070\001\084\001\079\001\079\001\
\074\001\075\001\084\001\077\001\078\001\079\001\055\000\041\001\
\057\000\083\001\082\001\078\001\144\002\087\001\088\001\089\001\
\090\001\091\001\092\001\093\001\094\001\095\001\096\001\097\001\
\098\001\076\001\076\001\081\001\102\001\103\001\104\001\105\001\
\106\001\107\001\076\001\109\001\084\001\078\001\083\001\113\001\
\114\001\115\001\116\001\117\001\118\001\119\001\076\001\078\001\
\033\001\123\001\124\001\076\001\127\001\127\001\000\001\001\001\
\002\001\083\001\077\001\005\001\006\001\007\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\083\001\023\001\024\001\025\001\
\026\001\027\001\028\001\029\001\030\001\031\001\032\001\033\001\
\034\001\035\001\036\001\037\001\038\001\039\001\040\001\082\001\
\041\001\042\001\043\001\044\001\045\001\046\001\047\001\048\001\
\049\001\050\001\051\001\006\001\083\001\082\001\076\001\076\001\
\081\001\076\001\060\001\061\001\062\001\078\001\076\001\065\001\
\066\001\078\001\078\001\069\001\070\001\000\001\077\001\082\001\
\074\001\075\001\083\001\077\001\031\001\079\001\083\001\034\001\
\035\001\083\001\077\001\038\001\007\001\087\001\088\001\089\001\
\090\001\091\001\092\001\093\001\094\001\095\001\096\001\097\001\
\098\001\081\001\083\001\013\001\102\001\103\001\104\001\105\001\
\106\001\107\001\094\001\109\001\101\001\082\001\041\001\113\001\
\114\001\115\001\116\001\117\001\118\001\119\001\082\001\080\001\
\083\001\123\001\124\001\076\001\083\001\127\001\000\001\001\001\
\002\001\083\001\004\001\005\001\007\001\007\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\082\001\054\001\055\001\
\084\001\076\001\083\001\102\001\079\001\081\001\077\001\082\001\
\083\001\084\001\066\001\067\001\068\001\076\001\070\001\071\001\
\115\001\082\001\117\001\076\001\076\001\039\001\040\001\076\001\
\084\001\081\001\082\001\083\001\084\001\076\001\086\001\150\001\
\151\001\152\001\153\001\154\001\155\001\156\001\157\001\158\001\
\159\001\160\001\060\001\061\001\062\001\078\001\000\001\065\001\
\066\001\083\001\076\001\069\001\070\001\076\001\078\001\076\001\
\074\001\075\001\080\001\077\001\078\001\079\001\076\001\078\001\
\120\001\083\001\083\001\078\001\076\001\087\001\088\001\089\001\
\090\001\091\001\092\001\093\001\094\001\095\001\096\001\097\001\
\098\001\082\001\078\001\083\001\078\001\083\001\104\001\077\001\
\106\001\107\001\081\001\109\001\076\001\076\001\083\001\113\001\
\114\001\076\001\116\001\076\001\076\001\002\000\136\000\208\002\
\231\001\214\002\124\001\082\001\126\001\127\001\000\001\001\001\
\002\001\004\000\004\001\005\001\006\001\007\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\006\002\023\001\024\001\025\001\
\026\001\027\001\028\001\029\001\030\001\031\001\032\001\033\001\
\034\001\035\001\036\001\037\001\038\001\039\001\040\001\101\002\
\152\000\102\002\174\001\119\002\144\002\033\003\046\003\175\001\
\248\001\176\000\196\002\176\001\017\001\177\001\194\001\037\001\
\178\001\038\001\060\001\061\001\062\001\039\001\041\001\065\001\
\066\001\040\001\076\003\069\001\070\001\074\003\094\003\255\255\
\074\001\075\001\255\255\077\001\078\001\079\001\255\255\255\255\
\255\255\083\001\255\255\255\255\255\255\087\001\088\001\089\001\
\090\001\091\001\092\001\093\001\094\001\095\001\096\001\097\001\
\098\001\255\255\255\255\255\255\102\001\103\001\104\001\105\001\
\106\001\107\001\255\255\109\001\255\255\255\255\112\001\113\001\
\114\001\115\001\116\001\117\001\118\001\119\001\255\255\255\255\
\255\255\123\001\124\001\255\255\126\001\000\001\001\001\002\001\
\255\255\004\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\039\001\040\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\060\001\061\001\062\001\255\255\255\255\065\001\066\001\
\255\255\255\255\069\001\070\001\255\255\255\255\255\255\074\001\
\075\001\255\255\077\001\078\001\079\001\255\255\255\255\255\255\
\083\001\255\255\255\255\255\255\087\001\088\001\089\001\090\001\
\091\001\092\001\093\001\094\001\095\001\096\001\097\001\098\001\
\255\255\255\255\255\255\102\001\103\001\104\001\105\001\106\001\
\107\001\255\255\109\001\255\255\255\255\255\255\113\001\114\001\
\115\001\116\001\117\001\118\001\119\001\255\255\255\255\255\255\
\123\001\124\001\255\255\126\001\000\001\001\001\002\001\255\255\
\004\001\005\001\006\001\007\001\008\001\009\001\010\001\011\001\
\012\001\013\001\014\001\015\001\016\001\017\001\018\001\019\001\
\020\001\021\001\255\255\023\001\024\001\025\001\026\001\027\001\
\028\001\029\001\030\001\031\001\032\001\033\001\034\001\035\001\
\036\001\037\001\038\001\039\001\040\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\060\001\061\001\062\001\255\255\255\255\065\001\066\001\255\255\
\255\255\069\001\070\001\255\255\255\255\255\255\074\001\075\001\
\255\255\077\001\255\255\079\001\255\255\255\255\255\255\083\001\
\255\255\255\255\255\255\087\001\088\001\089\001\090\001\091\001\
\092\001\093\001\255\255\255\255\255\255\097\001\098\001\255\255\
\255\255\255\255\102\001\103\001\104\001\105\001\106\001\107\001\
\255\255\109\001\255\255\255\255\255\255\113\001\114\001\115\001\
\116\001\117\001\255\255\255\255\255\255\255\255\255\255\123\001\
\124\001\255\255\126\001\000\001\001\001\002\001\255\255\004\001\
\005\001\255\255\007\001\008\001\009\001\010\001\011\001\012\001\
\013\001\014\001\255\255\255\255\052\001\053\001\054\001\055\001\
\056\001\057\001\058\001\059\001\060\001\061\001\062\001\063\001\
\064\001\255\255\066\001\067\001\068\001\255\255\070\001\071\001\
\072\001\073\001\039\001\040\001\076\001\255\255\255\255\255\255\
\080\001\081\001\082\001\083\001\084\001\255\255\086\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\060\001\
\061\001\062\001\255\255\255\255\065\001\066\001\255\255\255\255\
\069\001\070\001\255\255\255\255\255\255\074\001\075\001\255\255\
\077\001\255\255\079\001\255\255\255\255\255\255\083\001\255\255\
\120\001\255\255\087\001\088\001\089\001\090\001\091\001\092\001\
\093\001\094\001\095\001\096\001\097\001\098\001\255\255\255\255\
\255\255\255\255\255\255\104\001\013\001\106\001\107\001\255\255\
\109\001\255\255\255\255\255\255\113\001\114\001\255\255\116\001\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\124\001\
\255\255\126\001\000\001\001\001\002\001\255\255\004\001\005\001\
\255\255\007\001\008\001\009\001\010\001\011\001\012\001\013\001\
\014\001\255\255\255\255\052\001\053\001\054\001\055\001\056\001\
\057\001\058\001\059\001\060\001\061\001\062\001\063\001\064\001\
\255\255\066\001\067\001\068\001\255\255\070\001\071\001\072\001\
\073\001\039\001\040\001\076\001\255\255\255\255\255\255\080\001\
\081\001\082\001\083\001\084\001\255\255\086\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\060\001\061\001\
\062\001\255\255\255\255\065\001\066\001\255\255\255\255\069\001\
\070\001\106\001\107\001\255\255\074\001\075\001\255\255\077\001\
\255\255\079\001\255\255\255\255\255\255\083\001\255\255\120\001\
\255\255\087\001\088\001\089\001\090\001\091\001\092\001\093\001\
\255\255\255\255\255\255\097\001\098\001\255\255\255\255\255\255\
\255\255\255\255\104\001\255\255\106\001\107\001\255\255\109\001\
\255\255\255\255\255\255\113\001\114\001\000\001\116\001\255\255\
\255\255\255\255\255\255\006\001\007\001\255\255\124\001\255\255\
\126\001\012\001\255\255\255\255\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\052\001\053\001\054\001\055\001\
\056\001\057\001\058\001\059\001\060\001\061\001\062\001\063\001\
\064\001\255\255\066\001\067\001\068\001\255\255\070\001\071\001\
\072\001\073\001\255\255\062\001\076\001\255\255\255\255\255\255\
\080\001\081\001\082\001\083\001\084\001\255\255\086\001\255\255\
\255\255\076\001\077\001\255\255\255\255\080\001\255\255\082\001\
\083\001\084\001\071\001\255\255\255\255\255\255\255\255\076\001\
\255\255\255\255\255\255\255\255\081\001\082\001\083\001\084\001\
\255\255\086\001\255\255\102\001\103\001\255\255\105\001\255\255\
\120\001\255\255\255\255\255\255\255\255\000\001\255\255\255\255\
\115\001\255\255\117\001\006\001\007\001\255\255\255\255\255\255\
\123\001\012\001\125\001\255\255\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\120\001\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\052\001\053\001\054\001\055\001\
\056\001\057\001\058\001\059\001\060\001\061\001\062\001\063\001\
\064\001\255\255\066\001\067\001\068\001\255\255\070\001\071\001\
\072\001\073\001\255\255\062\001\076\001\255\255\255\255\255\255\
\080\001\081\001\082\001\083\001\084\001\255\255\086\001\255\255\
\255\255\076\001\077\001\255\255\255\255\080\001\255\255\082\001\
\083\001\084\001\071\001\255\255\255\255\255\255\255\255\076\001\
\255\255\255\255\255\255\255\255\081\001\082\001\083\001\084\001\
\255\255\086\001\255\255\102\001\103\001\255\255\105\001\255\255\
\120\001\255\255\255\255\255\255\255\255\000\001\255\255\255\255\
\115\001\255\255\117\001\006\001\007\001\255\255\255\255\255\255\
\123\001\012\001\125\001\255\255\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\120\001\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\255\255\255\255\054\001\055\001\
\056\001\057\001\058\001\059\001\060\001\061\001\062\001\063\001\
\064\001\255\255\066\001\067\001\068\001\255\255\070\001\071\001\
\072\001\073\001\255\255\062\001\076\001\255\255\255\255\255\255\
\255\255\081\001\082\001\083\001\084\001\255\255\086\001\255\255\
\255\255\076\001\077\001\255\255\255\255\080\001\255\255\082\001\
\083\001\084\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\102\001\103\001\255\255\105\001\255\255\
\120\001\255\255\255\255\255\255\255\255\000\001\255\255\255\255\
\115\001\255\255\117\001\006\001\007\001\255\255\255\255\255\255\
\123\001\012\001\125\001\255\255\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\255\255\255\255\054\001\055\001\
\056\001\057\001\058\001\059\001\060\001\061\001\062\001\063\001\
\064\001\255\255\066\001\067\001\068\001\255\255\070\001\071\001\
\072\001\073\001\255\255\062\001\076\001\255\255\255\255\255\255\
\255\255\081\001\082\001\083\001\084\001\006\001\086\001\255\255\
\255\255\076\001\077\001\255\255\000\001\080\001\255\255\082\001\
\083\001\084\001\255\255\007\001\008\001\009\001\010\001\011\001\
\012\001\013\001\014\001\255\255\255\255\255\255\031\001\255\255\
\255\255\034\001\035\001\102\001\103\001\038\001\105\001\255\255\
\120\001\255\255\255\255\255\255\255\255\033\001\255\255\255\255\
\115\001\255\255\117\001\039\001\040\001\041\001\070\001\071\001\
\123\001\255\255\125\001\255\255\076\001\255\255\255\255\255\255\
\255\255\081\001\082\001\083\001\084\001\255\255\086\001\255\255\
\060\001\061\001\062\001\255\255\077\001\065\001\066\001\080\001\
\255\255\069\001\070\001\255\255\255\255\255\255\074\001\075\001\
\076\001\077\001\255\255\079\001\080\001\081\001\082\001\083\001\
\084\001\255\255\255\255\255\255\255\255\102\001\255\255\104\001\
\120\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\115\001\255\255\117\001\255\255\106\001\107\001\
\255\255\109\001\255\255\255\255\255\255\113\001\114\001\255\255\
\255\255\255\255\255\255\255\255\255\255\000\001\001\001\002\001\
\124\001\125\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\039\001\040\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\060\001\061\001\062\001\255\255\255\255\065\001\066\001\
\255\255\255\255\069\001\070\001\255\255\255\255\255\255\074\001\
\075\001\255\255\077\001\078\001\079\001\255\255\255\255\255\255\
\083\001\255\255\255\255\255\255\087\001\088\001\089\001\090\001\
\091\001\092\001\093\001\094\001\095\001\096\001\097\001\098\001\
\255\255\255\255\255\255\102\001\103\001\104\001\105\001\106\001\
\107\001\255\255\109\001\255\255\255\255\255\255\113\001\114\001\
\115\001\116\001\117\001\118\001\119\001\000\001\001\001\002\001\
\123\001\124\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\039\001\040\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\060\001\061\001\062\001\255\255\255\255\065\001\066\001\
\255\255\255\255\069\001\070\001\255\255\255\255\255\255\074\001\
\075\001\255\255\077\001\255\255\079\001\255\255\255\255\255\255\
\083\001\255\255\255\255\255\255\087\001\088\001\089\001\090\001\
\091\001\092\001\093\001\255\255\255\255\255\255\097\001\098\001\
\255\255\255\255\255\255\102\001\103\001\104\001\105\001\106\001\
\107\001\255\255\109\001\255\255\255\255\255\255\113\001\114\001\
\115\001\116\001\117\001\118\001\119\001\000\001\255\255\255\255\
\123\001\124\001\255\255\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\039\001\040\001\070\001\071\001\
\255\255\255\255\255\255\255\255\076\001\255\255\255\255\255\255\
\255\255\081\001\082\001\083\001\084\001\255\255\086\001\255\255\
\255\255\060\001\061\001\062\001\255\255\255\255\065\001\066\001\
\255\255\255\255\069\001\070\001\255\255\255\255\255\255\074\001\
\075\001\255\255\077\001\255\255\079\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\120\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\102\001\103\001\255\255\105\001\106\001\
\107\001\255\255\109\001\255\255\255\255\255\255\113\001\114\001\
\115\001\255\255\117\001\255\255\000\001\255\255\002\001\255\255\
\123\001\124\001\006\001\007\001\008\001\009\001\010\001\011\001\
\012\001\013\001\014\001\015\001\016\001\017\001\018\001\019\001\
\020\001\021\001\255\255\023\001\024\001\025\001\026\001\027\001\
\028\001\029\001\030\001\031\001\032\001\033\001\034\001\035\001\
\036\001\037\001\038\001\039\001\040\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\060\001\061\001\062\001\255\255\255\255\065\001\066\001\255\255\
\255\255\069\001\070\001\000\001\001\001\002\001\074\001\075\001\
\005\001\077\001\007\001\008\001\009\001\010\001\011\001\012\001\
\013\001\014\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\102\001\103\001\255\255\105\001\106\001\107\001\
\255\255\109\001\039\001\040\001\255\255\113\001\114\001\115\001\
\255\255\117\001\255\255\255\255\255\255\255\255\255\255\123\001\
\124\001\255\255\255\255\255\255\255\255\255\255\255\255\060\001\
\061\001\062\001\255\255\255\255\065\001\066\001\255\255\255\255\
\069\001\070\001\255\255\255\255\255\255\074\001\075\001\255\255\
\077\001\255\255\079\001\255\255\255\255\255\255\083\001\255\255\
\255\255\255\255\087\001\088\001\089\001\090\001\091\001\092\001\
\093\001\255\255\255\255\255\255\097\001\098\001\255\255\255\255\
\255\255\255\255\255\255\104\001\255\255\106\001\107\001\255\255\
\109\001\255\255\255\255\255\255\113\001\114\001\255\255\116\001\
\002\001\003\001\255\255\255\255\006\001\007\001\255\255\124\001\
\255\255\255\255\012\001\255\255\255\255\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\255\255\023\001\024\001\025\001\
\026\001\027\001\028\001\029\001\030\001\031\001\032\001\033\001\
\034\001\035\001\036\001\037\001\038\001\007\001\255\255\255\255\
\010\001\255\255\012\001\013\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\006\001\255\255\255\255\255\255\031\001\255\255\255\255\
\034\001\000\001\255\255\255\255\255\255\039\001\040\001\255\255\
\007\001\008\001\009\001\010\001\011\001\255\255\013\001\014\001\
\255\255\083\001\031\001\255\255\255\255\034\001\035\001\255\255\
\255\255\038\001\060\001\061\001\062\001\255\255\255\255\065\001\
\066\001\255\255\033\001\069\001\102\001\103\001\104\001\105\001\
\039\001\040\001\255\255\077\001\255\255\255\255\255\255\255\255\
\255\255\115\001\255\255\117\001\118\001\119\001\255\255\121\001\
\122\001\123\001\255\255\093\001\126\001\060\001\061\001\062\001\
\077\001\255\255\065\001\066\001\255\255\255\255\069\001\070\001\
\106\001\107\001\255\255\074\001\075\001\000\001\077\001\255\255\
\255\255\255\255\081\001\255\255\007\001\008\001\009\001\010\001\
\011\001\102\001\013\001\014\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\115\001\255\255\
\117\001\255\255\255\255\106\001\107\001\255\255\109\001\255\255\
\255\255\255\255\113\001\114\001\039\001\040\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\124\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\060\001\061\001\062\001\255\255\255\255\065\001\066\001\
\255\255\255\255\069\001\070\001\000\001\255\255\255\255\074\001\
\075\001\255\255\077\001\007\001\008\001\009\001\010\001\011\001\
\083\001\013\001\014\001\067\001\068\001\255\255\070\001\071\001\
\255\255\255\255\255\255\006\001\076\001\255\255\255\255\255\255\
\255\255\081\001\082\001\083\001\084\001\255\255\086\001\106\001\
\107\001\255\255\109\001\039\001\040\001\255\255\113\001\114\001\
\255\255\255\255\255\255\255\255\031\001\255\255\255\255\034\001\
\035\001\124\001\255\255\038\001\255\255\255\255\255\255\255\255\
\060\001\061\001\062\001\255\255\255\255\065\001\066\001\255\255\
\120\001\069\001\070\001\255\255\255\255\255\255\074\001\075\001\
\000\001\077\001\255\255\255\255\255\255\081\001\255\255\007\001\
\008\001\009\001\010\001\011\001\071\001\013\001\014\001\255\255\
\255\255\076\001\255\255\078\001\255\255\006\001\081\001\082\001\
\083\001\084\001\085\001\086\001\255\255\255\255\106\001\107\001\
\255\255\109\001\255\255\255\255\255\255\113\001\114\001\039\001\
\040\001\255\255\255\255\102\001\255\255\255\255\031\001\255\255\
\124\001\034\001\035\001\255\255\255\255\038\001\255\255\255\255\
\115\001\255\255\117\001\255\255\060\001\061\001\062\001\255\255\
\255\255\065\001\066\001\255\255\255\255\069\001\070\001\255\255\
\255\255\255\255\074\001\075\001\000\001\077\001\255\255\255\255\
\255\255\081\001\255\255\007\001\008\001\009\001\010\001\011\001\
\255\255\013\001\014\001\076\001\255\255\078\001\255\255\255\255\
\081\001\082\001\083\001\084\001\085\001\255\255\255\255\255\255\
\255\255\255\255\106\001\107\001\255\255\109\001\255\255\255\255\
\255\255\113\001\114\001\039\001\040\001\102\001\255\255\255\255\
\255\255\255\255\255\255\255\255\124\001\255\255\255\255\255\255\
\255\255\255\255\115\001\255\255\117\001\255\255\255\255\255\255\
\060\001\061\001\062\001\255\255\255\255\065\001\066\001\255\255\
\255\255\069\001\070\001\000\001\255\255\255\255\074\001\075\001\
\255\255\077\001\007\001\008\001\009\001\010\001\011\001\255\255\
\013\001\014\001\255\255\255\255\255\255\255\255\000\001\255\255\
\255\255\255\255\255\255\255\255\255\255\007\001\008\001\009\001\
\010\001\011\001\255\255\013\001\014\001\255\255\106\001\107\001\
\255\255\109\001\039\001\040\001\255\255\113\001\114\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\124\001\255\255\255\255\255\255\255\255\039\001\040\001\060\001\
\061\001\062\001\255\255\255\255\065\001\066\001\255\255\255\255\
\069\001\070\001\255\255\255\255\255\255\074\001\075\001\255\255\
\077\001\255\255\060\001\061\001\062\001\255\255\255\255\065\001\
\066\001\255\255\255\255\069\001\070\001\000\001\255\255\255\255\
\074\001\075\001\255\255\077\001\007\001\008\001\009\001\010\001\
\011\001\255\255\013\001\014\001\255\255\106\001\107\001\255\255\
\109\001\255\255\255\255\255\255\113\001\114\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\124\001\
\106\001\107\001\255\255\109\001\039\001\040\001\255\255\113\001\
\114\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\124\001\255\255\255\255\255\255\255\255\255\255\
\255\255\060\001\061\001\062\001\255\255\255\255\065\001\066\001\
\255\255\255\255\069\001\070\001\000\001\255\255\255\255\074\001\
\075\001\255\255\077\001\007\001\008\001\009\001\010\001\011\001\
\255\255\013\001\014\001\067\001\068\001\255\255\070\001\071\001\
\255\255\255\255\255\255\255\255\076\001\255\255\255\255\255\255\
\255\255\081\001\082\001\083\001\084\001\255\255\086\001\106\001\
\107\001\255\255\109\001\039\001\040\001\255\255\113\001\114\001\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\124\001\255\255\255\255\255\255\255\255\255\255\255\255\
\060\001\061\001\062\001\255\255\255\255\065\001\066\001\000\001\
\120\001\069\001\070\001\255\255\255\255\006\001\074\001\075\001\
\255\255\077\001\255\255\012\001\255\255\255\255\015\001\016\001\
\017\001\018\001\019\001\020\001\021\001\255\255\023\001\024\001\
\025\001\026\001\027\001\028\001\029\001\030\001\031\001\032\001\
\033\001\034\001\035\001\036\001\037\001\038\001\106\001\107\001\
\255\255\109\001\255\255\255\255\255\255\113\001\114\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\124\001\255\255\054\001\055\001\056\001\057\001\058\001\059\001\
\060\001\061\001\062\001\063\001\064\001\255\255\066\001\067\001\
\068\001\255\255\070\001\071\001\072\001\073\001\255\255\255\255\
\076\001\255\255\083\001\255\255\255\255\081\001\082\001\083\001\
\084\001\255\255\086\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\102\001\103\001\255\255\
\105\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\115\001\002\001\117\001\118\001\119\001\006\001\
\007\001\255\255\123\001\010\001\120\001\012\001\255\255\255\255\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\255\255\
\023\001\024\001\025\001\026\001\027\001\028\001\029\001\030\001\
\031\001\032\001\033\001\034\001\035\001\036\001\037\001\038\001\
\255\255\255\255\041\001\042\001\043\001\044\001\045\001\046\001\
\047\001\048\001\049\001\050\001\051\001\052\001\053\001\054\001\
\055\001\056\001\057\001\058\001\059\001\060\001\061\001\062\001\
\063\001\064\001\255\255\066\001\067\001\068\001\255\255\070\001\
\071\001\072\001\073\001\074\001\075\001\076\001\077\001\078\001\
\079\001\080\001\081\001\082\001\083\001\084\001\085\001\086\001\
\255\255\255\255\255\255\255\255\255\255\255\255\093\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\102\001\
\103\001\255\255\105\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\115\001\255\255\117\001\255\255\
\255\255\120\001\002\001\003\001\123\001\255\255\006\001\007\001\
\255\255\255\255\255\255\255\255\012\001\255\255\255\255\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\255\255\023\001\
\024\001\025\001\026\001\027\001\028\001\029\001\030\001\031\001\
\032\001\033\001\034\001\035\001\036\001\037\001\038\001\002\001\
\255\255\255\255\255\255\006\001\255\255\255\255\255\255\255\255\
\255\255\012\001\255\255\255\255\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\255\255\255\255\255\255\054\001\
\055\001\056\001\057\001\058\001\059\001\060\001\061\001\255\255\
\255\255\255\255\255\255\066\001\067\001\068\001\255\255\070\001\
\071\001\072\001\073\001\255\255\255\255\076\001\102\001\103\001\
\104\001\105\001\081\001\082\001\083\001\084\001\255\255\086\001\
\255\255\255\255\255\255\115\001\079\001\117\001\118\001\119\001\
\255\255\121\001\122\001\123\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\102\001\103\001\255\255\105\001\255\255\
\255\255\120\001\255\255\255\255\255\255\255\255\255\255\255\255\
\115\001\255\255\117\001\006\001\007\001\255\255\255\255\255\255\
\123\001\012\001\255\255\255\255\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\255\255\255\255\054\001\055\001\
\056\001\057\001\058\001\059\001\060\001\061\001\255\255\255\255\
\255\255\255\255\066\001\067\001\068\001\255\255\070\001\071\001\
\072\001\073\001\255\255\062\001\076\001\255\255\255\255\255\255\
\255\255\081\001\082\001\083\001\084\001\255\255\086\001\255\255\
\255\255\255\255\077\001\255\255\255\255\255\255\255\255\082\001\
\083\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\102\001\103\001\255\255\105\001\255\255\
\120\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\115\001\255\255\117\001\255\255\255\255\255\255\255\255\255\255\
\123\001\255\255\125\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\039\001\040\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\060\001\061\001\062\001\255\255\255\255\065\001\066\001\
\255\255\255\255\069\001\070\001\255\255\255\255\255\255\074\001\
\075\001\255\255\077\001\255\255\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\102\001\103\001\255\255\105\001\106\001\
\107\001\255\255\109\001\255\255\039\001\040\001\113\001\114\001\
\115\001\255\255\117\001\255\255\255\255\255\255\255\255\255\255\
\123\001\124\001\053\001\255\255\255\255\255\255\255\255\255\255\
\255\255\060\001\061\001\062\001\255\255\255\255\065\001\066\001\
\255\255\255\255\069\001\070\001\255\255\255\255\255\255\074\001\
\075\001\255\255\077\001\255\255\079\001\080\001\255\255\066\001\
\067\001\068\001\255\255\070\001\071\001\255\255\255\255\255\255\
\255\255\076\001\255\255\255\255\255\255\255\255\081\001\082\001\
\083\001\084\001\255\255\086\001\255\255\255\255\255\255\106\001\
\107\001\255\255\109\001\255\255\255\255\255\255\113\001\114\001\
\255\255\255\255\255\255\255\255\255\255\006\001\255\255\255\255\
\255\255\124\001\125\001\012\001\013\001\255\255\015\001\016\001\
\017\001\018\001\019\001\020\001\021\001\120\001\023\001\024\001\
\025\001\026\001\027\001\028\001\029\001\030\001\031\001\032\001\
\033\001\034\001\035\001\036\001\037\001\038\001\006\001\255\255\
\255\255\255\255\255\255\255\255\012\001\255\255\255\255\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\255\255\023\001\
\024\001\025\001\026\001\027\001\028\001\029\001\030\001\031\001\
\032\001\033\001\034\001\035\001\036\001\037\001\038\001\066\001\
\067\001\068\001\255\255\070\001\071\001\255\255\255\255\255\255\
\255\255\076\001\255\255\255\255\255\255\255\255\081\001\082\001\
\083\001\084\001\255\255\086\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\102\001\103\001\255\255\
\105\001\106\001\107\001\255\255\255\255\077\001\255\255\255\255\
\255\255\255\255\115\001\255\255\117\001\085\001\255\255\255\255\
\255\255\255\255\123\001\255\255\255\255\120\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\102\001\103\001\
\255\255\105\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\115\001\006\001\117\001\255\255\255\255\
\255\255\255\255\012\001\123\001\255\255\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\255\255\023\001\024\001\025\001\
\026\001\027\001\028\001\029\001\030\001\031\001\032\001\033\001\
\034\001\035\001\036\001\037\001\038\001\006\001\007\001\255\255\
\255\255\255\255\255\255\012\001\255\255\255\255\015\001\016\001\
\017\001\018\001\019\001\020\001\021\001\255\255\023\001\024\001\
\025\001\026\001\027\001\028\001\029\001\030\001\031\001\032\001\
\033\001\034\001\035\001\036\001\037\001\038\001\255\255\255\255\
\255\255\255\255\076\001\077\001\255\255\054\001\055\001\056\001\
\057\001\058\001\059\001\060\001\061\001\255\255\255\255\255\255\
\255\255\066\001\067\001\068\001\255\255\070\001\071\001\072\001\
\073\001\255\255\255\255\076\001\102\001\103\001\255\255\105\001\
\081\001\082\001\083\001\084\001\255\255\086\001\255\255\255\255\
\255\255\115\001\255\255\117\001\255\255\255\255\255\255\255\255\
\255\255\123\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\102\001\103\001\255\255\
\105\001\255\255\255\255\255\255\255\255\255\255\255\255\120\001\
\255\255\255\255\115\001\006\001\117\001\255\255\255\255\255\255\
\255\255\012\001\123\001\255\255\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\006\001\255\255\255\255\255\255\
\255\255\255\255\012\001\255\255\255\255\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\255\255\023\001\024\001\025\001\
\026\001\027\001\028\001\029\001\030\001\031\001\032\001\033\001\
\034\001\035\001\036\001\037\001\038\001\255\255\255\255\255\255\
\255\255\255\255\077\001\255\255\054\001\055\001\056\001\057\001\
\058\001\059\001\255\255\255\255\255\255\255\255\255\255\255\255\
\066\001\067\001\068\001\255\255\070\001\071\001\072\001\073\001\
\255\255\255\255\076\001\102\001\103\001\255\255\105\001\081\001\
\082\001\083\001\084\001\255\255\086\001\255\255\255\255\255\255\
\115\001\255\255\117\001\085\001\255\255\255\255\255\255\255\255\
\123\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\102\001\103\001\255\255\105\001\
\255\255\255\255\255\255\255\255\255\255\255\255\120\001\255\255\
\255\255\115\001\006\001\117\001\255\255\255\255\255\255\255\255\
\012\001\123\001\255\255\015\001\016\001\017\001\018\001\019\001\
\020\001\021\001\255\255\023\001\024\001\025\001\026\001\027\001\
\028\001\029\001\030\001\031\001\032\001\033\001\034\001\035\001\
\036\001\037\001\038\001\006\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\255\255\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\067\001\255\255\255\255\070\001\
\071\001\255\255\255\255\255\255\255\255\076\001\255\255\255\255\
\255\255\255\255\081\001\082\001\083\001\084\001\255\255\086\001\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\102\001\103\001\255\255\105\001\255\255\007\001\
\008\001\009\001\010\001\011\001\255\255\013\001\014\001\115\001\
\255\255\117\001\255\255\255\255\255\255\255\255\255\255\123\001\
\255\255\120\001\255\255\255\255\067\001\255\255\255\255\070\001\
\071\001\255\255\255\255\102\001\103\001\076\001\105\001\039\001\
\040\001\041\001\081\001\082\001\083\001\084\001\255\255\086\001\
\115\001\255\255\117\001\255\255\255\255\255\255\255\255\255\255\
\123\001\255\255\255\255\255\255\060\001\061\001\062\001\255\255\
\255\255\065\001\066\001\255\255\255\255\069\001\070\001\255\255\
\255\255\255\255\074\001\075\001\255\255\077\001\255\255\079\001\
\255\255\120\001\255\255\007\001\008\001\009\001\010\001\011\001\
\255\255\013\001\014\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\106\001\107\001\255\255\109\001\255\255\255\255\
\255\255\113\001\114\001\039\001\040\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\124\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\060\001\061\001\062\001\255\255\255\255\065\001\066\001\255\255\
\255\255\069\001\070\001\255\255\255\255\255\255\074\001\075\001\
\255\255\077\001\255\255\079\001\255\255\255\255\255\255\007\001\
\008\001\009\001\010\001\011\001\255\255\013\001\014\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\106\001\107\001\
\255\255\109\001\255\255\255\255\255\255\113\001\114\001\039\001\
\040\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\124\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\060\001\061\001\062\001\255\255\
\255\255\065\001\066\001\255\255\255\255\069\001\070\001\255\255\
\255\255\255\255\074\001\075\001\255\255\077\001\255\255\079\001\
\255\255\255\255\255\255\007\001\008\001\009\001\010\001\011\001\
\255\255\013\001\014\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\106\001\107\001\255\255\109\001\255\255\255\255\
\255\255\113\001\114\001\039\001\040\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\124\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\060\001\061\001\062\001\255\255\255\255\065\001\066\001\255\255\
\255\255\069\001\070\001\255\255\255\255\255\255\074\001\075\001\
\255\255\077\001\255\255\079\001\255\255\255\255\255\255\007\001\
\008\001\009\001\010\001\011\001\255\255\013\001\014\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\106\001\107\001\
\255\255\109\001\255\255\255\255\255\255\113\001\114\001\039\001\
\040\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\124\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\060\001\061\001\062\001\255\255\
\255\255\065\001\066\001\255\255\255\255\069\001\070\001\255\255\
\255\255\255\255\074\001\075\001\255\255\077\001\007\001\008\001\
\009\001\010\001\011\001\255\255\013\001\014\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\007\001\008\001\009\001\010\001\011\001\255\255\013\001\
\014\001\255\255\106\001\107\001\255\255\109\001\039\001\040\001\
\255\255\113\001\114\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\124\001\255\255\255\255\255\255\
\255\255\039\001\040\001\060\001\061\001\062\001\255\255\255\255\
\065\001\066\001\255\255\255\255\069\001\070\001\255\255\255\255\
\255\255\074\001\075\001\255\255\077\001\255\255\060\001\061\001\
\062\001\255\255\255\255\065\001\066\001\255\255\255\255\069\001\
\070\001\255\255\255\255\255\255\074\001\075\001\255\255\077\001\
\007\001\008\001\009\001\010\001\011\001\255\255\013\001\014\001\
\255\255\106\001\107\001\255\255\109\001\255\255\255\255\255\255\
\113\001\114\001\255\255\007\001\008\001\009\001\010\001\011\001\
\255\255\013\001\014\001\124\001\106\001\107\001\255\255\109\001\
\039\001\040\001\255\255\113\001\114\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\124\001\255\255\
\255\255\255\255\255\255\039\001\040\001\060\001\061\001\062\001\
\255\255\255\255\065\001\066\001\255\255\255\255\069\001\070\001\
\255\255\255\255\255\255\074\001\075\001\255\255\077\001\255\255\
\060\001\061\001\062\001\255\255\255\255\065\001\066\001\255\255\
\255\255\069\001\070\001\255\255\255\255\255\255\074\001\075\001\
\255\255\077\001\007\001\008\001\009\001\010\001\011\001\255\255\
\013\001\014\001\255\255\106\001\107\001\255\255\109\001\255\255\
\255\255\255\255\113\001\114\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\124\001\106\001\107\001\
\255\255\109\001\039\001\040\001\255\255\113\001\114\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\124\001\255\255\255\255\255\255\255\255\255\255\255\255\060\001\
\061\001\062\001\255\255\255\255\065\001\066\001\255\255\255\255\
\069\001\070\001\255\255\255\255\255\255\074\001\075\001\255\255\
\077\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\006\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\106\001\107\001\255\255\
\109\001\255\255\255\255\255\255\113\001\114\001\031\001\255\255\
\255\255\034\001\035\001\255\255\255\255\038\001\255\255\124\001\
\041\001\042\001\043\001\044\001\045\001\046\001\047\001\048\001\
\049\001\050\001\051\001\052\001\053\001\054\001\055\001\056\001\
\057\001\058\001\059\001\060\001\061\001\062\001\063\001\064\001\
\255\255\066\001\067\001\068\001\255\255\070\001\071\001\072\001\
\073\001\074\001\075\001\076\001\077\001\078\001\255\255\080\001\
\081\001\082\001\083\001\084\001\085\001\086\001\255\255\255\255\
\255\255\255\255\255\255\255\255\006\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\102\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\115\001\255\255\117\001\031\001\255\255\120\001\
\034\001\035\001\255\255\255\255\038\001\255\255\255\255\041\001\
\042\001\043\001\044\001\045\001\046\001\047\001\048\001\049\001\
\050\001\051\001\255\255\255\255\054\001\055\001\056\001\057\001\
\058\001\059\001\060\001\061\001\062\001\063\001\064\001\255\255\
\066\001\067\001\068\001\255\255\070\001\071\001\072\001\073\001\
\255\255\255\255\076\001\255\255\078\001\255\255\255\255\081\001\
\082\001\083\001\084\001\085\001\086\001\255\255\255\255\255\255\
\255\255\255\255\255\255\006\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\102\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\115\001\255\255\117\001\031\001\255\255\120\001\034\001\
\035\001\255\255\255\255\038\001\255\255\255\255\041\001\042\001\
\043\001\044\001\045\001\046\001\047\001\048\001\049\001\050\001\
\051\001\255\255\255\255\054\001\055\001\056\001\057\001\058\001\
\059\001\060\001\061\001\062\001\063\001\064\001\255\255\066\001\
\067\001\068\001\255\255\070\001\071\001\072\001\073\001\255\255\
\255\255\076\001\255\255\078\001\255\255\255\255\081\001\082\001\
\083\001\084\001\085\001\086\001\255\255\255\255\255\255\255\255\
\255\255\255\255\006\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\102\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\115\001\255\255\117\001\031\001\255\255\120\001\034\001\035\001\
\255\255\255\255\038\001\255\255\255\255\041\001\042\001\043\001\
\044\001\045\001\046\001\047\001\048\001\049\001\050\001\051\001\
\255\255\255\255\054\001\055\001\056\001\057\001\058\001\059\001\
\060\001\061\001\062\001\063\001\064\001\255\255\066\001\067\001\
\068\001\255\255\070\001\071\001\072\001\073\001\255\255\255\255\
\076\001\255\255\078\001\255\255\255\255\081\001\082\001\083\001\
\084\001\085\001\086\001\007\001\255\255\255\255\010\001\255\255\
\012\001\013\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\102\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\031\001\255\255\255\255\034\001\115\001\
\255\255\117\001\255\255\255\255\120\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\052\001\053\001\054\001\055\001\056\001\057\001\058\001\059\001\
\060\001\061\001\062\001\063\001\064\001\255\255\066\001\067\001\
\068\001\255\255\070\001\071\001\072\001\073\001\255\255\255\255\
\076\001\077\001\255\255\255\255\080\001\081\001\082\001\083\001\
\084\001\007\001\086\001\255\255\010\001\255\255\012\001\013\001\
\255\255\093\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\106\001\107\001\
\255\255\031\001\255\255\255\255\034\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\120\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\052\001\053\001\
\054\001\055\001\056\001\057\001\058\001\059\001\060\001\061\001\
\062\001\063\001\064\001\255\255\066\001\067\001\068\001\255\255\
\070\001\071\001\072\001\073\001\255\255\255\255\076\001\077\001\
\255\255\255\255\080\001\081\001\082\001\083\001\084\001\007\001\
\086\001\255\255\010\001\255\255\012\001\013\001\255\255\093\001\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\106\001\107\001\255\255\031\001\
\255\255\255\255\034\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\120\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\052\001\053\001\054\001\055\001\
\056\001\057\001\058\001\059\001\060\001\061\001\062\001\063\001\
\064\001\255\255\066\001\067\001\068\001\255\255\070\001\071\001\
\072\001\073\001\255\255\255\255\076\001\255\255\255\255\255\255\
\080\001\081\001\255\255\083\001\084\001\007\001\086\001\255\255\
\010\001\255\255\012\001\013\001\255\255\093\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\106\001\107\001\255\255\031\001\255\255\255\255\
\034\001\255\255\255\255\255\255\255\255\039\001\040\001\255\255\
\120\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\060\001\061\001\062\001\255\255\255\255\065\001\
\066\001\255\255\255\255\069\001\255\255\255\255\255\255\006\001\
\255\255\255\255\255\255\077\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\093\001\255\255\255\255\255\255\255\255\
\031\001\255\255\255\255\034\001\035\001\255\255\255\255\038\001\
\106\001\107\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\120\001\054\001\
\055\001\056\001\057\001\058\001\059\001\060\001\061\001\062\001\
\063\001\064\001\255\255\066\001\067\001\068\001\255\255\070\001\
\071\001\072\001\073\001\255\255\006\001\076\001\255\255\078\001\
\255\255\255\255\081\001\082\001\083\001\084\001\085\001\086\001\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\031\001\255\255\102\001\
\034\001\035\001\255\255\255\255\038\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\115\001\255\255\117\001\255\255\
\255\255\255\255\255\255\255\255\054\001\055\001\056\001\057\001\
\058\001\059\001\060\001\061\001\255\255\255\255\255\255\255\255\
\066\001\067\001\068\001\255\255\070\001\071\001\072\001\073\001\
\255\255\006\001\076\001\255\255\078\001\255\255\255\255\081\001\
\082\001\083\001\084\001\085\001\086\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\031\001\255\255\102\001\034\001\035\001\255\255\
\255\255\038\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\115\001\255\255\117\001\255\255\255\255\255\255\255\255\
\255\255\054\001\055\001\056\001\057\001\058\001\059\001\060\001\
\061\001\255\255\255\255\255\255\255\255\066\001\067\001\068\001\
\255\255\070\001\071\001\072\001\073\001\255\255\006\001\076\001\
\255\255\078\001\255\255\255\255\081\001\082\001\083\001\084\001\
\085\001\086\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\031\001\
\255\255\102\001\034\001\035\001\255\255\255\255\038\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\115\001\255\255\
\117\001\255\255\255\255\255\255\255\255\255\255\054\001\055\001\
\056\001\057\001\058\001\059\001\060\001\061\001\255\255\255\255\
\255\255\255\255\066\001\067\001\068\001\255\255\070\001\071\001\
\072\001\073\001\255\255\006\001\076\001\255\255\078\001\255\255\
\255\255\081\001\082\001\083\001\084\001\085\001\086\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\031\001\255\255\102\001\034\001\
\035\001\255\255\255\255\038\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\115\001\255\255\117\001\255\255\255\255\
\255\255\255\255\255\255\054\001\055\001\056\001\057\001\058\001\
\059\001\255\255\255\255\255\255\255\255\255\255\255\255\066\001\
\067\001\068\001\255\255\070\001\071\001\072\001\073\001\255\255\
\006\001\076\001\255\255\078\001\255\255\255\255\081\001\082\001\
\083\001\084\001\085\001\086\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\031\001\255\255\102\001\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\115\001\255\255\117\001\255\255\255\255\255\255\255\255\255\255\
\054\001\055\001\056\001\057\001\058\001\059\001\255\255\255\255\
\255\255\255\255\255\255\255\255\066\001\067\001\068\001\255\255\
\070\001\071\001\072\001\073\001\255\255\006\001\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\031\001\255\255\
\102\001\034\001\035\001\255\255\255\255\038\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\255\255\117\001\
\255\255\255\255\255\255\255\255\255\255\054\001\055\001\056\001\
\057\001\058\001\059\001\255\255\255\255\255\255\255\255\255\255\
\255\255\066\001\067\001\068\001\255\255\070\001\071\001\072\001\
\073\001\255\255\006\001\076\001\255\255\078\001\255\255\255\255\
\081\001\082\001\083\001\084\001\085\001\086\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\031\001\255\255\102\001\034\001\035\001\
\255\255\255\255\038\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\115\001\255\255\117\001\255\255\255\255\255\255\
\255\255\255\255\054\001\055\001\056\001\057\001\058\001\059\001\
\255\255\255\255\255\255\255\255\255\255\255\255\066\001\067\001\
\068\001\255\255\070\001\071\001\006\001\255\255\255\255\255\255\
\076\001\255\255\078\001\255\255\255\255\081\001\082\001\083\001\
\084\001\085\001\086\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\031\001\255\255\255\255\
\034\001\035\001\102\001\255\255\038\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\115\001\
\255\255\117\001\255\255\255\255\054\001\055\001\056\001\057\001\
\058\001\059\001\255\255\255\255\255\255\255\255\255\255\255\255\
\066\001\067\001\068\001\255\255\070\001\071\001\006\001\255\255\
\255\255\255\255\076\001\255\255\078\001\255\255\255\255\081\001\
\082\001\083\001\084\001\085\001\086\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\031\001\
\255\255\255\255\034\001\035\001\102\001\255\255\038\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\115\001\255\255\117\001\255\255\255\255\054\001\055\001\
\056\001\057\001\058\001\059\001\255\255\255\255\255\255\255\255\
\255\255\255\255\066\001\067\001\068\001\255\255\070\001\071\001\
\006\001\255\255\255\255\255\255\076\001\255\255\078\001\255\255\
\255\255\081\001\082\001\083\001\084\001\085\001\086\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\031\001\255\255\255\255\034\001\035\001\102\001\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\115\001\255\255\117\001\255\255\255\255\
\054\001\055\001\056\001\057\001\058\001\059\001\255\255\255\255\
\006\001\255\255\255\255\255\255\066\001\067\001\068\001\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\054\001\055\001\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\066\001\067\001\068\001\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\054\001\055\001\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\066\001\067\001\068\001\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\054\001\055\001\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\066\001\067\001\068\001\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\066\001\067\001\068\001\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\066\001\067\001\068\001\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\255\255\067\001\068\001\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\255\255\067\001\068\001\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\255\255\067\001\255\255\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\255\255\067\001\255\255\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\255\255\117\001\
\006\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\031\001\255\255\255\255\034\001\035\001\255\255\255\255\
\038\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\000\001\117\001\
\255\255\255\255\255\255\255\255\255\255\007\001\255\255\255\255\
\010\001\071\001\012\001\013\001\255\255\255\255\076\001\255\255\
\078\001\255\255\255\255\081\001\082\001\083\001\084\001\085\001\
\086\001\255\255\255\255\255\255\255\255\031\001\255\255\255\255\
\034\001\255\255\255\255\255\255\255\255\039\001\040\001\255\255\
\102\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\115\001\255\255\117\001\
\255\255\255\255\060\001\061\001\062\001\255\255\255\255\065\001\
\066\001\255\255\255\255\069\001\255\255\000\001\255\255\255\255\
\255\255\255\255\076\001\077\001\007\001\255\255\255\255\010\001\
\255\255\012\001\013\001\255\255\255\255\054\001\055\001\056\001\
\057\001\058\001\059\001\093\001\255\255\255\255\255\255\255\255\
\255\255\066\001\067\001\068\001\031\001\070\001\071\001\034\001\
\106\001\107\001\255\255\076\001\039\001\040\001\255\255\255\255\
\081\001\082\001\083\001\084\001\255\255\086\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\060\001\061\001\062\001\000\001\255\255\065\001\066\001\
\255\255\255\255\069\001\007\001\255\255\255\255\010\001\255\255\
\012\001\013\001\077\001\255\255\255\255\255\255\255\255\120\001\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\093\001\031\001\255\255\255\255\034\001\255\255\
\255\255\255\255\255\255\039\001\040\001\255\255\255\255\106\001\
\107\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\060\001\061\001\062\001\000\001\255\255\065\001\066\001\255\255\
\255\255\069\001\007\001\255\255\255\255\010\001\255\255\012\001\
\013\001\077\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\093\001\031\001\255\255\255\255\034\001\255\255\255\255\
\255\255\255\255\039\001\040\001\255\255\255\255\106\001\107\001\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\060\001\
\061\001\062\001\255\255\255\255\065\001\066\001\255\255\255\255\
\069\001\054\001\055\001\056\001\057\001\058\001\059\001\255\255\
\077\001\255\255\255\255\255\255\255\255\066\001\067\001\068\001\
\255\255\070\001\071\001\072\001\073\001\255\255\255\255\076\001\
\093\001\255\255\255\255\255\255\081\001\082\001\083\001\084\001\
\255\255\086\001\255\255\255\255\255\255\106\001\107\001\255\255\
\255\255\054\001\055\001\056\001\057\001\058\001\059\001\255\255\
\255\255\255\255\255\255\255\255\255\255\066\001\067\001\068\001\
\255\255\070\001\071\001\072\001\073\001\255\255\255\255\076\001\
\255\255\255\255\255\255\120\001\081\001\082\001\083\001\084\001\
\255\255\086\001\054\001\055\001\056\001\057\001\058\001\059\001\
\255\255\255\255\255\255\255\255\255\255\255\255\066\001\067\001\
\068\001\255\255\070\001\071\001\255\255\255\255\255\255\255\255\
\076\001\255\255\255\255\255\255\255\255\081\001\082\001\083\001\
\084\001\255\255\086\001\120\001\255\255\054\001\055\001\056\001\
\057\001\058\001\059\001\255\255\255\255\255\255\255\255\255\255\
\255\255\066\001\067\001\068\001\255\255\070\001\071\001\255\255\
\255\255\255\255\255\255\076\001\255\255\255\255\255\255\255\255\
\081\001\082\001\083\001\084\001\120\001\086\001\054\001\055\001\
\056\001\057\001\058\001\059\001\255\255\255\255\255\255\255\255\
\255\255\255\255\066\001\067\001\068\001\255\255\070\001\071\001\
\255\255\255\255\255\255\255\255\076\001\255\255\255\255\255\255\
\255\255\081\001\082\001\083\001\084\001\255\255\086\001\120\001\
\255\255\054\001\055\001\056\001\057\001\058\001\059\001\255\255\
\255\255\255\255\255\255\255\255\255\255\066\001\067\001\068\001\
\255\255\070\001\071\001\255\255\255\255\255\255\255\255\076\001\
\054\001\055\001\255\255\255\255\081\001\082\001\083\001\084\001\
\120\001\086\001\255\255\255\255\066\001\067\001\068\001\255\255\
\070\001\071\001\255\255\255\255\255\255\255\255\076\001\054\001\
\055\001\255\255\255\255\081\001\082\001\083\001\084\001\255\255\
\086\001\255\255\255\255\066\001\067\001\068\001\255\255\070\001\
\071\001\255\255\255\255\120\001\255\255\076\001\255\255\255\255\
\255\255\255\255\081\001\082\001\083\001\084\001\255\255\086\001\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\120\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\120\001\041\001\042\001\043\001\044\001\045\001\046\001\
\047\001\048\001\049\001\050\001\051\001\052\001\053\001\054\001\
\055\001\056\001\057\001\058\001\059\001\060\001\061\001\062\001\
\063\001\064\001\255\255\066\001\067\001\068\001\255\255\070\001\
\071\001\072\001\073\001\074\001\075\001\255\255\077\001\078\001\
\255\255\080\001\255\255\082\001\083\001\084\001\255\255\086\001"

let yynames_const = "\
  EOF\000\
  EQ\000\
  PLUS_EQ\000\
  MINUS_EQ\000\
  STAR_EQ\000\
  SLASH_EQ\000\
  PERCENT_EQ\000\
  AND_EQ\000\
  PIPE_EQ\000\
  CIRC_EQ\000\
  INF_INF_EQ\000\
  SUP_SUP_EQ\000\
  ARROW\000\
  DOT\000\
  EQ_EQ\000\
  EXCLAM_EQ\000\
  INF\000\
  SUP\000\
  INF_EQ\000\
  SUP_EQ\000\
  SLASH\000\
  PERCENT\000\
  PIPE\000\
  CIRC\000\
  PIPE_PIPE\000\
  INF_INF\000\
  SUP_SUP\000\
  RPAREN\000\
  LBRACKET\000\
  RBRACKET\000\
  COLON\000\
  COMMA\000\
  ELLIPSIS\000\
  QUEST\000\
  ELSE\000\
  LABEL__\000\
  BUILTIN_VA_LIST\000\
  BLOCKATTRIBUTE\000\
  PRAGMA_EOL\000\
  AT_NAME\000\
  LGHOST\000\
  RGHOST\000\
  "

let yynames_block = "\
  FOR_SPEC\000\
  SPEC\000\
  DECL\000\
  CODE_ANNOT\000\
  LOOP_ANNOT\000\
  ATTRIBUTE_ANNOT\000\
  IDENT\000\
  CST_CHAR\000\
  CST_WCHAR\000\
  CST_INT\000\
  CST_FLOAT\000\
  NAMED_TYPE\000\
  CST_STRING\000\
  CST_WSTRING\000\
  BOOL\000\
  CHAR\000\
  INT\000\
  DOUBLE\000\
  FLOAT\000\
  VOID\000\
  INT64\000\
  INT32\000\
  ENUM\000\
  STRUCT\000\
  TYPEDEF\000\
  UNION\000\
  SIGNED\000\
  UNSIGNED\000\
  LONG\000\
  SHORT\000\
  VOLATILE\000\
  EXTERN\000\
  STATIC\000\
  CONST\000\
  RESTRICT\000\
  AUTO\000\
  REGISTER\000\
  THREAD\000\
  SIZEOF\000\
  ALIGNOF\000\
  PLUS\000\
  MINUS\000\
  STAR\000\
  TILDE\000\
  AND\000\
  EXCLAM\000\
  AND_AND\000\
  PLUS_PLUS\000\
  MINUS_MINUS\000\
  LPAREN\000\
  RBRACE\000\
  LBRACE\000\
  SEMICOLON\000\
  BREAK\000\
  CONTINUE\000\
  GOTO\000\
  RETURN\000\
  SWITCH\000\
  CASE\000\
  DEFAULT\000\
  WHILE\000\
  DO\000\
  FOR\000\
  IF\000\
  TRY\000\
  EXCEPT\000\
  FINALLY\000\
  ATTRIBUTE\000\
  INLINE\000\
  ASM\000\
  TYPEOF\000\
  FUNCTION__\000\
  PRETTY_FUNCTION__\000\
  BUILTIN_VA_ARG\000\
  ATTRIBUTE_USED\000\
  BUILTIN_TYPES_COMPAT\000\
  BUILTIN_OFFSETOF\000\
  DECLSPEC\000\
  MSASM\000\
  MSATTR\000\
  PRAGMA_LINE\000\
  PRAGMA\000\
  AT_TRANSFORM\000\
  AT_TRANSFORMEXPR\000\
  AT_SPECIFIER\000\
  AT_EXPR\000\
  "

let yyact = [|
  (fun _ -> failwith "parser")
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : (bool*Cabs.definition) list) in
    Obj.repr(
# 443 "cil/src/frontc/cparser.mly"
              (_1)
# 3216 "cil/src/frontc/cparser.ml"
               : (bool*Cabs.definition) list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : (bool*Cabs.definition) list) in
    Obj.repr(
# 445 "cil/src/frontc/cparser.mly"
                 (_1)
# 3223 "cil/src/frontc/cparser.ml"
               : (bool*Cabs.definition) list))
; (fun __caml_parser_env ->
    Obj.repr(
# 448 "cil/src/frontc/cparser.mly"
                                        ( [] )
# 3229 "cil/src/frontc/cparser.ml"
               : (bool*Cabs.definition) list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.definition) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : (bool*Cabs.definition) list) in
    Obj.repr(
# 449 "cil/src/frontc/cparser.mly"
                                        ( (false,_1) :: _2 )
# 3237 "cil/src/frontc/cparser.ml"
               : (bool*Cabs.definition) list))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'ghost_globals) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : (bool*Cabs.definition) list) in
    Obj.repr(
# 450 "cil/src/frontc/cparser.mly"
                                        ( _2 @ _3 )
# 3245 "cil/src/frontc/cparser.ml"
               : (bool*Cabs.definition) list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : (bool*Cabs.definition) list) in
    Obj.repr(
# 451 "cil/src/frontc/cparser.mly"
                                        ( _2 )
# 3253 "cil/src/frontc/cparser.ml"
               : (bool*Cabs.definition) list))
; (fun __caml_parser_env ->
    Obj.repr(
# 455 "cil/src/frontc/cparser.mly"
                               ( currentLoc () )
# 3259 "cil/src/frontc/cparser.ml"
               : Cabs.cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.definition) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'ghost_globals) in
    Obj.repr(
# 460 "cil/src/frontc/cparser.mly"
                                                      ( (true,_1)::_2 )
# 3267 "cil/src/frontc/cparser.ml"
               : 'ghost_globals))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.definition) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'ghost_globals) in
    Obj.repr(
# 461 "cil/src/frontc/cparser.mly"
                                                      ( (true,_1)::_2 )
# 3275 "cil/src/frontc/cparser.ml"
               : 'ghost_globals))
; (fun __caml_parser_env ->
    Obj.repr(
# 462 "cil/src/frontc/cparser.mly"
                                                      ( [] )
# 3281 "cil/src/frontc/cparser.ml"
               : 'ghost_globals))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Logic_ptree.decl list) in
    Obj.repr(
# 467 "cil/src/frontc/cparser.mly"
                                        ( GLOBANNOT _1 )
# 3288 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.definition) in
    Obj.repr(
# 468 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 3295 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.definition) in
    Obj.repr(
# 469 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 3302 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string * cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.definition) in
    Obj.repr(
# 473 "cil/src/frontc/cparser.mly"
                                        ( LINKAGE (fst _2, (*handleLoc*) (snd _2), [ _3 ]) )
# 3311 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : string * cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : (bool*Cabs.definition) list) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 475 "cil/src/frontc/cparser.mly"
      ( LINKAGE (fst _2, (*handleLoc*) (snd _2),
                 List.map
                   (fun (x,y) ->
                      if x then parse_error "invalid ghost in extern linkage specification"
		      else y)
                   _4)  )
# 3327 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : string * cabsloc) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 482 "cil/src/frontc/cparser.mly"
                                        ( GLOBASM (fst _3, (*handleLoc*) _1) )
# 3337 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'pragma) in
    Obj.repr(
# 483 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 3344 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : string list) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'old_pardef_list) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 488 "cil/src/frontc/cparser.mly"
    (
      let loc = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 1 in
      (* Convert pardecl to new style *)
      let pardecl, isva = doOldParDecl _3 _5 in
      (* Make the function declarator *)
      doDeclaration None loc []
        [((_1, PROTO(JUSTBASE, pardecl,isva), [], loc), NO_INIT)]
    )
# 3362 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 498 "cil/src/frontc/cparser.mly"
    ( (* Make the function declarator *)
      let loc = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 1 in
      doDeclaration None loc []
        [((_1, PROTO(JUSTBASE,[],false), [], loc), NO_INIT)]
    )
# 3375 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 7 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 6 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 5 : Cabs.definition) in
    let _4 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _5 = (Parsing.peek_val __caml_parser_env 3 : string) in
    let _6 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _7 = (Parsing.peek_val __caml_parser_env 1 : (bool*Cabs.definition) list) in
    let _8 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 504 "cil/src/frontc/cparser.mly"
                                                                        (
    checkConnective _5;
    TRANSFORMER(_3,
                  List.map
                    (fun (x,y) ->
                       if x then parse_error "invalid ghost transformer"
                       else y)
                    _7,
                _1)
  )
# 3398 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 7 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 6 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 5 : Cabs.expression) in
    let _4 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _5 = (Parsing.peek_val __caml_parser_env 3 : string) in
    let _6 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _7 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression) in
    let _8 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 515 "cil/src/frontc/cparser.mly"
                                                                                   (
    checkConnective _5;
    EXPRTRANSFORMER(_3, _7, _1)
  )
# 3415 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 519 "cil/src/frontc/cparser.mly"
                           ( PRAGMA (make_expr (VARIABLE "parse_error"), _1) )
# 3423 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 523 "cil/src/frontc/cparser.mly"
             ( _1 )
# 3430 "cil/src/frontc/cparser.ml"
               : 'id_or_typename_as_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 524 "cil/src/frontc/cparser.mly"
                  ( _1 )
# 3437 "cil/src/frontc/cparser.ml"
               : 'id_or_typename_as_id))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename_as_id) in
    Obj.repr(
# 528 "cil/src/frontc/cparser.mly"
                          ( _1 )
# 3444 "cil/src/frontc/cparser.ml"
               : 'id_or_typename))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : string) in
    Obj.repr(
# 529 "cil/src/frontc/cparser.mly"
                                        ( "@name(" ^ _3 ^ ")" )
# 3452 "cil/src/frontc/cparser.ml"
               : 'id_or_typename))
; (fun __caml_parser_env ->
    Obj.repr(
# 534 "cil/src/frontc/cparser.mly"
                                        ( () )
# 3458 "cil/src/frontc/cparser.ml"
               : 'maybecomma))
; (fun __caml_parser_env ->
    Obj.repr(
# 535 "cil/src/frontc/cparser.mly"
                                        ( () )
# 3464 "cil/src/frontc/cparser.ml"
               : 'maybecomma))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 541 "cil/src/frontc/cparser.mly"
         ( make_expr (VARIABLE _1) )
# 3471 "cil/src/frontc/cparser.ml"
               : 'primary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.constant * cabsloc) in
    Obj.repr(
# 542 "cil/src/frontc/cparser.mly"
                   ( make_expr (CONSTANT (fst _1)) )
# 3478 "cil/src/frontc/cparser.ml"
               : 'primary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression list) in
    Obj.repr(
# 544 "cil/src/frontc/cparser.mly"
          ( make_expr (PAREN (smooth_expression _1)) )
# 3485 "cil/src/frontc/cparser.ml"
               : 'primary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.block * cabsloc * cabsloc) in
    Obj.repr(
# 545 "cil/src/frontc/cparser.mly"
                       ( make_expr (GNU_BODY (fst3 _2)) )
# 3493 "cil/src/frontc/cparser.ml"
               : 'primary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : string) in
    Obj.repr(
# 547 "cil/src/frontc/cparser.mly"
                                            ( make_expr (EXPR_PATTERN _3) )
# 3502 "cil/src/frontc/cparser.ml"
               : 'primary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'primary_expression) in
    Obj.repr(
# 551 "cil/src/frontc/cparser.mly"
                     ( _1 )
# 3509 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'postfix_expression) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression list) in
    Obj.repr(
# 553 "cil/src/frontc/cparser.mly"
      (make_expr (INDEX (_1, smooth_expression _2)))
# 3517 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'postfix_expression) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    Obj.repr(
# 554 "cil/src/frontc/cparser.mly"
                                             (make_expr (CALL (_1, _3)))
# 3526 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : Cabs.expression) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : Cabs.spec_elem list * Cabs.decl_type) in
    Obj.repr(
# 556 "cil/src/frontc/cparser.mly"
      ( let b, d = _5 in
        let loc = Parsing.rhs_start_pos 5, Parsing.rhs_end_pos 5 in
        let loc_f = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 1 in
        make_expr
          (CALL
             ({ expr_loc = loc_f;
                expr_node = VARIABLE "__builtin_va_arg"},
              [_3; { expr_loc = loc;
                     expr_node = TYPE_SIZEOF (b, d)}]))
      )
# 3545 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : Cabs.spec_elem list * Cabs.decl_type) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : Cabs.spec_elem list * Cabs.decl_type) in
    Obj.repr(
# 567 "cil/src/frontc/cparser.mly"
      ( let b1,d1 = _3 in
        let b2,d2 = _5 in
        let loc_f = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 1 in
        let loc1 = Parsing.rhs_start_pos 3, Parsing.rhs_end_pos 3 in
        let loc2 = Parsing.rhs_start_pos 5, Parsing.rhs_end_pos 5 in
        make_expr
          (CALL
             ({expr_loc = loc_f;
               expr_node = VARIABLE "__builtin_types_compatible_p"},
              [ { expr_loc = loc1; expr_node = TYPE_SIZEOF(b1,d1)};
                { expr_loc = loc2; expr_node = TYPE_SIZEOF(b2,d2)}]))
      )
# 3566 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : Cabs.spec_elem list * Cabs.decl_type) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'offsetof_member_designator) in
    Obj.repr(
# 580 "cil/src/frontc/cparser.mly"
      ( transformOffsetOf _3 _5 )
# 3576 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'postfix_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename) in
    Obj.repr(
# 581 "cil/src/frontc/cparser.mly"
                                        ( make_expr (MEMBEROF (_1, _3)))
# 3584 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'postfix_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename) in
    Obj.repr(
# 582 "cil/src/frontc/cparser.mly"
                                          ( make_expr (MEMBEROFPTR (_1, _3)) )
# 3592 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'postfix_expression) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 583 "cil/src/frontc/cparser.mly"
                               ( make_expr (UNARY (POSINCR, _1)) )
# 3600 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'postfix_expression) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 584 "cil/src/frontc/cparser.mly"
                                 ( make_expr (UNARY (POSDECR, _1)) )
# 3608 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : Cabs.spec_elem list * Cabs.decl_type) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'initializer_list_opt) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 587 "cil/src/frontc/cparser.mly"
      ( make_expr (CAST(_2, COMPOUND_INIT _5)) )
# 3619 "cil/src/frontc/cparser.ml"
               : 'postfix_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename) in
    Obj.repr(
# 591 "cil/src/frontc/cparser.mly"
                  ( make_expr (VARIABLE _1) )
# 3626 "cil/src/frontc/cparser.ml"
               : 'offsetof_member_designator))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'offsetof_member_designator) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 593 "cil/src/frontc/cparser.mly"
   ( make_expr (MEMBEROF (_1, _3)) )
# 3634 "cil/src/frontc/cparser.ml"
               : 'offsetof_member_designator))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'offsetof_member_designator) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression list) in
    Obj.repr(
# 595 "cil/src/frontc/cparser.mly"
   ( make_expr (INDEX (_1, smooth_expression _2)) )
# 3642 "cil/src/frontc/cparser.ml"
               : 'offsetof_member_designator))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'postfix_expression) in
    Obj.repr(
# 600 "cil/src/frontc/cparser.mly"
                        ( _1 )
# 3649 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'unary_expression) in
    Obj.repr(
# 602 "cil/src/frontc/cparser.mly"
          (make_expr (UNARY (PREINCR, _2)))
# 3657 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'unary_expression) in
    Obj.repr(
# 604 "cil/src/frontc/cparser.mly"
          (make_expr (UNARY (PREDECR, _2)))
# 3665 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'unary_expression) in
    Obj.repr(
# 606 "cil/src/frontc/cparser.mly"
          (make_expr (EXPR_SIZEOF _2))
# 3673 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.spec_elem list * Cabs.decl_type) in
    Obj.repr(
# 608 "cil/src/frontc/cparser.mly"
          (let b, d = _3 in make_expr (TYPE_SIZEOF (b, d)) )
# 3682 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'unary_expression) in
    Obj.repr(
# 610 "cil/src/frontc/cparser.mly"
          ( make_expr (EXPR_ALIGNOF _2) )
# 3690 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.spec_elem list * Cabs.decl_type) in
    Obj.repr(
# 612 "cil/src/frontc/cparser.mly"
          (let b, d = _3 in make_expr (TYPE_ALIGNOF (b, d)) )
# 3699 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 614 "cil/src/frontc/cparser.mly"
          ( make_expr (UNARY (PLUS, _2)) )
# 3707 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 616 "cil/src/frontc/cparser.mly"
          ( make_expr (UNARY (MINUS, _2)) )
# 3715 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 618 "cil/src/frontc/cparser.mly"
          (make_expr (UNARY (MEMOF, _2)) )
# 3723 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 620 "cil/src/frontc/cparser.mly"
          (make_expr (UNARY (ADDROF, _2)))
# 3731 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 622 "cil/src/frontc/cparser.mly"
          ( make_expr (UNARY (NOT, _2)) )
# 3739 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 624 "cil/src/frontc/cparser.mly"
          ( make_expr (UNARY (BNOT, _2)) )
# 3747 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename_as_id) in
    Obj.repr(
# 626 "cil/src/frontc/cparser.mly"
                                             ( make_expr (LABELADDR _2) )
# 3755 "cil/src/frontc/cparser.ml"
               : 'unary_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'unary_expression) in
    Obj.repr(
# 630 "cil/src/frontc/cparser.mly"
                   ( _1 )
# 3762 "cil/src/frontc/cparser.ml"
               : 'cast_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.spec_elem list * Cabs.decl_type) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 632 "cil/src/frontc/cparser.mly"
      ( make_expr (CAST(_2, SINGLE_INIT _4)) )
# 3771 "cil/src/frontc/cparser.ml"
               : 'cast_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 636 "cil/src/frontc/cparser.mly"
                  ( _1 )
# 3778 "cil/src/frontc/cparser.ml"
               : 'multiplicative_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'multiplicative_expression) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 638 "cil/src/frontc/cparser.mly"
      ( make_expr (BINARY(MUL, _1, _3)) )
# 3787 "cil/src/frontc/cparser.ml"
               : 'multiplicative_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'multiplicative_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 640 "cil/src/frontc/cparser.mly"
      ( make_expr (BINARY(DIV, _1, _3)) )
# 3795 "cil/src/frontc/cparser.ml"
               : 'multiplicative_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'multiplicative_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'cast_expression) in
    Obj.repr(
# 642 "cil/src/frontc/cparser.mly"
      ( make_expr (BINARY(MOD, _1, _3)) )
# 3803 "cil/src/frontc/cparser.ml"
               : 'multiplicative_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'multiplicative_expression) in
    Obj.repr(
# 646 "cil/src/frontc/cparser.mly"
                                          ( _1 )
# 3810 "cil/src/frontc/cparser.ml"
               : 'additive_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'additive_expression) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'multiplicative_expression) in
    Obj.repr(
# 648 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(ADD, _1, _3)) )
# 3819 "cil/src/frontc/cparser.ml"
               : 'additive_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'additive_expression) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'multiplicative_expression) in
    Obj.repr(
# 650 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(SUB, _1, _3)) )
# 3828 "cil/src/frontc/cparser.ml"
               : 'additive_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'additive_expression) in
    Obj.repr(
# 654 "cil/src/frontc/cparser.mly"
                                    ( _1 )
# 3835 "cil/src/frontc/cparser.ml"
               : 'shift_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'shift_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'additive_expression) in
    Obj.repr(
# 656 "cil/src/frontc/cparser.mly"
   (make_expr (BINARY(SHL, _1, _3)) )
# 3843 "cil/src/frontc/cparser.ml"
               : 'shift_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'shift_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'additive_expression) in
    Obj.repr(
# 658 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(SHR, _1, _3)) )
# 3851 "cil/src/frontc/cparser.ml"
               : 'shift_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'shift_expression) in
    Obj.repr(
# 663 "cil/src/frontc/cparser.mly"
                                 ( _1 )
# 3858 "cil/src/frontc/cparser.ml"
               : 'relational_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'relational_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shift_expression) in
    Obj.repr(
# 665 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(LT, _1, _3)) )
# 3866 "cil/src/frontc/cparser.ml"
               : 'relational_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'relational_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shift_expression) in
    Obj.repr(
# 667 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(GT, _1, _3)) )
# 3874 "cil/src/frontc/cparser.ml"
               : 'relational_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'relational_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shift_expression) in
    Obj.repr(
# 669 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(LE, _1, _3)) )
# 3882 "cil/src/frontc/cparser.ml"
               : 'relational_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'relational_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shift_expression) in
    Obj.repr(
# 671 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(GE, _1, _3)) )
# 3890 "cil/src/frontc/cparser.ml"
               : 'relational_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'relational_expression) in
    Obj.repr(
# 675 "cil/src/frontc/cparser.mly"
                        ( _1 )
# 3897 "cil/src/frontc/cparser.ml"
               : 'equality_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'equality_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'relational_expression) in
    Obj.repr(
# 677 "cil/src/frontc/cparser.mly"
      ( make_expr (BINARY(EQ, _1, _3)) )
# 3905 "cil/src/frontc/cparser.ml"
               : 'equality_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'equality_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'relational_expression) in
    Obj.repr(
# 679 "cil/src/frontc/cparser.mly"
      ( make_expr (BINARY(NE, _1, _3)) )
# 3913 "cil/src/frontc/cparser.ml"
               : 'equality_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'equality_expression) in
    Obj.repr(
# 683 "cil/src/frontc/cparser.mly"
                                    ( _1 )
# 3920 "cil/src/frontc/cparser.ml"
               : 'bitwise_and_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'bitwise_and_expression) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'equality_expression) in
    Obj.repr(
# 685 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(BAND, _1, _3)) )
# 3929 "cil/src/frontc/cparser.ml"
               : 'bitwise_and_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_and_expression) in
    Obj.repr(
# 689 "cil/src/frontc/cparser.mly"
                                       ( _1 )
# 3936 "cil/src/frontc/cparser.ml"
               : 'bitwise_xor_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'bitwise_xor_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_and_expression) in
    Obj.repr(
# 691 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(XOR, _1, _3)) )
# 3944 "cil/src/frontc/cparser.ml"
               : 'bitwise_xor_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_xor_expression) in
    Obj.repr(
# 695 "cil/src/frontc/cparser.mly"
                                       ( _1 )
# 3951 "cil/src/frontc/cparser.ml"
               : 'bitwise_or_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'bitwise_or_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_xor_expression) in
    Obj.repr(
# 697 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(BOR, _1, _3)) )
# 3959 "cil/src/frontc/cparser.ml"
               : 'bitwise_or_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_or_expression) in
    Obj.repr(
# 701 "cil/src/frontc/cparser.mly"
                                      ( _1 )
# 3966 "cil/src/frontc/cparser.ml"
               : 'logical_and_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'logical_and_expression) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_or_expression) in
    Obj.repr(
# 703 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(AND, _1, _3)) )
# 3975 "cil/src/frontc/cparser.ml"
               : 'logical_and_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'logical_and_expression) in
    Obj.repr(
# 707 "cil/src/frontc/cparser.mly"
                                       ( _1 )
# 3982 "cil/src/frontc/cparser.ml"
               : 'logical_or_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'logical_or_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'logical_and_expression) in
    Obj.repr(
# 709 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(OR, _1, _3)) )
# 3990 "cil/src/frontc/cparser.ml"
               : 'logical_or_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'logical_or_expression) in
    Obj.repr(
# 713 "cil/src/frontc/cparser.mly"
                        ( _1 )
# 3997 "cil/src/frontc/cparser.ml"
               : 'conditional_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : 'logical_or_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : Cabs.expression) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : 'conditional_expression) in
    Obj.repr(
# 715 "cil/src/frontc/cparser.mly"
      ( make_expr (QUESTION (_1, _3, _5)) )
# 4006 "cil/src/frontc/cparser.ml"
               : 'conditional_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'conditional_expression) in
    Obj.repr(
# 722 "cil/src/frontc/cparser.mly"
                                       ( _1 )
# 4013 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 724 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(ASSIGN, _1, _3)) )
# 4021 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 726 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(ADD_ASSIGN, _1, _3)) )
# 4029 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 728 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(SUB_ASSIGN, _1, _3)) )
# 4037 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 730 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(MUL_ASSIGN, _1, _3)) )
# 4045 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 732 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(DIV_ASSIGN, _1, _3)) )
# 4053 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 734 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(MOD_ASSIGN, _1, _3)) )
# 4061 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 736 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(BAND_ASSIGN, _1, _3)) )
# 4069 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 738 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(BOR_ASSIGN, _1, _3)) )
# 4077 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 740 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(XOR_ASSIGN, _1, _3)) )
# 4085 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 742 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(SHL_ASSIGN, _1, _3)) )
# 4093 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cast_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 744 "cil/src/frontc/cparser.mly"
   ( make_expr (BINARY(SHR_ASSIGN, _1, _3)))
# 4101 "cil/src/frontc/cparser.ml"
               : 'assignment_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'assignment_expression) in
    Obj.repr(
# 748 "cil/src/frontc/cparser.mly"
                                      ( _1 )
# 4108 "cil/src/frontc/cparser.ml"
               : Cabs.expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 753 "cil/src/frontc/cparser.mly"
               (CONST_INT (fst _1), snd _1)
# 4115 "cil/src/frontc/cparser.ml"
               : Cabs.constant * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 754 "cil/src/frontc/cparser.mly"
                 (CONST_FLOAT (fst _1), snd _1)
# 4122 "cil/src/frontc/cparser.ml"
               : Cabs.constant * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int64 list * Cabs.cabsloc) in
    Obj.repr(
# 755 "cil/src/frontc/cparser.mly"
                (CONST_CHAR (fst _1), snd _1)
# 4129 "cil/src/frontc/cparser.ml"
               : Cabs.constant * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int64 list * Cabs.cabsloc) in
    Obj.repr(
# 756 "cil/src/frontc/cparser.mly"
                 (CONST_WCHAR (fst _1), snd _1)
# 4136 "cil/src/frontc/cparser.ml"
               : Cabs.constant * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * cabsloc) in
    Obj.repr(
# 757 "cil/src/frontc/cparser.mly"
                             (CONST_STRING (fst _1), snd _1)
# 4143 "cil/src/frontc/cparser.ml"
               : Cabs.constant * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int64 list * cabsloc) in
    Obj.repr(
# 758 "cil/src/frontc/cparser.mly"
                   (CONST_WSTRING (fst _1), snd _1)
# 4150 "cil/src/frontc/cparser.ml"
               : Cabs.constant * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int64 list Queue.t * cabsloc) in
    Obj.repr(
# 764 "cil/src/frontc/cparser.mly"
                                        (
     let queue, location = _1 in
     let buffer = Buffer.create (Queue.length queue) in
     Queue.iter
       (List.iter
	  (fun value ->
	    let char = int64_to_char value in
	    Buffer.add_char buffer char))
       queue;
     Buffer.contents buffer, location
   )
# 4167 "cil/src/frontc/cparser.ml"
               : string * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int64 list * Cabs.cabsloc) in
    Obj.repr(
# 778 "cil/src/frontc/cparser.mly"
                                        (intlist_to_string (fst _1) )
# 4174 "cil/src/frontc/cparser.ml"
               : 'one_string_constant))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'one_string) in
    Obj.repr(
# 781 "cil/src/frontc/cparser.mly"
                                        (
      let queue = Queue.create () in
      Queue.add (fst _1) queue;
      queue, snd _1
    )
# 4185 "cil/src/frontc/cparser.ml"
               : int64 list Queue.t * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : int64 list Queue.t * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'one_string) in
    Obj.repr(
# 786 "cil/src/frontc/cparser.mly"
                                        (
      Queue.add (fst _2) (fst _1);
      _1
    )
# 4196 "cil/src/frontc/cparser.ml"
               : int64 list Queue.t * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int64 list * Cabs.cabsloc) in
    Obj.repr(
# 793 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 4203 "cil/src/frontc/cparser.ml"
               : int64 list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : int64 list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'one_string) in
    Obj.repr(
# 794 "cil/src/frontc/cparser.mly"
                                        ( (fst _1) @ (fst _2), snd _1 )
# 4211 "cil/src/frontc/cparser.ml"
               : int64 list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : int64 list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : int64 list * Cabs.cabsloc) in
    Obj.repr(
# 795 "cil/src/frontc/cparser.mly"
                                        ( (fst _1) @ (fst _2), snd _1 )
# 4219 "cil/src/frontc/cparser.ml"
               : int64 list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int64 list * Cabs.cabsloc) in
    Obj.repr(
# 800 "cil/src/frontc/cparser.mly"
                  (_1)
# 4226 "cil/src/frontc/cparser.ml"
               : 'one_string))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 801 "cil/src/frontc/cparser.mly"
                                        ((Cabshelper.explodeStringToInts
					    !currentFunctionName), _1)
# 4234 "cil/src/frontc/cparser.ml"
               : 'one_string))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 803 "cil/src/frontc/cparser.mly"
                                        ((Cabshelper.explodeStringToInts
					    !currentFunctionName), _1)
# 4242 "cil/src/frontc/cparser.ml"
               : 'one_string))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression) in
    Obj.repr(
# 808 "cil/src/frontc/cparser.mly"
                        ( SINGLE_INIT _1 )
# 4249 "cil/src/frontc/cparser.ml"
               : Cabs.init_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'initializer_list_opt) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 810 "cil/src/frontc/cparser.mly"
   ( COMPOUND_INIT _2)
# 4258 "cil/src/frontc/cparser.ml"
               : Cabs.init_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.initwhat * Cabs.init_expression) in
    Obj.repr(
# 813 "cil/src/frontc/cparser.mly"
                                                   ( [_1] )
# 4265 "cil/src/frontc/cparser.ml"
               : (Cabs.initwhat * Cabs.init_expression) list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.initwhat * Cabs.init_expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'initializer_list_opt) in
    Obj.repr(
# 814 "cil/src/frontc/cparser.mly"
                                                   ( _1 :: _3 )
# 4273 "cil/src/frontc/cparser.ml"
               : (Cabs.initwhat * Cabs.init_expression) list))
; (fun __caml_parser_env ->
    Obj.repr(
# 817 "cil/src/frontc/cparser.mly"
                                            ( [] )
# 4279 "cil/src/frontc/cparser.ml"
               : 'initializer_list_opt))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : (Cabs.initwhat * Cabs.init_expression) list) in
    Obj.repr(
# 818 "cil/src/frontc/cparser.mly"
                                            ( _1 )
# 4286 "cil/src/frontc/cparser.ml"
               : 'initializer_list_opt))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.initwhat) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'eq_opt) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.init_expression) in
    Obj.repr(
# 821 "cil/src/frontc/cparser.mly"
                                            ( (_1, _3) )
# 4295 "cil/src/frontc/cparser.ml"
               : Cabs.initwhat * Cabs.init_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'gcc_init_designators) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.init_expression) in
    Obj.repr(
# 822 "cil/src/frontc/cparser.mly"
                                         ( (_1, _2) )
# 4303 "cil/src/frontc/cparser.ml"
               : Cabs.initwhat * Cabs.init_expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.init_expression) in
    Obj.repr(
# 823 "cil/src/frontc/cparser.mly"
                                        ( (NEXT_INIT, _1) )
# 4310 "cil/src/frontc/cparser.ml"
               : Cabs.initwhat * Cabs.init_expression))
; (fun __caml_parser_env ->
    Obj.repr(
# 826 "cil/src/frontc/cparser.mly"
                             ( () )
# 4316 "cil/src/frontc/cparser.ml"
               : 'eq_opt))
; (fun __caml_parser_env ->
    Obj.repr(
# 828 "cil/src/frontc/cparser.mly"
                                 ( () )
# 4322 "cil/src/frontc/cparser.ml"
               : 'eq_opt))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'id_or_typename) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.initwhat) in
    Obj.repr(
# 831 "cil/src/frontc/cparser.mly"
                                                 ( INFIELD_INIT(_2, _3) )
# 4330 "cil/src/frontc/cparser.ml"
               : Cabs.initwhat))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.expression) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Cabs.initwhat) in
    Obj.repr(
# 832 "cil/src/frontc/cparser.mly"
                                                       ( ATINDEX_INIT(_2, _4) )
# 4338 "cil/src/frontc/cparser.ml"
               : Cabs.initwhat))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 3 : Cabs.expression) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression) in
    Obj.repr(
# 834 "cil/src/frontc/cparser.mly"
        ( ATINDEXRANGE_INIT(_2, _4) )
# 4346 "cil/src/frontc/cparser.ml"
               : Cabs.initwhat))
; (fun __caml_parser_env ->
    Obj.repr(
# 837 "cil/src/frontc/cparser.mly"
                                        ( NEXT_INIT )
# 4352 "cil/src/frontc/cparser.ml"
               : Cabs.initwhat))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.initwhat) in
    Obj.repr(
# 838 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 4359 "cil/src/frontc/cparser.ml"
               : Cabs.initwhat))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'id_or_typename) in
    Obj.repr(
# 842 "cil/src/frontc/cparser.mly"
                                        ( INFIELD_INIT(_1, NEXT_INIT) )
# 4366 "cil/src/frontc/cparser.ml"
               : 'gcc_init_designators))
; (fun __caml_parser_env ->
    Obj.repr(
# 846 "cil/src/frontc/cparser.mly"
                                    ( [] )
# 4372 "cil/src/frontc/cparser.ml"
               : Cabs.expression list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression list) in
    Obj.repr(
# 847 "cil/src/frontc/cparser.mly"
                                    ( _1 )
# 4379 "cil/src/frontc/cparser.ml"
               : Cabs.expression list))
; (fun __caml_parser_env ->
    Obj.repr(
# 851 "cil/src/frontc/cparser.mly"
                     (make_expr NOTHING)
# 4385 "cil/src/frontc/cparser.ml"
               : Cabs.expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression list) in
    Obj.repr(
# 852 "cil/src/frontc/cparser.mly"
                           (smooth_expression _1 )
# 4392 "cil/src/frontc/cparser.ml"
               : Cabs.expression))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression) in
    Obj.repr(
# 856 "cil/src/frontc/cparser.mly"
                                           ( [_1] )
# 4399 "cil/src/frontc/cparser.ml"
               : Cabs.expression list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.expression) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression list) in
    Obj.repr(
# 857 "cil/src/frontc/cparser.mly"
                                                  ( _1 :: _3 )
# 4407 "cil/src/frontc/cparser.ml"
               : Cabs.expression list))
; (fun __caml_parser_env ->
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression list) in
    Obj.repr(
# 858 "cil/src/frontc/cparser.mly"
                                                  ( _3 )
# 4414 "cil/src/frontc/cparser.ml"
               : Cabs.expression list))
; (fun __caml_parser_env ->
    Obj.repr(
# 862 "cil/src/frontc/cparser.mly"
                                    ( make_expr NOTHING )
# 4420 "cil/src/frontc/cparser.ml"
               : 'comma_expression_opt))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression list) in
    Obj.repr(
# 863 "cil/src/frontc/cparser.mly"
                                    ( smooth_expression _1 )
# 4427 "cil/src/frontc/cparser.ml"
               : 'comma_expression_opt))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    Obj.repr(
# 867 "cil/src/frontc/cparser.mly"
                                                   ( _2 )
# 4435 "cil/src/frontc/cparser.ml"
               : Cabs.expression list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    Obj.repr(
# 868 "cil/src/frontc/cparser.mly"
                                                   ( [] )
# 4442 "cil/src/frontc/cparser.ml"
               : Cabs.expression list))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    Obj.repr(
# 872 "cil/src/frontc/cparser.mly"
                                                       ( _2 )
# 4449 "cil/src/frontc/cparser.ml"
               : Cabs.expression list))
; (fun __caml_parser_env ->
    Obj.repr(
# 873 "cil/src/frontc/cparser.mly"
                                                       ( [] )
# 4455 "cil/src/frontc/cparser.ml"
               : Cabs.expression list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : 'block_begin) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : string list) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'block_attrs) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : Cabs.statement list) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 880 "cil/src/frontc/cparser.mly"
                                         (!Lexerhack.pop_context();
                                          { blabels = _2;
                                            battrs = _3;
                                            bstmts = _4 },
					    _1, _5
                                         )
# 4471 "cil/src/frontc/cparser.ml"
               : Cabs.block * cabsloc * cabsloc))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 886 "cil/src/frontc/cparser.mly"
                                         ( { blabels = [];
                                             battrs  = [];
                                             bstmts  = [] },
					     _2, _3
                                         )
# 4483 "cil/src/frontc/cparser.ml"
               : Cabs.block * cabsloc * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 893 "cil/src/frontc/cparser.mly"
                           (!Lexerhack.push_context (); _1)
# 4490 "cil/src/frontc/cparser.ml"
               : 'block_begin))
; (fun __caml_parser_env ->
    Obj.repr(
# 897 "cil/src/frontc/cparser.mly"
                                                            ( [] )
# 4496 "cil/src/frontc/cparser.ml"
               : 'block_attrs))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'paren_attr_list_ne) in
    Obj.repr(
# 899 "cil/src/frontc/cparser.mly"
                                        ( [("__blockattribute__", _2)] )
# 4503 "cil/src/frontc/cparser.ml"
               : 'block_attrs))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'annot_list_opt) in
    Obj.repr(
# 904 "cil/src/frontc/cparser.mly"
                                                        ( _1 )
# 4510 "cil/src/frontc/cparser.ml"
               : Cabs.statement list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'annot_list_opt) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.definition) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.statement list) in
    Obj.repr(
# 906 "cil/src/frontc/cparser.mly"
        ( _1 @ no_ghost_stmt (DEFINITION(_2)) :: _3 )
# 4519 "cil/src/frontc/cparser.ml"
               : Cabs.statement list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'annot_list_opt) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'statement) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.statement list) in
    Obj.repr(
# 908 "cil/src/frontc/cparser.mly"
            ( _1 @ _2 @ _3 )
# 4528 "cil/src/frontc/cparser.ml"
               : Cabs.statement list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'annot_list_opt) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'pragma) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.statement list) in
    Obj.repr(
# 909 "cil/src/frontc/cparser.mly"
                                                        ( _1 @ _3 )
# 4537 "cil/src/frontc/cparser.ml"
               : Cabs.statement list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'annot_list_opt) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'id_or_typename_as_id) in
    Obj.repr(
# 912 "cil/src/frontc/cparser.mly"
    ( let loc = Parsing.rhs_start_pos 2, Parsing.rhs_end_pos 3 in
      _1 @ no_ghost [LABEL (_2, no_ghost_stmt (NOP loc), loc)] )
# 4546 "cil/src/frontc/cparser.ml"
               : Cabs.statement list))
; (fun __caml_parser_env ->
    Obj.repr(
# 917 "cil/src/frontc/cparser.mly"
                    ( [] )
# 4552 "cil/src/frontc/cparser.ml"
               : 'annot_list_opt))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'annot_list) in
    Obj.repr(
# 918 "cil/src/frontc/cparser.mly"
                    ( _1 )
# 4559 "cil/src/frontc/cparser.ml"
               : 'annot_list_opt))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Logic_ptree.code_annot * Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'annot_list_opt) in
    Obj.repr(
# 922 "cil/src/frontc/cparser.mly"
                             ( no_ghost [Cabs.CODE_ANNOT _1] @ _2)
# 4567 "cil/src/frontc/cparser.ml"
               : 'annot_list))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.statement list) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'annot_list_opt) in
    Obj.repr(
# 924 "cil/src/frontc/cparser.mly"
       ( (in_ghost _2) @ _4 )
# 4575 "cil/src/frontc/cparser.ml"
               : 'annot_list))
; (fun __caml_parser_env ->
    Obj.repr(
# 928 "cil/src/frontc/cparser.mly"
                                                     ( [] )
# 4581 "cil/src/frontc/cparser.ml"
               : string list))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : string list) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : string list) in
    Obj.repr(
# 929 "cil/src/frontc/cparser.mly"
                                                     ( _2 @ _4 )
# 4590 "cil/src/frontc/cparser.ml"
               : string list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename_as_id) in
    Obj.repr(
# 932 "cil/src/frontc/cparser.mly"
                                                     ( [ _1 ] )
# 4597 "cil/src/frontc/cparser.ml"
               : string list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'id_or_typename_as_id) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string list) in
    Obj.repr(
# 933 "cil/src/frontc/cparser.mly"
                                                     ( _1 :: _3 )
# 4605 "cil/src/frontc/cparser.ml"
               : string list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'statement) in
    Obj.repr(
# 938 "cil/src/frontc/cparser.mly"
              ( _1 )
# 4612 "cil/src/frontc/cparser.ml"
               : 'annotated_statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'annot_list) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'statement) in
    Obj.repr(
# 939 "cil/src/frontc/cparser.mly"
                         ( _1 @ _2 )
# 4620 "cil/src/frontc/cparser.ml"
               : 'annotated_statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 943 "cil/src/frontc/cparser.mly"
               ( no_ghost [NOP _1] )
# 4627 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc * string list * Logic_ptree.spec) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'annotated_statement) in
    Obj.repr(
# 945 "cil/src/frontc/cparser.mly"
      ( (* TODO: Do not forget behavior list. *)
        let loc,_bhv,spec = _1 in no_ghost [Cabs.CODE_SPEC (spec,loc)] @ _2)
# 4636 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Lexing.position * string) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'annotated_statement) in
    Obj.repr(
# 948 "cil/src/frontc/cparser.mly"
      (
        let bs = _2 in
        Cabshelper.continue_annot
	  (currentLoc())
	  (fun () ->
             let spec = Logic_lexer.spec _1 in
             let spec = no_ghost [Cabs.CODE_SPEC spec] in
             spec @ _2)
	  (fun () -> bs)
	  "Skipping annotation"
      )
# 4654 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 960 "cil/src/frontc/cparser.mly"
   ( let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
            no_ghost [COMPUTATION (smooth_expression _1,loc)])
# 4663 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.block * cabsloc * cabsloc) in
    Obj.repr(
# 962 "cil/src/frontc/cparser.mly"
          ( let (x,y,z) = _1 in no_ghost [BLOCK (x, y, z)])
# 4670 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'annotated_statement) in
    Obj.repr(
# 964 "cil/src/frontc/cparser.mly"
        ( let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
          no_ghost [IF (smooth_expression _2,
                       in_block _3, no_ghost_stmt (NOP loc), loc)])
# 4681 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : Cabs.expression list) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'annotated_statement) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : 'annotated_statement) in
    Obj.repr(
# 968 "cil/src/frontc/cparser.mly"
 ( let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
          no_ghost
            [IF (smooth_expression _2, in_block _3, in_block _5, loc)])
# 4693 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'annotated_statement) in
    Obj.repr(
# 972 "cil/src/frontc/cparser.mly"
        ( let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
          no_ghost [SWITCH (smooth_expression _2, in_block _3, loc)])
# 4703 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'opt_loop_annotations) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'annotated_statement) in
    Obj.repr(
# 976 "cil/src/frontc/cparser.mly"
    ( let loc = Parsing.rhs_start_pos 2, Parsing.symbol_end_pos () in
      no_ghost [WHILE (_1, smooth_expression _3, in_block _4, loc)] )
# 4714 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : 'opt_loop_annotations) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : 'annotated_statement) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 980 "cil/src/frontc/cparser.mly"
    ( let loc = Parsing.rhs_start_pos 2, Parsing.symbol_end_pos () in
      no_ghost [DOWHILE (_1, smooth_expression _5, in_block _3, loc)])
# 4727 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 8 : 'opt_loop_annotations) in
    let _2 = (Parsing.peek_val __caml_parser_env 7 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 6 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 5 : 'for_clause) in
    let _5 = (Parsing.peek_val __caml_parser_env 4 : Cabs.expression) in
    let _6 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _7 = (Parsing.peek_val __caml_parser_env 2 : Cabs.expression) in
    let _9 = (Parsing.peek_val __caml_parser_env 0 : 'annotated_statement) in
    Obj.repr(
# 985 "cil/src/frontc/cparser.mly"
    ( let loc = Parsing.rhs_start_pos 2, Parsing.symbol_end_pos () in
      no_ghost [FOR (_1, _4, _5, _7, in_block _9, loc)])
# 4742 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'id_or_typename_as_id) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'attribute_nocv_list) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'annotated_statement) in
    Obj.repr(
# 988 "cil/src/frontc/cparser.mly"
 ((* The only attribute that should appear here
            is "unused". For now, we drop this on the
            floor, since unused labels are usually
            removed anyways by Rmtmps. *)
          let loc = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 2 in
          no_ghost [LABEL (_1, in_block _4, loc)])
# 4756 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.expression) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'annotated_statement) in
    Obj.repr(
# 995 "cil/src/frontc/cparser.mly"
     ( let loc = Parsing.symbol_start_pos (), Parsing.rhs_end_pos 3 in
              no_ghost [CASE (_2, in_block _4, loc)])
# 4766 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : Cabs.expression) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : Cabs.expression) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : 'annotated_statement) in
    Obj.repr(
# 998 "cil/src/frontc/cparser.mly"
     ( let loc = Parsing.symbol_start_pos (), Parsing.rhs_end_pos 5 in
              no_ghost [CASERANGE (_2, _4, in_block _6, loc)])
# 4777 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    Obj.repr(
# 1001 "cil/src/frontc/cparser.mly"
     ( let loc = Parsing.symbol_start_pos(), Parsing.symbol_end_pos () in
              no_ghost [DEFAULT (no_ghost_stmt (NOP loc), loc)])
# 4785 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1003 "cil/src/frontc/cparser.mly"
                     (
      let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
      no_ghost [RETURN ({ expr_loc = loc; expr_node = NOTHING}, loc)]
    )
# 4796 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1007 "cil/src/frontc/cparser.mly"
                                      (
      let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
      no_ghost [RETURN (smooth_expression _2, loc)]
    )
# 4808 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1011 "cil/src/frontc/cparser.mly"
                        (
      let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
      no_ghost [BREAK loc]
    )
# 4819 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1015 "cil/src/frontc/cparser.mly"
                        (
      let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
      no_ghost [CONTINUE loc]
    )
# 4830 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'id_or_typename_as_id) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1019 "cil/src/frontc/cparser.mly"
                                        (
      let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
      no_ghost [GOTO (_2, loc)]
    )
# 4842 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1023 "cil/src/frontc/cparser.mly"
                                         (
      let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
      no_ghost [COMPGOTO (smooth_expression _3, loc) ]
    )
# 4855 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 6 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 5 : Cabs.attribute list) in
    let _3 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 3 : 'asmtemplate) in
    let _5 = (Parsing.peek_val __caml_parser_env 2 : 'asmoutputs) in
    let _7 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1027 "cil/src/frontc/cparser.mly"
                                                               (
      let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
      no_ghost [ASM (_2, _4, _5, loc)]
    )
# 4870 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 1031 "cil/src/frontc/cparser.mly"
            ( no_ghost [ASM ([], [fst _1], None, snd _1)])
# 4877 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : Cabs.block * cabsloc * cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression list) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Cabs.block * cabsloc * cabsloc) in
    Obj.repr(
# 1032 "cil/src/frontc/cparser.mly"
                                                  (
      let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
      let loc_e = Parsing.rhs_start_pos 4, Parsing.rhs_end_pos 4 in
      let b, _, _ = _2 in
      let h, _, _ = _5 in
      if not !Cprint.msvcMode then parse_error "try/except in GCC code";
      no_ghost
        [TRY_EXCEPT (b, {expr_loc = loc_e; expr_node = COMMA _4}, h, loc)]
    )
# 4896 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.block * cabsloc * cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Cabs.block * cabsloc * cabsloc) in
    Obj.repr(
# 1041 "cil/src/frontc/cparser.mly"
                            (
      let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos () in
      let b, _, _ = _2 in
      let h, _, _ = _4 in
      if not !Cprint.msvcMode then parse_error "try/finally in GCC code";
      no_ghost [TRY_FINALLY (b, h, loc)]
    )
# 4912 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1048 "cil/src/frontc/cparser.mly"
                                 ( no_ghost [NOP _2])
# 4920 "cil/src/frontc/cparser.ml"
               : 'statement))
; (fun __caml_parser_env ->
    Obj.repr(
# 1052 "cil/src/frontc/cparser.mly"
                ( [] )
# 4926 "cil/src/frontc/cparser.ml"
               : 'opt_loop_annotations))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'loop_annotations) in
    Obj.repr(
# 1053 "cil/src/frontc/cparser.mly"
                   ( _1 )
# 4933 "cil/src/frontc/cparser.ml"
               : 'opt_loop_annotations))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'loop_annotation) in
    Obj.repr(
# 1057 "cil/src/frontc/cparser.mly"
                  ( _1 )
# 4940 "cil/src/frontc/cparser.ml"
               : 'loop_annotations))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Logic_ptree.code_annot list * Cabs.cabsloc) in
    Obj.repr(
# 1069 "cil/src/frontc/cparser.mly"
               ( fst _1 )
# 4947 "cil/src/frontc/cparser.ml"
               : 'loop_annotation))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1073 "cil/src/frontc/cparser.mly"
                                 ( FC_EXP _1 )
# 4955 "cil/src/frontc/cparser.ml"
               : 'for_clause))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.definition) in
    Obj.repr(
# 1074 "cil/src/frontc/cparser.mly"
                                 ( FC_DECL _1 )
# 4962 "cil/src/frontc/cparser.ml"
               : 'for_clause))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.init_name list) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1079 "cil/src/frontc/cparser.mly"
      ( doDeclaration None ((snd _1)) (fst _1) _2 )
# 4971 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1081 "cil/src/frontc/cparser.mly"
      ( doDeclaration None ((snd _1)) (fst _1) [] )
# 4979 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Lexing.position * string) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : spec_elem list * cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.init_name list) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1083 "cil/src/frontc/cparser.mly"
          ( doDeclaration (Some _1) ((snd _2)) (fst _2) _3 )
# 4989 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Lexing.position * string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : spec_elem list * cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1085 "cil/src/frontc/cparser.mly"
      ( doDeclaration (Some _1) ((snd _2)) (fst _2) [] )
# 4998 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.init_name) in
    Obj.repr(
# 1088 "cil/src/frontc/cparser.mly"
                                                 ( [_1] )
# 5005 "cil/src/frontc/cparser.ml"
               : Cabs.init_name list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.init_name) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.init_name list) in
    Obj.repr(
# 1089 "cil/src/frontc/cparser.mly"
                                                 ( _1 :: _3 )
# 5013 "cil/src/frontc/cparser.ml"
               : Cabs.init_name list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.name) in
    Obj.repr(
# 1093 "cil/src/frontc/cparser.mly"
                                        ( (_1, NO_INIT) )
# 5020 "cil/src/frontc/cparser.ml"
               : Cabs.init_name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.name) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.init_expression) in
    Obj.repr(
# 1095 "cil/src/frontc/cparser.mly"
                                        ( (_1, _3) )
# 5028 "cil/src/frontc/cparser.ml"
               : Cabs.init_name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'decl_spec_list_opt) in
    Obj.repr(
# 1100 "cil/src/frontc/cparser.mly"
                                        ( SpecTypedef :: _2, _1  )
# 5036 "cil/src/frontc/cparser.ml"
               : spec_elem list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'decl_spec_list_opt) in
    Obj.repr(
# 1101 "cil/src/frontc/cparser.mly"
                                        ( SpecStorage EXTERN :: _2, _1 )
# 5044 "cil/src/frontc/cparser.ml"
               : spec_elem list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'decl_spec_list_opt) in
    Obj.repr(
# 1102 "cil/src/frontc/cparser.mly"
                                        ( SpecStorage STATIC :: _2, _1 )
# 5052 "cil/src/frontc/cparser.ml"
               : spec_elem list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'decl_spec_list_opt) in
    Obj.repr(
# 1103 "cil/src/frontc/cparser.mly"
                                        ( SpecStorage AUTO :: _2, _1 )
# 5060 "cil/src/frontc/cparser.ml"
               : spec_elem list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'decl_spec_list_opt) in
    Obj.repr(
# 1104 "cil/src/frontc/cparser.mly"
                                        ( SpecStorage REGISTER :: _2, _1)
# 5068 "cil/src/frontc/cparser.ml"
               : spec_elem list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : typeSpecifier * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'decl_spec_list_opt_no_named) in
    Obj.repr(
# 1106 "cil/src/frontc/cparser.mly"
                                          ( SpecType (fst _1) :: _2, snd _1 )
# 5076 "cil/src/frontc/cparser.ml"
               : spec_elem list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'decl_spec_list_opt) in
    Obj.repr(
# 1108 "cil/src/frontc/cparser.mly"
                                        ( SpecInline :: _2, _1 )
# 5084 "cil/src/frontc/cparser.ml"
               : spec_elem list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.spec_elem * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'decl_spec_list_opt) in
    Obj.repr(
# 1109 "cil/src/frontc/cparser.mly"
                                        ( (fst _1) :: _2, snd _1 )
# 5092 "cil/src/frontc/cparser.ml"
               : spec_elem list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'attribute_nocv) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'decl_spec_list_opt) in
    Obj.repr(
# 1110 "cil/src/frontc/cparser.mly"
                                        ( SpecAttr (fst _1) :: _2, snd _1 )
# 5100 "cil/src/frontc/cparser.ml"
               : spec_elem list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : string) in
    Obj.repr(
# 1112 "cil/src/frontc/cparser.mly"
                                        ( [ SpecPattern _3 ], _1 )
# 5109 "cil/src/frontc/cparser.ml"
               : spec_elem list * cabsloc))
; (fun __caml_parser_env ->
    Obj.repr(
# 1117 "cil/src/frontc/cparser.mly"
                                        ( [] )
# 5115 "cil/src/frontc/cparser.ml"
               : 'decl_spec_list_opt))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : spec_elem list * cabsloc) in
    Obj.repr(
# 1118 "cil/src/frontc/cparser.mly"
                                        ( fst _1 )
# 5122 "cil/src/frontc/cparser.ml"
               : 'decl_spec_list_opt))
; (fun __caml_parser_env ->
    Obj.repr(
# 1125 "cil/src/frontc/cparser.mly"
                                        ( [] )
# 5128 "cil/src/frontc/cparser.ml"
               : 'decl_spec_list_opt_no_named))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : spec_elem list * cabsloc) in
    Obj.repr(
# 1126 "cil/src/frontc/cparser.mly"
                                        ( fst _1 )
# 5135 "cil/src/frontc/cparser.ml"
               : 'decl_spec_list_opt_no_named))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1129 "cil/src/frontc/cparser.mly"
                    ( Tvoid, _1)
# 5142 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1130 "cil/src/frontc/cparser.mly"
                    ( Tchar, _1 )
# 5149 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1131 "cil/src/frontc/cparser.mly"
                    ( Tbool, _1 )
# 5156 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1132 "cil/src/frontc/cparser.mly"
                    ( Tshort, _1 )
# 5163 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1133 "cil/src/frontc/cparser.mly"
                    ( Tint, _1 )
# 5170 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1134 "cil/src/frontc/cparser.mly"
                    ( Tlong, _1 )
# 5177 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1135 "cil/src/frontc/cparser.mly"
                    ( Tint64, _1 )
# 5184 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1136 "cil/src/frontc/cparser.mly"
                    ( Tfloat, _1 )
# 5191 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1137 "cil/src/frontc/cparser.mly"
                    ( Tdouble, _1 )
# 5198 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1138 "cil/src/frontc/cparser.mly"
                    ( Tsigned, _1 )
# 5205 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1139 "cil/src/frontc/cparser.mly"
                    ( Tunsigned, _1 )
# 5212 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename) in
    Obj.repr(
# 1141 "cil/src/frontc/cparser.mly"
                                                   ( Tstruct (_2, None,    []), _1 )
# 5220 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'just_attributes) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename) in
    Obj.repr(
# 1143 "cil/src/frontc/cparser.mly"
                                                   ( Tstruct (_3, None,    _2), _1 )
# 5229 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'id_or_typename) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : Cabs.field_group list) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1145 "cil/src/frontc/cparser.mly"
                                                   ( Tstruct (_2, Some _4, []), _1 )
# 5240 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.field_group list) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1147 "cil/src/frontc/cparser.mly"
                                                   ( Tstruct ("", Some _3, []), _1 )
# 5250 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : 'just_attributes) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : 'id_or_typename) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : Cabs.field_group list) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1149 "cil/src/frontc/cparser.mly"
                                                   ( Tstruct (_3, Some _5, _2), _1 )
# 5262 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'just_attributes) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : Cabs.field_group list) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1151 "cil/src/frontc/cparser.mly"
                                                   ( Tstruct ("", Some _4, _2), _1 )
# 5273 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename) in
    Obj.repr(
# 1153 "cil/src/frontc/cparser.mly"
                                                   ( Tunion  (_2, None,    []), _1 )
# 5281 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'id_or_typename) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : Cabs.field_group list) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1155 "cil/src/frontc/cparser.mly"
                                                   ( Tunion  (_2, Some _4, []), _1 )
# 5292 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.field_group list) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1157 "cil/src/frontc/cparser.mly"
                                                   ( Tunion  ("", Some _3, []), _1 )
# 5302 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : 'just_attributes) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : 'id_or_typename) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : Cabs.field_group list) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1159 "cil/src/frontc/cparser.mly"
                                                   ( Tunion  (_3, Some _5, _2), _1 )
# 5314 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'just_attributes) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : Cabs.field_group list) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1161 "cil/src/frontc/cparser.mly"
                                                   ( Tunion  ("", Some _4, _2), _1 )
# 5325 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename) in
    Obj.repr(
# 1163 "cil/src/frontc/cparser.mly"
                                                   ( Tenum   (_2, None,    []), _1 )
# 5333 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : 'id_or_typename) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : Cabs.enum_item list) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'maybecomma) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1165 "cil/src/frontc/cparser.mly"
                                                   ( Tenum   (_2, Some _4, []), _1 )
# 5345 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : Cabs.enum_item list) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'maybecomma) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1167 "cil/src/frontc/cparser.mly"
                                                   ( Tenum   ("", Some _3, []), _1 )
# 5356 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 6 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 5 : 'just_attributes) in
    let _3 = (Parsing.peek_val __caml_parser_env 4 : 'id_or_typename) in
    let _4 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _5 = (Parsing.peek_val __caml_parser_env 2 : Cabs.enum_item list) in
    let _6 = (Parsing.peek_val __caml_parser_env 1 : 'maybecomma) in
    let _7 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1169 "cil/src/frontc/cparser.mly"
                                                   ( Tenum   (_3, Some _5, _2), _1 )
# 5369 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 5 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 4 : 'just_attributes) in
    let _3 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : Cabs.enum_item list) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'maybecomma) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1171 "cil/src/frontc/cparser.mly"
                                                   ( Tenum   ("", Some _4, _2), _1 )
# 5381 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 1172 "cil/src/frontc/cparser.mly"
                    (
      (Tnamed _1, (Parsing.symbol_start_pos (), Parsing.symbol_end_pos()))
      )
# 5390 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression) in
    Obj.repr(
# 1175 "cil/src/frontc/cparser.mly"
                                        ( TtypeofE _3, _1 )
# 5399 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.spec_elem list * Cabs.decl_type) in
    Obj.repr(
# 1176 "cil/src/frontc/cparser.mly"
                                        ( let s, d = _3 in
                                          TtypeofT (s, d), _1 )
# 5409 "cil/src/frontc/cparser.ml"
               : typeSpecifier * cabsloc))
; (fun __caml_parser_env ->
    Obj.repr(
# 1182 "cil/src/frontc/cparser.mly"
                                         ( [] )
# 5415 "cil/src/frontc/cparser.ml"
               : Cabs.field_group list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.field_group list) in
    Obj.repr(
# 1184 "cil/src/frontc/cparser.mly"
                                         ( FIELD (fst _1,
                                            [(missingFieldDecl, None)]) :: _3 )
# 5425 "cil/src/frontc/cparser.ml"
               : Cabs.field_group list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.field_group list) in
    Obj.repr(
# 1188 "cil/src/frontc/cparser.mly"
                                         ( _2 )
# 5433 "cil/src/frontc/cparser.ml"
               : Cabs.field_group list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : (Cabs.name * expression option) list) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Cabs.field_group list) in
    Obj.repr(
# 1190 "cil/src/frontc/cparser.mly"
                                          ( FIELD (fst _1, _2)
                                            :: _4 )
# 5444 "cil/src/frontc/cparser.ml"
               : Cabs.field_group list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'pragma) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.field_group list) in
    Obj.repr(
# 1193 "cil/src/frontc/cparser.mly"
                                          ( _2 )
# 5452 "cil/src/frontc/cparser.ml"
               : Cabs.field_group list))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.field_group list) in
    Obj.repr(
# 1196 "cil/src/frontc/cparser.mly"
                                          ( _3 )
# 5460 "cil/src/frontc/cparser.ml"
               : Cabs.field_group list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.name * expression option) in
    Obj.repr(
# 1199 "cil/src/frontc/cparser.mly"
                                         ( [_1] )
# 5467 "cil/src/frontc/cparser.ml"
               : (Cabs.name * expression option) list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.name * expression option) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : (Cabs.name * expression option) list) in
    Obj.repr(
# 1200 "cil/src/frontc/cparser.mly"
                                         ( _1 :: _3 )
# 5475 "cil/src/frontc/cparser.ml"
               : (Cabs.name * expression option) list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.name) in
    Obj.repr(
# 1203 "cil/src/frontc/cparser.mly"
                                    ( (_1, None) )
# 5482 "cil/src/frontc/cparser.ml"
               : Cabs.name * expression option))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.name) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Cabs.attribute list) in
    Obj.repr(
# 1205 "cil/src/frontc/cparser.mly"
                                    ( let (n,decl,al,loc) = _1 in
                                      let al' = al @ _4 in
                                     ((n,decl,al',loc), Some _3) )
# 5493 "cil/src/frontc/cparser.ml"
               : Cabs.name * expression option))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression) in
    Obj.repr(
# 1208 "cil/src/frontc/cparser.mly"
                                    ( (missingFieldDecl, Some _2) )
# 5500 "cil/src/frontc/cparser.ml"
               : Cabs.name * expression option))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.enum_item) in
    Obj.repr(
# 1212 "cil/src/frontc/cparser.mly"
                  ([_1])
# 5507 "cil/src/frontc/cparser.ml"
               : Cabs.enum_item list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.enum_item list) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.enum_item) in
    Obj.repr(
# 1213 "cil/src/frontc/cparser.mly"
                                       (_1 @ [_3])
# 5515 "cil/src/frontc/cparser.ml"
               : Cabs.enum_item list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.enum_item list) in
    Obj.repr(
# 1214 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 5522 "cil/src/frontc/cparser.ml"
               : Cabs.enum_item list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 1217 "cil/src/frontc/cparser.mly"
            (
      let loc = Parsing.symbol_start_pos (), Parsing.symbol_end_pos() in
      (_1, { expr_node = NOTHING; expr_loc = loc }, loc)
    )
# 5532 "cil/src/frontc/cparser.ml"
               : Cabs.enum_item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression) in
    Obj.repr(
# 1221 "cil/src/frontc/cparser.mly"
                         (
      (_1, _3, (Parsing.symbol_start_pos (),Parsing.symbol_end_pos()))
    )
# 5542 "cil/src/frontc/cparser.ml"
               : Cabs.enum_item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : attribute list list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string * Cabs.decl_type) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.attribute list) in
    Obj.repr(
# 1228 "cil/src/frontc/cparser.mly"
                                               (
     let (n, decl) = _2 in (n, applyPointer (fst _1) decl, _3, (snd _1))
   )
# 5553 "cil/src/frontc/cparser.ml"
               : Cabs.name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.attribute list) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'comma_expression_opt) in
    Obj.repr(
# 1235 "cil/src/frontc/cparser.mly"
                                  ( _1,_2 )
# 5561 "cil/src/frontc/cparser.ml"
               : 'attributes_or_static))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'attribute) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.attribute list) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression list) in
    Obj.repr(
# 1236 "cil/src/frontc/cparser.mly"
                                               (
    fst _1::_2  @ ["static",[]], smooth_expression _4
  )
# 5573 "cil/src/frontc/cparser.ml"
               : 'attributes_or_static))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.attribute list) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.expression list) in
    Obj.repr(
# 1239 "cil/src/frontc/cparser.mly"
                                     (
    ("static",[]) :: _2, smooth_expression _3
  )
# 5584 "cil/src/frontc/cparser.ml"
               : 'attributes_or_static))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename) in
    Obj.repr(
# 1247 "cil/src/frontc/cparser.mly"
                                   ( (_1, JUSTBASE) )
# 5591 "cil/src/frontc/cparser.ml"
               : string * Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.attribute list) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.name) in
    Obj.repr(
# 1250 "cil/src/frontc/cparser.mly"
                                   ( let (n,decl,al,_) = _3 in
                                     (n, PARENTYPE(_2,decl,al)) )
# 5601 "cil/src/frontc/cparser.ml"
               : string * Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : string * Cabs.decl_type) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'attributes_or_static) in
    Obj.repr(
# 1254 "cil/src/frontc/cparser.mly"
                                   ( let (n, decl) = _1 in
                                     let (attrs, size) = _3 in
                                     (n, ARRAY(decl, attrs, size)) )
# 5611 "cil/src/frontc/cparser.ml"
               : string * Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : string * Cabs.decl_type) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'parameter_list_startscope) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'rest_par_list) in
    Obj.repr(
# 1258 "cil/src/frontc/cparser.mly"
                                   ( let (n, decl) = _1 in
                                     let (params, isva) = _3 in
                                     !Lexerhack.pop_context ();
                                     (n, PROTO(decl, params, isva))
                                   )
# 5624 "cil/src/frontc/cparser.ml"
               : string * Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1265 "cil/src/frontc/cparser.mly"
                                   ( !Lexerhack.push_context () )
# 5631 "cil/src/frontc/cparser.ml"
               : 'parameter_list_startscope))
; (fun __caml_parser_env ->
    Obj.repr(
# 1268 "cil/src/frontc/cparser.mly"
                                   ( ([], false) )
# 5637 "cil/src/frontc/cparser.ml"
               : 'rest_par_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.single_name) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'rest_par_list1) in
    Obj.repr(
# 1269 "cil/src/frontc/cparser.mly"
                                   ( let (params, isva) = _2 in
                                     (_1 :: params, isva)
                                   )
# 5647 "cil/src/frontc/cparser.ml"
               : 'rest_par_list))
; (fun __caml_parser_env ->
    Obj.repr(
# 1274 "cil/src/frontc/cparser.mly"
                                        ( ([], false) )
# 5653 "cil/src/frontc/cparser.ml"
               : 'rest_par_list1))
; (fun __caml_parser_env ->
    Obj.repr(
# 1275 "cil/src/frontc/cparser.mly"
                                        ( ([], true) )
# 5659 "cil/src/frontc/cparser.ml"
               : 'rest_par_list1))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.single_name) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'rest_par_list1) in
    Obj.repr(
# 1276 "cil/src/frontc/cparser.mly"
                                        ( let (params, isva) = _3 in
                                          (_2 :: params, isva)
                                        )
# 5669 "cil/src/frontc/cparser.ml"
               : 'rest_par_list1))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.name) in
    Obj.repr(
# 1283 "cil/src/frontc/cparser.mly"
                                          ( (fst _1, _2) )
# 5677 "cil/src/frontc/cparser.ml"
               : Cabs.single_name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.decl_type * Cabs.attribute list) in
    Obj.repr(
# 1284 "cil/src/frontc/cparser.mly"
                                          ( let d, a = _2 in
                                            (fst _1, ("", d, a, (*CEA*) cabslu)) )
# 5686 "cil/src/frontc/cparser.ml"
               : Cabs.single_name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : spec_elem list * cabsloc) in
    Obj.repr(
# 1286 "cil/src/frontc/cparser.mly"
                                          ( (fst _1, ("", JUSTBASE, [], (*CEA*) cabslu)) )
# 5693 "cil/src/frontc/cparser.ml"
               : Cabs.single_name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.single_name) in
    Obj.repr(
# 1287 "cil/src/frontc/cparser.mly"
                                          ( _2 )
# 5701 "cil/src/frontc/cparser.ml"
               : Cabs.single_name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : attribute list list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'direct_old_proto_decl) in
    Obj.repr(
# 1292 "cil/src/frontc/cparser.mly"
                                      ( let (n, decl, a) = _2 in
					  (n, applyPointer (fst _1) decl,
                                           a, snd _1)
                                      )
# 5712 "cil/src/frontc/cparser.ml"
               : Cabs.name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : string * Cabs.decl_type) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : string list) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : 'old_pardef_list) in
    Obj.repr(
# 1300 "cil/src/frontc/cparser.mly"
                                                                  (
    let par_decl, isva = doOldParDecl _3 _5 in
    let n, decl = _1 in
    (n, PROTO(decl, par_decl, isva), [])
  )
# 5726 "cil/src/frontc/cparser.ml"
               : 'direct_old_proto_decl))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string * Cabs.decl_type) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    Obj.repr(
# 1305 "cil/src/frontc/cparser.mly"
                            (
    let n, decl = _1 in (n, PROTO(decl, [], false), [])
  )
# 5736 "cil/src/frontc/cparser.ml"
               : 'direct_old_proto_decl))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 1319 "cil/src/frontc/cparser.mly"
                                               ( [_1] )
# 5743 "cil/src/frontc/cparser.ml"
               : string list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string list) in
    Obj.repr(
# 1320 "cil/src/frontc/cparser.mly"
                                               ( _1::_3 )
# 5751 "cil/src/frontc/cparser.ml"
               : string list))
; (fun __caml_parser_env ->
    Obj.repr(
# 1324 "cil/src/frontc/cparser.mly"
                                          ( ([], false) )
# 5757 "cil/src/frontc/cparser.ml"
               : 'old_pardef_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'old_pardef) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    Obj.repr(
# 1326 "cil/src/frontc/cparser.mly"
                                          ( ([(fst _1, _2)], true) )
# 5766 "cil/src/frontc/cparser.ml"
               : 'old_pardef_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'old_pardef) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'old_pardef_list) in
    Obj.repr(
# 1328 "cil/src/frontc/cparser.mly"
                                          ( let rest, isva = _4 in
                                            ((fst _1, _2) :: rest, isva)
                                          )
# 5778 "cil/src/frontc/cparser.ml"
               : 'old_pardef_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.name) in
    Obj.repr(
# 1334 "cil/src/frontc/cparser.mly"
                                          ( [_1] )
# 5785 "cil/src/frontc/cparser.ml"
               : 'old_pardef))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.name) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'old_pardef) in
    Obj.repr(
# 1335 "cil/src/frontc/cparser.mly"
                                          ( _1 :: _3 )
# 5793 "cil/src/frontc/cparser.ml"
               : 'old_pardef))
; (fun __caml_parser_env ->
    Obj.repr(
# 1336 "cil/src/frontc/cparser.mly"
                                          ( [] )
# 5799 "cil/src/frontc/cparser.ml"
               : 'old_pardef))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.attribute list) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : attribute list list * cabsloc) in
    Obj.repr(
# 1341 "cil/src/frontc/cparser.mly"
                                ( _2 :: fst _3, _1 )
# 5808 "cil/src/frontc/cparser.ml"
               : attribute list list * cabsloc))
; (fun __caml_parser_env ->
    Obj.repr(
# 1344 "cil/src/frontc/cparser.mly"
                                 ( let l = currentLoc () in
                                   ([], l) )
# 5815 "cil/src/frontc/cparser.ml"
               : attribute list list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : attribute list list * cabsloc) in
    Obj.repr(
# 1346 "cil/src/frontc/cparser.mly"
                                 ( _1 )
# 5822 "cil/src/frontc/cparser.ml"
               : attribute list list * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.decl_type * Cabs.attribute list) in
    Obj.repr(
# 1350 "cil/src/frontc/cparser.mly"
                               ( let d, a = _2 in
                                 if a <> [] then parse_error "attributes in type name" ;
                                 (fst _1, d)
                               )
# 5833 "cil/src/frontc/cparser.ml"
               : Cabs.spec_elem list * Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : spec_elem list * cabsloc) in
    Obj.repr(
# 1354 "cil/src/frontc/cparser.mly"
                               ( (fst _1, JUSTBASE) )
# 5840 "cil/src/frontc/cparser.ml"
               : Cabs.spec_elem list * Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : attribute list list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.decl_type) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.attribute list) in
    Obj.repr(
# 1357 "cil/src/frontc/cparser.mly"
                                          ( applyPointer (fst _1) _2, _3 )
# 5849 "cil/src/frontc/cparser.ml"
               : Cabs.decl_type * Cabs.attribute list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : attribute list list * cabsloc) in
    Obj.repr(
# 1358 "cil/src/frontc/cparser.mly"
                                          ( applyPointer (fst _1) JUSTBASE, [] )
# 5856 "cil/src/frontc/cparser.ml"
               : Cabs.decl_type * Cabs.attribute list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.attribute list) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.decl_type * Cabs.attribute list) in
    Obj.repr(
# 1365 "cil/src/frontc/cparser.mly"
                                   ( let d, a = _3 in
                                     PARENTYPE (_2, d, a)
                                   )
# 5867 "cil/src/frontc/cparser.ml"
               : Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    Obj.repr(
# 1370 "cil/src/frontc/cparser.mly"
                                   ( JUSTBASE )
# 5874 "cil/src/frontc/cparser.ml"
               : Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.decl_type) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'comma_expression_opt) in
    Obj.repr(
# 1373 "cil/src/frontc/cparser.mly"
                                   ( ARRAY(_1, [], _3) )
# 5882 "cil/src/frontc/cparser.ml"
               : Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.decl_type) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'parameter_list_startscope) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'rest_par_list) in
    Obj.repr(
# 1376 "cil/src/frontc/cparser.mly"
                                   ( let (params, isva) = _3 in
                                     !Lexerhack.pop_context ();
                                     PROTO (_1, params, isva)
                                   )
# 5894 "cil/src/frontc/cparser.ml"
               : Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.decl_type) in
    Obj.repr(
# 1382 "cil/src/frontc/cparser.mly"
                                    ( _1 )
# 5901 "cil/src/frontc/cparser.ml"
               : Cabs.decl_type))
; (fun __caml_parser_env ->
    Obj.repr(
# 1383 "cil/src/frontc/cparser.mly"
                                    ( JUSTBASE )
# 5907 "cil/src/frontc/cparser.ml"
               : Cabs.decl_type))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Lexing.position * string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : cabsloc * spec_elem list * name) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : Cabs.block * cabsloc * cabsloc) in
    Obj.repr(
# 1387 "cil/src/frontc/cparser.mly"
          (
            let (loc, specs, decl) = _2 in
            let spec =
              Cabshelper.continue_annot loc
                (fun () -> Some (Logic_lexer.spec _1 ))
                (fun () -> None)
                "Ignoring specification of function %s" !currentFunctionName
            in
            currentFunctionName := "<__FUNCTION__ used outside any functions>";
            !Lexerhack.pop_context (); (* The context pushed by
                                    * announceFunctionName *)
            doFunctionDef spec loc (trd3 _3) specs decl (fst3 _3)
          )
# 5928 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : cabsloc * spec_elem list * name) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.block * cabsloc * cabsloc) in
    Obj.repr(
# 1401 "cil/src/frontc/cparser.mly"
          ( let (loc, specs, decl) = _1 in
            currentFunctionName := "<__FUNCTION__ used outside any functions>";
            !Lexerhack.pop_context (); (* The context pushed by
                                    * announceFunctionName *)
            (*OCAMLYACC BUG??? Format.printf "%a@." d_cabsloc (trd3 $2);*)
            doFunctionDef None ((*handleLoc*) loc) (trd3 _2) specs decl (fst3 _2)
          )
# 5942 "cil/src/frontc/cparser.ml"
               : Cabs.definition))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.name) in
    Obj.repr(
# 1412 "cil/src/frontc/cparser.mly"
                            ( announceFunctionName _2;
                              (fourth4 _2, fst _1, _2)
                            )
# 5952 "cil/src/frontc/cparser.ml"
               : cabsloc * spec_elem list * name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : spec_elem list * cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.name) in
    Obj.repr(
# 1418 "cil/src/frontc/cparser.mly"
                            ( announceFunctionName _2;
                              (snd _1, fst _1, _2)
                            )
# 5962 "cil/src/frontc/cparser.ml"
               : cabsloc * spec_elem list * name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'parameter_list_startscope) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'rest_par_list) in
    Obj.repr(
# 1423 "cil/src/frontc/cparser.mly"
    ( let (params, isva) = _3 in
      let loc = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 1 in
      let fdec =
        (_1, PROTO(JUSTBASE, params, isva), [], loc) in
      announceFunctionName fdec;
      (* Default is int type *)
      let defSpec = [SpecType Tint] in (loc, defSpec, fdec)
    )
# 5978 "cil/src/frontc/cparser.ml"
               : cabsloc * spec_elem list * name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : string list) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : 'old_pardef_list) in
    Obj.repr(
# 1434 "cil/src/frontc/cparser.mly"
    ( (* Convert pardecl to new style *)
      let pardecl, isva = doOldParDecl _3 _5 in
      let loc = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 1 in
      (* Make the function declarator *)
      let fdec = (_1, PROTO(JUSTBASE, pardecl,isva), [], loc) in
      announceFunctionName fdec;
      (* Default is int type *)
      let defSpec = [SpecType Tint] in (loc, defSpec, fdec)
    )
# 5996 "cil/src/frontc/cparser.ml"
               : cabsloc * spec_elem list * name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    Obj.repr(
# 1445 "cil/src/frontc/cparser.mly"
    ( (* Make the function declarator *)
      let loc = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 1 in
      let fdec = (_1, PROTO(JUSTBASE, [], false), [], loc) in
      announceFunctionName fdec;
      (* Default is int type *)
      let defSpec = [SpecType Tint] in (loc, defSpec, fdec)
    )
# 6010 "cil/src/frontc/cparser.ml"
               : cabsloc * spec_elem list * name))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1456 "cil/src/frontc/cparser.mly"
                             ( SpecCV(CV_CONST), _1 )
# 6017 "cil/src/frontc/cparser.ml"
               : Cabs.spec_elem * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1457 "cil/src/frontc/cparser.mly"
                             ( SpecCV(CV_VOLATILE), _1 )
# 6024 "cil/src/frontc/cparser.ml"
               : Cabs.spec_elem * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1458 "cil/src/frontc/cparser.mly"
                             ( SpecCV(CV_RESTRICT), _1 )
# 6031 "cil/src/frontc/cparser.ml"
               : Cabs.spec_elem * cabsloc))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 1459 "cil/src/frontc/cparser.mly"
                             ( let annot, loc = _1 in
			       SpecCV(CV_ATTRIBUTE_ANNOT annot), loc )
# 6039 "cil/src/frontc/cparser.ml"
               : Cabs.spec_elem * cabsloc))
; (fun __caml_parser_env ->
    Obj.repr(
# 1465 "cil/src/frontc/cparser.mly"
                   ( [])
# 6045 "cil/src/frontc/cparser.ml"
               : Cabs.attribute list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'attribute) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.attribute list) in
    Obj.repr(
# 1466 "cil/src/frontc/cparser.mly"
                                 ( fst _1 :: _2 )
# 6053 "cil/src/frontc/cparser.ml"
               : Cabs.attribute list))
; (fun __caml_parser_env ->
    Obj.repr(
# 1472 "cil/src/frontc/cparser.mly"
                                        ( [] )
# 6059 "cil/src/frontc/cparser.ml"
               : Cabs.attribute list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'attribute) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.attribute list) in
    Obj.repr(
# 1473 "cil/src/frontc/cparser.mly"
                                        ( fst _1 :: _2 )
# 6067 "cil/src/frontc/cparser.ml"
               : Cabs.attribute list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : string * cabsloc) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : Cabs.attribute list) in
    Obj.repr(
# 1474 "cil/src/frontc/cparser.mly"
                                                 (
      let loc = Parsing.rhs_start_pos 3, Parsing.rhs_end_pos 3 in
      ("__asm__",
       [{ expr_node = CONSTANT(CONST_STRING (fst _3)); expr_loc = loc}])
      :: _5
    )
# 6082 "cil/src/frontc/cparser.ml"
               : Cabs.attribute list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'paren_attr_list) in
    Obj.repr(
# 1485 "cil/src/frontc/cparser.mly"
                                        ( ("__attribute__", _3), _1 )
# 6091 "cil/src/frontc/cparser.ml"
               : 'attribute_nocv))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'paren_attr_list_ne) in
    Obj.repr(
# 1490 "cil/src/frontc/cparser.mly"
                                        ( ("__declspec", _2), _1 )
# 6099 "cil/src/frontc/cparser.ml"
               : 'attribute_nocv))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 1491 "cil/src/frontc/cparser.mly"
                                        ( (fst _1, []), snd _1 )
# 6106 "cil/src/frontc/cparser.ml"
               : 'attribute_nocv))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1493 "cil/src/frontc/cparser.mly"
                                        ( ("__thread",[]), _1 )
# 6113 "cil/src/frontc/cparser.ml"
               : 'attribute_nocv))
; (fun __caml_parser_env ->
    Obj.repr(
# 1497 "cil/src/frontc/cparser.mly"
                   ( [])
# 6119 "cil/src/frontc/cparser.ml"
               : 'attribute_nocv_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'attribute_nocv) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'attribute_nocv_list) in
    Obj.repr(
# 1498 "cil/src/frontc/cparser.mly"
                                        ( fst _1 :: _2 )
# 6127 "cil/src/frontc/cparser.ml"
               : 'attribute_nocv_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'attribute_nocv) in
    Obj.repr(
# 1503 "cil/src/frontc/cparser.mly"
                          ( _1 )
# 6134 "cil/src/frontc/cparser.ml"
               : 'attribute))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1504 "cil/src/frontc/cparser.mly"
                          ( ("const", []), _1 )
# 6141 "cil/src/frontc/cparser.ml"
               : 'attribute))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1505 "cil/src/frontc/cparser.mly"
                          ( ("restrict",[]), _1 )
# 6148 "cil/src/frontc/cparser.ml"
               : 'attribute))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1506 "cil/src/frontc/cparser.mly"
                          ( ("volatile",[]), _1 )
# 6155 "cil/src/frontc/cparser.ml"
               : 'attribute))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 1507 "cil/src/frontc/cparser.mly"
                          ( let annot, loc = _1 in
			    ("$annot:" ^ annot, []), loc )
# 6163 "cil/src/frontc/cparser.ml"
               : 'attribute))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'paren_attr_list) in
    Obj.repr(
# 1516 "cil/src/frontc/cparser.mly"
                                        ( ("__attribute__", _3) )
# 6172 "cil/src/frontc/cparser.ml"
               : 'just_attribute))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'paren_attr_list_ne) in
    Obj.repr(
# 1517 "cil/src/frontc/cparser.mly"
                                        ( ("__declspec", _2) )
# 6180 "cil/src/frontc/cparser.ml"
               : 'just_attribute))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'just_attribute) in
    Obj.repr(
# 1523 "cil/src/frontc/cparser.mly"
                                        ( [_1] )
# 6187 "cil/src/frontc/cparser.ml"
               : 'just_attributes))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'just_attribute) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'just_attributes) in
    Obj.repr(
# 1524 "cil/src/frontc/cparser.mly"
                                        ( _1 :: _2 )
# 6195 "cil/src/frontc/cparser.ml"
               : 'just_attributes))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    Obj.repr(
# 1529 "cil/src/frontc/cparser.mly"
                             (
    PRAGMA (make_expr (VARIABLE ("")), _1)
  )
# 6204 "cil/src/frontc/cparser.ml"
               : 'pragma))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'attr) in
    Obj.repr(
# 1532 "cil/src/frontc/cparser.mly"
                          ( PRAGMA (_2, _1) )
# 6212 "cil/src/frontc/cparser.ml"
               : 'pragma))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    Obj.repr(
# 1533 "cil/src/frontc/cparser.mly"
                                   ( PRAGMA (_2, _1) )
# 6221 "cil/src/frontc/cparser.ml"
               : 'pragma))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 1534 "cil/src/frontc/cparser.mly"
                                        (
    PRAGMA (make_expr (VARIABLE (fst _1)), snd _1)
  )
# 6230 "cil/src/frontc/cparser.ml"
               : 'pragma))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 1542 "cil/src/frontc/cparser.mly"
          ( make_expr (VARIABLE _1) )
# 6237 "cil/src/frontc/cparser.ml"
               : 'var_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 1543 "cil/src/frontc/cparser.mly"
               ( make_expr (VARIABLE _1) )
# 6244 "cil/src/frontc/cparser.ml"
               : 'var_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 1544 "cil/src/frontc/cparser.mly"
                          ( make_expr (VARIABLE ("default:" ^ fst _3)) )
# 6252 "cil/src/frontc/cparser.ml"
               : 'var_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1546 "cil/src/frontc/cparser.mly"
          ( make_expr (VARIABLE "aconst") )
# 6259 "cil/src/frontc/cparser.ml"
               : 'var_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Cabs.cabsloc) in
    Obj.repr(
# 1548 "cil/src/frontc/cparser.mly"
             ( make_expr (VARIABLE ("__noreturn__")) )
# 6266 "cil/src/frontc/cparser.ml"
               : 'var_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 1549 "cil/src/frontc/cparser.mly"
                        ( make_expr (VARIABLE (_1 ^ ":" ^ fst _3)) )
# 6274 "cil/src/frontc/cparser.ml"
               : 'var_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 1550 "cil/src/frontc/cparser.mly"
                             ( make_expr (VARIABLE (_1 ^ ":" ^ fst _3)) )
# 6282 "cil/src/frontc/cparser.ml"
               : 'var_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string * Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 1553 "cil/src/frontc/cparser.mly"
                          ( make_expr (VARIABLE (fst _1 ^ ":" ^ fst _3)) )
# 6290 "cil/src/frontc/cparser.ml"
               : 'var_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * Cabs.cabsloc) in
    Obj.repr(
# 1557 "cil/src/frontc/cparser.mly"
            ( make_expr (CONSTANT(CONST_INT (fst _1))) )
# 6297 "cil/src/frontc/cparser.ml"
               : 'basic_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'var_attr) in
    Obj.repr(
# 1558 "cil/src/frontc/cparser.mly"
             ( _1 )
# 6304 "cil/src/frontc/cparser.ml"
               : 'basic_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'basic_attr) in
    Obj.repr(
# 1561 "cil/src/frontc/cparser.mly"
                  ( [_1] )
# 6311 "cil/src/frontc/cparser.ml"
               : 'basic_attr_list_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'basic_attr) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'basic_attr_list_ne) in
    Obj.repr(
# 1562 "cil/src/frontc/cparser.mly"
                                  ( _1::_2 )
# 6319 "cil/src/frontc/cparser.ml"
               : 'basic_attr_list_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'basic_attr_list_ne) in
    Obj.repr(
# 1566 "cil/src/frontc/cparser.mly"
                         ( _1 )
# 6326 "cil/src/frontc/cparser.ml"
               : 'parameter_attr_list_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'basic_attr_list_ne) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : string * cabsloc) in
    Obj.repr(
# 1567 "cil/src/frontc/cparser.mly"
                                       (
      _1 @ [make_expr (CONSTANT(CONST_STRING (fst _2)))]
    )
# 6336 "cil/src/frontc/cparser.ml"
               : 'parameter_attr_list_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'basic_attr_list_ne) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string * cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'parameter_attr_list_ne) in
    Obj.repr(
# 1570 "cil/src/frontc/cparser.mly"
                                                              (
      _1 @ ([make_expr (CONSTANT(CONST_STRING (fst _2)))] @ _3)
    )
# 6347 "cil/src/frontc/cparser.ml"
               : 'parameter_attr_list_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'parameter_attr_list_ne) in
    Obj.repr(
# 1575 "cil/src/frontc/cparser.mly"
                           ( _1 )
# 6354 "cil/src/frontc/cparser.ml"
               : 'param_attr_list_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * cabsloc) in
    Obj.repr(
# 1576 "cil/src/frontc/cparser.mly"
                    ( [make_expr (CONSTANT(CONST_STRING (fst _1)))] )
# 6361 "cil/src/frontc/cparser.ml"
               : 'param_attr_list_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'basic_attr) in
    Obj.repr(
# 1579 "cil/src/frontc/cparser.mly"
               ( _1 )
# 6368 "cil/src/frontc/cparser.ml"
               : 'primary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'attr) in
    Obj.repr(
# 1580 "cil/src/frontc/cparser.mly"
                       ( _2 )
# 6376 "cil/src/frontc/cparser.ml"
               : 'primary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string * cabsloc) in
    Obj.repr(
# 1581 "cil/src/frontc/cparser.mly"
                    ( make_expr (CONSTANT(CONST_STRING (fst _1))) )
# 6383 "cil/src/frontc/cparser.ml"
               : 'primary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'primary_attr) in
    Obj.repr(
# 1584 "cil/src/frontc/cparser.mly"
                 ( _1 )
# 6390 "cil/src/frontc/cparser.ml"
               : 'postfix_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'id_or_typename_as_id) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'paren_attr_list_ne) in
    Obj.repr(
# 1585 "cil/src/frontc/cparser.mly"
                                            (
        let loc = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 1 in
        make_expr (CALL({ expr_loc = loc; expr_node = VARIABLE _1}, _2)) )
# 6400 "cil/src/frontc/cparser.ml"
               : 'postfix_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'id_or_typename_as_id) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    Obj.repr(
# 1589 "cil/src/frontc/cparser.mly"
                                        (
      let loc1 = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 1 in
      let loc2 = Parsing.rhs_start_pos 2, Parsing.rhs_end_pos 3 in
      let f = { expr_node = VARIABLE _1; expr_loc = loc1 } in
      let arg = { expr_node = VARIABLE ""; expr_loc = loc2 } in
      make_expr (CALL(f, [arg]))
    )
# 6414 "cil/src/frontc/cparser.ml"
               : 'postfix_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'basic_attr) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'param_attr_list_ne) in
    Obj.repr(
# 1598 "cil/src/frontc/cparser.mly"
                                   (
      let loc = Parsing.rhs_start_pos 1, Parsing.rhs_end_pos 1 in
      make_expr (CALL({ expr_node = VARIABLE ""; expr_loc = loc}, _1::_2)) )
# 6424 "cil/src/frontc/cparser.ml"
               : 'postfix_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'postfix_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename) in
    Obj.repr(
# 1602 "cil/src/frontc/cparser.mly"
                                        ( make_expr (MEMBEROFPTR (_1, _3)))
# 6432 "cil/src/frontc/cparser.ml"
               : 'postfix_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'postfix_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'id_or_typename) in
    Obj.repr(
# 1603 "cil/src/frontc/cparser.mly"
                                        ( make_expr (MEMBEROF (_1, _3)) )
# 6440 "cil/src/frontc/cparser.ml"
               : 'postfix_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : 'postfix_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'attr) in
    Obj.repr(
# 1604 "cil/src/frontc/cparser.mly"
                                        ( make_expr (INDEX (_1, _3)) )
# 6448 "cil/src/frontc/cparser.ml"
               : 'postfix_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'postfix_attr) in
    Obj.repr(
# 1611 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 6455 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'unary_expression) in
    Obj.repr(
# 1612 "cil/src/frontc/cparser.mly"
                                        ( make_expr (EXPR_SIZEOF _2) )
# 6463 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.spec_elem list * Cabs.decl_type) in
    Obj.repr(
# 1614 "cil/src/frontc/cparser.mly"
                          (let b, d = _3 in
                                         make_expr (TYPE_SIZEOF (b, d)) )
# 6473 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'unary_expression) in
    Obj.repr(
# 1617 "cil/src/frontc/cparser.mly"
                                        (make_expr (EXPR_ALIGNOF _2) )
# 6481 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : Cabs.spec_elem list * Cabs.decl_type) in
    Obj.repr(
# 1618 "cil/src/frontc/cparser.mly"
                                        (let b, d = _3 in
                                         make_expr (TYPE_ALIGNOF (b, d)) )
# 6491 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_attr) in
    Obj.repr(
# 1620 "cil/src/frontc/cparser.mly"
                                        (make_expr (UNARY (PLUS, _2)))
# 6499 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_attr) in
    Obj.repr(
# 1621 "cil/src/frontc/cparser.mly"
                                        (make_expr (UNARY (MINUS, _2)) )
# 6507 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_attr) in
    Obj.repr(
# 1622 "cil/src/frontc/cparser.mly"
                            (make_expr (UNARY (MEMOF, _2)) )
# 6515 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_attr) in
    Obj.repr(
# 1624 "cil/src/frontc/cparser.mly"
                                 ( make_expr (UNARY (ADDROF, _2)) )
# 6523 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_attr) in
    Obj.repr(
# 1625 "cil/src/frontc/cparser.mly"
                                 ( make_expr (UNARY (NOT, _2)) )
# 6531 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'cast_attr) in
    Obj.repr(
# 1626 "cil/src/frontc/cparser.mly"
                                        ( make_expr (UNARY (BNOT, _2)) )
# 6539 "cil/src/frontc/cparser.ml"
               : 'unary_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'unary_attr) in
    Obj.repr(
# 1630 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 6546 "cil/src/frontc/cparser.ml"
               : 'cast_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'cast_attr) in
    Obj.repr(
# 1634 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 6553 "cil/src/frontc/cparser.ml"
               : 'multiplicative_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'multiplicative_attr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'cast_attr) in
    Obj.repr(
# 1635 "cil/src/frontc/cparser.mly"
                                        (make_expr (BINARY(MUL ,_1 , _3)))
# 6562 "cil/src/frontc/cparser.ml"
               : 'multiplicative_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'multiplicative_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'cast_attr) in
    Obj.repr(
# 1636 "cil/src/frontc/cparser.mly"
                                        (make_expr (BINARY(DIV ,_1 , _3)))
# 6570 "cil/src/frontc/cparser.ml"
               : 'multiplicative_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'multiplicative_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'cast_attr) in
    Obj.repr(
# 1637 "cil/src/frontc/cparser.mly"
                                          (make_expr (BINARY(MOD ,_1 , _3)))
# 6578 "cil/src/frontc/cparser.ml"
               : 'multiplicative_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'multiplicative_attr) in
    Obj.repr(
# 1642 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 6585 "cil/src/frontc/cparser.ml"
               : 'additive_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'additive_attr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'multiplicative_attr) in
    Obj.repr(
# 1643 "cil/src/frontc/cparser.mly"
                                            (make_expr (BINARY(ADD ,_1 , _3)))
# 6594 "cil/src/frontc/cparser.ml"
               : 'additive_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'additive_attr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'multiplicative_attr) in
    Obj.repr(
# 1644 "cil/src/frontc/cparser.mly"
                                            (make_expr (BINARY(SUB ,_1 , _3)))
# 6603 "cil/src/frontc/cparser.ml"
               : 'additive_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'additive_attr) in
    Obj.repr(
# 1648 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 6610 "cil/src/frontc/cparser.ml"
               : 'shift_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'shift_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'additive_attr) in
    Obj.repr(
# 1649 "cil/src/frontc/cparser.mly"
                                     (make_expr (BINARY(SHL ,_1 , _3)))
# 6618 "cil/src/frontc/cparser.ml"
               : 'shift_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'shift_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'additive_attr) in
    Obj.repr(
# 1650 "cil/src/frontc/cparser.mly"
                                     (make_expr (BINARY(SHR ,_1 , _3)))
# 6626 "cil/src/frontc/cparser.ml"
               : 'shift_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'shift_attr) in
    Obj.repr(
# 1654 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 6633 "cil/src/frontc/cparser.ml"
               : 'relational_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'relational_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shift_attr) in
    Obj.repr(
# 1655 "cil/src/frontc/cparser.mly"
                                   (make_expr (BINARY(LT ,_1 , _3)))
# 6641 "cil/src/frontc/cparser.ml"
               : 'relational_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'relational_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shift_attr) in
    Obj.repr(
# 1656 "cil/src/frontc/cparser.mly"
                                   (make_expr (BINARY(GT ,_1 , _3)))
# 6649 "cil/src/frontc/cparser.ml"
               : 'relational_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'relational_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shift_attr) in
    Obj.repr(
# 1657 "cil/src/frontc/cparser.mly"
                                      (make_expr (BINARY(LE ,_1 , _3)))
# 6657 "cil/src/frontc/cparser.ml"
               : 'relational_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'relational_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shift_attr) in
    Obj.repr(
# 1658 "cil/src/frontc/cparser.mly"
                                      (make_expr (BINARY(GE ,_1 , _3)))
# 6665 "cil/src/frontc/cparser.ml"
               : 'relational_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'relational_attr) in
    Obj.repr(
# 1662 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 6672 "cil/src/frontc/cparser.ml"
               : 'equality_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'equality_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'relational_attr) in
    Obj.repr(
# 1663 "cil/src/frontc/cparser.mly"
                                            (make_expr (BINARY(EQ ,_1 , _3)))
# 6680 "cil/src/frontc/cparser.ml"
               : 'equality_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'equality_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'relational_attr) in
    Obj.repr(
# 1664 "cil/src/frontc/cparser.mly"
                                            (make_expr (BINARY(NE ,_1 , _3)))
# 6688 "cil/src/frontc/cparser.ml"
               : 'equality_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'equality_attr) in
    Obj.repr(
# 1669 "cil/src/frontc/cparser.mly"
                                        ( _1 )
# 6695 "cil/src/frontc/cparser.ml"
               : 'bitwise_and_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'bitwise_and_attr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'equality_attr) in
    Obj.repr(
# 1670 "cil/src/frontc/cparser.mly"
                                       (make_expr (BINARY(BAND ,_1 , _3)))
# 6704 "cil/src/frontc/cparser.ml"
               : 'bitwise_and_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_and_attr) in
    Obj.repr(
# 1674 "cil/src/frontc/cparser.mly"
                                           ( _1 )
# 6711 "cil/src/frontc/cparser.ml"
               : 'bitwise_xor_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'bitwise_xor_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_and_attr) in
    Obj.repr(
# 1675 "cil/src/frontc/cparser.mly"
                                           (make_expr (BINARY(XOR ,_1 , _3)))
# 6719 "cil/src/frontc/cparser.ml"
               : 'bitwise_xor_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_xor_attr) in
    Obj.repr(
# 1679 "cil/src/frontc/cparser.mly"
                                          ( _1 )
# 6726 "cil/src/frontc/cparser.ml"
               : 'bitwise_or_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'bitwise_or_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_xor_attr) in
    Obj.repr(
# 1680 "cil/src/frontc/cparser.mly"
                                          (make_expr (BINARY(BOR ,_1 , _3)))
# 6734 "cil/src/frontc/cparser.ml"
               : 'bitwise_or_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_or_attr) in
    Obj.repr(
# 1684 "cil/src/frontc/cparser.mly"
                                                ( _1 )
# 6741 "cil/src/frontc/cparser.ml"
               : 'logical_and_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'logical_and_attr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'bitwise_or_attr) in
    Obj.repr(
# 1686 "cil/src/frontc/cparser.mly"
        (make_expr (BINARY(AND ,_1 , _3)))
# 6750 "cil/src/frontc/cparser.ml"
               : 'logical_and_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'logical_and_attr) in
    Obj.repr(
# 1690 "cil/src/frontc/cparser.mly"
                                               ( _1 )
# 6757 "cil/src/frontc/cparser.ml"
               : 'logical_or_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'logical_or_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'logical_and_attr) in
    Obj.repr(
# 1692 "cil/src/frontc/cparser.mly"
        (make_expr (BINARY(OR ,_1 , _3)))
# 6765 "cil/src/frontc/cparser.ml"
               : 'logical_or_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'logical_or_attr) in
    Obj.repr(
# 1696 "cil/src/frontc/cparser.mly"
                                           ( _1 )
# 6772 "cil/src/frontc/cparser.ml"
               : 'conditional_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : 'logical_or_attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'conditional_attr) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : 'conditional_attr) in
    Obj.repr(
# 1699 "cil/src/frontc/cparser.mly"
    ( make_expr (QUESTION(_1, _3, _5)) )
# 6781 "cil/src/frontc/cparser.ml"
               : 'conditional_attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'conditional_attr) in
    Obj.repr(
# 1702 "cil/src/frontc/cparser.mly"
                                          ( _1 )
# 6788 "cil/src/frontc/cparser.ml"
               : 'attr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'attr) in
    Obj.repr(
# 1706 "cil/src/frontc/cparser.mly"
                                         ( [_1] )
# 6795 "cil/src/frontc/cparser.ml"
               : 'attr_list_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'attr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'attr_list_ne) in
    Obj.repr(
# 1707 "cil/src/frontc/cparser.mly"
                                         ( _1 :: _3 )
# 6803 "cil/src/frontc/cparser.ml"
               : 'attr_list_ne))
; (fun __caml_parser_env ->
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'attr_list_ne) in
    Obj.repr(
# 1708 "cil/src/frontc/cparser.mly"
                                         ( _3 )
# 6810 "cil/src/frontc/cparser.ml"
               : 'attr_list_ne))
; (fun __caml_parser_env ->
    Obj.repr(
# 1711 "cil/src/frontc/cparser.mly"
                                         ( [] )
# 6816 "cil/src/frontc/cparser.ml"
               : 'attr_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'attr_list_ne) in
    Obj.repr(
# 1712 "cil/src/frontc/cparser.mly"
                                         ( _1 )
# 6823 "cil/src/frontc/cparser.ml"
               : 'attr_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'attr_list_ne) in
    Obj.repr(
# 1715 "cil/src/frontc/cparser.mly"
                                         ( _2 )
# 6831 "cil/src/frontc/cparser.ml"
               : 'paren_attr_list_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    Obj.repr(
# 1716 "cil/src/frontc/cparser.mly"
                                         ( [] )
# 6838 "cil/src/frontc/cparser.ml"
               : 'paren_attr_list_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'attr_list) in
    Obj.repr(
# 1719 "cil/src/frontc/cparser.mly"
                                         ( _2 )
# 6846 "cil/src/frontc/cparser.ml"
               : 'paren_attr_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    Obj.repr(
# 1720 "cil/src/frontc/cparser.mly"
                                         ( [] )
# 6853 "cil/src/frontc/cparser.ml"
               : 'paren_attr_list))
; (fun __caml_parser_env ->
    Obj.repr(
# 1724 "cil/src/frontc/cparser.mly"
                                        ( [] )
# 6859 "cil/src/frontc/cparser.ml"
               : Cabs.attribute list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.attribute list) in
    Obj.repr(
# 1725 "cil/src/frontc/cparser.mly"
                                        ( ("volatile", []) :: _2 )
# 6867 "cil/src/frontc/cparser.ml"
               : Cabs.attribute list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Cabs.cabsloc) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : Cabs.attribute list) in
    Obj.repr(
# 1726 "cil/src/frontc/cparser.mly"
                                        ( ("const", []) :: _2 )
# 6875 "cil/src/frontc/cparser.ml"
               : Cabs.attribute list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'one_string_constant) in
    Obj.repr(
# 1729 "cil/src/frontc/cparser.mly"
                                                 ( [_1] )
# 6882 "cil/src/frontc/cparser.ml"
               : 'asmtemplate))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'one_string_constant) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'asmtemplate) in
    Obj.repr(
# 1730 "cil/src/frontc/cparser.mly"
                                                 ( _1 :: _2 )
# 6890 "cil/src/frontc/cparser.ml"
               : 'asmtemplate))
; (fun __caml_parser_env ->
    Obj.repr(
# 1733 "cil/src/frontc/cparser.mly"
                        ( None )
# 6896 "cil/src/frontc/cparser.ml"
               : 'asmoutputs))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'asmoperands) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'asminputs) in
    Obj.repr(
# 1735 "cil/src/frontc/cparser.mly"
                        ( let (ins, clobs) = _3 in
                          Some {aoutputs = _2; ainputs = ins; aclobbers = clobs} )
# 6905 "cil/src/frontc/cparser.ml"
               : 'asmoutputs))
; (fun __caml_parser_env ->
    Obj.repr(
# 1739 "cil/src/frontc/cparser.mly"
                                        ( [] )
# 6911 "cil/src/frontc/cparser.ml"
               : 'asmoperands))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'asmoperandsne) in
    Obj.repr(
# 1740 "cil/src/frontc/cparser.mly"
                                        ( List.rev _1 )
# 6918 "cil/src/frontc/cparser.ml"
               : 'asmoperands))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'asmoperand) in
    Obj.repr(
# 1743 "cil/src/frontc/cparser.mly"
                                        ( [_1] )
# 6925 "cil/src/frontc/cparser.ml"
               : 'asmoperandsne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'asmoperandsne) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'asmoperand) in
    Obj.repr(
# 1744 "cil/src/frontc/cparser.mly"
                                        ( _3 :: _1 )
# 6933 "cil/src/frontc/cparser.ml"
               : 'asmoperandsne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : 'asmopname) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : string * cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : Cabs.expression) in
    Obj.repr(
# 1747 "cil/src/frontc/cparser.mly"
                                                           ( (_1, fst _2, _4) )
# 6943 "cil/src/frontc/cparser.ml"
               : 'asmoperand))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : 'asmopname) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : string * cabsloc) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : Cabs.cabsloc) in
    Obj.repr(
# 1748 "cil/src/frontc/cparser.mly"
                                                           (
       let loc = Parsing.rhs_start_pos 4, Parsing.rhs_end_pos 4 in
       (_1, fst _2, { expr_loc = loc; expr_node = NOTHING} )
     )
# 6955 "cil/src/frontc/cparser.ml"
               : 'asmoperand))
; (fun __caml_parser_env ->
    Obj.repr(
# 1754 "cil/src/frontc/cparser.mly"
                             ( ([], []) )
# 6961 "cil/src/frontc/cparser.ml"
               : 'asminputs))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'asmoperands) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'asmclobber) in
    Obj.repr(
# 1756 "cil/src/frontc/cparser.mly"
                        ( (_2, _3) )
# 6969 "cil/src/frontc/cparser.ml"
               : 'asminputs))
; (fun __caml_parser_env ->
    Obj.repr(
# 1759 "cil/src/frontc/cparser.mly"
                                        ( None )
# 6975 "cil/src/frontc/cparser.ml"
               : 'asmopname))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string) in
    Obj.repr(
# 1760 "cil/src/frontc/cparser.mly"
                                        ( Some _2 )
# 6982 "cil/src/frontc/cparser.ml"
               : 'asmopname))
; (fun __caml_parser_env ->
    Obj.repr(
# 1764 "cil/src/frontc/cparser.mly"
                                        ( [] )
# 6988 "cil/src/frontc/cparser.ml"
               : 'asmclobber))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'asmcloberlst_ne) in
    Obj.repr(
# 1765 "cil/src/frontc/cparser.mly"
                                        ( _2 )
# 6995 "cil/src/frontc/cparser.ml"
               : 'asmclobber))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'one_string_constant) in
    Obj.repr(
# 1768 "cil/src/frontc/cparser.mly"
                                                 ( [_1] )
# 7002 "cil/src/frontc/cparser.ml"
               : 'asmcloberlst_ne))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'one_string_constant) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'asmcloberlst_ne) in
    Obj.repr(
# 1769 "cil/src/frontc/cparser.mly"
                                                 ( _1 :: _3 )
# 7010 "cil/src/frontc/cparser.ml"
               : 'asmcloberlst_ne))
(* Entry interpret *)
; (fun __caml_parser_env -> raise (Parsing.YYexit (Parsing.peek_val __caml_parser_env 0)))
(* Entry file *)
; (fun __caml_parser_env -> raise (Parsing.YYexit (Parsing.peek_val __caml_parser_env 0)))
|]
let yytables =
  { Parsing.actions=yyact;
    Parsing.transl_const=yytransl_const;
    Parsing.transl_block=yytransl_block;
    Parsing.lhs=yylhs;
    Parsing.len=yylen;
    Parsing.defred=yydefred;
    Parsing.dgoto=yydgoto;
    Parsing.sindex=yysindex;
    Parsing.rindex=yyrindex;
    Parsing.gindex=yygindex;
    Parsing.tablesize=yytablesize;
    Parsing.table=yytable;
    Parsing.check=yycheck;
    Parsing.error_function=parse_error;
    Parsing.names_const=yynames_const;
    Parsing.names_block=yynames_block }
let interpret (lexfun : Lexing.lexbuf -> token) (lexbuf : Lexing.lexbuf) =
   (Parsing.yyparse yytables 1 lexfun lexbuf : (bool*Cabs.definition) list)
let file (lexfun : Lexing.lexbuf -> token) (lexbuf : Lexing.lexbuf) =
   (Parsing.yyparse yytables 2 lexfun lexbuf : (bool*Cabs.definition) list)
;;
