/* -*- C -*-

  Copyright (c) 1994-2001 by the Massachusetts Institute of Technology.
  Copyright (c) 2001-2002 by beingmeta, inc. (A Delaware Corporation)
  All rights reserved.

  $Id: odb.h,v 1.18 2002/07/24 02:05:47 haase Exp $

  This file is part of FramerD, a representation language and semantic
  database developed by Kenneth B. Haase and his students at the Media
  Laboratory at the Massachusetts Institute of Technology in Cambridge,
  Massachusetts.  Research at the Media Lab is supported by funds and
  equipment from a variety of corporations and government sponsors whose
  contributions are gratefully acknowledged.

    Use, modification, and redistribution of this program is permitted
    under the terms of either (at the developer's discretion) the GNU
    General Public License (GPL) Version 2, the GNU Lesser General Public
    License.

    This program is based on the FramerD library released in Fall 2001 by
    MIT under both the GPL and the LGPL licenses, both of which accompany
    this distribution.  Subsequent modifications by beingmeta, inc. are
    also released under both the GPL and LGPL licenses (at the developer's
    discretion).

*************************************************************************/

#ifndef FRAMERD_ODB_H
#define FRAMERD_ODB_H

#include "framerd/common.h"
#include "framerd/cons.h"
#include "framerd/os.h"


/* FramerD Exceptions */

FRAMERD_EXPORT fd_exception fd_PoolNOOP;

FRAMERD_EXPORT fd_exception
  fd_Non_Atomic_Slot, fd_Read_Only_OID,
  fd_Cant_Lock_Remote_OID, fd_Homeless_OID,
  fd_Invalid_Frame, fd_Invalid_Slot, fd_NoSuchValue,
  fd_FSVInvalid,
  fd_UnknownPool,
  fd_FilePoolExhausted,
  fd_UnlockablePool,
  fd_ReadOnlyPool,
  fd_NotAFilePool,
  fd_NotASuperPool,
  fd_UnregisteredPool,
  fd_BadPoolSpec,
  fd_UnWritablePool,
  fd_NotASuperPool,
  fd_UnallocatedOID,
  fd_InvalidOID,
  fd_FSVInvalid,
  fd_AmbiguousFrameName,
  fd_UnknownFrameName;


/* Pools */

#define FD_SUPER_POOL_MAGIC_NUMBER 320147474
#define FD_REGISTERED_SUPER_POOL_MAGIC_NUMBER 320147475
#define FD_FILE_POOL_MAGIC_NUMBER  67179521
#define FD_FILE_POOL_SNAPSHOT_MAGIC_NUMBER 269684240

#define FD_POOL_WRITABLE 0
#define FD_POOL_LOCKABLE 1
#define FD_POOL_READ_ONLY 2

/* This is how all pools start */
#define FD_POOL_FIELDS() \
  FD_OID base; unsigned int capacity;                              \
  enum FD_POOL_TYPE type;                                          \
  char *id, *prefix_id; fd_lisp label;                             \
  int modifiedp; int read_only;                                    \
  struct FD_HASHSET modified;                                      \
  struct FD_POOL_HANDLER *handler

#define FD_POOL_LOCK_OID 1
#define FD_POOL_UNLOCK_OID 2

typedef struct FD_POOL { 
  FD_POOL_FIELDS();} *fd_pool;

struct FD_POOL_HANDLER {
  fd_lisp (*new_oid)(struct FD_POOL *p);
  fd_lisp (*lookup_oid)(struct FD_POOL *p,fd_lisp id);
  void (*commit_pool)(struct FD_POOL *p);
  void (*commit_oid)(struct FD_POOL *p,fd_lisp oid);
  int (*bulk_fetch)(struct FD_POOL *p,fd_lisp *o,fd_lisp *v,int n);
  void (*close_pool)(struct FD_POOL *p);
  void (*oid_locker)(struct FD_POOL *p,fd_lisp oid,int action);
  unsigned int (*get_load)(struct FD_POOL *p);
  void (*spend_memory)(struct FD_POOL *p);};
  

/* File pools are stored in the file system */
typedef struct FD_FILE_POOL {
  FD_POOL_FIELDS();
  char *buf; /* cached big buffer */
  char *filename;
  FILE *store;
#if FD_USING_THREADS
  fd_mutex lock;
#endif
  unsigned int load;
  unsigned int end_pos;
  unsigned int offsets_size;
  unsigned int *offsets;} *fd_file_pool;

/* Network pools are accessed over the network */
typedef struct FD_NETWORK_POOL {
  FD_POOL_FIELDS();
  fd_server conn;} *fd_network_pool;

/* An array of these works for now */
typedef struct FD_POOL_TABLE_ENTRY {
  FD_OID base; unsigned int capacity;
  fd_pool p;} *pool_entry;

FRAMERD_EXPORT struct FD_POOL_TABLE_ENTRY *_fd_pool_table;
FRAMERD_EXPORT int fd_n_pools;

FRAMERD_EXPORT
void fd_register_file_pool_opener
 (int prefix_code,fd_pool (*opener)(fd_u8char *c));

struct FD_FILE_POOL_OPENER {
  int magic_number; fd_pool (*opener)(fd_u8char *);
  struct FD_FILE_POOL_OPENER *next;};


/* Accessing OIDs */

FRAMERD_EXPORT fd_lisp fd_get_oid_value(fd_lisp oref);

#if (FD_USING_THREADS)
FASTOP fd_lisp fd_oid_value(fd_lisp) UNUSED;
FASTOP fd_lisp fd_oid_value(fd_lisp obj)
{
  fd_lisp v;
  FD_LOCK_OID(obj);
  v=_fd_oid_current_value_nolock(obj);
  FD_UNLOCK_OID(obj);
  if (FD_VOIDP(v)) return fd_get_oid_value(obj);
  else return v;
}
#else
#define fd_oid_value(obj) \
  ((FD_OIDP(obj)) ? \
   ((fd_oid_loadedp(obj)) ? (fd_oid_current_value(obj)) : \
    (fd_get_oid_value(obj))) : \
   (fd_type_error(_("fd_oid_value: not an OID"),obj),(FD_VOID)))
#endif


/* Pool Functions */

#if FD_USING_THREADS
static fd_mutex _fd_pool_table_lock;
#endif

FRAMERD_EXPORT fd_pool fd_use_pool(fd_u8char *c);
FRAMERD_EXPORT 
  void fd_for_pools(void (*fcn)(fd_pool p,void *arg),void *arg);
FRAMERD_EXPORT int fd_get_pool_count(void);
FRAMERD_EXPORT fd_pool fd_interpret_pool(fd_lisp spec);
FRAMERD_EXPORT struct FD_POOL *fd_get_pool(fd_lisp x);

FRAMERD_EXPORT int fd_register_pool(fd_pool p);
FRAMERD_EXPORT fd_pool fd_locate_pool(fd_lisp oid);

FRAMERD_EXPORT fd_pool fd_find_pool_named(fd_u8char *string);

FRAMERD_EXPORT fd_pool fd_use_file_pool(fd_u8char *filename);
FRAMERD_EXPORT
fd_network_pool fd_use_network_pool(char *hostname,int port,fd_u8char *id);
FRAMERD_EXPORT int fd_lock_file_pool(fd_file_pool p);

FRAMERD_EXPORT void fd_auto_cache_file_pools(void);
FRAMERD_EXPORT void fd_make_file_pool
   (char *filename,FD_OID base,unsigned int capacity,
    int major_version,fd_lisp metadata);
FRAMERD_EXPORT FD_OID fd_allocate_pool
   (char *super_pool,unsigned int capacity,char *label);
FRAMERD_EXPORT FD_OID fd_recovered_pool
   (char *super_pool,FD_OID base,unsigned int capacity,fd_lisp tag);
FRAMERD_EXPORT void fd_new_file_pool
   (char *filename,unsigned int capacity,char *super_pool);
FRAMERD_EXPORT void fd_cache_file_pool(fd_pool p);
FRAMERD_EXPORT fd_lisp fd_read_file_pool_metadata
  (FILE *f,int *revnum,int *size,time_t *make,time_t *repack,time_t *change);


/* Declarations */

FRAMERD_EXPORT int fd_loaded_oids;
FRAMERD_EXPORT int fd_oids_loaded;
FRAMERD_EXPORT int fd_new_oids;

FRAMERD_EXPORT void fd_never_save(void);
FRAMERD_EXPORT void fd_homeless_ok(void);
FRAMERD_EXPORT int fd_ephemeralp(void);

FRAMERD_EXPORT void fd_for_file_pool(void (*fcn)(fd_lisp obj),fd_file_pool fp);

/* Maintenance functions */

FRAMERD_EXPORT void fd_repack_file_pool(char *origin,char *destination);
FRAMERD_EXPORT unsigned int fd_make_super_pool
  (char *filename,unsigned int base,unsigned int load);
FRAMERD_EXPORT unsigned int fd_make_new_super_pool(char *filename);
FRAMERD_EXPORT void fd_report_framerd_stats(FILE *to);

FRAMERD_EXPORT unsigned int fd_file_pool_load(fd_u8char *filename);
FRAMERD_EXPORT unsigned int fd_file_pool_capacity(fd_u8char *filename);
FRAMERD_EXPORT unsigned int fd_file_pool_freespace(fd_u8char *filename);

FRAMERD_EXPORT FD_OID fd_super_pool_base(char *id);
FRAMERD_EXPORT FD_OID fd_super_pool_top(char *id);
FRAMERD_EXPORT float fd_super_pool_loading(char *id);

FRAMERD_EXPORT void fd_make_pool_snapshot(char *filename,char *snapshot);
FRAMERD_EXPORT void fd_restore_pool_snapshot(char *snapshot,char *filename);

/* Autoindexing declarations */

FRAMERD_EXPORT void fd_use_autoindex(struct FD_INDEX *ix);

FRAMERD_EXPORT int fd_pool_load(fd_pool p);
FRAMERD_EXPORT fd_lisp fd_random_oid(fd_pool p);

FRAMERD_EXPORT void fd_label_file_pool(char *filename,fd_lisp label);

FRAMERD_EXPORT fd_lisp fd_try_oid_value(fd_lisp oref);
FRAMERD_EXPORT void fd_mark_modified(fd_lisp oref);
FRAMERD_EXPORT fd_lisp *fd_get_modified(fd_pool p,int *n_oids,int reset);
FRAMERD_EXPORT void fd_set_oid_value(fd_lisp oref,fd_lisp value);
FRAMERD_EXPORT fd_lisp fd_new_oid(fd_pool p);

/* Saving, reversion, and swapping */

FRAMERD_EXPORT int fd_revert_oid(fd_lisp oref);
FRAMERD_EXPORT void fd_swap_out(fd_lisp oref);

FRAMERD_EXPORT void fd_close_pool(fd_pool p);
FRAMERD_EXPORT void fd_commit_pool(fd_pool p);
FRAMERD_EXPORT void fd_revert_pool(fd_pool p);
FRAMERD_EXPORT void fd_swap_out_pool(fd_pool p);

FRAMERD_EXPORT void fd_commit_pools(void);
FRAMERD_EXPORT void fd_revert_pools(void);
FRAMERD_EXPORT void fd_swap_out_oids(void);
/* For old code */
#define fd_swap_out_pools fd_swap_out_oids

FRAMERD_EXPORT void fd_save_all_and_exit(void);

FRAMERD_EXPORT void fd_commit_file_pool(fd_file_pool p);
FRAMERD_EXPORT void fd_revert_file_pool(fd_file_pool p);

FRAMERD_EXPORT void fd_prefetch_oids(fd_lisp oids);

/* Frame operations */

#define FD_FRAMEP(x) ((FD_OIDP(x)) && (FD_SLOTMAPP(fd_oid_value(x))))

typedef enum SLOT_OP 
  { slot_get, slot_add, slot_remove, slot_test , slot_validate} slot_op;
struct FD_SLOT_STACK {
  slot_op op;
  fd_lisp frame, slotid, value, goal; 
  struct FD_SLOT_STACK *next;};
extern struct FD_SLOT_STACK *fd_slot_stack;

FRAMERD_EXPORT fd_lisp fd_prim_get(fd_lisp frame,fd_lisp slot);
FRAMERD_EXPORT void fd_prim_set(fd_lisp frame,fd_lisp slot,fd_lisp value);
FRAMERD_EXPORT void fd_prim_set_consed(fd_lisp frame,fd_lisp slot,fd_lisp value);
FRAMERD_EXPORT int fd_prim_test(fd_lisp frame,fd_lisp slot,fd_lisp value);
FRAMERD_EXPORT void fd_prim_add(fd_lisp frame,fd_lisp slot,fd_lisp value);
FRAMERD_EXPORT void fd_prim_add_consed(fd_lisp frame,fd_lisp slot,fd_lisp value);
FRAMERD_EXPORT void fd_prim_drop(fd_lisp frame,fd_lisp slot,fd_lisp value);
FRAMERD_EXPORT void fd_prim_rename_slot
  (fd_lisp frame,fd_lisp old_name,fd_lisp new_name);

FRAMERD_EXPORT fd_lisp fd_overlay_get(fd_lisp frame,fd_lisp slotid);
FRAMERD_EXPORT int fd_overlay_test(fd_lisp frame,fd_lisp slotid,fd_lisp v);

FRAMERD_EXPORT fd_lisp fd_frame_create(fd_pool p);
FRAMERD_EXPORT fd_lisp fd_overlay_create(fd_pool x,fd_lisp base);

FRAMERD_EXPORT fd_lisp fd_copy_frame (fd_lisp f,fd_pool p);
FRAMERD_EXPORT fd_lisp fd_frame_get  (fd_lisp f,fd_lisp s);
FRAMERD_EXPORT void fd_frame_add     (fd_lisp f,fd_lisp s,fd_lisp v);
FRAMERD_EXPORT int fd_frame_validate (fd_lisp f,fd_lisp s,fd_lisp v);
FRAMERD_EXPORT int fd_frame_test     (fd_lisp f,fd_lisp s,fd_lisp v);
FRAMERD_EXPORT void fd_frame_set     (fd_lisp f,fd_lisp s,fd_lisp v);
FRAMERD_EXPORT void fd_frame_remove  (fd_lisp f,fd_lisp s,fd_lisp v);
FRAMERD_EXPORT fd_lisp fd_frame_get_star  (fd_lisp f,fd_lisp s);

/* Caching computed slot values */

FRAMERD_EXPORT void fd_clear_slot_cache(fd_lisp slot,fd_lisp frame);
FRAMERD_EXPORT void fd_enable_slot_cache(fd_lisp slot);
FRAMERD_EXPORT void fd_disable_slot_cache(fd_lisp slot);

/* Useful predicates */

FRAMERD_EXPORT fd_lisp fd_inherit_values(fd_lisp root,fd_lisp slotid,fd_lisp through);
FRAMERD_EXPORT int fd_inherits_valuep(fd_lisp root,fd_lisp slotid,fd_lisp through,fd_lisp value);
FRAMERD_EXPORT int fd_pathp(fd_lisp root,fd_lisp slotid,fd_lisp to);

/* I/O related functions */

FRAMERD_EXPORT void fd_control_frame_printing(int level);
FRAMERD_EXPORT void fd_show_poolids(int use_ids);

FRAMERD_EXPORT void fd_describe_slot(FILE *stream,fd_lisp slot,fd_lisp value);
FRAMERD_EXPORT void fd_describe_frame(fd_lisp frame,FILE *stream);

FRAMERD_EXPORT void fd_import_frame(fd_lisp frame,fd_lisp slots,int noisy);
FRAMERD_EXPORT fd_lisp fd_export_frame(fd_lisp frame,fd_lisp focus_slots,int only);


/* Dealing with slots */

#define FD_SLOTP(x) ((FD_SYMBOLP(x)) || (FD_OIDP(x)))

#define FD_DO_SLOTS(s,v,obj) \
  fd_lisp _doslots_u=obj; \
  fd_lisp _doslots_ov=fd_get_slotmap(_doslots_u);  \
  fd_lisp s, v, *_doslots_slots;  \
  int _size=_fd_slotmap_data(_doslots_ov,(void **)&_doslots_slots); \
  fd_lisp *_doslots_scan=_doslots_slots; \
  fd_lisp *_doslots_limit=_doslots_slots+_size; \
  while ((_doslots_scan<_doslots_limit) ? \
	 (s=*_doslots_scan++,v=*_doslots_scan++,1) : \
	 (fd_decref(_doslots_ov),_fd_done_with_slotmap_data(_doslots_slots,_size),0))

FRAMERD_EXPORT void fd_for_slots
  (void (*fcn)(fd_lisp frame,fd_lisp slot,fd_lisp value),fd_lisp frame);
FRAMERD_EXPORT void fd_for_values
  (void (*fcn)(fd_lisp frame,fd_lisp slot,fd_lisp value),fd_lisp frame);
FRAMERD_EXPORT fd_lisp fd_frame_slots(fd_lisp frame);
FRAMERD_EXPORT fd_lisp fd_get_slotmap(fd_lisp oid);

#if FD_INLINE_OIDS
#if (FD_LIGHTWEIGHT_OIDS)
STATIC_INLINE fd_pool FD_GET_POOL(fd_lisp x) 
{
  if (FD_OIDP(x)) {
    unsigned int d=FD_PTR_DATA(x,oidaddr);
    unsigned int pid=(d>>24);
    unsigned int off=(d&0xFFFFFF);
    if (_fd_pool_buckets[pid].pool)
      if (off < _fd_pool_buckets[pid].capacity)
	return (fd_pool) _fd_pool_buckets[pid].pool;
      else return (fd_pool) _fd_get_pool_from_bucket(pid,off);
    else return (fd_pool) _fd_get_pool_from_bucket(pid,off);}
  else {
    fd_type_error(_("not an OID"),x);
    return NULL;}
}
#else /* FD_LIGHTWEIGHT_OIDS */
FASTOP fd_pool FD_GET_POOL(lisp oid)
{
  if (FD_OIDP(oid)) {
    fd_pool p=NULL;
    if (fd_n_pools) {
      FD_OID id=FD_OID_ADDR(oid);
      int bot=0, top=fd_n_pools-1, direction;
      unsigned int hi=FD_OID_HIGH(id), lo=FD_OID_LOW(id);
      fd_lock_mutex(&_fd_pool_table_lock);
      while (top >= bot) {
	int i=bot+(top-bot)/2;
	if (FD_OID_HIGH(_fd_pool_table[i].base) < hi) direction=1;
	else if (FD_OID_HIGH(_fd_pool_table[i].base) > hi) direction=-1;
	else if (FD_OID_LOW(_fd_pool_table[i].base) > lo) direction=-1;
	else if (lo < FD_OID_LOW(_fd_pool_table[i].base)+_fd_pool_table[i].capacity) {
	  p=_fd_pool_table[i].p; break;}
	else direction=1;
	if (direction < 0) top=i-1;
	else bot=i+1;}
      fd_unlock_mutex(&_fd_pool_table_lock);}
    return p;}
  else {
    fd_type_error(_("not an OID"),oid);
    return NULL;}
}
#endif /* not FD_LIGHTWEIGHT_OIDS */
#else /* FD_INLINE_OIDS */
#define FD_GET_POOL(x) fd_get_pool(x)
#endif

#if (FD_SOURCE)
#define OID_ADDR_HIGH FD_OID_ADDR_HIGH
#define OID_ADDR_LOW FD_OID_ADDR_LOW
#define oid_value fd_oid_value
#define oid_current_value fd_oid_current_value
#define store_oid_value fd_store_oid_value

#define DO_SLOTS FD_DO_SLOTS
#define FRAMEP FD_FRAMEP
#define SLOTP FD_SLOTP
#endif


#endif /* ndef FRAMERD_ODB_H */




/* File specific stuff */

/* The CVS log for this file
   $Log: odb.h,v $
   Revision 1.18  2002/07/24 02:05:47  haase
   Removed 'new' symbols from include files to allow inclusion in C++ files

   Revision 1.17  2002/06/23 11:51:02  haase
   Fixed some race conditions with OID saving and multi threaded processes (where one thread is saving an OID while another one is modifying it)

   Revision 1.16  2002/06/21 13:46:03  haase
   Made file pools use an offset table limited to the load of the pool, growing it when neccessary.

   Revision 1.15  2002/05/01 21:46:31  haase
   Renamed mutex/condvar/rwlock types to have fd_ prefixes

   Revision 1.14  2002/04/22 14:23:07  haase
   Added extended metadata to file pools and indices

   Revision 1.13  2002/04/11 00:26:23  haase
   Removed API for naming pools

   Revision 1.12  2002/04/10 18:57:08  haase
       Orthogonalized associations between pool names and pools and between
   OIDs and their pools.
       Made canonicalization of paths or server specs be done directly
   at pool creation time rather than kludged through the name mechanism.
       Added use of dotted syntax in pool labels.

   Revision 1.11  2002/04/10 03:01:50  haase
   Added version information to file pools and indices

   Revision 1.10  2002/04/02 21:41:09  haase
   Added log and emacs init entries to C source files

*/

/* Emacs local variables
;;;  Local variables: ***
;;;  compile-command: "cd ../..; make" ***
;;;  End: ***
*/
