//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Cylinder.hpp,v 1.4 2004/02/08 18:44:19 delpinux Exp $

// This class provides a toolkit to manipulate Cylinders. It inherits from the
// Shape class.

#ifndef CYLINDER_HPP
#define CYLINDER_HPP

#include <Shape.hpp>

/**
 * @file   Cylinder.hpp
 * @author Stephane Del Pino
 * @date   Sat Feb  7 17:13:00 2004
 * 
 * @brief  This is the class which defines POV-Ray's cylinder.
 * 
 * 
 */

class Cylinder
  : public Shape
{
private:
  friend class InfiniteCylinder;
  friend class Plane;

  //! The center of the upper face.
  TinyVector<3, real_t> __c1;

  //! The center of the lower face
  TinyVector<3, real_t> __c2;

  //! This vertex is usefull to determine the space inside the Cylinder
  TinyVector<3, real_t> __center;

  //! The radius of the Cylinder.
  real_t __radius;

  /*!  The square of the radius of the Cylinder (since it is usefull very
    often, it is a good idea to store it).
  */
  real_t __radius2;

  /*!  Half of the height of the cylinder. Usefull to detect the space inside
    the Cylinder.
  */
  real_t __half_height;

 //! The unary vector along the \a c1 \a c2 axis.
  TinyVector<3, real_t> __unaryVector;

  //! Returns true if the point \a p is inside the Cylinder.
  inline bool inShape(const TinyVector<3, real_t>& p) const;

  //! Prints the definition of the Cylinder in the stream \a s.
  std::ostream& put(std::ostream& s) const;
public:

  //! Copies a Cylinder.
  const Cylinder& operator = (const Cylinder&);

  /*! Constructs the Cylinder whose axis is defined \a a and \a b, and of
    radius r
   */
  Cylinder(const TinyVector<3, real_t>& a,
	   const TinyVector<3, real_t>& b,
	   const real_t& r);

  //! Default constructor.
  Cylinder();

  //! Copy constructor.
  Cylinder(const Cylinder& C);

  //! Destructor.
  ~Cylinder()
  {
    ;
  }
};

//! Returns true if the point \a p is inside the Cylinder.
inline bool Cylinder::inShape(const TinyVector<3>& v) const
{
  TinyVector<3, real_t> x = v-__center;

  real_t ux = __unaryVector * x;
  TinyVector<3, real_t> normal (x - (ux * __unaryVector));
  return ((std::abs(ux) < __half_height) && (normal * normal < __radius2));
}

#endif // CYLINDER_HPP

