//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: MeshOfHexahedra.hpp,v 1.6 2003/12/26 18:38:02 uid68082 Exp $

#ifndef MESHOFHEXAHEDRA_HPP
#define MESHOFHEXAHEDRA_HPP

/**
 * @file   MeshOfHexahedra.hpp
 * @author Stephane Del Pino
 * @date   Tue Nov 26 17:34:43 2002
 * 
 * @brief  Describes meshes composed of hexahedra
 * 
 * 
 */

#include <Mesh.hpp>

#include <SurfaceMeshOfQuadrangles.hpp>

#include <Hexahedron.hpp>

#include <Connectivity.hpp>
#include <Octree.hpp>

#include <list>
#include <map>

class MeshOfHexahedra
  : public Mesh
{
public:
  typedef Hexahedron ElementGeometry;
  typedef MeshOfHexahedra Transformed;

  typedef SurfaceMeshOfQuadrangles BorderMeshType;

  enum {
    family = Mesh::volume
  };
private:
  ReferenceCounting<Vector<Hexahedron> > __cells; /**< Cells list */

  ReferenceCounting<BorderMeshType>
  __surfaceMesh; /**< The various border lists*/

  ReferenceCounting<Connectivity<Hexahedron> > __connectivity; /**< Connectivity */

  ReferenceCounting<Octree<size_t> > __octree; /**< Octree refering to the cells */

  TinyVector<3,real_t> __a;	/**< bounding box lower  corner */
  TinyVector<3,real_t> __b;	/**< bounding box higher corner */


  /** 
   * Forbids copy contructor
   * 
   * @param m another mesh
   * 
   */
  MeshOfHexahedra(const MeshOfHexahedra& m)
    : Mesh(Mesh::hexahedraMesh,0)
  {
    ;
  }

public:
  inline bool hasSurfaceMesh() const
  {
    return (__surfaceMesh != 0);
  }

  inline BorderMeshType& surfaceMesh()
  {
    return (*__surfaceMesh);
  }

  inline const BorderMeshType& surfaceMesh() const
  {
    return (*__surfaceMesh);
  }

  //! Read-only access to the number of cells.
  inline const size_t& numberOfCells() const
  {
    return (*__cells).size();
  }

  const size_t cellNumber(const Hexahedron& h) const
  {
    return (*__cells).number(h);
  }

  typedef Mesh::T_iterator<MeshOfHexahedra, Hexahedron> iterator;
  typedef Mesh::T_iterator<const MeshOfHexahedra, const Hexahedron> const_iterator;

  void buildLocalizationTools();

  MeshOfHexahedra::const_iterator find(const double& x,
				       const double& y,
				       const double& z) const;

  MeshOfHexahedra::const_iterator find(const TinyVector<3>& X) const
  {
    return find(X[0], X[1], X[2]);
  }


  //! Writes the mesh to the OpenDX Mesh format
  void plotOpenDX(std::ostream& fmesh,
		  const std::string& CR) const
  {
    fferr(0) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
    std::exit(1);
  }

  //! Access to the Cell \a i of the mesh.
  inline Hexahedron& cell(const size_t i)
  {
    return (*__cells)[i];
  }

  //! Read-only access to the Cell \a i of the mesh.
  inline const Hexahedron& cell(const size_t i) const
  {
    return (*__cells)[i];
  }

  MeshOfHexahedra(const size_t numberOfVertices,
		  const size_t numberOfCells)
    : Mesh(Mesh::hexahedraMesh,
	   numberOfVertices), 
      __cells(new Vector<Hexahedron>(numberOfCells))
  {
    ;
  }

  MeshOfHexahedra(ReferenceCounting<VerticesSet> vertices,
		  ReferenceCounting<Vector<Hexahedron> > hexahedra,
		  ReferenceCounting<BorderMeshType> surfmesh)
    : Mesh(Mesh::hexahedraMesh,
	   vertices),
      __cells(hexahedra),
      __surfaceMesh(surfmesh)
  {
    this->buildLocalizationTools();
  }

  /** 
   * With this constructor, border has to be built
   * 
   * @param vertices 
   * @param hexahedra 
   * 
   */
  MeshOfHexahedra(ReferenceCounting<VerticesSet> vertices,
		  ReferenceCounting<Vector<Hexahedron> > hexahedra)
    : Mesh(Mesh::hexahedraMesh,
	   vertices),
      __cells(hexahedra)
  {
    this->buildLocalizationTools();
  }

  ~MeshOfHexahedra()
  {
    ;
  }
};

#endif // MESHOFHEXAHEDRA_HPP
