//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: BiConjugateGradient.hpp,v 1.1.1.1 2003/02/17 16:32:52 delpinux Exp $

#ifndef _BICONJUGATEGRADIENT_HPP_
#define _BICONJUGATEGRADIENT_HPP_

/*!
  \class BiConjugateGradient
  Bi Conjugate Gradient.

  \par Resolve the linear system \f$ Au=b \f$ using the preconditionner P.

  \author Stphane Del Pino.
*/

#include <BiConjugateGradientOptions.hpp>
#include <GetParameter.hpp>

class BiConjugateGradient
{
 private:
  real_t epsilon;
  int    max_iter;

  GetParameter<BiConjugateGradientOptions> __options;

 public:

  template <typename VectorType,
	    typename MatrixType,
	    typename PreconditionerType>
  BiConjugateGradient(const VectorType& b,
		      const MatrixType& A, 
		      PreconditionerType& P,
		      VectorType& u)
  {
    epsilon  = __options.value().epsilon();
    max_iter = __options.value().maxiter();

    real_t alpha, beta;
    real_t rho_1 = 0;
    real_t rho_2 = 1;

    VectorType z(b.size());
    VectorType zTilda(b.size());
    VectorType p(b.size());
    VectorType pTilda(b.size());
    VectorType q(b.size());
    VectorType qTilda(b.size());
    z = 0;
    p = 0;

    VectorType r = b;
    A.timesX(u,q);
    r -= q;
    VectorType rTilda = r;

    real_t residu = Norm(r);
    if (residu==0)
      residu = 1.;
    real_t resid0 = residu;

    ffout(2) << "initial residu: " << resid0 << '\n';
    for (size_t i = 1; i <= (size_t)max_iter; i++) {
      ffout(3) << "biCG iteration: " << i << " relative residu: "
	       << residu/resid0 << '\n';
      // z = P^{-1}(r)
      P.Computes(r,z);

      // zTilda = P^{-T}(rTilda) Here we only use symetric preconditionners.
      P.Computes(rTilda,zTilda);

      rho_1 = z * rTilda;
      if (rho_1 == 0) { 
	return;
      }

      if (i == 1) {
	p = z;
	pTilda = zTilda;
      } else {
	beta = rho_1/rho_2;
	// p = z + beta * p;
	p *= beta;
	p += z;

	// pTilda = zTilda + beta * pTilda;
	pTilda *= beta;
	pTilda += zTilda;
      }

      A.transposedTimesX(pTilda,qTilda);
      A.timesX(p,q);
      alpha = rho_1/(pTilda*q);
      u += alpha * p;
      r -= alpha * q;
      rTilda -= alpha * qTilda;

      rho_2 = rho_1;
      if ((residu = (Norm(r)))/resid0<epsilon) {
	if (i==1) {
	  resid0=residu;
	} else {
	  ffout(2) << "leaving biCG:\n";
	  ffout(2) << "\tresidu = " << residu << '\n';
	  ffout(2) << "\tepsilon= " << epsilon << '\n';
	  return;
	}
      }
    }

    fferr(2) << "### NOT CONVERGED ###\nleaving CG:\n";
    fferr(2) << "\tresidu      = " << residu << '\n';

    fferr(2) << "\tepsilon     = " << epsilon << '\n';
  }

};
  
#endif // _BICONJUGATEGRADIENT_HPP_

