//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: ConjugateGradient.hpp,v 1.1.1.1 2003/02/17 16:32:51 delpinux Exp $


#ifndef _CONJUGATEGRADIENT_HPP_
#define _CONJUGATEGRADIENT_HPP_

#include <ConjugateGradientOptions.hpp>
#include <GetParameter.hpp>

/*!
  \class ConjugateGradient
  Conjugate Gradient.

  \par Resolve the linear system \f$ Au=b \f$ using the preconditionner P.

  \author Stphane Del Pino.
 */

class ConjugateGradient
{
private:
  real_t __epsilon;
  int    __maxiter;

  GetParameter<ConjugateGradientOptions> __options;

public:
  template <typename VectorType,
	    typename MatrixType,
	    typename PreconditionerType>
  ConjugateGradient(const VectorType& f,
		    const MatrixType& A, 
		    const PreconditionerType& C,
		    VectorType& x)
  {
    __epsilon  = __options.value().epsilon();
    __maxiter = __options.value().maxiter();

    VectorType h(x.size());
    VectorType b(f);

    VectorType g(b.size());
    VectorType cg = b;

    real_t gcg=0;
    real_t gcg0=1;

    for (int i=1; i<=__maxiter; ++i) {
      if (i==1) {
	A.timesX(x,h);

	cg -= h;

	C.Computes(cg, g);

	gcg = g * cg;

	h=g;
      }

      A.timesX(h,b);

      real_t hAh = h*b;

      if (hAh==0) {
	hAh=1.;
      }
      real_t ro = gcg/hAh;
      cg -= ro*b;

      VectorType b2 = b;
      C.Computes(b2,b);
      
      x+=ro*h;
      g-=ro*b;

      real_t gamma=gcg;
      gcg = g * cg;

      if ((i == 1)&&(gcg != 0)) {
 	__epsilon*=gcg;
	gcg0=gcg;
      }
      ffout(3) << "pcg: iteration "
	       << i << " gcg= " << gcg/gcg0;
      ffout(4) << "\tabsolute: " << gcg;
      ffout(3) << '\n';

      if (gcg<__epsilon)
	return;
      gamma=gcg/gamma;

      h *= gamma;
      h += g;
    }
  }
};
#endif // _CONJUGATEGRADIENT_HPP_

