//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stéphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: QGLvtkWindow.hpp,v 1.1 2004/10/24 16:58:05 delpinux Exp $

#ifndef QGL_VTK_WINDOW_HPP
#define QGL_VTK_WINDOW_HPP

#include <qgl.h>
#include <vtkXOpenGLRenderWindow.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkCommand.h>

/**
 * @file   QGLvtkWindow.hpp
 * @author Stephane Del Pino
 * @date   Sat Sep 25 12:32:42 2004
 * 
 * @brief  vtkWindow integration inside a Qt widget
 * 
 * 
 */

class QGLvtkWindow
  : public QGLWidget
{
  Q_OBJECT

private:
  /**
   * @class   QGLvtkWindow.hpp
   * 
   * @brief  this sub-class is used to make the iteractor a QObject
   * 
   * If using directly the vtkRenderWindowInteractor, anoter openGL
   * window is create.
   */
  struct QGLInteractor
    : public QObject, public vtkRenderWindowInteractor
  {
    /** 
     * That's all what we need
     * 
     * @return a new QGLInteractor object
     */
    static QGLInteractor* New()
    {
      return new QGLInteractor;
    }
  };

  vtkXOpenGLRenderWindow* __vtkRenderWindow; /**< The rendering window */
  QGLInteractor*  __interactor;	/**< Need to interact with the scene */
  Qt::ButtonState __mouseButton; /**< Remerber which is the last pressed button */

protected:

  /** 
   * openGL initialization
   * 
   */
  void initializeGL()
  {
    __vtkRenderWindow->OpenGLInit();
  }

  /** 
   * Passes new window size information to the vtk component
   * 
   * @param with new width
   * @param height new height
   */
  void resizeGL(int width, int height)
  {
    __vtkRenderWindow->SetSize(width, height);
  }

  /** 
   * Paints the Qt widget using vtk rendering
   * 
   */
  void paintGL()
  {
    __vtkRenderWindow->Render();
  }

public slots:

  /** 
   * updates the window
   * 
   */
  void updateGL()
  {
    __vtkRenderWindow->Render();
  }

public:
  /** 
   * adds the renderer
   * 
   * @param renderer a renderer
   */
#warning renderer presence should be checked before rendering
  void addRenderer( vtkRenderer *renderer )
  {
    __vtkRenderWindow->AddRenderer( renderer );
  }

  /** 
   * Enter event loop: the widget is activated
   * 
   * @param event an event
   */
  void enterEvent(QEvent* event)
  {
    this->QWidget::setFocus();
    __interactor->InvokeEvent(vtkCommand::EnterEvent);
  }

  /** 
   * Leaves event loop: the widget is disactivated
   * 
   * @param event an event
   */
  void leaveEvent(QEvent* event)
  {
    __interactor->InvokeEvent(vtkCommand::LeaveEvent);
  }

  /** 
   * Passes mouse motions to vtk
   * 
   * @param event the mouse even to be treated
   */
  void mouseMoveEvent(QMouseEvent* event)
  {      
    __interactor->SetEventInformationFlipY(event->x(),
					   event->y(),
					   (event->state() & ControlButton),
					   (event->state() & ShiftButton));
    __interactor->InvokeEvent(vtkCommand::MouseMoveEvent);
  }

  /** 
   * Passes mouse button pressed event to vtk
   * 
   * @param event the mouse event
   */
  void mousePressEvent(QMouseEvent* event)
  {
    __interactor->SetEventInformationFlipY(event->x(),
					   event->y(),
					   (event->state() & ControlButton),
					   (event->state() & ShiftButton));
    switch (event->button()) {
    case Qt::LeftButton:{
      __interactor->InvokeEvent(vtkCommand::LeftButtonPressEvent);
      break;
    }
    case Qt::RightButton:{
      __interactor->InvokeEvent(vtkCommand::RightButtonPressEvent);
      break;
    }
    case Qt::MidButton:{
      __interactor->InvokeEvent(vtkCommand::MiddleButtonPressEvent);
      break;
    }
    default:{
      std::cerr << "not implemented\n";
    }
    } 
  }

  /** 
   * Passes mouse release event to vtk
   * 
   * @param event the mouse event
   */
  void mouseReleaseEvent(QMouseEvent* event)
  {
    __interactor->SetEventInformationFlipY(event->x(),
					   event->y(),
					   (event->state() & ControlButton),
					   (event->state() & ShiftButton));
    switch (event->button()) {
    case Qt::LeftButton:{
      __interactor->InvokeEvent(vtkCommand::LeftButtonReleaseEvent);
      break;
    }
    case Qt::RightButton:{
      __interactor->InvokeEvent(vtkCommand::RightButtonReleaseEvent);
      break;
    }
    case Qt::MidButton:{
      __interactor->InvokeEvent(vtkCommand::MiddleButtonReleaseEvent);
      break;
    }
    default:{
      std::cerr << "not implemented\n";
    }
    }
  }

  void keyPressEvent(QKeyEvent* event)
  {
    __interactor->SetKeyEventInformation(event->state() & ControlButton, 
					 event->state() & ShiftButton, 
					 event->ascii(), 
					 event->count());
    __interactor->InvokeEvent(vtkCommand::KeyPressEvent, NULL);
    __interactor->InvokeEvent(vtkCommand::CharEvent, NULL);
  }

  void keyReleaseEvent(QKeyEvent* event)
  {
    __interactor->SetKeyEventInformation(event->state() & ControlButton, 
					 event->state() & ShiftButton, 
					 event->ascii(), 
					 event->count());
    __interactor->InvokeEvent(vtkCommand::KeyReleaseEvent, NULL);
    __interactor->InvokeEvent(vtkCommand::CharEvent, NULL);    
  }

  /** 
   * The constructor
   * 
   * @param parent the parent widget
   * @param name the widget name
   * 
   */
  QGLvtkWindow( QWidget* parent = 0, const char* name = 0);

  /** 
   * Destructor
   * 
   */
  ~QGLvtkWindow();
};


#endif // QGL_VTK_WINDOW_HPP
