//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: BoundaryConditionCommonFEMDiscretization.hpp,v 1.8 2005/01/29 20:20:33 delpinux Exp $


#ifndef BOUNDARY_CONDITION_COMMON_FEM_DISCRETIZATION_HPP
#define BOUNDARY_CONDITION_COMMON_FEM_DISCRETIZATION_HPP

#include <FiniteElementTraits.hpp>

#include <BoundaryConditionDiscretization.hpp>

#include <SurfaceMeshGenerator.hpp>

#include <SurfaceMeshOfQuadrangles.hpp>
#include <SurfaceMeshOfTriangles.hpp>

#include <MatrixManagement.hpp>

#include <AutoPointer.hpp>
#include <ConformTransformation.hpp>

#include <BoundaryMeshAssociation.hpp>
#include <BoundaryConditionSurfaceMeshAssociation.hpp>

#include <Stringify.hpp>
#include <ErrorHandler.hpp>

/**
 * @file   BoundaryConditionCommonFEMDiscretization.hpp
 * @author Stephane Del Pino
 * @date   Sun Nov 24 17:37:59 2002
 * 
 * @brief Standard templates for finite element boundary condition
 * discretization
 * 
 */
template <typename MeshType>
class BoundaryConditionCommonFEMDiscretization
  : public BoundaryConditionDiscretization
{
public:
  typedef
  typename MeshType::CellType
  CellType;			/**< Geometry of the finite
				     element */

  typedef
  typename FiniteElementTraits<CellType>::Type
  FiniteElement;		/**< The finite element type */

  typedef
  typename FiniteElementTraits<CellType>::Transformation
  ConformTransformation;	/**< Conform tranformation */

protected:
  const MeshType& __mesh;	/**< The mesh */

  ReferenceCounting<BoundaryConditionSurfaceMeshAssociation>
  __bcMeshAssociation;		/**< The boundary mesh association */

  /** 
   * Checks if boundary mesh association is correct
   * 
   * @param bma the BoundaryMeshAssociation
   */
  void __checkBoundaryMeshAssociation(const BoundaryMeshAssociation& bma)
  {
    for (BoundaryMeshAssociation::const_iterator i = bma.begin();
	 i != bma.end(); ++i) {
      if (i->second == 0) {
	throw ErrorHandler(__FILE__,__LINE__,
			   "The mesh of boundary "
			   +stringify(*(i->first))+" was not generated",
			   ErrorHandler::unexpected);
      } else {
	const SurfaceMesh& s = *(i->second);
	if (not(s.isAssociatedTo(__mesh))) {
	  throw ErrorHandler(__FILE__,__LINE__,
			     "The mesh of boundary "
			     +stringify(*(i->first))
			     +" is not related to the volume mesh"
			     +"(check your mesh and/or report the error)",
			     ErrorHandler::normal);
	}
      }
    }
  }

  /** 
   * Buils the BoundaryConditionSurfaceMeshAssociation using a given
   * boundary and mesh association
   * 
   * @param bma the BoundaryMeshAssociation
   */
  void __associatesDefinedMeshToBoundaryConditions(const BoundaryMeshAssociation& bma)
  {
    __bcMeshAssociation = new BoundaryConditionSurfaceMeshAssociation(__problem,bma);
  }

public:
  /** 
   * Associates meshes to boundary conditions
   * 
   */
  virtual void associatesMeshesToBoundaryConditions()
  {
    BoundaryMeshAssociation bma(__problem, __mesh);
    this->__checkBoundaryMeshAssociation(bma);
    this->__associatesDefinedMeshToBoundaryConditions(bma);
  }

  /** 
   * Read only access to the mesh
   * 
   * @return __mesh
   */
  const MeshType& mesh() const
  {
    return __mesh;
  }

  /** 
   * Access to the mesh
   * 
   * @return __mesh
   */
  MeshType& mesh()
  {
    return __mesh;
  }

  /** 
   * Constructs Boundary condition discretization using a problem, a
   * mesh and a set of degrees of freedom
   * 
   * @param problem the given problem
   * @param aMesh a mesh
   * @param dof a set of degrees of freedom
   * 
   */
  BoundaryConditionCommonFEMDiscretization(const Problem& problem,
					   const MeshType& aMesh,
					   const DegreeOfFreedomSet& dof)
    : BoundaryConditionDiscretization(problem,dof),
      __mesh(aMesh)
  {
    ;
  }

  /** 
   * Destructor
   * 
   */
  virtual ~BoundaryConditionCommonFEMDiscretization()
  {
    ;
  }

  //! Here comes the template structures
protected:
  template <typename MatrixType>
  class __SetVariationalBoundaryConditionsAlphaUV
  {
  private:
    const UserFunction& __Alpha;

    const size_t __equationNumber;

    const size_t __unknownNumber;

    MatrixType& __A;

    const BoundaryConditionCommonFEMDiscretization<MeshType>& __bc;

  public:

    template <typename SurfaceMeshType>
    void eval(const SurfaceMeshType& surfaceMesh) const
    {
      typedef typename SurfaceMeshType::CellType SurfaceCellType;
      typedef typename MeshType::CellType CellType;

      typedef
	typename FiniteElementTraits<SurfaceCellType>::Transformation
	SurfaceConformTransformation;

      const FiniteElement& finiteElement
	= FiniteElement::instance();

      AutoPointer<ConformTransformation> pT;

      const CellType* currentCell = 0;

      for (typename SurfaceMeshType::const_iterator icell(surfaceMesh);
	   not(icell.end()); ++icell) {
	const SurfaceCellType& cell = *icell;
	const SurfaceConformTransformation CT(cell);

	const CellType& K
	  = static_cast<const CellType&>(cell.mother());

	if(currentCell != &K) {
	  pT = new ConformTransformation(K);
	  currentCell = &K;
	}

	const ConformTransformation& T = *pT;

	for (size_t k=0; k<CT.numberOfQuadratureVertices(); k++) {
	  TinyVector<3> q = CT.quadratureVertex(k);
	  const real_t weight = CT.quadratureWeight(k); // Suface of SurfElem

	  TinyVector<3> coordinates;
	  T.invertT(q, coordinates);

	  const real_t AlphaValue = __Alpha(q);

	  typename FiniteElement::ElementaryVector W;
	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++)
	    W[l] = finiteElement.W(l,coordinates);

	  typename FiniteElement::ElementaryMatrix WiWj;

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++)
	    for (size_t m=0; m<FiniteElement::numberOfDegreesOfFreedom; m++)
	      WiWj(l,m) = W[l] * W[m];

	  size_t indicesI[FiniteElement::numberOfDegreesOfFreedom];
	  size_t indicesJ[FiniteElement::numberOfDegreesOfFreedom];
	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	    indicesI[l] = __bc.__degreeOfFreedomSet(__unknownNumber,
						    __bc.mesh().vertexNumber(K(l)));
	    indicesJ[l] = __bc.__degreeOfFreedomSet(__equationNumber,
						    __bc.mesh().vertexNumber(K(l)));
	  }

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) {
	    const size_t I = indicesI[l];
	    for (size_t m=0; m<FiniteElement::numberOfDegreesOfFreedom; m++) {
	      const size_t J = indicesJ[m];
	      __A(I,J) += weight * AlphaValue * WiWj(l,m);
	    }
	  }
	}
      }
    }

    __SetVariationalBoundaryConditionsAlphaUV(const UserFunction& alpha,
					      const size_t equationNumber,
					      const size_t unknownNumber,
					      MatrixType& A,
					      const BoundaryConditionCommonFEMDiscretization<MeshType>& bc)
      : __Alpha(alpha),
	__equationNumber(equationNumber),
	__unknownNumber(unknownNumber),
	__A(A),
	__bc(bc)
    {
      ;
    }

    __SetVariationalBoundaryConditionsAlphaUV(const __SetVariationalBoundaryConditionsAlphaUV& S)
      : __Alpha(S.__Alpha),
	__equationNumber(S.__equationNumber),
	__unknownNumber(S.__unknownNumber),
	__A(S.__A),
	__bc(S.__bc)
    {
      ;
    }

    ~__SetVariationalBoundaryConditionsAlphaUV()
    {
      ;
    }
  };

  class __SetSecondMemberDirichlet
  {
  private:
    const Dirichlet& __dirichlet;

    const size_t __equationNumber;

    const BoundaryConditionCommonFEMDiscretization& __bc;

  public:

    template <typename SurfaceMeshType>
    void eval(const SurfaceMeshType& surfMesh) const
    {
      typedef FiniteElementTraits<typename SurfaceMeshType::CellType>
	FEType;

      typedef typename FEType::Transformation ConformTransformation;

      for (typename SurfaceMeshType::const_iterator icell(surfMesh);
	   not(icell.end()); ++icell) {
	const typename SurfaceMeshType::CellType& cell = *icell;

	for (size_t i=0; i<SurfaceMeshType::CellType::NumberOfVertices; ++i) {
	  const Vertex& V = cell(i);
	  const size_t I = __bc.__degreeOfFreedomSet(__equationNumber,
						     __bc.mesh().vertexNumber(V));
	  if (not(__bc.__dirichletList[I])) {
	    const real_t GValue = __dirichlet.g(V);
	    __bc.__dirichletValues[I] = GValue;
	    __bc.__dirichletList[I] = true;
	  }
	}
      }
    }

    __SetSecondMemberDirichlet(const Dirichlet &D ,
			       const size_t equationNumber,
			       const BoundaryConditionCommonFEMDiscretization& bc)
      : __dirichlet(D),
	__equationNumber(equationNumber),
	__bc(bc)
    {
      ;
    }

    __SetSecondMemberDirichlet(const __SetSecondMemberDirichlet& s)
      : __dirichlet(s.__dirichlet),
	__equationNumber(s.__equationNumber),
	__bc(s.__bc)
    {
      ;
    }

    ~__SetSecondMemberDirichlet()
    {
      ;
    }
  };

  template <typename VectorType>
  class __GetDiagonalNaturalBoundaryConditions
  {
  private:
    const UserFunction& __Alpha;

    const size_t __equationNumber;

    VectorType& __z;

    const BoundaryConditionCommonFEMDiscretization& __bc;

  public:
    template <typename SurfaceMeshType>
    void eval(const SurfaceMeshType& surfaceMesh) const
    {
      typedef typename SurfaceMeshType::CellType SurfaceCellType;
      typedef typename MeshType::CellType CellType;

      typedef
	typename FiniteElementTraits<SurfaceCellType>::Transformation
	SurfaceConformTransformation;

      const FiniteElement& finiteElement
	= FiniteElement::instance();

      AutoPointer<ConformTransformation> pT;

      const CellType* currentCell = 0;

      for (typename SurfaceMeshType::const_iterator icell(surfaceMesh);
	   not(icell.end()); ++icell) {
	const SurfaceCellType& cell = *icell;
	const SurfaceConformTransformation CT(cell);

	const CellType& K
	  = static_cast<const CellType&>(cell.mother());

	if(currentCell != &K) {
	  pT = new ConformTransformation(K);
	  currentCell = &K;
	}

	const ConformTransformation& T = *pT;

	for (size_t k=0; k<CT.numberOfQuadratureVertices(); k++) {
	  TinyVector<3> q = CT.quadratureVertex(k);
	  const real_t weight = CT.quadratureWeight(k); // Suface of SurfElem

	  TinyVector<3> coordinates;
	  T.invertT(q, coordinates);

	  const real_t AlphaValue = __Alpha(q);

	  typename FiniteElement::ElementaryVector W;

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) 
	    W[l] = finiteElement.W(l,coordinates);

	  size_t indices[FiniteElement::numberOfDegreesOfFreedom];
	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	    indices[l] = __bc.__degreeOfFreedomSet(__equationNumber,
						   __bc.mesh().vertexNumber(K(l)));
	  }

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) {
	    const size_t& I = indices[l];
	    __z[I] += W[l] * W[l] * weight * AlphaValue;
	  }
	}
      }
    }

    __GetDiagonalNaturalBoundaryConditions(const UserFunction& alpha,
					   const size_t equationNumber,
					   VectorType& z,
					   const BoundaryConditionCommonFEMDiscretization& bc)
      : __Alpha(alpha),
	__equationNumber(equationNumber),
	__z(z),
	__bc(bc)
    {
      ;
    }

    __GetDiagonalNaturalBoundaryConditions(const __GetDiagonalNaturalBoundaryConditions& G)
      : __Alpha(G.__Alpha),
	__equationNumber(G.__equationNumber),
	__z(G.__z),
	__bc(G.__bc)
    {
      ;
    }

    ~__GetDiagonalNaturalBoundaryConditions()
    {
      ;
    }
  };

  template <typename VectorType>
  class __SetVariationalBoundaryConditionsFV
  {
  private:
    const UserFunction& __g;

    const size_t __equationNumber;

    VectorType& __b;

    const BoundaryConditionCommonFEMDiscretization<MeshType>& __bc;

  public:

    template <typename SurfaceMeshType>
    void eval(const SurfaceMeshType& surfaceMesh) const
    {
      typedef typename SurfaceMeshType::CellType SurfaceCellType;
      typedef typename MeshType::CellType CellType;

      typedef
	typename FiniteElementTraits<SurfaceCellType>::Transformation
	SurfaceConformTransformation;

      const FiniteElement& finiteElement
	= FiniteElement::instance();

      AutoPointer<ConformTransformation> pT;

      const CellType* currentCell = 0;

      for (typename SurfaceMeshType::const_iterator icell(surfaceMesh);
	   not(icell.end()); ++icell) {
	const SurfaceCellType& cell = *icell;
	const SurfaceConformTransformation CT(cell);

	const CellType& K
	  = static_cast<const CellType&>(cell.mother());

	if (currentCell != &K) {
	    pT = new ConformTransformation(K);
	    currentCell = &K;
	}

	const ConformTransformation& T = *pT;

	for (size_t k=0; k<CT.numberOfQuadratureVertices(); k++) {
	  TinyVector<3> q = CT.quadratureVertex(k);
	  const real_t weight = CT.quadratureWeight(k); // Suface of SurfElem

	  TinyVector<3> coordinates;
	  T.invertT(q, coordinates);

	  const real_t GValue = __g(q);

	  typename FiniteElement::ElementaryVector W;
	  size_t indices[FiniteElement::numberOfDegreesOfFreedom];

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	    W[l] = finiteElement.W(l,coordinates);
	    indices[l] = __bc.mesh().vertexNumber(K(l));
	  }

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) {
	    const size_t I = __bc.__degreeOfFreedomSet(__equationNumber,
						       indices[l]);
	    __b[I] += weight * GValue * W[l];
	  }
	}
      }
    }

    __SetVariationalBoundaryConditionsFV(const UserFunction& g,
					 const size_t equationNumber,
					 VectorType& b,
					 const BoundaryConditionCommonFEMDiscretization& bc)
      : __g(g),
	__equationNumber(equationNumber),
	__b(b),
	__bc(bc)
    {
      ;
    }

    __SetVariationalBoundaryConditionsFV(const __SetVariationalBoundaryConditionsFV& S)
      : __g(S.__g),
	__equationNumber(S.__equationNumber),
	__b(S.__b),
	__bc(S.__bc)
    {
      ;
    }

    ~__SetVariationalBoundaryConditionsFV()
    {
      ;
    }
  };

  template <typename VectorType>
  class __VariationalBoundaryConditionsAlphaUVTimesX
  {
  private:
    const UserFunction& __Alpha;

    const size_t __equationNumber;

    const size_t __unknownNumber;

    const VectorType& __x;

    VectorType& __z;

    const BoundaryConditionCommonFEMDiscretization<MeshType>& __bc;

  public:


    template <typename SurfaceMeshType>
    void eval(SurfaceMeshType& surfaceMesh) const
    {
      typedef typename SurfaceMeshType::CellType SurfaceCellType;
      typedef typename MeshType::CellType CellType;

      typedef
	typename FiniteElementTraits<SurfaceCellType>::Transformation
	SurfaceConformTransformation;

      const FiniteElement& finiteElement
	= FiniteElement::instance();

      AutoPointer<ConformTransformation> pT;

      const CellType* currentCell = 0;

      for (typename SurfaceMeshType::const_iterator icell(surfaceMesh);
	   not(icell.end()); ++icell) {
	const SurfaceCellType& cell = *icell;
	const SurfaceConformTransformation CT(cell);

	const CellType& K = static_cast<const CellType&>(cell.mother());

	if(currentCell != &K) {
	  pT = new ConformTransformation(K);
	  currentCell = &K;
	}

	const ConformTransformation& T = *pT;

	for (size_t k=0; k<CT.numberOfQuadratureVertices(); k++) {
	  TinyVector<3> q = CT.quadratureVertex(k);

	  const real_t weight = CT.quadratureWeight(k); // Suface of SurfElem

	  TinyVector<3> coordinates;
	  T.invertT(q, coordinates);

	  const real_t AlphaValue = __Alpha(q);

	  typename FiniteElement::ElementaryVector W;
	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) 
	    W[l] = finiteElement.W(l,coordinates);

	  typename FiniteElement::ElementaryMatrix WiWj;

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++)
	    for (size_t m=0; m<FiniteElement::numberOfDegreesOfFreedom; m++)
	      WiWj(l,m) = W[l] * W[m];

	  size_t indicesI[FiniteElement::numberOfDegreesOfFreedom];
	  size_t indicesJ[FiniteElement::numberOfDegreesOfFreedom];
	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	    indicesI[l] = __bc.__degreeOfFreedomSet(__unknownNumber,
						    __bc.mesh().vertexNumber(K(l)));
	    indicesJ[l] = __bc.__degreeOfFreedomSet(__equationNumber,
						    __bc.mesh().vertexNumber(K(l)));
	  }
	    
	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) {
	    const size_t I = indicesI[l];
	    for (size_t m=0; m<FiniteElement::numberOfDegreesOfFreedom; m++) {
	      const size_t J = indicesJ[m];
	      __z[J] += WiWj(l,m) * weight * AlphaValue * __x[I];
	    }
	  }
	}
      }
    }

    __VariationalBoundaryConditionsAlphaUVTimesX(const UserFunction& alpha,
						 const size_t& equationNumber,
						 const size_t& unknownNumber,
						 const VectorType& x,
						 VectorType& z,
						 const BoundaryConditionCommonFEMDiscretization<MeshType>& bc)
      : __Alpha(alpha),
	__equationNumber(equationNumber),
	__unknownNumber(unknownNumber),
	__x(x),
	__z(z),
	__bc(bc)
    {
      ;
    }

    __VariationalBoundaryConditionsAlphaUVTimesX(const __VariationalBoundaryConditionsAlphaUVTimesX& V)
      : __Alpha(V.__Alpha),
	__equationNumber(V.__equationNumber),
	__unknownNumber(V.__unknownNumber),
	__x(V.__x),
	__z(V.__z),
	__bc(V.__bc)
    {
      ;
    }

    ~__VariationalBoundaryConditionsAlphaUVTimesX()
    {
      ;
    }
  };

  template <typename FunctionClass>
  static void __meshWrapper(const SurfaceMesh& surfmesh, const FunctionClass& F)
  {
    switch (surfmesh.type()) {
    case Mesh::surfaceMeshTriangles: {
      const SurfaceMeshOfTriangles& surfTria
	= dynamic_cast<const SurfaceMeshOfTriangles&>(surfmesh);
      F.eval(surfTria);
      break;
    }
    case Mesh::surfaceMeshQuadrangles: {
      const SurfaceMeshOfQuadrangles& surfQuad
	= dynamic_cast<const SurfaceMeshOfQuadrangles&>(surfmesh);
      F.eval(surfQuad);
      break;
    }
    default: {
      throw ErrorHandler(__FILE__,__LINE__,
			 "unknown surface mesh type",
			 ErrorHandler::unexpected);
    }
    }
  }



  template <typename VectorType>
  static void
  __StandardGetDiagonalVariationalBorderBilinearOperator(const BoundaryConditionCommonFEMDiscretization<MeshType>& __bc,
							 VectorType& z)
  {
    const BoundaryConditionSurfaceMeshAssociation& bcMeshAssociation = *__bc.__bcMeshAssociation;

    for (BoundaryConditionSurfaceMeshAssociation
	   ::BilinearBorderOperatorMeshAssociation
	   ::const_iterator i = bcMeshAssociation.beginOfBilinear();
	 i != bcMeshAssociation.endOfBilinear(); ++i) {

      switch ((*(*i).first).type()) {
      case VariationalBilinearBorderOperator::alphaUV: {
	const VariationalBorderOperatorAlphaUV& V
	  = dynamic_cast<const VariationalBorderOperatorAlphaUV&>(*(*i).first);

	const SurfaceMesh& surfMesh = *((*i).second);

	BoundaryConditionCommonFEMDiscretization<MeshType>
	  ::__meshWrapper(surfMesh,
			  typename BoundaryConditionCommonFEMDiscretization<MeshType>
			  ::template __GetDiagonalNaturalBoundaryConditions<VectorType> (V.alpha(),
											 V.testFunctionNumber(), z,
											 __bc));
	break;
      }
      default: {
	throw ErrorHandler(__FILE__,__LINE__,
			   "not implemented",
			   ErrorHandler::unexpected);
      }
      }
    }
  }

  template <typename VectorType>
  static void
  __StandardGetDiagonalDirichletBorderBilinearOperator(const BoundaryConditionCommonFEMDiscretization<MeshType>& bc,
						       VectorType& z)
  {
    for (size_t i=0; i<bc.__dirichletList.size(); ++i) {
      if (bc.__dirichletList[i]) {
	z[i] = 1;
      }
    }
  }

  template <typename MatrixType>
  static void
  __StandardVariationalBorderBilinearOperator(const BoundaryConditionCommonFEMDiscretization<MeshType>& __bc,
					      MatrixType& A)
  {
    const BoundaryConditionSurfaceMeshAssociation& bcMeshAssociation = *__bc.__bcMeshAssociation;

    for (BoundaryConditionSurfaceMeshAssociation
	   ::BilinearBorderOperatorMeshAssociation
	   ::const_iterator i = bcMeshAssociation.beginOfBilinear();
	 i != bcMeshAssociation.endOfBilinear(); ++i) {

      switch ((*(*i).first).type()) {
      case VariationalBilinearBorderOperator::alphaUV: {
	const VariationalBorderOperatorAlphaUV& V
	  = dynamic_cast<const VariationalBorderOperatorAlphaUV&>(*(*i).first);

	const SurfaceMesh& surfMesh = *((*i).second);

	BoundaryConditionCommonFEMDiscretization<MeshType>
	  ::__meshWrapper(surfMesh,
			  typename BoundaryConditionCommonFEMDiscretization<MeshType>
			  ::template __SetVariationalBoundaryConditionsAlphaUV<MatrixType> (V.alpha(),
											    V.testFunctionNumber(),
											    V.unknownNumber(),
											    A,
											    __bc));
	break;
      }
      default: {
	throw ErrorHandler(__FILE__,__LINE__,
			   "not implemented",
			   ErrorHandler::unexpected);
      }
      }
    }
  }

  template <typename MatrixType>
  static void
  __StandardDirichletBorderBilinearOperator(const BoundaryConditionCommonFEMDiscretization<MeshType>& __bc,
					    MatrixType& A)

  {
    for (size_t i=0; i<__bc.__dirichletList.size(); ++i) {
      if (__bc.__dirichletList[i]) {
	for (DoubleHashedMatrix::iterator lineBrowser = A.beginOfLine(i);
	     lineBrowser != A.endOfLine(i); ++lineBrowser) {
#warning should use a syntax like "M.line(i) = 0;"
	  lineBrowser.second() = 0;
	}
	// makes the column i to be zero. Here the elimination modifies the second member.
	for (DoubleHashedMatrix::iterator columnBrowser = A.beginOfColumn(i);
	     columnBrowser != A.endOfColumn(i); ++columnBrowser) {
	  columnBrowser.second() = 0;
	}
	A(i,i) = 1.; // sets the diagonal to 1
      }
    }
  }

  template <typename VectorType>
  static void
  __StandardVariationalBorderLinearOperator(const BoundaryConditionCommonFEMDiscretization<MeshType>& __bc,
					    VectorType& b)
  {
    const BoundaryConditionSurfaceMeshAssociation& bcMeshAssociation = *__bc.__bcMeshAssociation;

    for (BoundaryConditionSurfaceMeshAssociation
	   ::LinearBorderOperatorMeshAssociation
	   ::const_iterator i = bcMeshAssociation.beginOfLinear();
	 i != bcMeshAssociation.endOfLinear(); ++i) {

      switch ((*(*i).first).type()) {
      case VariationalLinearBorderOperator::FV: {
	const VariationalBorderOperatorFV& V
	  = dynamic_cast<const VariationalBorderOperatorFV&>(*(*i).first);

	const SurfaceMesh& surfMesh = *((*i).second);

	BoundaryConditionCommonFEMDiscretization<MeshType>
	  ::__meshWrapper(surfMesh,
			  typename BoundaryConditionCommonFEMDiscretization<MeshType>
			  ::template __SetVariationalBoundaryConditionsFV<VectorType> (V.f(),
										       V.testFunctionNumber(), b,
										       __bc));
	break;
      }
      default: {
	throw ErrorHandler(__FILE__,__LINE__,
			   "not implemented",
			   ErrorHandler::unexpected);
      }
      }
    }
  }

  template <typename MatrixType,
	    typename VectorType>
  static void
  __StandardDirichletBorderLinearOperator(const BoundaryConditionCommonFEMDiscretization<MeshType>& bc,
					  MatrixType& A,
					  VectorType& b)
  {
    bc.__dirichletValues.resize(bc.__degreeOfFreedomSet.size());
    bc.__dirichletValues = 0;
    const BoundaryConditionSurfaceMeshAssociation& bcMeshAssociation = *bc.__bcMeshAssociation;

    // Dirichlet on the mesh border ...
    for (size_t i=0; i<bc.problem().numberOfUnknown(); ++i) {
      for (BoundaryConditionSurfaceMeshAssociation
	     ::DirichletMeshAssociation
	     ::const_iterator ibcMesh = bcMeshAssociation.bc(i).begin();
	   ibcMesh != bcMeshAssociation.bc(i).end();
	   ++ibcMesh) {
	const Dirichlet& D = *(*ibcMesh).first;
	const SurfaceMesh& surfMesh = *(*ibcMesh).second;

	BoundaryConditionCommonFEMDiscretization<MeshType>
	  ::__meshWrapper(surfMesh,
			  typename BoundaryConditionCommonFEMDiscretization<MeshType>
			  ::__SetSecondMemberDirichlet(D, i, bc));
      }
    }
    if (A.type() != BaseMatrix::unAssembled) {
      //! proceed to clean elimination
      Vector<real_t> y (bc.__dirichletValues.size());
      A.timesX(bc.__dirichletValues,y);
      b -= y;
    }

    for (size_t i=0; i<b.size(); ++i) {
      if (bc.__dirichletList[i]) {
	b[i] = bc.__dirichletValues[i];
      }
    }
  }

  template <typename VectorType>
  static void
  __StandardVariationalBorderBilinearOperatorTimesX(const BoundaryConditionCommonFEMDiscretization<MeshType>& __bc,
						    const VectorType& x,
						    VectorType& z)
  {
    const BoundaryConditionSurfaceMeshAssociation& bcMeshAssociation = *__bc.__bcMeshAssociation;
    for (BoundaryConditionSurfaceMeshAssociation
	   ::BilinearBorderOperatorMeshAssociation
	   ::const_iterator i = bcMeshAssociation.beginOfBilinear();
	 i != bcMeshAssociation.endOfBilinear(); ++i) {

      switch ((*(*i).first).type()) {
      case VariationalBilinearBorderOperator::alphaUV: {

	const VariationalBorderOperatorAlphaUV& V
	  = dynamic_cast<const VariationalBorderOperatorAlphaUV&>(*(*i).first);

	const SurfaceMesh& surfMesh = *((*i).second);

	BoundaryConditionCommonFEMDiscretization<MeshType>
	  ::__meshWrapper(surfMesh,
			  __VariationalBoundaryConditionsAlphaUVTimesX<VectorType>(V.alpha(),
										   V.testFunctionNumber(),
										   V.unknownNumber(),
										   x,
										   z,
										   __bc));
	break;
      }
      default: {
	throw ErrorHandler(__FILE__,__LINE__,
			   "not implemented",
			   ErrorHandler::unexpected);
      }
      }
    }
  }

  template <typename VectorType>
  static void
  __StandardDirichletBorderBilinearOperatorTimesX(const BoundaryConditionCommonFEMDiscretization<MeshType>& bc,
						  const VectorType& x,
						  VectorType& z)
  {
    for (size_t i=0; i<bc.__dirichletList.size(); ++i) {
      if (bc.__dirichletList[i]) {
	z[i] = x[i];
      }
    }
  }
};

#endif // BOUNDARY_CONDITION_COMMON_FEM_DISCRETIZATION_HPP

