//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Mesh.hpp,v 1.15 2005/02/02 21:40:15 delpinux Exp $

#ifndef MESH_HPP
#define MESH_HPP

#include <Vector.hpp>

#include <ReferenceCounting.hpp>

#include <VerticesSet.hpp>
#include <VerticesCorrespondance.hpp>
#include <EdgesSet.hpp>

#include <Cell.hpp>
#include <Edge.hpp>

#include <string>

template <typename MeshType>
class MeshExtractor;

class Mesh
{
public:
  typedef enum {
    cartesianHexahedraMesh,
    hexahedraMesh,
    tetrahedraMesh,
    surfaceMeshTriangles,
    surfaceMeshQuadrangles,
  } Type;

  // Family
  typedef enum {
    volume,
    surface
  } Family;

private:
  //! Mesh extractors are friends
  template <typename MeshType> friend class MeshExtractor;

  //! The type of the mesh
  const Type __type;
  const Family __family;
protected:

  template <typename MeshType,
	    typename CellType = typename MeshType::CellType>
  class T_iterator
  {
  private:
    MeshType* __mesh;
    CellType* __iterator;
    size_t __number;

  public:
    typedef enum {
      Begin,
      End
    } Position;

    const size_t& number() const
    {
      return __number;
    }

    inline CellType& operator*()
    {
      return *__iterator;
    }

    inline CellType* pointer() const
    {
      return __iterator;
    }

    inline bool end() const
    {
      return (__number >= __mesh->numberOfCells());
    }

    inline bool operator<(const T_iterator<MeshType, CellType>& i) const
    {
      return (__number < i.__number);
    }

    const T_iterator<MeshType, CellType>&
    operator=(const T_iterator<MeshType, CellType>& i)
    {
      __mesh = i.__mesh;
      __iterator = i.__iterator;
      __number = i.__number;

      return *this;
    }

    const T_iterator<MeshType, CellType>&
    operator=(CellType* iterator)
    {
      __iterator = iterator;
      __number = static_cast<size_t>(__iterator-&(__mesh->cell(0)));

      return *this;
    }

    inline T_iterator<MeshType, CellType> operator++(int)
    {
      T_iterator<MeshType, CellType> i = *this;
      ++(*this);
      return i;
    }

    inline T_iterator<MeshType, CellType> operator++()
    {
      for(;;) {
	++__number;
	if (this->end()) {
	  break;
	}
	++__iterator;
	if (not(__iterator->isFictitious())) {
	  break;
	}
      }
      return *this;
    }

    T_iterator(MeshType& m,
	       T_iterator::Position position = T_iterator::Begin)
      : __mesh(&m),
	__number(0)
    {
      if ((m.numberOfCells() == 0)
	  or (position == T_iterator<MeshType, CellType>::End)) {
	__number = __mesh->numberOfCells();
      } else {
	__iterator = & m.cell(0);
	if (__iterator->isFictitious()) {
	  ++(*this);
	}
      }
    }

    T_iterator(MeshType& m,
	       const size_t& cellNumber)
      : __mesh(&m),
	__number(cellNumber)
    {
      if (cellNumber < m.numberOfCells()) {
	__iterator = & m.cell(cellNumber);
      }
    }

    T_iterator(const T_iterator<MeshType, CellType>& i)
      : __mesh(i.__mesh),
	__iterator(i.__iterator),
	__number(i.__number)
    {
      ;
    }

    ~T_iterator()
    {
      ;
    }
  };

  //! Container for vertices.
  ReferenceCounting<VerticesSet> __verticesSet;

  //! Container for vertices correspondances.
  ReferenceCounting<VerticesCorrespondance> __verticesCorrespondance;

  //! Container for edges.
  ReferenceCounting<EdgesSet> __edgesSet;

  //! Copies of meshes are almost forbidden
  Mesh(const Mesh& M)
    : __type(M.__type),
      __family(M.__family),
      __verticesSet(M.__verticesSet),
      __edgesSet(M.__edgesSet)
  {
    ;
  }

public:
  //! Returns \p true if the point \a p is inside the mesh.
  virtual bool inside(const real_t& x,
		      const real_t& y,
		      const real_t& z) const =0;

  //! Returns \p true if the point \a p is inside the mesh.
  virtual bool inside(const TinyVector<3>& p) const = 0;

  //! The type of the mesh.
  const Mesh::Type& type() const
  {
    return __type;
  }

  //! The family of the mesh.
  const Mesh::Family& family() const
  {
    return __family;
  }

  /** 
   * Returns the number of the vertex v in the list
   * 
   * @param v a vertex
   * 
   * @return the vertex \a v number
   * 
   * @note polymorphism is not used for clarity reason
   */
  inline size_t vertexNumber(const Vertex& v) const
  {
    return (*__verticesSet).number(v);
  }

  /** 
   *  Returns the number of the Edge e in the list
   * 
   * @param e an Edge
   * 
   * @return the edge \a E number
   * 
   * @note polymorphism is not used for clarity reason
   */
  inline size_t edgeNumber(const Edge& e) const
  {
    return (*__edgesSet).number(e);
  }

  //! Read-only access to the number of vertices.
  inline const size_t& numberOfVertices() const
  {
    return (*__verticesSet).numberOfVertices();
  }

  /** 
   * Changes the size of the vertices container.
   * 
   * @param size vertices set new size
   * @bug this function should not be allowed!
   */
  inline void setNumberOfVertices(const size_t& size)
  {
    if (__verticesSet == 0) {
      __verticesSet = new VerticesSet(size);
      __verticesCorrespondance = new VerticesCorrespondance(size);
    } else {
      (*__verticesSet).setNumberOfVertices(size);
      __verticesCorrespondance = new VerticesCorrespondance(size);
    }
  }

  virtual const size_t& numberOfCells() const = 0;

  /** 
   * Sets the edges set a posteriori
   * 
   * @param e the edge set
   * @remark it up to the user to manage the consistency between data
   */
  inline void setEdges(ReferenceCounting<EdgesSet> e)
  {
    __edgesSet = e;
  }

  //! Read-only access to the number of edges.
  inline size_t numberOfEdges() const
  {
    return (*__edgesSet).numberOfEdges();
  }

  /** 
   * Access to the vertices set of the mesh
   * 
   * @return __verticesSet
   */
  inline ReferenceCounting<VerticesSet> verticesSet()
  {
    return __verticesSet;
  }

  /** 
   * Read only access to the vertices set of the mesh
   * 
   * @return __verticesSet
   */
  inline ConstReferenceCounting<VerticesSet> verticesSet() const
  {
    return __verticesSet;
  }


  /** 
   * Access to the vertices correspondance table
   * 
   * @return __verticesCorrespondance
   */
  inline ReferenceCounting<VerticesCorrespondance> verticesCorrespondance()
  {
    return __verticesCorrespondance;
  }

  /** 
   * Read only access to the vertices correspondance table
   * 
   * @return __verticesCorrespondance
   */
  inline ConstReferenceCounting<VerticesCorrespondance> verticesCorrespondance() const
  {
    return __verticesCorrespondance;
  }

  /** 
   * Read only access to the ith vertex of the mesh
   * 
   * @param i the number of the vertex to access
   * 
   * @return the ith vertex
   */
  inline const Vertex& vertex(const size_t& i) const
  {
    return (*__verticesSet)[i];
  }

  /** 
   * Access to the ith vertex of the mesh
   * 
   * @param i the number of the vertex to access
   * 
   * @return the ith vertex
   */
  inline Vertex& vertex(const size_t& i)
  {
    return (*__verticesSet)[i];
  }

  /** 
   * Read only access to the ith vertex "real" number (ie: when
   * dealing with periodic boundaries)
   * 
   * @param i the number of the vertex 
   * 
   * @return the vertex "real" number
   */
  inline const size_t& correspondance(const size_t& i) const
  {
    return (*__verticesCorrespondance)[i];
  }

  //! Read-only access to the Edge \a i.
  inline const Edge& edge(const size_t& i) const
  {
    return (*__edgesSet)[i];
  }

  //! Access to the Edge \a i.
  inline Edge& edge(const size_t& i)
  {
    return (*__edgesSet)[i];
  }

  Mesh(const Mesh::Type& type,
       const Mesh::Family& family,
       const size_t& numberOfVertices,
       const size_t& numberOfEdges)
    : __type(type),
      __family(family),
      __verticesSet(new VerticesSet(numberOfVertices)),
      __verticesCorrespondance(new VerticesCorrespondance(numberOfVertices)),
      __edgesSet(new EdgesSet(numberOfEdges))
  {
    ;
  }

  Mesh(const Mesh::Type& type,
       const Mesh::Family& family,
       ReferenceCounting<VerticesSet> vertices,
       ReferenceCounting<VerticesCorrespondance> correspondance,
       ReferenceCounting<EdgesSet> edges)
    : __type(type),
      __family(family),
      __verticesSet(vertices),
      __verticesCorrespondance(correspondance),
      __edgesSet(edges)
  {
    ;
  }

  virtual ~Mesh()
  {
    ;
  }
};

#endif // MESH_HPP

