//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: MeshTetrahedrizor.cpp,v 1.8 2005/01/29 20:20:33 delpinux Exp $

#include <MeshTetrahedrizor.hpp>

#include <MeshOfTetrahedra.hpp>
#include <Structured3DMesh.hpp>

#include <set>

template <>
void MeshTetrahedrizor::
__build<Structured3DMesh>(const bool& builtLocalizationTools)
{
  // In this function hexahedra are divide into 5 tetrahedra
  const Structured3DMesh& M
    = static_cast<const Structured3DMesh&>(*__input);

  // copies the set of vertices
  VerticesSet* pVerticesSet = new VerticesSet(*M.verticesSet());
  VerticesSet& verticesSet = *pVerticesSet;

  // copies vertices correspondances
  VerticesCorrespondance* pCorrespondance
    = new VerticesCorrespondance(*M.verticesCorrespondance());
  VerticesCorrespondance& correspondance = *pCorrespondance;

  // reserves the space to store the tetrahedra
  Vector<Tetrahedron>* pTetrahedra = new Vector<Tetrahedron>(5*M.numberOfCells());
  Vector<Tetrahedron>& tetrahedra = *pTetrahedra;

  size_t cellNumber=0;
  for(size_t i=0; i<M.shape().nx()-1; ++i)
    for(size_t j=0; j<M.shape().ny()-1; ++j)
      for(size_t k=0; k<M.shape().nz()-1; ++k) {
	const CartesianHexahedron& H = M.cell(i,j,k);

	if ( (i+j+k) % 2 == 0) {
	  Tetrahedron T0(verticesSet[M.vertexNumber(H(4))],
			 verticesSet[M.vertexNumber(H(0))],
			 verticesSet[M.vertexNumber(H(7))],
			 verticesSet[M.vertexNumber(H(5))],
			 H.reference());

	  Tetrahedron T1(verticesSet[M.vertexNumber(H(1))],
			 verticesSet[M.vertexNumber(H(2))],
			 verticesSet[M.vertexNumber(H(0))],
			 verticesSet[M.vertexNumber(H(5))],
			 H.reference());

	  Tetrahedron T2(verticesSet[M.vertexNumber(H(0))],
			 verticesSet[M.vertexNumber(H(2))],
			 verticesSet[M.vertexNumber(H(7))],
			 verticesSet[M.vertexNumber(H(5))],
			 H.reference());

	  Tetrahedron T3(verticesSet[M.vertexNumber(H(6))],
			 verticesSet[M.vertexNumber(H(2))],
			 verticesSet[M.vertexNumber(H(5))],
			 verticesSet[M.vertexNumber(H(7))],
			 H.reference());

	  Tetrahedron T4(verticesSet[M.vertexNumber(H(3))],
			 verticesSet[M.vertexNumber(H(0))],
			 verticesSet[M.vertexNumber(H(2))],
			 verticesSet[M.vertexNumber(H(7))],
			 H.reference());

	  tetrahedra[5*cellNumber  ] = T0;
	  tetrahedra[5*cellNumber+1] = T1;
	  tetrahedra[5*cellNumber+2] = T2;
	  tetrahedra[5*cellNumber+3] = T3;
	  tetrahedra[5*cellNumber+4] = T4;
	} else {
	  Tetrahedron T0(verticesSet[M.vertexNumber(H(0))],
			 verticesSet[M.vertexNumber(H(1))],
			 verticesSet[M.vertexNumber(H(3))],
			 verticesSet[M.vertexNumber(H(4))],
			 H.reference());

	  Tetrahedron T1(verticesSet[M.vertexNumber(H(5))],
			 verticesSet[M.vertexNumber(H(1))],
			 verticesSet[M.vertexNumber(H(4))],
			 verticesSet[M.vertexNumber(H(6))],
			 H.reference());

	  Tetrahedron T2(verticesSet[M.vertexNumber(H(1))],
			 verticesSet[M.vertexNumber(H(4))],
			 verticesSet[M.vertexNumber(H(6))],
			 verticesSet[M.vertexNumber(H(3))],
			 H.reference());

	  Tetrahedron T3(verticesSet[M.vertexNumber(H(2))],
			 verticesSet[M.vertexNumber(H(1))],
			 verticesSet[M.vertexNumber(H(6))],
			 verticesSet[M.vertexNumber(H(3))],
			 H.reference());

	  Tetrahedron T4(verticesSet[M.vertexNumber(H(7))],
			 verticesSet[M.vertexNumber(H(3))],
			 verticesSet[M.vertexNumber(H(6))],
			 verticesSet[M.vertexNumber(H(4))],
			 H.reference());

	  tetrahedra[5*cellNumber  ] = T0;
	  tetrahedra[5*cellNumber+1] = T1;
	  tetrahedra[5*cellNumber+2] = T2;
	  tetrahedra[5*cellNumber+3] = T3;
	  tetrahedra[5*cellNumber+4] = T4;
	}
	cellNumber++;
      }

  // Computes tetrahedra mesh edges
  std::set<Edge> edges;
  for (size_t i=0; i<tetrahedra.size(); ++i) {
    Tetrahedron& T = tetrahedra[i];
    edges.insert(Edge(T(0),T(1)));
    edges.insert(Edge(T(0),T(2)));
    edges.insert(Edge(T(0),T(3)));
    edges.insert(Edge(T(1),T(2)));
    edges.insert(Edge(T(1),T(3)));
    edges.insert(Edge(T(2),T(3)));
  }

  EdgesSet* edgesSet = new EdgesSet(edges.size());
  size_t n=0;
  for (std::set<Edge>::const_iterator i=edges.begin();
       i != edges.end(); ++i, ++n) {
    (*edgesSet)[n] = *i;
  }

//   std::vector<ReferenceCounting<SurfaceMeshOfTriangles> > faces;

//   faces.resize(M.numberOfSurfaceMeshes());

//   for (size_t i =0; i<M.numberOfSurfaceMeshes(); ++i) {
//     const SurfaceMeshOfQuadrangles& sq = M.surfaceMeshOfQuadrangles(i);
//     faces[i] = new SurfaceMeshOfTriangles(2 * sq.numberOfCells());
//     size_t quadrangleNumber = 0;
//     for (SurfaceMeshOfQuadrangles::const_iterator j=(sq);
// 	 not(j.end()); ++j) {

//       const Quadrangle& Q = *j;
//       Triangle T1(Q(0), Q(1), Q(2), Q.reference());
//       Triangle T2(Q(2), Q(3), Q(0), Q.reference());

//       (*faces[i]).cell(quadrangleNumber*2)     = T1;
//       (*faces[i]).cell(quadrangleNumber*2 + 1) = T2;
//       quadrangleNumber++;
//     }
//   }

  MeshOfTetrahedra* meshOfTetrahedra
    = new MeshOfTetrahedra(pVerticesSet,
			   pCorrespondance,
			   pTetrahedra,
			   0, // <= Border mesh of triangles
			   edgesSet,
			   builtLocalizationTools);

  __mesh = meshOfTetrahedra;
  std::cerr << __FILE__ << ':' << __LINE__ << ": WARNING implemention incomplete\n";
}

template<>
ReferenceCounting<MeshTetrahedrizor::CellMapping>
MeshTetrahedrizor::__motherCells<Structured3DMesh>()
{
  const Structured3DMesh& M
    = static_cast<const Structured3DMesh&>(*__input);

  // In this function hexahedra supposed to be divided into 5
  // tetrahedra
  MeshTetrahedrizor::CellMapping* pCellMapping
    = new MeshTetrahedrizor::CellMapping(5*M.numberOfCells());
  MeshTetrahedrizor::CellMapping& cellMapping = *pCellMapping;

  size_t tetrahedronNumber = 0;
  for(size_t i=0; i<M.shape().nx()-1; ++i)
    for(size_t j=0; j<M.shape().ny()-1; ++j)
      for(size_t k=0; k<M.shape().nz()-1; ++k) {
	const CartesianHexahedron& H = M.cell(i,j,k);
	for (size_t l = 0; l<5; ++l) {
	  cellMapping[tetrahedronNumber] = &H;
	  tetrahedronNumber++;
	}
      }

  return pCellMapping;
}



ReferenceCounting<MeshTetrahedrizor::CellMapping>
MeshTetrahedrizor::motherCells()
{
  switch ((*__input).type()) {
  case Mesh::cartesianHexahedraMesh: {
    return __motherCells<Structured3DMesh>();
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "not implemented: unexpected mesh type",
		       ErrorHandler::unexpected);
    return 0;
  }
  }
}

void
MeshTetrahedrizor::run(const bool& builtLocalizationTools)
{
  switch ((*__input).type()) {
  case Mesh::cartesianHexahedraMesh: {
    this->__build<Structured3DMesh>(builtLocalizationTools);
    break;
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "not implemented: unexpected mesh type",
		       ErrorHandler::unexpected);
  }
  }
}
