//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: UserFunction.cpp,v 1.8 2004/12/31 16:38:58 delpinux Exp $

#include <fstream>

#include <Types.hpp>

#include <Structured3DMesh.hpp>

#include <UserFunction.hpp>

#include <Scene.hpp>
#include <FunctionExpression.hpp>

/*! Unary minus for UserFunctions
  \todo Make this function virtual.
*/
UserFunction UserFunction::operator- () const
{
  return UnaryMinusUserFunction(this);
}

real_t UserFunctionLanguage::operator()(const TinyVector<3>& X) const
{
  return (*__function).value(X[0], X[1], X[2]);
}

void UserFunctionLanguage::gradient(TinyVector<3>& gradient,
				    const TinyVector<3>& X) const
{
#warning BAD TRICK TO AVOID CONST PROBLEM
  FunctionExpression* f = __function;

  switch((*(*f).value()).type()) {
  case (FunctionExpression::fem): {
    const FunctionExpressionFEM& F
      = dynamic_cast<const FunctionExpressionFEM&>(*(*f).value());


    gradient[0] = F.dx(X,FunctionExpressionDx::x);
    gradient[1] = F.dx(X,FunctionExpressionDx::y);
    gradient[2] = F.dx(X,FunctionExpressionDx::z);
    break;
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unexpected function type",
		       ErrorHandler::unexpected);
  }
  }
}

UserFunctionLanguage::
UserFunctionLanguage(UserFunctionLanguage& U)
  : __function(U.__function)
{
  ;
}

UserFunctionLanguage::
UserFunctionLanguage(ReferenceCounting<FunctionExpression> f)
  : __function(f)
{
  ;
}

UserFunctionLanguage::~UserFunctionLanguage()
{
  ;
}



//! Constructs a OneUserFunction using the POV-Ray reference \a r.
OneUserFunction::OneUserFunction(TinyVector<3>& r)
{
  fferr(2) << __FILE__ << ':' << __LINE__ << ": Not Implemented\n";
  /*
  extern Scene* CurrentScene;

  for (size_t i =0; i<CurrentScene->NbObj() ; i++) {
    if (CurrentScene->Objects(i).Ref() == r)
      o.push_back(&(CurrentScene->Objects(i)));
    }
  if (o.size() == 0)
    fferr(2) << "warning: no object in the Scene has reference " << r << '\n';
  */
}


//! Evalutes \f$ 1_{\cup_i o_i} \f$ at point \a X.
real_t OneUserFunction::operator()(const TinyVector<3>& X) const
{
  for (size_t i=0; i<o.size(); i++)
    if (o[i]->inside(X))
      return 1;
  
  return 0;
}

//! Constructor.
OneMeshUserFunction::OneMeshUserFunction(ConstReferenceCounting<Structured3DMesh> givenMesh)
  : __mesh(givenMesh)
{
  ;
}

//! Destructor.
OneMeshUserFunction::~OneMeshUserFunction()
{
  ;
}

real_t OneDomainUserFunction::operator()(const TinyVector<3>& x) const
{
  return (*__domain).inside(x);
}

OneDomainUserFunction::
OneDomainUserFunction(ConstReferenceCounting<Domain> domain)
  : __domain(domain)
{
  ;
}

OneDomainUserFunction::~OneDomainUserFunction()
{
  ;
}


//! Evalutes the characteristic function of the \a mesh at point \a X
real_t OneMeshUserFunction::operator()(const TinyVector<3>& X) const
{
  return ((*__mesh).inside(X));
}

//! Constructs a FEMUserFunction using a pointer on a Structured3DMesh and a UserFunction
FEMUserFunction::FEMUserFunction(ReferenceCounting<Mesh> pmesh,
				 const Vector<real_t>& F)
  : __mesh(pmesh),
    __values((*__mesh).numberOfVertices())
{
  for (size_t i=0; i<__values.size(); i++)
    __values[i] = F[i];
}

/*!
  Constructs a FEMUserFunction using a pointer on a Structured3DMesh and a Vector of data.
  \warning Will have to change this for the vectorial case.
*/
FEMUserFunction::FEMUserFunction(ReferenceCounting<Mesh> pmesh, const UserFunction& F)
  : __mesh(pmesh),
    __values((*__mesh).numberOfVertices())
{
  for (size_t i=0; i<__values.size(); i++) {
    const TinyVector<3>& X = (*__mesh).vertex(i);
    __values[i] = F(X);
  }
}

//! Constructs a FEMUserFunction using a pointer on a Structured3DMesh and a real_t
FEMUserFunction::FEMUserFunction(ReferenceCounting<Mesh> pmesh, const real_t d)
  : __mesh(pmesh),
    __values((*__mesh).numberOfVertices())
{
  __values = d;
}

//! Evaluates the FEMUserFunction at point \a X.
real_t FEMUserFunction::operator()(const TinyVector<3>& V) const
{
  throw ErrorHandler(__FILE__,__LINE__,
		     "not implemented",
		     ErrorHandler::unexpected);

  return 0;
}

//! Affects a \a Vector of data to the FEMUserFunction.
void FEMUserFunction::operator=(Vector<real_t>& u)
{
  __values = u;
}

FEMUserFunction::~FEMUserFunction()
{
  ;
}






//! Constructs a FEM0UserFunction using a pointer on a Structured3DMesh and a UserFunction
FEM0UserFunction::FEM0UserFunction(ReferenceCounting<Mesh> pmesh,
				   const Vector<real_t>& F)
  : __mesh(pmesh),
    __values(F)
{
  assert((*__mesh).numberOfCells() == __values.size());
}

/*!
  Constructs a FEM0UserFunction using a pointer on a Structured3DMesh and a Vector of data.
  \warning Will have to change this for the vectorial case.
*/
FEM0UserFunction::FEM0UserFunction(ReferenceCounting<Mesh> pmesh, const UserFunction& F)
  : __mesh(pmesh),
    __values((*__mesh).numberOfVertices())
{
  throw ErrorHandler(__FILE__,__LINE__,
		     "not implemented",
		     ErrorHandler::unexpected);
}

//! Constructs a FEM0UserFunction using a pointer on a Structured3DMesh and a real_t
FEM0UserFunction::FEM0UserFunction(ReferenceCounting<Mesh> pmesh, const real_t d)
  : __mesh(pmesh),
    __values((*__mesh).numberOfVertices())
{
  __values = d;
}

//! Evaluates the FEM0UserFunction at point \a X.
real_t FEM0UserFunction::operator()(const TinyVector<3>& V) const
{
  switch((*__mesh).type()) {
  case Mesh::cartesianHexahedraMesh: {
    const Structured3DMesh& mesh = static_cast<const Structured3DMesh&>(*__mesh);
    Structured3DMesh::const_iterator icell = mesh.find(V);
    if (not(icell.end())) {
      assert(icell.number() < __values.size());
      return __values(icell.number());
    } else {
      return 0;
    }
    break;
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unexpected mesh type",
		       ErrorHandler::unexpected);
  }
  }
  return 0;
}

//! Affects a \a Vector of data to the FEM0UserFunction.
void FEM0UserFunction::operator=(Vector<real_t>& u)
{
  assert(__values.size() == u.size());
  __values = u;
}

FEM0UserFunction::~FEM0UserFunction()
{
  ;
}

