/*
---------------------------------------------------------------------------
 Authentication library (Using SASL)
 $Id: auth.c,v 1.4 2003/08/11 15:18:22 jfboud Exp $
---------------------------------------------------------------------------
Copyright (c) 2001-2003 Hexago Inc. All rights reserved.

     The contents of this file are subject to the Hexago
     Public License Version 1.0 (the "License"); you may not
     use this file except in compliance with the License.

     Software distributed under the License is distributed on
     an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either
     express or implied. See the License for the specific
     language governing rights and limitations under the
     License.

     The Original Code is _source code of the tunnel server
     protocol-client side_.

     The Initial Developer of the Original Code is Hexago .

     All Rights Reserved.

     Contributor(s): ______________________________________.

--------------------------------------------------------------------------
*/
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <regex.h>
#include "log.h"
#include "tspc.h"
#include "network.h"
#include "base64.h"
#include "md5.h"
#include "auth.h"

/*
   Read server capability and call the supported function.
*/
int Authenticate(int Socket, char *Method, char *User, char *Passwd, char *host) {
   char Data[1024];
   tCapability capability;
   tCapability Mechanism; /* Prefered auth mechanism */
   int rc = -100;

   if (NetReadLine(Socket, Data, sizeof Data) <= 0) {
      Display(1,ELError, "Authenticate", "Disconnected from server");
      return -1;
   }
   if(memcmp("CAPABILITY ", Data, 11)==0)
      capability = ExtractCapability(Data);
   else {
      Display(1,ELError, "Authenticate", "Error on receiving server capabilities");
      return -1;
   }

   if(!(capability & TUNNEL_V6V4))
      return -1; /* Server does not support ipv6 tunnels */

   /*----------------------------------------------------------*/

   if(memcmp(Method, "ANY", 4) && memcmp(Method, "any", 4))
      Mechanism = SetCapability("AUTH", Method);
   else
      Mechanism = AUTH_ANY;

   if(strcmp(User, "anonymous")) { /* If not anonymous user */
      if(Mechanism & capability & AUTH_DIGEST_MD5) {
         rc = AuthDIGEST_MD5(Socket, User, Passwd, host);
         goto EndAuthenticate;
      }
      if(Mechanism & capability & AUTH_PLAIN) {
         rc = AuthPLAIN(Socket, User, Passwd);
         goto EndAuthenticate;
      }
   } else { /* ANONYMOUS user */
      if(capability & AUTH_ANONYMOUS) {
         rc = AuthANONYMOUS(Socket);
         goto EndAuthenticate;
      }
   }

   /*----------------------------------------------------------*/

   EndAuthenticate:
      if(rc == -100)
         Display(1,ELError, "Authenticate", "No common authentication mechanism found!");

   return rc; 
}

/* Return the capability flag corrsponding to the token */
tCapability SetCapability(char *Token, char *Value) {

   if((strcmp("TUNNEL", Token)==0) && (strcmp("V6V4", Value)==0))
      return TUNNEL_V6V4;

   if(strcmp("AUTH", Token)==0) {
      if(strcmp("DIGEST-MD5", Value)==0)
         return AUTH_DIGEST_MD5;
      if(strcmp("ANONYMOUS", Value)==0)
         return AUTH_ANONYMOUS;
      if(strcmp("PLAIN", Value)==0)
         return AUTH_PLAIN;
   }

   return 0;
}

/* Convert a CAPABILITY string in corresponding bit in capability flag */
tCapability ExtractCapability(char *String) {
   tCapability flags;
   char *s, *e, Token[strlen(String)+1], Value[strlen(String)+1];
   int len;

   flags = 0;
   *Token=*Value=NULL;
   for(s=e=String+11; *e; e++) {
      if(*e== ' ' || *e == '\r' || *e == '\n' || *e==NULL) {
        if(s!=e) {
           if(*Token && (*Value==NULL)) {
               len = (int)((char *)e-(char *)s);
               memcpy(Value, s, len);
               Value[len] = NULL;
           }
           if(*Token && *Value) {
              flags += SetCapability(Token,Value); 
              *Value = *Token = NULL;
           }
         }
         s = ++e;
      }
      if((*e=='=' || *e== ' ' || *e == '\r' || *e == '\n' || *e==NULL) && (e != s)) {
         len = (int)((char *)e-(char *)s);
         memcpy(Token, s, len);
         Token[len] = NULL;
         s = ++e;
      }
   }
   return flags;
}

int AuthANONYMOUS(int Socket)
{
   char Buffer[1024];
   if (NetPrintf(Socket, "AUTHENTICATE ANONYMOUS\r\n") == 0) {
      Display(1,ELError, "Authenticate", "Not able to write to server socket");
      return 1;
   }
   if (NetReadLine(Socket, Buffer, sizeof Buffer ) == 0) {
      Display(1,ELError, "Authenticate", "Not able to read from server socket");
      return -1;
   }
   if (memcmp(Buffer, "200", 3)) { /* Not successful */
      Display(1,ELError, "Authenticate", "No success: %s", Buffer);
      return -1;
   }
   return 0;
}

int AuthPLAIN(int Socket, char *User, char *Passwd) {
   char Buffer[1024];
   int Length;

   if (NetPrintf(Socket, "AUTHENTICATE PLAIN\r\n") == 0) {
      Display(1,ELError, "Authenticate", "Not able to write to server socket");
      return -1;
   }

   memset(Buffer, 0, sizeof(Buffer));
   Length = snprintf(Buffer, sizeof Buffer, "%c%s%c%s\r\n",
                      '\0', User, '\0', Passwd);

   if (NetWrite(Socket, Buffer, Length) == 0) {
     Display(1,ELError, "Authenticate", "Not able to write to server socket");
     return -1;
   }
   if (NetReadLine(Socket, Buffer, sizeof(Buffer)) == 0) {
      Display(1,ELError, "Authenticate", "Not able to read from server socket");
      return -1;
   }
   if (memcmp(Buffer, "200", 3)) { /* Not successful */
      Display(1,ELError, "Authenticate", "No success: %s", Buffer);
      return -1;
   }

  return 0;
}

int InsertInChallegeStruct(tChallenge *c, char *Token, char *Value) {
   if(strcmp(Token, "realm")==0) {
      c->realm = strdup(Value);
      return 0;
   }
   if(strcmp(Token, "nonce")==0) {
      c->nonce = strdup(Value);
      return 0;
   }
   if(strcmp(Token, "qop")==0) {
      c->qop = strdup(Value);
      return 0;
   }
   if(strcmp(Token, "algorithm")==0) {
      c->algorithm = strdup(Value);
      return 0;
   }
   if(strcmp(Token, "charset")==0) {
      c->charset = strdup(Value);
      return 0;
   }
   if(strcmp(Token, "rspauth")==0) {
      c->rspauth = strdup(Value);
      return 0;
   }
   return -1;
}

void ExtractChallenge(tChallenge *c, char *String) {
   char *s, *e, Token[strlen(String)+1], Value[strlen(String)+1];
   int len;
   memset(c, 0, sizeof(tChallenge));
   c->realm     = "";
   c->nonce     = "";
   c->qop       = "";
   c->algorithm = "";
   c->charset   = "";
   c->rspauth   = "";
   *Token=*Value=NULL;
   for(s=e=String; ; e++) {
      if(*e== ',' || *e == '\r' || *e == '\n' || *e==NULL) {
        if(s!=e) {
           if(*Token && (*Value==NULL)) {
               len = (int)((char *)e-(char *)s);
               /* Chop the quotes */
               if((*s == '"') && len) { s++; len--; }
               if((s[len-1] == '"') && len) len--;

               if(len) memcpy(Value, s, len);
               Value[len] = NULL;
           }
           if(*Token && *Value) {
              InsertInChallegeStruct(c, Token,Value); 
              *Value = *Token = NULL;
           }
         }
         if(*e == NULL) break;
         s = ++e;
      }
      if((*e=='=' || *e== ',' || *e == '\r' || *e == '\n' || *e==NULL) && (*Token == NULL) && (e != s)) {
         len = (int)((char *)e-(char *)s);
         memcpy(Token, s, len);
         Token[len] = NULL;
         if(*e == NULL) break;
         s = ++e;
      }
   }
}

int AuthDIGEST_MD5(int Socket, char *User, char *Passwd, char *Host) {
   char Buffer[4096], Response[33], cResponse[33], *ChallengeString;
   time_t cnonce = time(NULL);
   tChallenge c;

   if (NetPrintf(Socket, "AUTHENTICATE DIGEST-MD5\r\n") == 0) {
      Display(1,ELError, "AuthDIGEST_MD5", "Not able to write to server socket");
      return 1;
   }
   if (NetReadLine(Socket, Buffer, sizeof Buffer ) == 0) {
      Display(1,ELError, "Authenticate", "Not able to read from server socket");
      return -1;
   }
   if (memcmp(Buffer, "300", 3) == 0) { /* Not successful */
      Display(1,ELError, "Authenticate", "No success: %s", Buffer+3);
      return -1;
   }
   if((ChallengeString = malloc(strlen(Buffer) + 1)) == NULL) {
      Display(1,ELError, "AuthDIGEST_MD5", "Not enough memory for ChallengeString!");
      return -1;
   }
   base64decode(ChallengeString, Buffer);
   ExtractChallenge(&c, ChallengeString);
   free(ChallengeString);

   {
   /*-----------------------------------------------------------*/
   /*
      Extract from : RFC 2831 Digest SASL Mechanism

      Let H(s) be the 16 octet MD5 hash [RFC 1321] of the octet string s.

      Let KD(k, s) be H({k, ":", s}), i.e., the 16 octet hash of the string
      k, a colon and the string s.

      Let HEX(n) be the representation of the 16 octet MD5 hash n as a
      string of 32 hex digits (with alphabetic characters always in lower
      case, since MD5 is case sensitive).

      response-value  =
         HEX( KD ( HEX(H(A1)),
                 { nonce-value, ":" nc-value, ":",
                   cnonce-value, ":", qop-value, ":", HEX(H(A2)) }))

      If authzid is not specified, then A1 is

         A1 = { H( { username-value, ":", realm-value, ":", passwd } ),
           ":", nonce-value, ":", cnonce-value }

      If the "qop" directive's value is "auth", then A2 is:

         A2       = { "AUTHENTICATE:", digest-uri-value }

   */
      char *A1_1Fmt      = "%s:%s:%s",
           *A1Fmt        = ":%s:%lu",
           *A2Fmt        = "%s:tsp/%s",
           *ChallRespFmt = "%s:%s:00000001:%lu:%s:%s",
           *ResponseFmt   = "charset=%s,username=\"%s\",realm=\"%s\",nonce=\"%s\",nc=00000001,cnonce=\"%lu\",digest-uri=\"tsp/%s\",response=%s,qop=auth",
           A1[33], A1_1[33], A2[33], cA2[33], *String;
      int len;

      /*-----------------------------------------------------------*/
      /* Build HEX(H(A2)) & HEX(H(cA2))                            */

      len = strlen(A2Fmt) + 12 /* AUTHENTICATE */ + strlen(Host) + 1;
      if((String = malloc(len)) == NULL) {
         Display(1,ELError, "AuthDIGEST_MD5", "Not enough memory for String!");
         return -1;
      }
      snprintf(String, len, A2Fmt, "AUTHENTICATE", Host);
#if DEBUG
      printf("A2 = %s\n", String);
#endif
      strncpy(A2, md5(String), 33);
      snprintf(String, len, A2Fmt, "", Host);
#if DEBUG
      printf("cA2 = %s\n", String);
#endif
      strncpy(cA2, md5(String), 33);
      free(String);

      /*-----------------------------------------------------------*/
      /* Build HEX(H(A1))                                          */
      /* A1_1 = { username-value, ":", realm-value, ":", passwd }  */
      /* A1 = { H( A1_1 ), ":", nonce-value, ":", cnonce-value }   */

      len = strlen(A1_1Fmt) + strlen(User) + strlen(c.realm) + 
            strlen(Passwd) +  1;
      if((String = malloc(len)) == NULL) {
         Display(1,ELError, "AuthDIGEST_MD5", "Not enough memory for String!");
         return -1;
      }
      snprintf(String, len, A1_1Fmt, User, c.realm, Passwd);
#if DEBUG
      printf("A1_1 = %s\n", String);
#endif

      md5digest(String, A1_1);
      free(String);

      len = 16 /* A1_1 */ + 1 +						   
	  strlen(c.nonce) + 16 /* cnonce */ +  1;
      if((String = malloc(len)) == NULL) {
         Display(1,ELError, "AuthDIGEST_MD5", "Not enough memory for String!");
         return -1;
      }

      memcpy(String, A1_1, 16);
      snprintf(String + 16, len - 16, A1Fmt, c.nonce, cnonce);
      strncpy(A1, md5(String), 33);
      free(String);
#if DEBUG
      printf("A1 = [%s]\n", A1);
#endif      

      /*-----------------------------------------------------------*/
      /* Build server's and client's challenge responses           */

      len = strlen(ChallRespFmt) + 32 /* md5(A1) */ + strlen(c.nonce) + 
            16 /* cnonce */ + strlen(c.qop) + 32 /* md5(A2) */ +  1;
      if((String = malloc(len)) == NULL) {
         Display(1,ELError, "AuthDIGEST_MD5", "Not enough memory for String!");
         return -1;
      }
      snprintf(String, len, ChallRespFmt, A1, c.nonce, cnonce, c.qop, A2);
#if DEBUG
      printf("Response = [%s]\n", String);
#endif
      strncpy(Response, md5(String), 33);
#if DEBUG
      printf("MD5 Response = %s\n", Response);
#endif
      snprintf(String, len, ChallRespFmt, A1, c.nonce, cnonce, c.qop, cA2);
#if DEBUG
      printf("cResponse = [%s]\n", String);
#endif
      strncpy(cResponse, md5(String), 33);
#if DEBUG
      printf("MD5 cResponse = %s\n", cResponse);
#endif
      free(String);

      /*-----------------------------------------------------------*/
      /* Build Response                                            */

      len = strlen(ResponseFmt) + strlen(c.charset) + strlen(User) +
            strlen(c.realm) + strlen(c.nonce) + 16 /*cnonce*/ +
            strlen(Host)    + 32 /* md5 response */;
      if((String = malloc(len)) == NULL) {
         Display(1,ELError, "AuthDIGEST_MD5", "Not enough memory for String!");
         return -1;
      }
      snprintf(String, len, ResponseFmt, c.charset, 
         User, c.realm, c.nonce, cnonce, Host, Response);
      memset(Buffer, 0, sizeof(Buffer));
      base64encode(Buffer, String, strlen(String));
      free(String);
   }

   if (NetPrintf(Socket, "%s\r\n", Buffer) == 0) {
      Display(1,ELError, "AuthDIGEST_MD5", "Not able to write to server socket");
      return -1;
   }

   /*-----------------------------------------------------------*/
   /* Verify server response                                    */

   if (NetReadLine(Socket, Buffer, sizeof Buffer ) == 0) {
      Display(1,ELError, "Authenticate", "Not able to read from server socket");
      return -1;
   }
   if (memcmp(Buffer, "300", 3) == 0) { /* Not successful */
      Display(1,ELError, "Authenticate", "No success: %s", Buffer+4);
      return -1;
   }

   if((ChallengeString = malloc(strlen(Buffer) + 1)) == NULL) {
      Display(1,ELError, "AuthDIGEST_MD5", "Not enough memory for ChallengeString!");
      return -1;
   }
   base64decode(ChallengeString, Buffer);
   ExtractChallenge(&c, ChallengeString);
   free(ChallengeString);

   if(memcmp(c.rspauth, cResponse, 32)) {
      Display(1,ELError, "Authenticate", "Invalid response from server");
      return -1;
   }
   if (NetReadLine(Socket, Buffer, sizeof Buffer ) == 0) {
      Display(1,ELError, "Authenticate", "Not able to read from server socket");
      return -1;
   }
   if (memcmp(Buffer, "200", 3)) { /* Not successful */
      Display(1,ELError, "Authenticate", "No success: %s", Buffer+4);
      return -1;
   }

   return 0;
}
