from fusil.mangle_agent import MangleAgent
from fusil.incr_mangle_op import OPERATIONS
from random import randint, choice
from array import array

class DataVersion:
    """
    Immutable data version
    """
    def __init__(self, versions, data, operations):
        self.versions = versions
        self.data = data
        self.operations = operations
        self.version = versions.getVersionNumber()

    def createVersion(self, datalen, max_operation):
        operations = list(self.operations)
        wanted_len = len(operations) + randint(1, max_operation)
        while len(operations) < wanted_len:
            operation_cls = choice(OPERATIONS)
            operation = operation_cls(datalen)
            if any(item.overlaps(operation) for item in operations):
                continue
            operations.append(operation)
        return ModifiedVersion(self.versions, tuple(operations))

    def getPrevious(self):
        return self.versions.getPrevious(self)

    def createData(self):
        raise NotImplementedError()

    def revert(self):
        """
        Revert this version and return new last version
        """
        self.versions.removeVersion(self)
        return self.versions.getLast()

    def __str__(self):
        return "<DataVersion version=%s operations=%s>" % (
            self.version, len(self.operations))

    def clearCache(self):
        pass

class OriginalVersion(DataVersion):
    def __init__(self, versions, data):
        DataVersion.__init__(self, versions, data, tuple())

    def createData(self):
        return array('B', self.data)

    def revert(self):
        return self

    def __str__(self):
        return "<OriginalVersion>"

class ModifiedVersion(DataVersion):
    def __init__(self, versions, operations):
        DataVersion.__init__(self, versions, None, operations)

    def createData(self):
        # Cached result?
        if self.data:
            return array('B', self.data)

        # Get previous complete data
        previous = self
        while True:
            previous = previous.getPrevious()
            if previous.data:
                break

        # Apply new operations (since previous version)
        data = array('B', previous.data)
        for operation in self.operations[len(previous.operations):]:
            operation(data)

        # Cache result
        self.data = data.tostring()
        return data

    def clearCache(self):
        self.data = None

class DataVersions:
    def __init__(self):
        self.versions = []

    def getVersionNumber(self):
        return len(self.versions)+1

    def addVersion(self, version):
        self.versions.append(version)

        # Clear cache of old versions
        for version in self.versions[:-2]:
            version.clearCache()

    def removeVersion(self, version):
        self.versions.remove(version)

    def getLast(self):
        try:
            return self.versions[-1]
        except IndexError:
            return None

    def getPrevious(self, version):
        index = self.versions.index(version)
        return self.versions[index-1]

    def rollback(self, version_number):
        del self.versions[version_number:]
        return self.versions[-1]

class IncrMangle(MangleAgent):
    def __init__(self, project, source):
        MangleAgent.__init__(self, project, source, 1)
        self.versions = DataVersions()
        self.previous_score = None

        # User config
        self.operation_per_version = 1
        self.max_version = 50

    def on_session_done(self, score):
        self.previous_score = score

    def checkPreviousVersion(self, version):
        if self.previous_score is not None:
            if (self.project().success_score <= self.previous_score):
                # Rollback to original version
                original = self.versions.rollback(1)
                self.error("Rollback to %s" % original)
                return original

            if self.previous_score < 0:
                # Failure: revert version
                self.warning("Revert previous version: %s (score %+.1f%%)" % (version, self.previous_score*100))
                return version.revert()

        if self.max_version <= version.version:
            number = randint(1, version.version-1)
            version = self.versions.rollback(number)
            self.error("Rollback to %s" % version)
            return version

        # No change
        self.error("Accepted version: %s" % version)
        return version

    def mangleData(self, data, file_index):
        # Get last version
        version = self.versions.getLast()
        if version:
            version = self.checkPreviousVersion(version)
        else:
            version = OriginalVersion(self.versions, data.tostring())
            self.warning("Create first version: %s" % version)
            self.versions.addVersion(version)

        # New version
        previous = version
        version = previous.createVersion(len(data), self.operation_per_version)
        self.warning("New version %s based on %s" % (version, previous))
        self.versions.addVersion(version)

        # Create data
        return version.createData()

