/* 

                          Firewall Builder

                 Copyright (C) 2002 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: CertificateDruid.cc,v 1.20 2002/09/09 19:12:43 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include "CertificateDruid.hh"
#include "MessageDialog.hh"
#include "SimpleProgressDialog.hh"
#include "KeyGenerationProgressIndicator.hh"

#include "CertDruidP20Dialog.hh"
#include "CertDruidP30Dialog.hh"
#include "CertDruidP40Dialog.hh"

#include "Preferences.hh"
#include "helpers.hh"

#include "fwbuilder/crypto.hh"

#include <gtk--.h>

#include "glademm_support.hh"

#include <iostream>


using namespace libfwbuilder;


enum {
    PAGE_S,
    PAGE_20,
    PAGE_30,
    PAGE_40,
    PAGE_F
} CertDruidPages;


CertificateDruid::CertificateDruid(const string &t,
                                   const string& l) : Druid(t,l), key(NULL), cert(NULL)
{
    set_default_size(450,400);
    set_position(GTK_WIN_POS_CENTER);


/************************************************************************
 *    Start Page 
 */

    createStartPage(PAGE_S,_("This druid will help you generate digital certificate which \n\
can be used to identify you when accessing Firewall Building Daemon.\n\
\n\
This is self-signed Certificate. This means it won't be signed by any \n\
Certificate authority, ensuring your identiry. We assume that you manage \n\
you firewall machine and Firewall Builder Daemon running on it and can \n\
install public key on it in a secure manner. This level of security should \n\
be sufficient for most home and small company users.\n\
\n\
Corporate users or others who require higher level of security can use \n\
'Import' function in Options dialog to import certificat properly signed \n\
by Certificate Authority\n\
\n\
Press 'Next' to start"));



/************************************************************************
 *    Page #20
 */

    p20=manage(new CertDruidP20Dialog());
    createStandardPage(PAGE_20,p20);


/************************************************************************
 *    Page #30
 */

    p30=manage(new CertDruidP30Dialog());
    createStandardPage(PAGE_30,p30);


/************************************************************************
 *    Page #40
 */

    p40=manage(new CertDruidP40Dialog());
    createStandardPage(PAGE_40,p40);


/************************************************************************
 *    Final Page 
 */
 
    createFinishPage(PAGE_F,
_("Congratulations! You successfully generated X509 digital certificate.\n\
You can use this certificate to authenticate to firewall builder \n\
daemon. Use 'Export Certificate' functionin the 'Options' dialog and\n\
follow instructions there"));

    glademm_set_Widget("CertificateDruid", this);

    running=false;
    delete_event.connect( SigC::slot(this,&CertificateDruid::on_delete_event));
}

gint CertificateDruid::on_delete_event(GdkEventAny* ev)
{
    Gtk::Main::quit();
    return true;  // ignore this even, but break main loop
}

void CertificateDruid::run()
{
    show_all();
#ifdef __MINGW32__
    __fwb_win32_loadIcon("FWBuilderWin32Ico",this);
#endif

    running=true;

    Gtk::Main::grab_add(*this);
    Gtk::Main::run();
    Gtk::Main::grab_remove(*this);

    running=false;

    destroy();
}


/*
 *  this method is called BEFORE druid page is shown to user. This is 
 *  convenient place to execute any action which is supposed to happen
 *  when user switches from previous page to this one.
 *
 *  access data fields in druid pages like this:
 *
 *  to access passphrase on page P20:
 *  p20->cert_passphrase1
 *  p20->cert_passphrase2
 *
 *  and so on.
 */
void CertificateDruid::on_prepare()
{
    int current_page=get_current_page_id();

    switch (current_page) {

    case PAGE_20:  break;
    case PAGE_30:  break;

    }
}

/*
 * this method is called when user clicks "Next" but when current page
 * is still the "old" one. This method should decide what page will be
 * shown next and call  setPage
 */
gboolean CertificateDruid::on_next()
{
    int current_page=get_current_page_id();

    switch (current_page) {

    case PAGE_S:
	setPage(PAGE_20);
	return true;

    case PAGE_20:
    {
        // --- Generate Key pair ---

        string pass1 = p20->getPassphrase1();
        string pass2 = p20->getPassphrase2();
	if (pass1=="") {
	    MessageDialog::Error(_("Empty passphrase is not allowed"),this);
	    return true;
	}
        if (pass1!=pass2) {
            // two copies of the passphrase do not match. Show MessageDialog and do not
            // switch druid page to PAGE_30
	    MessageDialog::Error(_("Passphrases do not match"),this);
	    return true;
	}

        int bits = p20->getKeyLength();
        
        password = pass1;

        KeyGenerationProgressIndicator  progress;

        try
        {
            key  = new Key(bits, &progress); 
        } catch(const FWException &ex)
        {
	    MessageDialog::Error(string(_("Error generating key: "))+ex.toString(),this);
	    return true;
        }

	setPage(PAGE_30);
	return true;
    }

    case PAGE_30:
    {
        long not_before = p30->getNotValidBefore();
        long not_after  = p30->getNotValidAfter();

        if (not_before==-1l || not_after==-1l || not_before>=not_after) return true;

        string description = p30->getDescription();
	if(description.empty()) 
        {
	    MessageDialog::Error(_("Please enter some description for this certificate"),this);
	    return true;
	}

	setPage(PAGE_40);
	return true;
    }

    case PAGE_40:
    {
        // --- Generate certificate ---

	std::vector<X509_entry> entries;

	p30->getEntries(entries);
	p40->getEntries(entries);

        long not_before = p30->getNotValidBefore();
        long not_after  = p30->getNotValidAfter();

        string description = p30->getDescription();

        try
        {
            cert = new Certificate(key, entries, not_before, not_after);
        } catch(const FWException &ex)
        {
	    MessageDialog::Error(string(_("Error generating certificate: "))+ex.toString(),this);
	    return true;
        }

        try
        {
            KeyAndCert kc(*key, *cert);
            char id[80];
            sprintf(id,"%ld",time(NULL));
            Preferences::global_prefs->addKeyAndCert(string(id), description, kc,
                                                     password);
            Preferences::global_prefs->savePrefs();
        } catch(const FWException &ex)
        {
	    MessageDialog::Error(string(_("Error saving certificate: "))+ex.toString(),this);
	    return true;
        }


	setPage(PAGE_F);
	return true;
    }
    }
    return true;
}

/*
 * this method acts in a manner opposite to on_next: it is called
 * when user clicks "Back". It should decide what page should be shown 
 * and call setPage
 */
gboolean CertificateDruid::on_back()
{
    int current_page=get_current_page_id();

    switch (current_page) {

    case PAGE_20:
	setPage(PAGE_S);
	return true;

    case PAGE_30:
	setPage(PAGE_20);
	return true;

    case PAGE_40:
	setPage(PAGE_30);
	return true;

    case PAGE_F:
	setPage(PAGE_40);
	return true;
    }
    return  true;
}

gboolean CertificateDruid::on_cancel()
{
    if (running) Gtk::Main::quit();
    else         destroy();
    return true;
}

/*
 *  whatever needs to be done when user clicks "Finish"
 */
void CertificateDruid::on_finish()
{


    if (running)  Gtk::Main::quit();
    else         destroy();
}

