/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: Preferences.cc,v 1.57 2003/01/29 07:49:16 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <sys/types.h>
#include <sys/stat.h>
#ifndef __MINGW32__
#include <pwd.h>
#endif
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>


#include <fstream>
#include <cstdlib>

#include <config.h>

#include "Preferences.hh"
#include "fwbuilder/FWObject.hh"
#include "FWObjectDatabaseGUI.hh"
#include "fwbuilder/Tools.hh"
#include "helpers.hh"

using namespace libfwbuilder;

const string Preferences::PREFS_DIR_NIX               = ".fwbuilder"                ;
const string Preferences::PREFS_DIR_WIN               = "Firewall_Builder"          ;
const string Preferences::PREFS_TEMPLATE_FILE_NAME    = "fwbuilder_prefs.xml"       ;
const string Preferences::OLD_PREFS_FILE_NAME         = ".fwbuilder.xml"            ;
const string Preferences::NEW_PREFS_FILE_NAME         = "prefs.xml"                 ;
const string Preferences::DTD_FILE_NAME               = "fwbuilder_preferences.dtd" ;
const string Preferences::TYPENAME                    = "FWBuilderPreferences"      ;

Preferences*  Preferences::global_prefs = NULL;

Preferences::Preferences() throw(FWException)
{
    g_assert(global_prefs==NULL);

    global_prefs=this;
    doc=NULL;

    string home;
    string prefsDir;

#ifndef __MINGW32__
    struct passwd *pwd=getpwuid(getuid());
    g_assert(pwd!=NULL);

    home=string(pwd->pw_dir);
    if(!home.length())  throw FWException(_("Empty user home dir"));

    if(home[home.length()-1]!='/')  home = home + "/";
    prefsDir = home+PREFS_DIR_NIX+"/";
#else
    home     = __fwb_win32_getCurrentProfileDir() + "\\";
    prefsDir = home+PREFS_DIR_WIN+"\\";
#endif


    if(access(prefsDir.c_str() , R_OK )!=0 && mkdir(prefsDir.c_str() , 0755)!=0)
    {
        string err=strerror(errno);
        throw FWException(_("Could not create directory for the preferences file: ")+prefsDir+"\n"+err);
    }

    prefsFile = prefsDir + NEW_PREFS_FILE_NAME;
    
    if(access(prefsFile.c_str() , R_OK )==0) 
    {
        loadPrefs();
        return;
    }

/* check if there is preferences file in the old location  */
    prefsFile = home + OLD_PREFS_FILE_NAME;

    if(access(prefsFile.c_str() , R_OK )==0)
    {
/* found preferences in the old location. Open it, then change prefsFile
 * to point to the new location and save */
        string oldPrefsFile=prefsFile;
        loadPrefs();
        prefsFile = prefsDir + NEW_PREFS_FILE_NAME;
        savePrefs();
        unlink(oldPrefsFile.c_str());
        return;
    } else 
    { 
/* preferences file was not found in either new or old location */

        prefsFile = prefsDir + NEW_PREFS_FILE_NAME;
    
        string prefsTemplateFile = TEMPLATE_DIR  "/" + PREFS_TEMPLATE_FILE_NAME;

        std::ifstream in(prefsTemplateFile.c_str());
        if(in.fail())
        {
            string err=strerror(errno);
            throw FWException(_("Could not open user preferences template file: ")+prefsTemplateFile+"\n"+err);
        }

        std::ofstream out(prefsFile.c_str());
        if(out.fail()) 
        {
            string err=strerror(errno);
            throw FWException(_("Could not create user preferences file: ")+prefsFile+"\n"+err);
        }

        int c;
        while((c=in.get()) && c!=EOF)
            out.put(c);	    

        in.close();
        out.close();

        loadPrefs();
    }
}


xmlNodePtr Preferences::getXmlNode(const string& path)
{
    return XMLTools::getXmlNodeByPath(root,path);
}

void  Preferences::setOpt(const string &path, const string &val)
{
    setOpt(path, val.c_str());
}


void  Preferences::setOpt(const string &path, const char* val)
{
    xmlNodePtr node;

    node=XMLTools::getXmlNodeByPath(root,path);

    if (node)  
	xmlNodeSetContent(node,TOXMLCAST(val));
}

void  Preferences::setOptInt(const string &path, int val)
{
    char str[32];
    sprintf(str,"%d",val);
    setOpt(path,str);
}

void  Preferences::setOptBool(const string &path, bool val)
{
    setOpt(path,(val)?"true":"false");
}

int Preferences::getOptInt(const string &path)
{
    string  s=getOptStr(path);
    return (atoi(s.c_str()));
}

int Preferences::getOptInt(const string& name, int default_value)
{
    xmlNodePtr node=XMLTools::getXmlNodeByPath(root, name);
    char *res=NULL;
    
    if (node) 
	res= (char*)( xmlNodeGetContent(node) );

    if (res) {
      int r=atoi(res);
      free(res);
      return r;
    }
    else     
        return default_value;
}

bool Preferences::getOptBool(const string &path)
{
    string  s=getOptStr(path);
    return (s=="true" || s=="1" );
}

string Preferences::getOptStr(const string &path)
{
    xmlNodePtr node=XMLTools::getXmlNodeByPath(root,path);
    char *res=NULL;

    if (node) 
	res= (char*)( xmlNodeGetContent(node) );

    string str;
    if (res!=NULL) str=res;
    free(res);
    return str;
}

void Preferences::setProp(const string& path,
			  const char* attrname,
			  const char* val)
{
    xmlNodePtr node;
    node=XMLTools::getXmlNodeByPath(root,path);
    if (node) 
	setProp(node,attrname,val);
}

void Preferences::setProp(xmlNodePtr node,
			  const char* attrname,
			  const char* val)
{
    if (node) 
	FROMXMLCAST( xmlSetProp(node,
				TOXMLCAST(attrname),
				TOXMLCAST(val)
	));
}

string Preferences::getPropStr(const string& path,const char* attrname)
{
    xmlNodePtr node=XMLTools::getXmlNodeByPath(root,path);
    return  getPropStr(node,attrname);
}

string Preferences::getPropStr(xmlNodePtr node,
			       const char* attrname)
{
    char *res=NULL;
    if (node)
	res=(char*)( xmlGetProp(node,TOXMLCAST(attrname)));
    string str=res;
    free(res);
    return str;
}

void Preferences::loadPrefs() throw(FWException)
{
    MessageBoxUpgradePredicate upgrade_predicate;
    doc = XMLTools::loadFile(prefsFile, TYPENAME, DTD_FILE_NAME, &upgrade_predicate, TEMPLATE_DIR, VERSION);
    root= xmlDocGetRootElement(doc);
}

void Preferences::savePrefs() throw(FWException)
{
    if(!doc)
        throw FWException(_("Preferences file not saved - it's empty"));

    
    XMLTools::saveFile(doc, prefsFile, TYPENAME, DTD_FILE_NAME);
}

string Preferences::getWdir()
{
    return  Preferences::global_prefs->getOptStr("/FWBuilderPreferences/Paths/Wdir");
#if 0
    if (FWObjectDatabase::db->getFileName()=="")
    {
	return  Preferences::global_prefs->getOptStr("/FWBuilderPreferences/Paths/Wdir");
    }
    else 
	return FWObjectDatabase::db->getFileDir();
#endif
}

xmlNodePtr Preferences::findCert(const string &id) throw(libfwbuilder::FWException)
{
    if(id.empty())
        throw FWException(_("Empty certificate ID"));


    xmlNodePtr sec = XMLTools::getXmlNodeByPath(root, "/FWBuilderPreferences/Security");

    if(!sec)
        throw FWException(_("No certificates are present."));

    
    for(xmlNodePtr cur=sec->xmlChildrenNode; cur; cur=cur->next)
    {
        if(cur                  && 
           !xmlIsBlankNode(cur) &&
           cur->name!=NULL      && 
           strcasecmp(FROMXMLCAST(cur->name), "PKCS12Identity")==0 && 
           getPropStr(cur,"id")==id
        )
            return cur;
    }
    throw FWException(_("Ceritifcate not found"));

}

map<string, pair<string, string> > Preferences::getCerificates()
{
    map<string, pair<string, string> > res;

    xmlNodePtr sec = XMLTools::getXmlNodeByPath(root, "/FWBuilderPreferences/Security");
    if(!sec)
        return res;

    for(xmlNodePtr cur=sec->xmlChildrenNode; cur; cur=cur->next) 
    {
        if(cur && !xmlIsBlankNode(cur))  
        {
            const char *name=FROMXMLCAST(cur->name);
            if(name!=NULL && strcasecmp(name, "PKCS12Identity")==0) 
            {
                string id    = getPropStr(cur,"id");
                string descr = getPropStr(cur,"description");
                char *data = (char*)(xmlNodeGetContent(cur));
                if(!id.empty() && !descr.empty() && data) {
                    res[id] = pair<string,string>(descr,string(XMLTools::unquote_linefeeds(data)));
                    free(data);
                }
            }
        }
    }
    return res;
}

void Preferences::addKeyAndCert(const string &id, const string &description, const KeyAndCert &c,
                                const string &password) throw(FWException)
{
    if(id.empty() || description.empty())
        throw FWException(_("Empty ID or Description"));

    
    xmlNodePtr sec = XMLTools::getXmlNodeByPath(root, "/FWBuilderPreferences/Security");
    if(!sec)
    {
        xmlNodePtr p = XMLTools::getXmlNodeByPath(root, "/FWBuilderPreferences");

        if(!p)
            throw FWException(_("Missing root 'FWBuilderPreferences' element"));

        sec = xmlNewChild(p, NULL, TOXMLCAST("Security"), NULL);

    }
    
    xmlNodePtr node = xmlNewChild(sec, NULL, TOXMLCAST("PKCS12Identity"), 
                                  TOXMLCAST(XMLTools::quote_linefeeds(c.asString(password)).c_str())
    );
    
    xmlNewProp(node,
               TOXMLCAST("id") , 
               STRTOXMLCAST(id)
    );
    
    xmlNewProp(node,
               TOXMLCAST("description") , 

               STRTOXMLCAST(description)
    );
}

void Preferences::removeKeyAndCert(const string &id) throw(FWException)
{
    xmlNodePtr cur = findCert(id);
    assert(cur!=NULL);
    xmlUnlinkNode(cur);
    xmlFreeNode(cur);
}

void Preferences::exportPKCS12(const string &id, const string &filename) throw(FWException)
{
    if(filename.empty())
        throw FWException(_("Empty Filename"));


    xmlNodePtr cur = findCert(id);
    assert(cur!=NULL);
    
    char *data = (char*)(xmlNodeGetContent(cur));
    if(!data)
        throw FWException(_("Empty certificate found"));

    
    char *res=NULL;
    int len = unbase64(XMLTools::unquote_linefeeds(data).c_str(), &res);
    free(data);

    if(len<=0 || !res)
        throw FWException(_("BASE64 decoding error"));

    
    try
    {
        FILE *f = fopen(filename.c_str(),"w");

        if(!f)
            throw FWException(_("Error opening file for certificate export"));

        
        if(fwrite(res,1,len,f)!=(unsigned)len)
        {
            fclose(f);
            throw FWException(_("Error writing certificate to file"));

        }
        
        fclose(f);
        delete res;
        return;
    } catch(...)
    {
        delete res;
        throw;
    }
}

void Preferences::exportPublicKey(const string &id, const string &password, const string &filename) throw(libfwbuilder::FWException)
{
    if(filename.empty())
        throw FWException(_("Empty Filename"));

    
    KeyAndCert kc = getKeyAndCert(id, password);
    string s = kc.getPrivateKey().getPublicKey();
    
    FILE *f = fopen(filename.c_str(),"w");

    if(!f)
        throw FWException(_("Error opening file for public key export"));

    
    unsigned int len=s.length();
    if(fwrite(s.c_str(),1,len,f)!=len)
    {
        fclose(f);
        throw FWException(_("Error writing public key to file"));

    }
    
    fclose(f);
    return;
    
}

libfwbuilder::KeyAndCert Preferences::getKeyAndCert(const string &id, const string &password) throw(libfwbuilder::FWException)
{
    xmlNodePtr cur = findCert(id);
    assert(cur!=NULL);
    
    char *data = (char*)(xmlNodeGetContent(cur));
    if(!data)
        throw FWException(_("Empty certificate found"));

    
    char *res=NULL;

    int len = unbase64(XMLTools::unquote_linefeeds(data).c_str(), &res);
    free(data);

    if(len<=0 || !res)
        throw FWException(_("BASE64 decoding error"));

    
    try
    {
        KeyAndCert c(res, len, password);
        delete res;
        return c;
    } catch(...)
    {
        delete res; 
        throw;
    }
}

string Preferences::importPKCS12(const string &filename, const string &descr) throw(libfwbuilder::FWException)
{
#ifndef HAVE_LIBSSL
    throw FWException(_("Compiled without crypto support."));

#else
    if(filename.empty() || descr.empty())
        throw FWException(_("Empty Filename or Description"));



    // Read data from file, write to memory in BASE64 encoding

    BIO *bio_out = BIO_push(BIO_new(BIO_f_base64()), BIO_new(BIO_s_mem()));
    BIO *bio_in  = BIO_new_file(filename.c_str(), "r");
    
    
    char inbuf[256];
    int  inlen;
    
    while((inlen = BIO_read(bio_in, inbuf, sizeof(inbuf))) > 0)
        BIO_write(bio_out, inbuf, inlen);
    
    *inbuf='\0'; // null terminate
    BIO_write(bio_out, inbuf, 1);

    BIO_flush(bio_out);
    
    char   *ptr;
    BIO_get_mem_data(bio_out, &ptr);
    assert(ptr!=NULL);
    
    string base64data = ptr;
    
    BIO_free_all(bio_in  );
    BIO_free_all(bio_out );
    
    xmlNodePtr sec = XMLTools::getXmlNodeByPath(root, "/FWBuilderPreferences/Security");
    if(!sec)
    {
        xmlNodePtr p = XMLTools::getXmlNodeByPath(root, "/FWBuilderPreferences");

        if(!p)
            throw FWException(_("Missing root 'FWBuilderPreferences' element"));

        sec = xmlNewChild(p, NULL, TOXMLCAST("Security"), NULL);

    }
    
    xmlNodePtr node = xmlNewChild(sec, NULL, TOXMLCAST("PKCS12Identity"), 
                                  TOXMLCAST(XMLTools::quote_linefeeds(base64data).c_str())
    );

    char id[80];
    sprintf(id,"%ld",time(NULL));
    
    
    xmlNewProp(node,
               TOXMLCAST("id") , 
               TOXMLCAST(id)
    );
    
    xmlNewProp(node,
               TOXMLCAST("description") , 
               STRTOXMLCAST(descr)
    );
    
    return id;
#endif
}
