/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: main_window.cc,v 1.136 2003/06/29 06:38:25 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <config.h>

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <vector>

#include <assert.h>

#include <gtk--.h>

#include <gdk/gdkkeysyms.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "glademm_support.hh"


#include "main_window.hh"
#include "helpers.hh"
#include "SimplePixmap.hh"

#include "fwbuilder/Firewall.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/NAT.hh"
#include "fwbuilder/FWOptions.hh"
#include "fwbuilder/Constants.hh"
#include "fwbuilder/Management.hh"

#include "FWObjectDatabaseGUI.hh"
#include "DialogPlugin.hh"
#include "PrintDialog.hh"
#include "PolicyDialog.hh"
#include "NATDialog.hh"
#include "FirewallDialog.hh"
#include "PolicyList.hh"
#include "FWObjectBook.hh"
#include "AboutDialog.hh"
#include "OptionsDlg.hh"
#include "ExecDialog.hh"
#include "MessageDialog.hh"
#include "FileSel.hh"

#include "StandardRulesDruid.hh"
#include "DiscoveryDruid.hh"

using namespace libfwbuilder;

std::list<main_window*> main_window::windows;


main_window::main_window() :  Gtk::Window(GTK_WINDOW_TOPLEVEL)
{

    windows.push_back(this);


    custom_item=NULL;
    policy_in_right_pane=false;
    current_right_pane_dialog=NULL;
    pane_docked=true;

    left_pane_width=150;
    stored_divider_pos=left_pane_width;

    navbar=NULL;
    needs_saving_icon=NULL;
    main_statusbar_txt=NULL;

    main_vb          = new class Gtk::VBox();
    main_menubar1    = new class Gtk::MenuBar();
    save_statusbar   = new class Gtk::Viewport();
    main_statusbar   = new class Gtk::Viewport();
   
    hpaned           = new class Gtk::HPaned();

    hpaned->set_handle_size(10);
    hpaned->set_gutter_size(6);
    hpaned->set_position(-1);

    main_vb->pack_start(*main_menubar1 , false, false );
    main_vb->pack_start(*hpaned        , true , true  );

    statusbar_hbox = new class Gtk::HBox();
    statusbar_hbox->set_spacing(6);

    main_vb->pack_start(*statusbar_hbox, false, false );

    statusbar_hbox->pack_start(*save_statusbar, false, false );
    statusbar_hbox->pack_start(*main_statusbar, true,  true  );

    save_statusbar->set_shadow_type(GTK_SHADOW_IN);
    main_statusbar->set_shadow_type(GTK_SHADOW_IN);

    save_statusbar->set_usize(32 ,-1);

    showStatusbarMsg(" ");

    int sw=gdk_screen_width();
    int sh=gdk_screen_height();

    if (sw<=1024)
	set_default_size( 800,600);
    else  
	set_default_size( int(float(sw)/1.1), int (float(sh)/1.4) );

    set_position(GTK_WIN_POS_CENTER);

    set_policy(false,true,false);

    add(*main_vb);

    hpaned->show();
    save_statusbar->show();
    main_statusbar->show();
    main_vb->show_all();



    obook=manage(new FWObjectBook(this));

/*
 *  do not manage() left_pane_port and right_pane_port because we will
 *  unparent them later
 */
    left_pane_port     = new Gtk::ScrolledWindow();
    Gtk::Viewport *vp  = manage( new Gtk::Viewport() );
    
//    ((Gtk::ScrolledWindow*)left_pane_port)->set_policy(GTK_POLICY_AUTOMATIC, 
//						       GTK_POLICY_AUTOMATIC);
    ((Gtk::ScrolledWindow*)left_pane_port)->set_policy(GTK_POLICY_NEVER, 
						       GTK_POLICY_NEVER);
    ((Gtk::ScrolledWindow*)left_pane_port)->add(*vp);
    vp->add( *obook );


    obook->setMode(
	Preferences::global_prefs->getOptStr(
	    "/FWBuilderPreferences/UI/ObjectTreeMode") );

    left_pane_port->set_usize(150,-1);
    left_pane_port->size_allocate.connect(SigC::slot(this, 
			  &main_window::on_left_pane_size_allocate_event));
    hpaned->pack1( *left_pane_port , false, false );

    right_vb  = new class Gtk::VBox();

    hpaned->pack2( *right_vb , true, true);

    right_pane_port = new Gtk::Viewport();
    right_pane_port->add( *(new Gtk::Label("")) ); // placeholder
    right_pane_port->size_allocate.connect(SigC::slot(this, 
			  &main_window::on_right_pane_size_allocate_event));

    right_vb->pack_start(*right_pane_port, true, true);

    main_menubar1->set_shadow_type( GTK_SHADOW_NONE );  // glade bug?

    agr = (Gtk::AccelGroup*)Gtk::AccelGroup::get_default();
/*
 * Mask here __enables__ different (actually all) modifiers. If mask is
 * set to 0, then all accelerators silently ignore modifiers
 * (Ctrl, Shift etc)
 */
    agr->set_default_mod_mask( GDK_MODIFIER_MASK );
    add_accel_group( *agr );

    build_menu();  // needs AccelGroup object 

    delete_event.connect( SigC::slot(this,&main_window::on_delete_event));

    if ( ! Preferences::global_prefs->getOptBool("/FWBuilderPreferences/UI/HideNavigationBar") )
        showNavbar();
}

main_window::~main_window()
{
    if ( Preferences::global_prefs->getOptBool("/FWBuilderPreferences/UI/RememberWindowPositionAndSize") ) 
	saveWindowConfiguration();

    try
    {
	Preferences::global_prefs->savePrefs();
    } catch(FWException &ex)
    {
	MessageDialog::Error(_("Error saving preferences."),ex.toString(),this);
    }

    left_pane_port->hide();
    delete left_pane_port;

    right_pane_port->hide();
    delete right_pane_port;

    if (navbar!=NULL) delete navbar;

    delete right_vb;

    delete save_statusbar;
    delete main_statusbar;
    delete statusbar_hbox;
    delete hpaned;
    delete main_menubar1;
}

/*
 * This event is generated in responce to the window manager's signal
 * sent to the window when user clicks "close" button on window's
 * title bar 
 */
gint main_window::on_delete_event(GdkEventAny* ev)
{
    if (windows.size()>1) {
        if ( safe_to_close_right_pane()) {
            windows.remove(this);
            return false;
        }
        return true;
    }

// this is the last window
    if ( checkAndSaveData() ) {
        windows.remove(this);
        Gtk::Main::quit();
	return false;
    }
    return true;
}

void main_window::on_close_activate()
{   
    if (windows.size()>1) {
        if ( safe_to_close_right_pane()) {
            windows.remove(this);
            delete this;
        }
        return;
    }

// this is the last window
    if ( checkAndSaveData() ) {
        windows.remove(this);
        Gtk::Main::quit();
    }
}

void main_window::on_exit1_activate()
{   
    if ( checkAndSaveData() ) {
//        ClearRightPane(false);
	Gtk::Main::quit();
    }
}


void main_window::showNavbar()
{
    if (navbar==NULL) {
        navbar=new Navbar(this);
        right_vb->pack_start(*navbar, false, false );
        right_vb->reorder_child(*navbar,0);
    }
}

void main_window::hideNavbar()
{
    if (navbar!=NULL) {
        right_vb->remove(*navbar);
        navbar=NULL;
    }
}

void main_window::updateNavbar(libfwbuilder::FWObject* obj)
{
    if (navbar!=NULL) navbar->update(obj);
}

void main_window::updateNavbar(const std::string &id)
{
    if (navbar!=NULL) navbar->update(id);
}

void main_window::on_left_pane_size_allocate_event(GtkAllocation *all)
{
    left_pane_width  = all->width;
    left_pane_height = all->height;
}

void main_window::on_right_pane_size_allocate_event(GtkAllocation *all)
{
    right_pane_width  = all->width;
    right_pane_height = all->height;
}

void main_window::LoadObjDB(const string &data_file,bool check_perm)
{   
    assert(FWObjectDatabaseGUI::getInstance() != NULL);

    try
    {
        FWObjectDatabaseGUI::getInstance()->load(data_file,check_perm);
    } catch(FWException &ex)
    {
        string trans = ex.getProperties()["failed_transformation"];
        string elem  = ex.getProperties()["failed_element"];

        if(!trans.empty() && !elem.empty())
        {
            string msg = _("Exception: : ") + ex.toString();
            if(!trans.empty())
                msg+=_("\nFailed transformation : ") + trans;
            if(!elem.empty())
                msg+=_("\nXML element : ") + elem;

            MessageDialog::LongTextError(_("Error loading data file. "), msg, this);
        } else
            MessageDialog::Error(_("Error loading data file. "), ex.toString(), this);
    }

    showSaveStatusbar(false);
    refreshAllObjectBooks();

//    setCurrentLibrary("");
}

main_window *main_window::createNewMainWindow()
{
    main_window* mw = new main_window();
    if ( Preferences::global_prefs->getOptBool(
	     "/FWBuilderPreferences/UI/RememberWindowPositionAndSize") ) 
    {
	mw->restoreWindowConfiguration();
    } 
    mw->restoreWindowPosition();
    mw->obook->build();
    mw->show_all();
    return mw;
}

void main_window::on_new_wnd_activate()
{
    main_window* mw=createNewMainWindow();
    mw->schedule_open_object(FWObjectDatabaseGUI::db->getRootId(),"");
}

void main_window::on_new1_activate()
{   
    if(safe_to_close_right_pane()) 
    {
        string dbTemplateFile = TEMPLATE_DIR  "/" + FWObjectDatabaseGUI::DB_TEMPLATE_FILE_NAME;
	ClearRightPane(true);
	LoadObjDB(dbTemplateFile,false);
        FWObjectDatabase::db->setFileName("");
        setTitle("");
    }
}

void main_window::openFile(const string &file,bool update_recent)
{
    if (safe_to_close_right_pane()) 
    {
	ClearRightPane(true);
	LoadObjDB(file);
        setTitle( FWObjectDatabaseGUI::getInstance()->getFileName() );
        if (update_recent) updateRecentlyOpenedFiles( FWObjectDatabaseGUI::getInstance()->getFileName() );
    }
}

void main_window::on_open1_activate()
{   
    FileSel *fs=new FileSel(_("Select file"),
                            Preferences::global_prefs->getWdir(),
                            "*.xml");
    string df=fs->run();
    delete fs;
    if (df=="") return;

    openFile(df,true);
}

void main_window::on_open_recent_file1_activate()
{
    openFile( Preferences::global_prefs->getOptStr("/FWBuilderPreferences/History/File1") ,false );
}

void main_window::on_open_recent_file2_activate()
{
    openFile( Preferences::global_prefs->getOptStr("/FWBuilderPreferences/History/File2") ,false );
}

void main_window::on_open_recent_file3_activate()
{
    openFile( Preferences::global_prefs->getOptStr("/FWBuilderPreferences/History/File3") ,false );
}

void main_window::on_open_recent_file4_activate()
{
    openFile( Preferences::global_prefs->getOptStr("/FWBuilderPreferences/History/File4") ,false );
}

void main_window::on_save1_activate()
{   
    assert (FWObjectDatabase::db!=NULL);
    if ( ! safe_to_close_right_pane()) return ;
    try
    {
        FWObjectDatabaseGUI::getInstance()->save(); 
        showStatusbarMsg(string(_("Saved as: "))+FWObjectDatabaseGUI::getInstance()->getFileName() );
        showSaveStatusbar(false);
    } catch(FWException &ex)
    {
	MessageDialog::Error(_("Error saving data file."),ex.toString(),this);
    }
}

void main_window::on_saveas1_activate()
{   
    assert (FWObjectDatabase::db!=NULL);
    if ( ! safe_to_close_right_pane()) return ;
    try
    {
        FWObjectDatabaseGUI::getInstance()->saveAs(); 
        setTitle( FWObjectDatabaseGUI::getInstance()->getFileName() );
        showStatusbarMsg(string(_("Saved as: "))+FWObjectDatabaseGUI::getInstance()->getFileName() );
        showSaveStatusbar(false);
    } catch(FWException &ex)
    {
        if ( ex.toString()!="" )
            MessageDialog::Error(_("Error saving data file."),ex.toString(),this);
    }
}

bool main_window::checkAndSaveData()
{
    if ( ! safe_to_close_right_pane()) return false;
    if (FWObjectDatabase::db->isDirty(true) ) 
    {
        try  
        {
            if (! FWObjectDatabaseGUI::getInstance()->saveIfModified() ) return false;  // "Cancel"
        } catch(FWException &ex)  {
	    MessageDialog::Error(_("Error saving data file."),ex.toString(),this);
            return false;
        }
    }
    return true;
}

void main_window::on_print_activate()
{   
    if(checkAndSaveData())
    {
        PrintDialog *pd = 
	    new PrintDialog(obook->getCurrentSelection()!="" ?
			    obook->getCurrentSelection().c_str():NULL
        );
        try
        {
            pd->set_transient_for( *this );
            pd->run();
        } catch(FWException &ex)
        {
            MessageDialog::Error(_("Print error."),ex.toString(),this);
        }
	delete pd;
    }
}

void main_window::on_edit1_activate()
{   
}

void main_window::on_options_activate()
{   
    bool show_properties=Preferences::global_prefs->getOptBool(
	"/FWBuilderPreferences/UI/ShowObjectPropertiesInTree");
    bool show_icons=Preferences::global_prefs->getOptBool(
	"/FWBuilderPreferences/UI/ShowIconsInTree");

    OptionsDlg *pd=new OptionsDlg();
    bool ok=pd->run();
    delete pd;

    if (ok) 
    {
	obook->setMode(Preferences::global_prefs->getOptStr(
			   "/FWBuilderPreferences/UI/ObjectTreeMode") );

	if (show_properties!=Preferences::global_prefs->getOptBool(
		"/FWBuilderPreferences/UI/ShowObjectPropertiesInTree") 
            ||
            show_icons!=Preferences::global_prefs->getOptBool(
                "/FWBuilderPreferences/UI/ShowIconsInTree") 
        )
	    obook->build();

        if (Preferences::global_prefs->getOptBool(
                "/FWBuilderPreferences/UI/HideNavigationBar") )
            hideNavbar();
        else
            showNavbar();

        Gtk::MenuBar *main_menubar=getMainMenuBar();
        Gtk::Widget *itm;

        itm=find_widget("sort_by_prop",main_menubar);
        if (itm) itm->set_sensitive(
            (Preferences::global_prefs->getOptBool(
                "/FWBuilderPreferences/UI/ShowObjectPropertiesInTree")));

/*
 * can't simply call schedule_open_object because if currently shown dialog
 * has unsaved data, it will be lost
 */
	BuiltinDialog *dlg=getRightPaneDialog();
	if (dlg) 
        {
/*
 * User might have changes something in certificates, let's refresh list if
 * what we have in the right pane is firewall dialog
 */
	    DialogPlugin*  dp=dlg->getDialog();
	    if (dp!=NULL && dynamic_cast<FirewallDialog*>(dp)!=NULL) {
		dynamic_cast<FirewallDialog*>(dp)->fillListOfCertificates();
	    }
	}
    }
}

void main_window::on_about1_activate()
{   
    AboutDialog::showAboutDialog();
}

               

void main_window::on_release_notes_activate()
{
    string file=DOCDIR "/ReleaseNotes_" VERSION;
    std::ifstream rn( file.c_str() );
    if (!rn) {
	MessageDialog::Error(_("Can not open Release Notes file ") + file , this);
	return;
    }
    string buf, s;

    while (getline(rn,s)) { buf+=s; buf+="\n"; }

    MessageDialog::LongTextInfo( buf , this);
}

void  main_window::setTitle(const string &msg)
{
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) 
    {
        (*i)->set_title(_("Firewall Builder: ")+msg);
    }
}

void  main_window::showSaveStatusbar(bool needs_saving)
{
    if (FWObjectDatabase::db->isDirty(true) ) 
    {
        for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) {
            if ( (*i)->needs_saving_icon ==NULL ) {
                string icn = Resources::global_res->getIconPath("Floppy");
                SimplePixmap *pix=new SimplePixmap(icn);
                (*i)->save_statusbar->remove();
                (*i)->save_statusbar->add(*pix);
                pix->show();
            }
        }
    }
    else 
    {
        for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) {
            (*i)->save_statusbar->remove();
        }
    }
}


void main_window::showStatusbarMsg(const string& msg)
{
    clearStatusbar();
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) 
    {
        (*i)->main_statusbar->remove();
        (*i)->main_statusbar->add_label(string("  ")+msg );
    }
}

void main_window::clearStatusbar()
{
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) 
    {
        (*i)->main_statusbar->remove();
    }
}

RuleSetList* main_window::getRuleSetListWidget()
{
    RuleSetList *pl=NULL;

    if (current_right_pane_dialog) {
	DialogPlugin *dp= current_right_pane_dialog->getDialog();
	
	if (dp->get_name()=="policy_dialog")
	    pl=((PolicyDialog*)dp)->getPolicyList();
	
	if (dp->get_name()=="nat_dialog")
	    pl=((NATDialog*)dp)->getPolicyList();

//	if (dp->get_name()=="firewall_dialog")
//	    pl=((FirewallDialog*)dp)->getPolicyList();
    }
    return pl;
}

/*
 * Submenu Policy
 */
void main_window::on_insert_top()    
{ 
    if (current_right_pane_dialog==NULL) return;
    RuleSetList *pl=getRuleSetListWidget();
    if (pl!=NULL) pl->insertRuleAtTop(); 
}

void main_window::on_append_bottom() 
{ 
    if (current_right_pane_dialog==NULL) return;
    RuleSetList *pl=getRuleSetListWidget();
    if (pl!=NULL) pl->appendRuleAtBottom();  
}

void main_window::on_add_rule()
{
    if (current_right_pane_dialog==NULL) return;
    RuleSetList *pl=getRuleSetListWidget();
    if (pl!=NULL) pl->appendRuleAfter(-1);
}

void main_window::on_remove_rule()   
{ 
    if (current_right_pane_dialog==NULL) return;
    RuleSetList *pl=getRuleSetListWidget();
    if (pl!=NULL) pl->delRule(-1);
}

void main_window::on_move_up()       
{ 
    if (current_right_pane_dialog==NULL) return;
    RuleSetList *pl=getRuleSetListWidget();
    if (pl!=NULL) pl->moveRuleUp(-1);
}

void main_window::on_move_down()     
{
    if (current_right_pane_dialog==NULL) return;
    RuleSetList *pl=getRuleSetListWidget();
    if (pl!=NULL) pl->moveRuleDown(-1);
}

void main_window::on_build_autorules()
{
    if (current_right_pane_dialog==NULL) return;

    if (safe_to_close_right_pane()) {

	FWObject   *obj=nowShowingObject();

	while (obj && ! Firewall::isA(obj)) obj=obj->getParent();
	if (obj) {
//		ClearRightPane(false);

	    StandardRulesDruid *dr=new StandardRulesDruid( obj->getId(),
						   _("Standard policy rules"),
		       Resources::global_res->getIconPath("RulesDruidLogo") );


	    dr->show_all();
#ifdef __MINGW32__
            __fwb_win32_loadIcon("FWBuilderWin32Ico",dr);
#endif

	}

    }
}

void main_window::on_compile()
{
    if (current_right_pane_dialog==NULL) return;

    if (safe_to_close_right_pane()) 
    {
	FWObject   *obj=nowShowingObject();
	if (obj) 
        {
            while (obj!=NULL && !Firewall::isA(obj)) obj=obj->getParent();
            
	    if (Firewall::isA(obj))   
            {
		on_save1_activate();
// call compiler now

		Firewall  *fw=Firewall::cast(obj);
		FWOptions *fwopt=fw->getOptionsObject();

/*
 * I want to be able to specify custom compiler for firewall with
 * no platform (e.g. for experiments)
 */
		string compiler=fwopt->getStr("compiler");
		if (compiler=="") 
                {
                    compiler=Resources::platform_res[fw->getStr("platform")]->getCompiler();
		}

		if (compiler=="")
                {
                    MessageDialog::Error(
_("Firewall platform is not specified in this object.\n\
Can't compile firewall policy.")
                    );
                    return;
                }
                string wdir=Preferences::global_prefs->getWdir();
                if (wdir.empty())
                {
                    string file=FWObjectDatabase::db->getFileName();
                    string::size_type n=file.rfind("/");
                    if (n!=string::npos) wdir=file.substr(0,n);
                }
		ExecDialog *cmpl=
		    new ExecDialog(compiler , 
				   fwopt->getStr("cmdline") ,
				   FWObjectDatabase::db->getFileName(),
				   fw->getName() ,
				   wdir
		    );

		cmpl->execute();
		delete cmpl;
	    }
	}
    }
}

void main_window::on_install()
{
    if (current_right_pane_dialog==NULL) return;

    if (safe_to_close_right_pane()) 
    {
	FWObject   *obj=nowShowingObject();
	if (obj) 
        {
            while (obj!=NULL && !Firewall::isA(obj) ) obj=obj->getParent();

	    if (obj!=NULL && Firewall::isA(obj))   
            {
		on_save1_activate();
// call install script now

		Management *mgmt=(Firewall::cast(obj))->getManagementObject();
		assert(mgmt!=NULL);
		PolicyInstallScript *pis   = mgmt->getPolicyInstallScript();

		if ( ! pis->isEnabled() )
                {
		    MessageDialog::Error(_("Install script is disabled"),this);
		    return;
		}

		if ( pis->getCommand()=="" )
                {
		    MessageDialog::Error(_("Install script is not configured"),this);
		    return;
                }

		string inst_script=pis->getCommand();
            
                string wdir=Preferences::global_prefs->getWdir();
                if (wdir.empty())
                {
                    string file=FWObjectDatabase::db->getFileName();
                    string::size_type n=file.rfind("/");
                    if (n!=string::npos) wdir=file.substr(0,n);
                }
		ExecDialog *inst=
		    new ExecDialog(inst_script , 
				   pis->getArguments(), 
				   FWObjectDatabase::db->getFileName(),
				   obj->getName() ,
				   wdir
		    );

		inst->execute();
		delete inst;
	    }
	}
    }
}

void main_window::RefreshRightPane()
{
    BuiltinDialog *bd=getRightPaneDialog();
    if (bd!=NULL) bd->getDialog()->conditionalRefresh();
}

void  main_window::refreshAllWindows()
{
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) {
        BuiltinDialog *bd=(*i)->getRightPaneDialog();
        if (bd!=NULL) {
            bd->getDialog()->conditionalRefresh();
        }
    }
}

void main_window::conditionalRefreshAllDialogs(libfwbuilder::FWObject* obj,
                                               main_window *mwin)
{
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) 
    {
        if (mwin!=NULL && (*i)==mwin) continue;
        BuiltinDialog *bd=(*i)->getRightPaneDialog();
        if (bd!=NULL) {
            bd->getDialog()->conditionalRefresh(obj);
        }
    }
}


void  main_window::refreshAllObjectBooks()
{
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) 
    {
        (*i)->getObjectBook()->rebuild();
    }
}

void  main_window::sortSubtreeInAllObjectBooks(const string &id,int col)
{
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) 
    {
        (*i)->getObjectBook()->sortSubtree(id,col);
    }
}


void  main_window::insertInAllObjectBooks(FWObject *o)
{
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) 
    {
	(*i)->getObjectBook()->showPage(0);  // User-defined objects
        (*i)->getObjectBook()->insertObject(o);
    }
}

void  main_window::removeFromAllObjectBooks(FWObject *o)
{
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) 
    {
        (*i)->getObjectBook()->removeObject(o);
    }
}
     
void  main_window::changeLabelInAllObjectBooks(const std::string &id)
{
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) 
    {
        (*i)->getObjectBook()->changeTreeLabel( id );
    }
}

main_window* main_window::getMainWindowForWidget(Gtk::Widget *w)
{
    Gtk::Widget *p=w;
    while (p!=NULL && dynamic_cast<main_window*>(p)==NULL) p=p->get_parent();
    return dynamic_cast<main_window*>(p);
}

main_window* main_window::getAnotherMainWindowForWidget(Gtk::Widget *w)
{
    main_window *mw=getMainWindowForWidget(w);
    for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) {
        if (mw!=(*i)) return (*i);
    }
/*
 * there is only one window and widget w is in it, create new window.
 */
    return createNewMainWindow();
}

string main_window::nowShowingObjectId()
{
    FWObject   *obj=nowShowingObject();
    return (obj)?obj->getId():"";
}

FWObject* main_window::nowShowingObject()
{
    if (current_right_pane_dialog) {
	DialogPlugin *dp= current_right_pane_dialog->getDialog();
	return (dp)?dp->getObject():NULL;
    }
    return NULL;
}

void main_window::on_tools_scan()
{
    string icn=
	Resources::global_res->getResourceStr("/FWBuilderResources/Paths/Icndir")+"/"+
	Resources::global_res->getResourceStr("/FWBuilderResources/UI/Icons/Discover");



    DiscoveryDruid *dr=new DiscoveryDruid(this,_("Objects Discovery"),icn);



    dr->show_all();
#ifdef __MINGW32__
    __fwb_win32_loadIcon("FWBuilderWin32Ico",dr);
#endif

}

/*
 *  for reference:
 *
    struct windowGeometry {
	Gdk_Rectangle wnd;
	Gdk_Rectangle trwnd;
	Gdk_Rectangle objwnd;
	int           divider;
    };
*/
void main_window::readGeometry(windowGeometry &c)
{
    int x,y,w,h;

    x  =Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/MainWindowX");
    y  =Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/MainWindowY");
    w  =Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/MainWindowW");
    h  =Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/MainWindowH");
    c.wnd=Gdk_Rectangle(x,y,w,h);


    x=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/TreeWindowX");
    y=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/TreeWindowY");
    w=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/TreeWindowW");
    h=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/TreeWindowH");
    c.trwnd=Gdk_Rectangle(x,y,w,h);


    x=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/ObjWindowX" );
    y=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/ObjWindowY" );
    w=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/ObjWindowW" );
    h=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/ObjWindowH" );
    c.objwnd=Gdk_Rectangle(x,y,w,h);

    c.divider=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/MainWindowDivider");

}

void main_window::writeGeometry(windowGeometry &c)
{
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/MainWindowX",c.wnd->x);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/MainWindowY",c.wnd->y);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/MainWindowW",c.wnd->width);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/MainWindowH",c.wnd->height);

    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/MainWindowDivider",c.divider);

    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/TreeWindowX",c.trwnd->x);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/TreeWindowY",c.trwnd->y);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/TreeWindowW",c.trwnd->width);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/TreeWindowH",c.trwnd->height);
    


    
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/ObjWindowX" ,c.objwnd->x);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/ObjWindowY" ,c.objwnd->y);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/ObjWindowW" ,c.objwnd->width);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/ObjWindowH" ,c.objwnd->height);
}


void main_window::restoreWindowConfiguration()
{
    windowGeometry g;

    readGeometry(g);

    setWindowConfiguration(g.wnd);
    setDividerPosition(g.divider);
}

void main_window::restoreWindowPosition()
{
    int i, w_display, h_display;

    windowGeometry g;

    readGeometry(g);

    gdk_window_get_geometry(NULL, &i, &i, &w_display, &h_display, &i);

    if (g.wnd->x>0      && g.wnd->x<w_display-100 && 
	g.wnd->y>0      && g.wnd->y<h_display-100 )
    {
        setWindowPosition(g.wnd->x,g.wnd->y);
    }
}

void main_window::saveWindowConfiguration()
{
    windowGeometry g;


/*
 *  if user closed window by clicking on "x" (via "delete" event) and not
 *  via menu item Quit, then window is going to be invisible at this point.
 *  In this case its size and coordinates are going to be equal to 0 and we
 *  should not store them in preferences file
 */
    if (is_visible()) 
    {
	getWindowConfiguration(g.wnd);
	g.divider=getDividerPosition();
    } else 
    {
	readGeometry(g);
    }

    writeGeometry(g);

}

void main_window::getWindowConfiguration(Gdk_Rectangle &r)
{
    int a,b;

    GtkWidget *widget=GTK_WIDGET(gtkobj());
    if (widget->window) {
	gdk_window_get_position(widget->window, &a, &b);
	r->x=a; r->y=b;
	gdk_window_get_size(widget->window, &a, &b);
	r->width=a; r->height=b;
    }
}

void main_window::setWindowConfiguration(Gdk_Rectangle &r)
{
    setWindowPosition(r->x,r->y);
    set_default_size( r->width,r->height );
}

void main_window::setWindowPosition(gint16 x , gint16 y)
{
    if (is_visible()) {
//	GtkWidget *widget=GTK_WIDGET(gtkobj());
//	gdk_window_move(widget->window, x, y);
	set_uposition(x,y);
    }
}


int  main_window::getDividerPosition()         { return left_pane_width;     }
void main_window::setDividerPosition(int size) 
{ 
    left_pane_width=size;
    hpaned->set_position(size); 
}

void main_window::show_impl()
{
    Gtk::Widget::show_impl();
}

void main_window::show_all_impl()
{
    Gtk::Widget::show_all_impl();
}
