//====================================================================
//                
// Copyright (C) 2006,  Goedson Teixeira Paixao
//                
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307, USA.
//                
// Filename:      IconTheme.cc
// Author:        Goedson Teixeira Paixao <goedson@debian.org>
// Description:   Implementation of the support for icon themes in the
//                format used by gaim emoticon themes.
//                
// Created at:    Wed May 10 17:45:33 2006
// Modified at:   Thu May 11 20:02:39 2006
// Modified by:   Goedson Teixeira Paixao <goedson@debian.org>
//====================================================================

#include "IconTheme.hh"
#include "intl.h"

#include <fstream>
#include <iostream>
#include <list>
#include <glibmm/miscutils.h>
#include <glibmm/unicode.h>
#include <algorithm>
#include <functional>
#include "GabberApp.hh"
#include "ConfigPaths.hh"

Gabber::IconTheme::IconTheme(std::string theme_name, std::string base_dir)
{
    std::list<std::string> path_elements;
    path_elements.push_back(G_App.getConfigurator().get_string(Keys::paths.pixmapdir));
    path_elements.push_back(base_dir);
    path_elements.push_back(theme_name);
    _theme_path = Glib::build_filename(path_elements);
    path_elements.push_back("theme");
    std::string theme_file_name = Glib::build_filename(path_elements);
    std::ifstream theme_file(theme_file_name.c_str());
    
    if (theme_file)
    {
        const int buffer_size = 1000;
        char buffer[buffer_size];
        std::string current_section = "header";
        std::string spaces = " \t";
        while(!theme_file.eof())
        {
            theme_file.getline(buffer, buffer_size);
            std::string line(buffer);

            std::string::iterator first_non_space = find_if(line.begin(), line.end(), not1(ptr_fun(&Glib::Unicode::isspace)));
            
// if it's an empty or comment line, skip it
            if (first_non_space == line.end() || *first_non_space == '#' || *first_non_space == '!')
            {
                continue;
            }
            
            if (*first_non_space == '[')
            {
                current_section = std::string(first_non_space,find(line.begin(), line.end(), ']'));
                continue;
            }
            
            if (current_section == "header")
            {
                std::string::iterator equal_sign = find(first_non_space,line.end(),'=');
                std::string property_name(first_non_space,equal_sign);
                std::string property_value(++equal_sign,line.end());
                _properties[property_name] = property_value;
            }
            else
                {
                    std::string icon_file(first_non_space,find_if(first_non_space, line.end(), ptr_fun(&Glib::Unicode::isspace)));
                    std::string::iterator end_of_value = find_if(first_non_space, line.end(), ptr_fun(&Glib::Unicode::isspace));
                    std::string::iterator start_of_value = first_non_space;
                    do
                    {
                        start_of_value = find_if(end_of_value, line.end(), not1(ptr_fun(&Glib::Unicode::isspace)));
                        end_of_value = find_if(start_of_value, line.end(), ptr_fun(&Glib::Unicode::isspace));
                        
                        if (end_of_value != start_of_value)
                        {
                            std::string icon_text(start_of_value, end_of_value);
                            _icons[icon_text] = icon_file;
                        }

                    }
                    while (end_of_value != line.end());
                }
        }
    }
    else // Could not open the file
    {
        std::cerr << "Could not load theme file:" << theme_file_name << std::endl;
    }
    
}

Gabber::IconTheme::~IconTheme()
{
    _icons.clear();
    _properties.clear();
}

std::string Gabber::IconTheme::get_icon_path(std::string name)
{
    if (_icons.find(name) == _icons.end())
    {
        return "";
    }
    else
    {
        return build_icon_path(_icons[name]);
    }
}

std::string Gabber::IconTheme::build_icon_path(std::string icon)
{
    return Glib::build_filename(_theme_path, icon);
}

Gabber::EmoticonTheme::EmoticonTheme(std::string theme): Gabber::IconTheme::IconTheme(theme, "emoticons")
{
}

Gabber::UserInterfaceTheme::UserInterfaceTheme(std::string theme): IconTheme(theme, "ui")
{
}
