/*
 * tiers.c
 * Thomas Nemeth, le 22.02.2000
 *
 * Gestion de la liste des donnes de gAcc, gestionnaire de comptes
 * banquaires personnels avec GTK+.
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <string.h>
#include "structs.h"
#include "tiers.h"
#include "operations.h"
#include "fileaccess.h"
#include "usefull.h"


/* ------++++++======*** TIERS ******* 3RD PARTY ***======++++++----- */

void read_trs_from_last_version (FILE *file) {
    TRS_ELT *lst, *prev = NULL;
    char     tmpstr[MAXSTRLEN+1];

	fgets (tmpstr, (int) MAXSTRLEN, file); // read separator
	while ( ! feof (file) ) {
		if (tmpstr[0] != 0 && strlen(tmpstr)>5) {
#if DEBUG_LEVEL>4
			printf ("read_trs_from_last_version line=%s",tmpstr);
#endif
			MY_ALLOC (lst, 1, TRS_ELT);
			lst->next = NULL;
			MY_ALLOC (lst->tiers, 1, TIERS);
			lst->tiers = get_tiers_from_line (tmpstr);
			if(prev != NULL) {
				prev->next = lst;
			}
			prev = lst;
			if(config.trs_list_head==NULL) {
				config.trs_list_head = lst;
			}
		}
		fgets (tmpstr, (int) MAXSTRLEN, file);
	}
}

void read_tiers_from_file () {
    FILE    *file;
    char    *FileVersion;

    FileVersion = open_file (config.trsfilename, &file);
	if(FileVersion != NULL) {
#if DEBUG_LEVEL>2
		printf ("Version du fichier tiers : %s\n", FileVersion);
		printf ("Version du programme : %s\n", VERSION);
#endif
		if (in_versions ("0.6.9", VERSION, FileVersion) == VRAI) {
			read_trs_from_last_version (file);
		} else if (strcmp (FileVersion, VERSION) > 0) {
			printf ("File version : %s -- Program version : %s\n",
					FileVersion, VERSION);
			printf (_("tiers file version not supported by this one. Sorry !\n") );
		} else {
			printf ("File version : %s -- Program version : %s\n",
					FileVersion, VERSION);
			printf ("Cannot open %s\n",config.trsfilename);
		}
		// do not change TrsDataModified status
		//    config.TrsDataModified = FAUX;
		if(strlen(FileVersion)>3)
			fclose (file);
		free (FileVersion);
	}
}

void write_tiers_to_file () {
    TRS_ELT *lst = config.trs_list_head;
    FILE    *file;
    char    *line;

    if ( (file = fopen(config.trsfilename, "w") ) == NULL) {
        fatal_error (config.trsfilename);
    } else {
        create_file_header (file, "Tiers description file");
        while (lst != NULL) {
            line = create_line_from_tiers (lst->tiers);
            fprintf (file, "%s", line);
#if DEBUG_LEVEL>4
			printf ("write_tiers_to_file line=%s",line);
#endif
            free (line);
            lst = lst->next;
        }
        fclose (file);
        config.TrsDataModified = FAUX;
    }
}

void add_tiers (TIERS *tiers) {
    TRS_ELT *trs        = config.trs_list_head;
    TRS_ELT *prev_trs   = NULL;
    TRS_ELT *support    = NULL;
    int      go_to_next = VRAI;

    if (tiers->is_in_file == FAUX) {
#if DEBUG_LEVEL>4
		printf ("STORING a \"NOT IN FILE\" tiers\n");
		if(trs != NULL) {
			printf ("tiers \"%s\" is a %s tiers\n",
					trs->tiers->name,
					trs->tiers->is_in_file == VRAI ?
					"\"IN FILE\"" : "\"NOT IN FILE\"");
		} else {
			printf ("No tiers yet...\n");
		}
#endif
        while ( (trs != NULL) && (trs->tiers->is_in_file == VRAI) ) {
            prev_trs = trs;
            trs      = trs->next;
#if DEBUG_LEVEL>4
			if (trs != NULL) {
				printf ("tiers \"%s\" is a %s tiers\n",
						trs->tiers->name,
						trs->tiers->is_in_file == VRAI ?
						"\"IN FILE\"" : "\"NOT IN FILE\"");
			} else {
				printf ("There is no other tiers...\n");
			}
#endif
        }
    }

    while ( (trs != NULL) && (go_to_next == VRAI) ) {
        prev_trs = trs;
        trs      = trs->next;
        if (trs != NULL)
            go_to_next = (tiers->is_in_file == trs->tiers->is_in_file);
    }
    MY_ALLOC (support, 1, TRS_ELT);
    support->tiers      = tiers;
    support->next       = trs;
    if (prev_trs != NULL)
        prev_trs->next  = support;
    else
        config.trs_list_head = support;
#if DEBUG_LEVEL>4
	printf ("add_tiers %s=%s %d=%d %d=%d\n",
			support->tiers->name,tiers->name,
			support->tiers->num,tiers->num,
			support->tiers->is_in_file,tiers->is_in_file);
#endif
    config.TrsDataModified   = VRAI;
}

void remove_tiers (TIERS *tiers) {
    TRS_ELT *trs, *prev;

    trs  = config.trs_list_head;
    prev = NULL;
    while ( (trs != NULL) && (trs->tiers != tiers) ) {
        prev = trs;
        trs  = trs->next;
    }
    if (trs->tiers == tiers) {
        #ifdef DEBUG_GACC
            printf ("REMOVING \"%s\" before \"%s\"...\n",
                    trs->tiers->name,
                    trs->next != NULL ? trs->next->tiers->name : "END");
        #endif
        if (prev != NULL)
            prev->next = trs->next;
        else
            config.trs_list_head = trs->next;

        free (trs);
        config.TrsDataModified = VRAI;
    } else
        printf ("remove_tiers : tiers not found !\n");
}

void del_tiers (TRS_ELT *element) {
    TRS_ELT *trs, *prev;
    ACC_ELT *acc = config.acc_list_head;
    OPE_ELT *ope = NULL;
    int num = element->tiers->num;

    while (acc != NULL) {
        ope = acc->account->ope_list_head;
        while (ope != NULL) {
            if (ope->operation->tiers == num)
                ope_set_tiers (ope->operation, UNKNOWN_TIERS);
            ope = ope->next;
        }
        acc = acc->next;
    }

    prev = NULL;
    trs  = config.trs_list_head;
    while ( (trs->next != NULL) && (trs != element) ) {
        prev = trs;
        trs  = trs->next;
    }

    if (prev != NULL)
        prev->next = element->next;
    else
        config.trs_list_head = element->next;

    if ( (element->next == NULL) && (prev != NULL) ) prev->next = NULL;

    free (element->tiers->name);
    free (element->tiers);
    free (element);
    config.TrsDataModified = VRAI;
}

TRS_ELT *get_tiers (int trs_num) {
    TRS_ELT *trs = config.trs_list_head;
    int      i = 0;

    while ( (i != trs_num) && (trs != NULL) ){
        trs = trs->next;
        i++;
    }

    return trs;
}

TRS_ELT *get_tiers_with_name (const char *trs_name) {
    TRS_ELT *trs = config.trs_list_head;

    while ( (trs != NULL) && (strcmp (trs->tiers->name, trs_name) != 0) ){
        trs = trs->next;
    }

    return trs;
}

TRS_ELT *get_tiers_with_num (int trs_num) {
    TRS_ELT *trs = config.trs_list_head;

    while ( (trs != NULL) && (trs->tiers->num != trs_num) ){
        trs = trs->next;
    }

    return trs;
}

int get_tiers_next_num () {
    TRS_ELT *trs     = config.trs_list_head;
    int      max_num = UNKNOWN_TIERS;

    while (trs != NULL && trs->tiers != NULL) {
        if (trs->tiers->num > max_num) {
			max_num = trs->tiers->num;
		}
        trs = trs->next;
    }
#if DEBUG_LEVEL>4
	printf ("get_tiers_next_num %d\n",max_num + 1);
#endif
    return max_num + 1;
}

char *tiers_name (int trs_num) {
    TRS_ELT *trs = config.trs_list_head;
    char    *name = NULL;

    if (trs != NULL)
        while ( (trs != NULL) && (trs->tiers->num != trs_num) )
            trs = trs->next;

    if (trs != NULL)
        name = trs->tiers->name;
    else if (trs_num != UNKNOWN_TIERS) {
        MY_ALLOC (name, 6, char);
        sprintf (name , "%d", trs_num);
    }

    return name;
}

int tiers_number (char *trs_name) {
    TRS_ELT *trs = config.trs_list_head;
	int num = UNKNOWN_TIERS;

	if(trs_name == NULL || strlen(trs_name)==0) {
		return tiers_number(_("Unknown"));
	} else {
#if DEBUG_LEVEL>4
		printf ("Recherche du numro du tiers : %s\n", trs_name);
#endif
		while (trs != NULL
			   && (trs->tiers == NULL
				   || trs->tiers->name == NULL
				   || strcmp (trs->tiers->name, trs_name) != 0)) {
#if DEBUG_LEVEL>4
			printf ("Le tiers %s n'est pas %s\n",
					trs->tiers->name, trs_name);
#endif
			trs = trs->next;
		}
	}
	if (trs != NULL) {
		num = trs->tiers->num;
	}
	return num;
}

void free_tiers () {
    TRS_ELT *p = config.trs_list_head, *q;

    while (p != NULL) {
        q = p->next;
        free (p->tiers->name);
        free (p->tiers);
        free (p);
        p = q;
    }
    config.trs_list_head = NULL;
}

TIERS *tiers_new () {
    TIERS *trs;

    MY_ALLOC (trs, 1, TIERS);

    trs->name       = NULL;
    trs->num        = UNKNOWN_TIERS;
    trs->is_in_file = FAUX;

    return trs;
}


TIERS *get_tiers_from_line (const char *line) {
    TIERS *trs = tiers_new ();
    char  *elt = NULL;
    int    i = 0, j = 0, k = 0;

    MY_ALLOC (elt, MAXSTRLEN + 1, char);
    for (i = 0 ; i < strlen (line) ; i++) {
        if ( (line[i] != '\t') && (line[i] != '\n') ){
            elt[j] = line[i];
            j++;
        } else {
            elt[j] = 0;
            j      = 0;
#if DEBUG_LEVEL>4
			printf ("Niveau %s -- ",
					k ==  0 ? "NAME" :
					k ==  1 ? "NUM " : "IS_IN_FILE");
			printf ("lement reconnu : %s\n", elt);
#endif
            switch (k) {
			case  0 : if (elt[0] != 0) {
				MY_ALLOC (trs->name, strlen (elt) + 1, char);
				strcpy (trs->name, elt);
			} else trs->name = NULL;
			break;
			case  1 : trs->num  = atoi (elt);
				break;
			case  2 : trs->is_in_file = atoi (elt);
				break;
			default : MY_STOP;
            }
            k++;
        }
    }
#if DEBUG_LEVEL>4
	printf ("get_tiers_from_line %s %d %d\n",
			trs->name, trs->num, trs->is_in_file);
#endif
    free (elt);
	return trs;
}

char *create_line_from_tiers (TIERS *trs) {
    char *line;

    MY_ALLOC (line, MAXSTRLEN + 1, char);
    sprintf (line, "%s\t%d\t%d\n",
             trs->name != NULL ? trs->name : "",
             trs->num,
             trs->is_in_file);
    return line;
}

TRS_ELT *trs_list_elt_new () {
    TRS_ELT *elt;

    MY_ALLOC (elt, 1, TRS_ELT);

    elt->tiers = NULL;
    elt->next  = NULL;

    return elt;
}

void trs_set_name (TIERS *trs, const char *name) {
	if(name == NULL || strlen(name)==0) {
		trs_set_name(trs,_("Unknown"));
	} else {
		if (trs->name != NULL) free (trs->name);
		MY_ALLOC (trs->name, (strlen (name) + 1), char);
		strcpy (trs->name, name);
		config.TrsDataModified = VRAI;
#if DEBUG_LEVEL > 0
		printf("trs_set_name %d %d %d \n",
			   config,
			   config.CatDataModified,
			   config.TrsDataModified);
#endif
	}
}

void trs_set_num (TIERS *trs, int num) {
    if (num > UNKNOWN_TIERS) {
        trs->num = num;
        config.TrsDataModified = VRAI;
    }
}

void trs_set_flag (TIERS *trs, int is_in_file) {
    if ( (is_in_file == VRAI) || (is_in_file == FAUX) ) {
        trs->is_in_file = is_in_file;
        config.TrsDataModified = VRAI;
    }
}
