 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Project: gafix                                                         *
 * File: file.c                                                           *
 * Copyright: (c) 2007 Sebastian Reichel                                  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * This program is free software; you can redistribute it and/or modidy   *
 * it under the terms of GNU Public License as published by               *
 * the Free Software Foundation; either version 2 of the License, or      *
 * (at your option) any later version.                                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <gtk/gtk.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "gafix.h"
#include "tab.h"
#include "file.h"
#include "statusbar.h"
#include "convert.h"

void
new_file ( void )
{
	gint page;

	/* Create new tab */
	page = new_tab("** unsaved **", "");

	/* set tab to the new created one */
	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);
}

void
open_file ( void )
{
	const gchar *filename;
	gchar *extension, *text;
	GString * string;
	gint page;
	GtkWidget *dialog;
	GtkFileFilter *filter;
	
	/* configure file filter */
	filter = gtk_file_filter_new();
	gtk_file_filter_add_pattern (filter, "*.cafix");
	gtk_file_filter_add_pattern (filter, "*.cat");
	gtk_file_filter_set_name(filter, "Gafix compatible files");

	/* User message */
	statusbar_write ("Open File...");

	/* create dialog window */
	dialog = gtk_file_chooser_dialog_new ("Open File",
				      NULL,
				      GTK_FILE_CHOOSER_ACTION_OPEN,
				      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
				      NULL);
	
	/* Set file filter */
	gtk_file_chooser_add_filter( GTK_FILE_CHOOSER(dialog), filter);

	/* Set default dir to Home dir */
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), g_get_home_dir ());


	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		/* get new filename */
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

		/* get extension of filename */
		extension = get_extension_of_file ( filename );

		/* save text string into file */
		g_file_get_contents (filename, &text, NULL, NULL);
	}

	gtk_widget_destroy (dialog);

	if ( strcmp ( extension, "cafix" ) == 0) {
		/* Write user message */
		statusbar_write ("Open cafix file \"%s\"", filename);

		/* get string */
		string = raw_to_cat("muh", text);

		/* Create new tab */
		page = new_tab(filename, string->str);

		/* set tab to the new created one */
		gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);
		
		/* free string */
		g_string_free(string, TRUE);
	}
	else if ( strcmp ( extension, "cat" ) == 0) {
		/* Write user message */
		statusbar_write ("Open cat file \"%s\"", filename);

		/* Create new tab */
		page = new_tab(filename, text);
	
		/* set tab to the new created one */
		gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);
	}
	else {
		statusbar_write ("File is not supported!");
		return;
	}

	/* free filename & text */
	g_free (text);
}

void
save_file ( void )
{
	gint page;
	gchar *text;
	gchar *file;

	statusbar_write ("Save File!");
	
	/* get current notebook page */
	page = gtk_notebook_get_current_page( GTK_NOTEBOOK(notebook) );

	/* get page's text */
	text = get_text_of_tab (page);

	/* get filename of tab */
	file = get_filename_of_tab (page);

	/* start "Save as" if file is unsaved so far */
	if( strcmp ("** unsaved **", file) == 0) {
		return save_file_as();
	}

	/* save text string into file */
	g_file_set_contents (file, text, -1, NULL);
}

void
save_file_as ( void )
{
	gint page;
	char *filename;
	gchar *text;
	GtkWidget *dialog;

	statusbar_write ("Save File as!");
	
	/* get current notebook page */
	page = gtk_notebook_get_current_page( GTK_NOTEBOOK(notebook) );

	/* get page's text */
	text = get_text_of_tab (page);

	/* get filename */
	filename = get_filename_of_tab (page);

	/* file selection */

	dialog = gtk_file_chooser_dialog_new ("Save File",
				      NULL,
				      GTK_FILE_CHOOSER_ACTION_SAVE,
				      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				      GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
				      NULL);

	gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);

	if (strcmp (filename, "** unsaved **") == 0) {
		gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), g_get_home_dir ());
		gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialog), "Untitled document");
	}
	else
		gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (dialog), filename);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		/* get new filename */
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

		/* save text string into file */
		g_file_set_contents (filename, text, -1, NULL);

		/* free filename & text */
		g_free (filename);
		g_free (text);
	}

	gtk_widget_destroy (dialog);
}

void
close_file ( void )
{
	rem_tab ( gtk_notebook_get_current_page( GTK_NOTEBOOK(notebook) ) );
}

void
quit ( void )
{
	gtk_main_quit ();
}

gchar * 
get_extension_of_file ( const gchar * filename )
{
	gchar file[10];
	static gchar output[10];
	gchar *temp;
	unsigned int i;

	/* get the data after the last "." */
	temp = g_strrstr(filename, ".");

	/* make sure, that there are not 
	more then ten chars, but at least one */
	if(temp != NULL) {
		strncpy(file, temp, 10);
	} else {
		return "ERROR!";
	}

	/* remove the "." from the extension, e.g. ".txt" -> "txt" */
	if(file != NULL) {
		for(i = 1; i <= strlen(file); i++) {
			output[i-1] = file[i];
		}
	}

	/* return string */
	return output;
}

gboolean
file_existance ( const gchar * filename )
{
   FILE *f = fopen ( filename, "r" );
   if( errno != ENOENT ) {
         fclose(f);
         return TRUE; /* file exists */
      }
   else
      return FALSE; /* file does not exist */
}
