 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Project: gafix                                                         *
 * File: tab.c                                                            *
 * Copyright: (c) 2007 Sebastian Reichel                                  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

 /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * This program is free software; you can redistribute it and/or modidy   *
 * it under the terms of GNU Public License as published by               *
 * the Free Software Foundation; either version 2 of the License, or      *
 * (at your option) any later version.                                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <gtk/gtk.h>
#include <gtksourceview/gtksourceview.h>
#include <gtksourceview/gtksourcebuffer.h>
#include <gtksourceview/gtksourcetagtable.h>
#include <glib/gprintf.h>
#include <string.h>

#include "statusbar.h"
#include "gafix.h"
#include "tab.h"

gint new_tab( const gchar *file, gchar *text)
{
	GtkWidget *content, *header;
	gchar *title;
	gint page;
	gchar string[30];

	title = get_title_from_filename ( file );
	header = new_tab_header();
	content = new_tab_content();

	/* create new notebook page */
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), GTK_WIDGET(content), GTK_WIDGET(header));

	page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), GTK_WIDGET(content));

	if(strcmp (file, "** unsaved **") == 0 || strcmp (text, "") == 0) {
		/* new unsaved doc */
		unsaved_docs++;

		/* new title */
		g_snprintf(string, 30, "Ungespeichertes Dokument %i", unsaved_docs);

		/* set title */
		set_label_of_tab( page, string );

		/* set tab text */
		set_text_of_tab( page, text );

		/* set filename */
		set_filename_of_tab ( page, file );

		statusbar_write ("Added Tab \"%s\"", string);
	} else {
		/* set tab label */
		set_label_of_tab( page, title );

		/* set tab text */
		set_text_of_tab( page, text );

		/* set filename */
		set_filename_of_tab ( page, file );

		statusbar_write ("Added Tab \"%s\"", title);
	}
	
	/* refresh view */
	gtk_widget_show_all (glade_xml_get_widget (xml, "main"));

	/* return page number */
	return page;
}

void
rem_tab(gint page)
{
	statusbar_write ("Remove tab \"%s\"!", get_label_of_tab( page ));
	
	gtk_notebook_remove_page (GTK_NOTEBOOK(notebook), page);
	
	if(gtk_notebook_get_n_pages ( GTK_NOTEBOOK(notebook) ) == 0) {
		new_tab("** unsaved **", "");
	}
}

GtkWidget*
new_tab_header( void )
{
	GtkWidget *hbox, *label, *button, *event_box, *image;
	GtkSettings *settings;
	gint w, h;

	/* event box */
	event_box = gtk_event_box_new ();
	gtk_event_box_set_visible_window (GTK_EVENT_BOX (event_box), FALSE);

	/* horizontal box */
	hbox = gtk_hbox_new (FALSE, 2);

	/* label */
	label = gtk_label_new("");
	gtk_container_add (GTK_CONTAINER (event_box), label);

	/* button */
	button = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	gtk_button_set_focus_on_click (GTK_BUTTON (button), FALSE);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);

	/* image settings */
	settings = gtk_widget_get_settings (GTK_WIDGET (label));
	gtk_icon_size_lookup_for_settings (settings, GTK_ICON_SIZE_MENU, &w, &h);
	gtk_widget_set_size_request (button, w + 4, h + 4);

	/* image for button */
	image = gtk_image_new_from_stock (GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU);
	gtk_container_add (GTK_CONTAINER (button), image);

	/* connects the button with the close callback */
	g_signal_connect ( button, "clicked", G_CALLBACK(button_callback), NULL );

	/* pack the box */
	gtk_box_pack_start (GTK_BOX (hbox), event_box, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);

	gtk_widget_show_all(hbox);

	/* return the box */
	return hbox;
}

void
set_label_of_tab(gint page, gchar *text)
{
	gtk_label_set_text ( //set text to label
		GTK_LABEL(
			gtk_container_get_children ( //get content of event box (text label)
				GTK_CONTAINER(
					gtk_container_get_children ( //get content of head (event box)
						GTK_CONTAINER(
							gtk_notebook_get_tab_label ( //get head field of tab
								GTK_NOTEBOOK(notebook),
								gtk_notebook_get_nth_page ( //get notebooks page
									GTK_NOTEBOOK(notebook),
									page
								)
							)
						)
					)->data
				)
			)->data
		),
		text
	);
}

const gchar *
get_label_of_tab(gint page)
{
	return 
	gtk_label_get_text ( //get text of label
		GTK_LABEL (
			gtk_container_get_children ( //get content of event box (text label)
				GTK_CONTAINER (
					gtk_container_get_children ( //get content of head (event box)
						GTK_CONTAINER (
							gtk_notebook_get_tab_label ( //get head field of tab
								GTK_NOTEBOOK(notebook),
								gtk_notebook_get_nth_page ( //get notebook page
									GTK_NOTEBOOK(notebook),
									page
								)
							)
						)
					)->data
				)
			)->data
		)
	);
}

GtkWidget*
new_tab_content (void)
{
	GtkWidget *scrolledwindow, *textview;
	
	GtkSourceTagTable *tagtable;
	GtkSourceBuffer *buffer;

	tagtable = gtk_source_tag_table_new ();
	buffer = gtk_source_buffer_new(tagtable);
	
	scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
	textview = gtk_source_view_new();

	/* source view & source buffer settings */
	gtk_source_view_set_show_line_numbers (GTK_SOURCE_VIEW(textview), TRUE);
	gtk_source_view_set_show_line_markers (GTK_SOURCE_VIEW(textview), TRUE);
	gtk_source_view_set_highlight_current_line (GTK_SOURCE_VIEW(textview), TRUE);
	gtk_source_buffer_set_check_brackets (GTK_SOURCE_BUFFER(buffer), TRUE);
	gtk_source_buffer_set_highlight (GTK_SOURCE_BUFFER(buffer), TRUE);
	
	/* set buffer */
	gtk_text_view_set_buffer (GTK_TEXT_VIEW(textview), GTK_TEXT_BUFFER(buffer));
	
	/* scrolled window settings */
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add ( GTK_CONTAINER ( scrolledwindow ), textview );

	return scrolledwindow;
}

void
set_text_of_tab(gint page, gchar *text)
{
	GtkSourceBuffer *buffer;
	
	buffer = get_buffer_of_tab (page);

	gtk_text_buffer_set_text ( GTK_TEXT_BUFFER(buffer), text, -1 );
}

gchar *
get_text_of_tab(gint page)
{
	GtkSourceBuffer *buffer;
	GtkTextIter start, end;
	
	buffer = get_buffer_of_tab ( page );

	gtk_text_buffer_get_start_iter ( GTK_TEXT_BUFFER(buffer), &start );

	gtk_text_buffer_get_end_iter ( GTK_TEXT_BUFFER(buffer), &end );
	
	return gtk_text_buffer_get_text ( GTK_TEXT_BUFFER(buffer) , &start, &end, TRUE);
}

GtkWidget *
get_textview_of_tab(gint page)
{
	GtkWidget *scrolled_window, *textview;
	
	scrolled_window = gtk_notebook_get_nth_page ( GTK_NOTEBOOK(notebook), page );
	textview = gtk_container_get_children ( GTK_CONTAINER( scrolled_window ) )->data;
	
	return textview;
}

GtkSourceBuffer *
get_buffer_of_tab(gint page)
{
	GtkWidget *textview;
	GtkSourceBuffer *buffer;

	textview = get_textview_of_tab(page);
	buffer = GTK_SOURCE_BUFFER ( gtk_text_view_get_buffer( GTK_TEXT_VIEW( textview ) ) );

	return buffer;
}

void
button_callback( GtkWidget * widget )
{
	const gchar *label;
	gint page;

	/* get label of tab, because it is unique */	
	label = gtk_label_get_text ( //get text label's text
		GTK_LABEL (
			gtk_container_get_children ( //get text label
				GTK_CONTAINER (
					gtk_container_get_children ( //get event box
						GTK_CONTAINER (
							gtk_widget_get_parent ( //get head of tab
								widget
							)
						)
					)->data
				)
			)->data
		)
	);

	page = get_page_by_label (label);

	statusbar_write("Close Tab: \"%s\" on tab position %i!", label, page);

	rem_tab ( page );
}

gint
get_page_by_label (const gchar * label)
{
	gint i;

	/* scan the tabs for a title like the one in the parameter */
	for(i = 0; i <= gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook)); i++)
		if(g_str_equal(label, get_label_of_tab(i)))
			return i;

	return -1;
}

gchar * 
get_title_from_filename (const gchar * filename )
{
	gchar file[30];
	static gchar output[30];
	gchar *temp;
	unsigned int i;

	/* if filename == unsaved just return this */
	if(strcmp(filename, "** unsaved **") == 0) {
		return "** unsaved **";
	}

	/* get everything after the last "/" */
	temp = g_strrstr(filename, "/");

	/* make sure there is at least something in temp string */
	if(temp != NULL) {
		strncpy(file, temp, 30);
	} else {
		return "ERROR!";
	}

	/* remove the "/" from the temp string */
	if(file != NULL) {
		for(i = 1; i <= strlen(file); i++) {
			output[i-1] = file[i];
		}
	}

	return output;
}

void
set_filename_of_tab ( gint page, const gchar * filename )
{
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();

	gtk_tooltips_set_tip ( //set tooltip of event box
		tooltips,
		GTK_WIDGET (
			gtk_container_get_children ( //get event box of header of tab
				GTK_CONTAINER (
					gtk_notebook_get_tab_label ( //get header of tab
						GTK_NOTEBOOK(notebook),
						gtk_notebook_get_nth_page ( //get tab
							GTK_NOTEBOOK(notebook),
							page
						)
					)
				)
			)->data
		),
		filename,
		filename
	);
}

gchar * 
get_filename_of_tab ( gint page )
{
	return gtk_tooltips_data_get ( //get tooltip of event box
		GTK_WIDGET (
			gtk_container_get_children ( //get event box of header of tab
				GTK_CONTAINER (
					gtk_notebook_get_tab_label ( //get header of tab
						GTK_NOTEBOOK(notebook),
						gtk_notebook_get_nth_page ( //get tab
							GTK_NOTEBOOK(notebook),
							page
						)
					)
				)
			)->data
		)
	)->tip_text;
}
