
/*
 * GAI - General Applet Interface Library
 * Copyright (C) 2003-2004 Jonas Aaberg <cja@gmx.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * - Preference window generator v2.0
 */


#include <stdio.h>
#include <string.h>
#include "../config.h"
#include "gai.h"
#include "gai-private.h"

#include "colourbutton.xpm"


#define GAI_PREF_MASK 0x3f

typedef struct {
    GtkWidget *widget;
    GaiPrefTypes type;
    void *result, *result2;
    int id, group;
} GaiIW;

/* Selector structure */
typedef struct {
    int type;
    GdkPixbuf *pixbuf;
    GtkImage *image;
    GaiColor color, old_color;
    GtkWidget *sd, *entry;
    char *name;
} GaiSelS;


static GaiSelS *ss;
static GaiIW *iw;
static GtkWidget *pref_window = NULL;
static GtkAdjustment *rox_adj = NULL;
static int num_items = 0, num_notebooks = 0, ptr, iw_ptr = 0, radio_group_number = 0, ss_ptr = 0;
static float align = 0.0;


static GtkWidget *gai_create_page(GaiPI *);


static void 
gai_change_colour(GtkImage *image, GdkPixbuf *pixbuf, GaiColor c)
{

    unsigned char *buf;
    int x, y, w, h, rs, alpha, yshort;

    w =     gdk_pixbuf_get_width(pixbuf);
    h =     gdk_pixbuf_get_height(pixbuf);
    rs =    gdk_pixbuf_get_rowstride(pixbuf);
    alpha = gdk_pixbuf_get_has_alpha(pixbuf);
    
    buf = gdk_pixbuf_get_pixels(pixbuf);

    for (y=2; y < (h-2) ; y++)
    {
	yshort = rs*y;
	for(x=2; x < (w-2) ; x++)
	{
	    buf[yshort+x*(3+alpha)+0] = c.r;
	    buf[yshort+x*(3+alpha)+1] = c.g;
	    buf[yshort+x*(3+alpha)+2] = c.b;
	}
    }

    gtk_image_set_from_pixbuf(image, pixbuf);
}



static void on_sel_cancel_clicked(GtkButton *b, gpointer number)
{

    if(ss[(int)number].sd != NULL)
	gtk_widget_destroy(ss[(int)number].sd);
    ss[(int)number].sd = NULL;
}


static void on_sel_ok_clicked(GtkButton *b, gpointer number)
{
    int alpha;
    GdkColor color;

    if(ss[(int)number].type == GAI_COLORSELECTOR){
	ss[(int)number].old_color = ss[(int)number].color;

	gtk_color_selection_get_current_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(ss[(int)number].sd)->colorsel),
					      &color);

	alpha = gtk_color_selection_get_current_alpha(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(ss[(int)number].sd)->colorsel));
	ss[(int)number].color.alpha = (unsigned char)(alpha>>8);
	ss[(int)number].color.r = (unsigned char)(color.red>>8);
	ss[(int)number].color.g = (unsigned char)(color.green>>8);
	ss[(int)number].color.b = (unsigned char)(color.blue>>8);

	gai_change_colour(ss[(int)number].image, ss[(int)number].pixbuf, ss[(int)number].color);
    }

    if(ss[(int)number].type == GAI_FILESELECTOR){
	gtk_entry_set_text(GTK_ENTRY(ss[(int)number].entry),
			   gtk_file_selection_get_filename(GTK_FILE_SELECTION(ss[(int)number].sd)));
    }


    on_sel_cancel_clicked(b, number);

}


static void selector_button(GtkButton *b, gpointer number)
{

    GdkColor colour;
    GtkColorSelection *colorsel;


    if (ss[(int)number].sd == NULL)
    {	
	if(ss[(int)number].type == GAI_COLORSELECTOR){
	    ss[(int)number].sd = gtk_color_selection_dialog_new (ss[(int)number].name);
	    gtk_window_set_resizable (GTK_WINDOW (ss[(int)number].sd), FALSE);
	    gtk_widget_realize(ss[(int)number].sd);
	
	    colour.red =   ((int)ss[(int)number].color.r)<<8;
	    colour.green = ((int)ss[(int)number].color.g)<<8;
	    colour.blue =  ((int)ss[(int)number].color.b)<<8;

	    colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(ss[(int)number].sd) ->colorsel);

	    gtk_color_selection_set_current_color(colorsel, &colour);
	    gtk_color_selection_set_current_alpha(colorsel,(int)(ss[(int)number].color.alpha <<8));
	
	    colour.red =   ((int)ss[(int)number].old_color.r)<<8;
	    colour.green = ((int)ss[(int)number].old_color.g)<<8;
	    colour.blue =  ((int)ss[(int)number].old_color.b)<<8;
	
	    gtk_color_selection_set_previous_color(colorsel, &colour);
	    gtk_color_selection_set_previous_alpha(colorsel, (int)ss[(int)number].old_color.alpha <<8);
	    gtk_color_selection_set_has_opacity_control (colorsel, TRUE);

	    g_signal_connect((gpointer)GTK_COLOR_SELECTION_DIALOG (ss[(int)number].sd)->ok_button,
			     "clicked", G_CALLBACK (on_sel_ok_clicked), number);

	    g_signal_connect((gpointer)GTK_COLOR_SELECTION_DIALOG(ss[(int)number].sd)->cancel_button,
			     "clicked", G_CALLBACK(on_sel_cancel_clicked), number);

	}
	
	if(ss[(int)number].type == GAI_FILESELECTOR){
	    ss[(int)number].sd = gtk_file_selection_new(ss[(int)number].name);

	    gtk_file_selection_set_filename(GTK_FILE_SELECTION(ss[(int)number].sd),
					gtk_entry_get_text(GTK_ENTRY(ss[(int)number].entry)));
	    g_signal_connect((gpointer)GTK_FILE_SELECTION(ss[(int)number].sd)->ok_button,
			     "clicked",G_CALLBACK(on_sel_ok_clicked), number);
	    g_signal_connect((gpointer)GTK_FILE_SELECTION(ss[(int)number].sd)->cancel_button,
			     "clicked",G_CALLBACK(on_sel_cancel_clicked),number);
	}


	gtk_widget_show_all(ss[(int)number].sd);
    } else 
	gtk_window_present(GTK_WINDOW(ss[(int)number].sd));


}


/* ======================================================================== */

static GtkWidget *gai_gen_label(GaiPI *g)
{
    GtkWidget *label;

    if(g->name != NULL)
	label = gtk_label_new(g->name);
    else
	label = gtk_label_new(" ");

    if((g->type&GAI_NO_TEXT_MARKUP) != GAI_NO_TEXT_MARKUP)
	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);

    if((g->type&GAI_LEFT) == GAI_LEFT)
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    else
    if((g->type&GAI_RIGHT) == GAI_RIGHT)
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    else
    if((g->type&GAI_CENTER) == GAI_CENTER)
	gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
    else
        gtk_misc_set_alignment(GTK_MISC(label), align, 0.5);

    return label;
}


static GtkWidget *gai_gen_checkbutton(GaiPI *g)
{
    GtkWidget *item;

    if(g->name != NULL)
	item = iw[iw_ptr].widget = gtk_check_button_new_with_label(g->name);
    else
	item = iw[iw_ptr].widget = gtk_check_button_new_with_label(" ");
    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(iw[iw_ptr].widget),
				 ((int *)g->default_val)[0]);

    iw[iw_ptr].type = GAI_CHECKBUTTON;
    iw[iw_ptr].result = g->result_val;
    iw_ptr++;
    ptr++;
    return item;
}


static GtkWidget *gai_gen_textentry(GaiPI *g, int passwd)
{
    GtkWidget *label, *item;
    label = gai_gen_label(g);

    item = gtk_hbox_new (FALSE, 2);
    gtk_box_pack_start(GTK_BOX(item), label, FALSE, TRUE, 0);

    iw[iw_ptr].type = GAI_TEXTENTRY;
    iw[iw_ptr].widget = gtk_entry_new();

    gtk_entry_set_max_length(GTK_ENTRY(iw[iw_ptr].widget), 1024);
    
    if (((char **)(g->default_val))[0] != NULL)
	gtk_entry_set_text(GTK_ENTRY(iw[iw_ptr].widget), 
			   ((char **)(g->default_val))[0]);

    /* Makes it a password entry */
    if(passwd)
	gtk_entry_set_visibility (GTK_ENTRY(iw[iw_ptr].widget), FALSE);


    gtk_box_pack_end(GTK_BOX(item), iw[iw_ptr].widget, TRUE, TRUE,0);


    gtk_label_set_mnemonic_widget(GTK_LABEL(label), iw[iw_ptr].widget);
    iw[iw_ptr].result = g->result_val;
    iw_ptr++;
    ptr++;
    return item;
}


static void gai_gen_radiobutton(GaiPI *g, int *i, GtkWidget *box)
{
    int j;
    GSList *radio_group = NULL;
	    
    for(j = 0; ; j++){
	if(((char **)g->name)[j] == NULL)
	    break;

	iw[iw_ptr].widget = gtk_radio_button_new_with_mnemonic(NULL, ((char **)g->name)[j]);

	gtk_radio_button_set_group (GTK_RADIO_BUTTON(iw[iw_ptr].widget), radio_group);

	radio_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(iw[iw_ptr].widget));

	if(j == ((int *)g->default_val)[0])
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(iw[iw_ptr].widget), TRUE);


	gtk_table_attach (GTK_TABLE(box), iw[iw_ptr].widget,
			  0, 1, i[0], i[0]+1,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 2, 2);


	iw[iw_ptr].type = GAI_RADIOBUTTON;
	iw[iw_ptr].group = radio_group_number;
	iw[iw_ptr].result = g->result_val;
	iw[iw_ptr].id = j;
	i[0]++;
	iw_ptr++;
    }


    /* Just to make the main function uniform */
    i--;

    radio_group_number++;
    /* We can't dealloc the list now. It is needed by the layout.
       Wonder if it deallocates it on widget destroy...*/
    /* g_slist_free(radio_group);*/
    ptr++;
}



static GtkWidget *gai_gen_spinbutton(GaiPI *g, int use_int)
{
    GtkWidget *item, *label;
    label = gai_gen_label(g);

    item = gtk_hbox_new(FALSE,2);

    gtk_box_pack_start (GTK_BOX(item), label, FALSE, TRUE, 0);

    if(use_int){
	iw[iw_ptr].type = GAI_SPINBUTTON;
	iw[iw_ptr].widget = gtk_spin_button_new(GTK_ADJUSTMENT(gtk_adjustment_new(((int *)g->default_val)[0],
										  ((GaiSS *)g->extra)->min,
										  ((GaiSS *)g->extra)->max,
										  ((GaiSS *)g->extra)->step, 10, 10)), 1, 0);
    } else {
	    iw[iw_ptr].type = GAI_SPINBUTTON_FLOAT;
	    iw[iw_ptr].widget = gtk_spin_button_new(GTK_ADJUSTMENT(gtk_adjustment_new(((float *)g->default_val)[0],
								   ((GaiSSF *)g->extra)->min,
								   ((GaiSSF *)g->extra)->max,
								   ((GaiSSF *)g->extra)->step, 10.0, 10.0)), 
						    1, ((GaiSSF *)g->extra)->decimals);
    }

    gtk_box_pack_end(GTK_BOX(item), iw[iw_ptr].widget, FALSE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(label), iw[iw_ptr].widget);
    iw[iw_ptr].result = g->result_val;
    iw_ptr++;
    ptr++;
    return item;
}



static GtkWidget *gai_gen_selector(GaiPI *g, int sel_type) 
{
    GtkWidget *item, *label, *button_align, *button_box, *button = NULL;

    label = gai_gen_label(g);

    button_box = gtk_hbox_new(FALSE, 1);
    item = gtk_hbox_new(FALSE, 2);
    button_align = gtk_alignment_new(0.5, 0.5, 0, 0);
    gtk_box_pack_start(GTK_BOX(item), label, FALSE, FALSE, 2);

    ss[ss_ptr].sd = NULL;
    ss[ss_ptr].name = g_strdup(g->name);
    ss[ss_ptr].type = sel_type;


    if(sel_type == GAI_COLORSELECTOR){

	button = iw[iw_ptr].widget = gtk_button_new();
	ss[ss_ptr].pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)colourbutton_xpm);
	ss[ss_ptr].image = GTK_IMAGE(gtk_image_new_from_pixbuf(ss[ss_ptr].pixbuf));
	ss[ss_ptr].color = ((GaiColor *)g->default_val)[0];
	ss[ss_ptr].old_color.r = ss[ss_ptr].old_color.g = ss[ss_ptr].old_color.b = ss[ss_ptr].old_color.alpha = 0xff; 

	gai_change_colour(ss[ss_ptr].image, ss[ss_ptr].pixbuf, ss[ss_ptr].color);

	gtk_box_pack_start(GTK_BOX(button_box), GTK_WIDGET(ss[ss_ptr].image), FALSE, FALSE, 0); 
	iw[iw_ptr].type = GAI_COLORSELECTOR;

	gtk_container_add(GTK_CONTAINER(button_align), button_box);
	gtk_container_add(GTK_CONTAINER(iw[iw_ptr].widget), button_align);
	gtk_box_pack_end(GTK_BOX(item), iw[iw_ptr].widget, FALSE, FALSE, 2);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), iw[iw_ptr].widget);


    }
    if(sel_type == GAI_FILESELECTOR){

	ss[ss_ptr].entry = iw[iw_ptr].widget = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(iw[iw_ptr].widget), 1024);


	if(((char **)(g->default_val))[0] != NULL)
	    gtk_entry_set_text(GTK_ENTRY(iw[iw_ptr].widget), 
			   ((char **)(g->default_val))[0]);


	gtk_box_pack_start(GTK_BOX(item),
			   iw[iw_ptr].widget, FALSE, FALSE, 0); 

	button = gtk_button_new();

	gtk_container_add(GTK_CONTAINER(button), button_align);
	gtk_container_add(GTK_CONTAINER(button_align), button_box);

	gtk_box_pack_start(GTK_BOX(button_box), 
			   gtk_image_new_from_stock ("gtk-find", GTK_ICON_SIZE_BUTTON),
			   FALSE, FALSE, 0);
	label = gtk_label_new_with_mnemonic("Ch_ange");

	gtk_label_set_mnemonic_widget(GTK_LABEL(label), button);
		
	gtk_box_pack_start(GTK_BOX(button_box), 
			       label,
			       FALSE, FALSE, 0);


	gtk_box_pack_end(GTK_BOX(item), button, FALSE, FALSE, 0); 
	iw[iw_ptr].type = GAI_FILESELECTOR;

    }


    g_signal_connect(G_OBJECT(button),"clicked",
		     G_CALLBACK(selector_button),
		     (void *)ss_ptr);	/* This is abit ugly. Convert an int to a pointer */

    iw[iw_ptr].result  = g->result_val;
    iw[iw_ptr].result2 = (void *)ss_ptr;
    ss_ptr++;
    ptr++;
    iw_ptr++;
    return item;
}

				       
static GtkWidget *gai_gen_option_menu(GaiPI *g)
{
    int j;
    GtkWidget *label, *item, *menu;

    item = gtk_hbox_new(FALSE, 2);

    label = gai_gen_label(g);

    gtk_box_pack_start(GTK_BOX(item), label, FALSE, TRUE, 0);

    iw[iw_ptr].widget = gtk_option_menu_new();
	
    menu = gtk_menu_new();

    for (j=0 ;; j++){
	if((((char **)g->extra)[j]) == NULL) 
	    break;
	gtk_container_add(GTK_CONTAINER(menu),
			  gtk_menu_item_new_with_mnemonic(((char **)g->extra)[j]));
    }

    gtk_option_menu_set_menu(GTK_OPTION_MENU(iw[iw_ptr].widget), menu);

    gtk_option_menu_set_history(GTK_OPTION_MENU(iw[iw_ptr].widget), 
				((int *)g->default_val)[0]);

    gtk_box_pack_end(GTK_BOX(item), iw[iw_ptr].widget, FALSE, TRUE, 0);

    iw[iw_ptr].type = GAI_OPTIONMENU;
    iw[iw_ptr].result = g->result_val;
    iw_ptr++;
    ptr++;
    return item;
}

static GtkWidget *gai_gen_frame(GaiPI *g)
{
    GtkWidget *label, *item;
    item = gtk_frame_new(NULL);

    label = gtk_label_new(g[ptr].name);

    gtk_frame_set_label_widget(GTK_FRAME(item), label);

    ptr++;
    gtk_container_add(GTK_CONTAINER(item), gai_create_page(g));

    return item;
}


static GtkWidget *gai_gen_button(GaiPI *g, int button_type)
{
    GtkWidget *item, *label, *button_box, *button_align;
    GdkPixbuf *pixbuf;

    item = gtk_hbox_new(FALSE, 2);

    iw[iw_ptr].widget = gtk_button_new();

    label = gai_gen_label(g);


    button_align = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
    button_box = gtk_hbox_new(FALSE, 2);

    if(button_type == GAI_BUTTON_IMAGE){

	pixbuf = gdk_pixbuf_new_from_file((char *)g->default_val, NULL);

	if(pixbuf != NULL){
	    
	    gtk_box_pack_start(GTK_BOX(button_box), 
			       gtk_image_new_from_pixbuf(pixbuf),
			       FALSE, FALSE, 0); 
	    g_object_unref(pixbuf);
	} 
	iw[iw_ptr].type = GAI_BUTTON_IMAGE;

	g_signal_connect (G_OBJECT(iw[iw_ptr].widget),"clicked",
			  G_CALLBACK(g->result_val), NULL);
    }
    if(button_type == GAI_BUTTON_TEXT){
	iw[iw_ptr].type = GAI_BUTTON_TEXT;
	g_signal_connect (G_OBJECT(iw[iw_ptr].widget),"clicked",
			  G_CALLBACK(g->default_val), NULL);

    }
    if(button_type == GAI_BUTTON_STOCK){
	gtk_box_pack_start(GTK_BOX(button_box), 
			   gtk_image_new_from_stock ((char *)g->default_val, GTK_ICON_SIZE_BUTTON),
			   FALSE, FALSE, 0);
	iw[iw_ptr].type = GAI_BUTTON_STOCK;
	g_signal_connect (G_OBJECT(iw[iw_ptr].widget),"clicked",
			  G_CALLBACK(g->result_val), NULL);
    }



    gtk_box_pack_start(GTK_BOX(button_box), label, FALSE, FALSE, 2);
    gtk_container_add(GTK_CONTAINER(button_align), button_box);
    gtk_container_add(GTK_CONTAINER(iw[iw_ptr].widget), button_align);
    gtk_box_pack_end(GTK_BOX(item),iw[iw_ptr].widget, FALSE, FALSE, 2);

    gtk_label_set_mnemonic_widget(GTK_LABEL(label), iw[iw_ptr].widget);
    ptr++;
    iw_ptr++;
    return item;
}


static GtkWidget *gai_gen_combo(GaiPI *g)
{
    GtkWidget *label, *item;

    label = gai_gen_label(g);

    item = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(item), label, FALSE, FALSE,0);

    iw[iw_ptr].type = GAI_COMBO;
    iw[iw_ptr].widget = gtk_combo_new();
    gtk_box_pack_end(GTK_BOX(item), iw[iw_ptr].widget, FALSE, FALSE,0);


    gtk_entry_set_max_length(GTK_ENTRY(GTK_COMBO(iw[iw_ptr].widget)->entry), 
			     1024);

    gtk_combo_set_popdown_strings(GTK_COMBO(iw[iw_ptr].widget), 
				  (GList *)((int *)g->extra)[0]);

    gtk_label_set_mnemonic_widget(GTK_LABEL(label), iw[iw_ptr].widget);

    if(g_list_nth_data((GList *)((int *)g->extra)[0],
		       ((int *)g->default_val)[0]) != NULL)
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(iw[iw_ptr].widget)->entry), 
			   g_list_nth_data((GList *)((int *)g->extra)[0],
					   ((int *)g->default_val)[0]));

    iw[iw_ptr].result = g->extra;
    iw[iw_ptr].result2 = g->result_val;
    iw_ptr++;
    ptr++;
    return item;
}


static int gai_get_entries(GaiPI *g, int start)
{
    int i, entries=0, depth = 0;
 
    for(i = start; g[i].type != GAI_END; i++){

	if(g[i].type == GAI_FRAME || g[i].type == GAI_FRAME_R){
	    if(depth==0)
		if((g[i].type&GAI_PREF_MASK) != GAI_ALL_LEFT ||
		   (g[i].type&GAI_PREF_MASK) != GAI_ALL_RIGHT ||
		   (g[i].type&GAI_PREF_MASK) != GAI_ALL_CENTER ||
		   g[i].type != GAI_NOTEBOOK_E ||
		   g[i].type != GAI_NOTEBOOK)
		    entries++;
	    depth++;
	    continue;
	}

	if(g[i].type == GAI_FRAME_E && depth>0){
	    depth--;
	    continue;
	}

	if(depth == 0)
	    if((g[i].type&GAI_PREF_MASK) != GAI_ALL_LEFT ||
	       (g[i].type&GAI_PREF_MASK) != GAI_ALL_RIGHT ||
	       (g[i].type&GAI_PREF_MASK) != GAI_ALL_CENTER ||
	       g[i].type != GAI_NOTEBOOK_E ||
	       g[i].type != GAI_NOTEBOOK)

		entries++;

	if(g[i].type == GAI_NOTEBOOK_E)
	    break;

	if(g[i].type == GAI_FRAME_E)
	    break;
    }
    return entries;
}



static GtkWidget *gai_create_page(GaiPI *g)
{
    int i, quit=0,  entries;
    GtkWidget *item = NULL, *box;


    entries = gai_get_entries(g, ptr);

    box = gtk_table_new(2, entries, FALSE);

    i = 0;
    while(g[ptr].type != GAI_END && !quit){


	switch(g[ptr].type&GAI_PREF_MASK){

	case GAI_END:
	    quit=1;
	    break;

	case GAI_CHECKBUTTON:
	    item = gai_gen_checkbutton(&g[ptr]);

	    gtk_table_attach (GTK_TABLE (box), 
			      item, 0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),
			      (GtkAttachOptions) (0), 2, 2);
	    i++;
	    break;
	case GAI_TEXTENTRY:

	    item = gai_gen_textentry(&g[ptr], FALSE);

	    gtk_table_attach (GTK_TABLE (box), item,
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),
			      (GtkAttachOptions) (0), 2, 2);
	    i++;
	    break;
	case GAI_TEXT:
	    item = gai_gen_label(&g[ptr]);

	    gtk_table_attach (GTK_TABLE (box), item,
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),
			      (GtkAttachOptions) (0), 2, 2);
	    ptr++;
	    i++;
	    break;
	case GAI_NOTEBOOK:
	    ptr++;
	    break;
	case GAI_RADIOBUTTON:
	    gai_gen_radiobutton(&g[ptr], &i, box);
	    i++;
	    break;
	case GAI_SPINBUTTON:
	    item = gai_gen_spinbutton(&g[ptr], TRUE);
	    gtk_table_attach(GTK_TABLE(box), item,
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL),
			      (GtkAttachOptions) (0), 2, 2);
	    i++;

	    break;
	case GAI_COLORSELECTOR:
	    item = gai_gen_selector(&g[ptr], GAI_COLORSELECTOR);
	    gtk_table_attach (GTK_TABLE(box), item,
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL),
			      (GtkAttachOptions) (0), 2, 2);
	    i++;
	    break;
	case GAI_HLINE:
	    item = gtk_hseparator_new();
	    gtk_table_attach (GTK_TABLE (box), item,
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL),
			      (GtkAttachOptions) (0), 2, 2);
	    i++;
	    ptr++;
	    break;
	case GAI_FILESELECTOR:
	    item = gai_gen_selector(&g[ptr], GAI_FILESELECTOR);

	    gtk_table_attach(GTK_TABLE(box), item,
			     0, 1, i, i+1,
			     (GtkAttachOptions) (GTK_FILL),
			     (GtkAttachOptions) (0), 2, 2);
	    i++;
	    break;
	case GAI_FRAME:
	    item = gai_gen_frame(g);

	    gtk_table_attach(GTK_TABLE(box), item, 0, 1, i, i+1,
			     (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
			     (GtkAttachOptions)(GTK_EXPAND | GTK_FILL), 
			     5, 5);
	    i++;
	    break;
	case GAI_FRAME_R:
	    item = gai_gen_frame(g);

	    gtk_table_attach(GTK_TABLE(box), item, 1, 2, i-1, i,
			     (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
			     (GtkAttachOptions)(GTK_EXPAND | GTK_FILL), 
			     5, 5);

	    i++;
	    break;

	case GAI_OPTIONMENU:
	    item = gai_gen_option_menu(&g[ptr]);
	    gtk_table_attach (GTK_TABLE (box), item,
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL),
			      (GtkAttachOptions) (0), 2, 2);

	    i++;
	    break;
	case GAI_BUTTON_TEXT:

	    item = gai_gen_button(&g[ptr], GAI_BUTTON_TEXT);

	    gtk_table_attach (GTK_TABLE(box), item,
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL),
			      (GtkAttachOptions) (0), 2, 2);
	    i++;
	    break;
	case GAI_BUTTON_IMAGE:
	    item = gai_gen_button(&g[ptr], GAI_BUTTON_IMAGE);
	    gtk_table_attach (GTK_TABLE(box), item,
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL),
			      (GtkAttachOptions) (0), 2, 2);
	    i++;
	    break;
	case GAI_BUTTON_STOCK:
	    item = gai_gen_button(&g[ptr], GAI_BUTTON_STOCK);
	    gtk_table_attach (GTK_TABLE(box), item, 
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL),
			      (GtkAttachOptions) (0), 2, 2);
	    i++;
	    break;
	case GAI_PASSWORDENTRY:
	    item = gai_gen_textentry(&g[ptr], TRUE);
	    gtk_table_attach (GTK_TABLE (box), item,
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL),
			      (GtkAttachOptions) (0), 2, 2);
	    i++;
	    break;
	case GAI_SPINBUTTON_FLOAT:
	    item = gai_gen_spinbutton(&g[ptr], FALSE);
	    gtk_table_attach(GTK_TABLE(box), item,
			      0, 1, i, i+1,
			      (GtkAttachOptions) (GTK_FILL),
			      (GtkAttachOptions) (0), 2, 2);
	    i++;

	    break;
	case GAI_COMBO:
	    item = gai_gen_combo(&g[ptr]);
	    gtk_table_attach(GTK_TABLE(box), item,
			     0, 1, i, i+1,
			     (GtkAttachOptions)(GTK_FILL),
			     (GtkAttachOptions)(0), 2, 2);
	    i++;
	    break;
	case GAI_ALL_LEFT:
	    align = 0.0;
	    ptr++;
	    break;
	case GAI_ALL_CENTER:
	    align = 0.5;
	    ptr++;
	    break;
	case GAI_ALL_RIGHT:
	    align = 1.0;
	    ptr++;
	    break;
	case GAI_FRAME_E:
	    ptr++;
	    quit = 1;
	    break;
	case GAI_NOTEBOOK_E:
	    ptr++;
	    quit = 1;
	    break;
	default:
	    printf("Unknown command (%d)!\n",g[i].type&GAI_PREF_MASK);
	    break;
	}

    }


    return box;
}

/* 
   And the world is like a shiny diamond,
   the way it glitters if you polish it right,
   if the light should burn and leave you blinded
   there'll be treasure on the other side

   - Masterplan
*/

static gboolean on_close_button_clicked(GtkWidget *w, gpointer d)
{
    int i,j;
    char *buff;

    for(i=0;i<iw_ptr;i++){
	switch(iw[i].type){
	case GAI_CHECKBUTTON:
	    if(iw[i].result != NULL)
		((int *)iw[i].result)[0] = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(iw[i].widget));
	    break;
	case GAI_RADIOBUTTON:
	    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(iw[i].widget))){
		if(iw[i].result != NULL)
		    ((int *)iw[i].result)[0] = iw[i].id;
	    }
	    break;

	case GAI_COMBO:
	    buff = (char *)gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(iw[i].widget)->entry));
	    for(j=0;j<g_list_length((GList *)((int *)iw[i].result)[0]);j++){

		/* Build the resulting list */
		if(!strcmp(buff,g_list_nth_data((GList *)((int *)iw[i].result)[0],j)))
		    break;
	    }

	    
	    if(j == g_list_length((GList *)((int *)iw[i].result)[0])){
		(GList *)((int *)iw[i].result)[0] = g_list_append((GList *)((int *)iw[i].result)[0], 
								  g_strdup(buff));
	    } 
	    ((int *)iw[i].result2)[0] = j;
	    break;

	case GAI_FILESELECTOR:
	    if(((char **)iw[i].result)[0] != NULL)
		g_free(((char **)iw[i].result)[0]);
	    ((char **)iw[i].result)[0] = (char *)g_strdup(gtk_entry_get_text(GTK_ENTRY(iw[i].widget)));
	    break;

	case GAI_TEXTENTRY:
	    if(((char **)iw[i].result)[0] != NULL)
		g_free(((char **)iw[i].result)[0]);
	    ((char **)iw[i].result)[0] = (char *)g_strdup(gtk_entry_get_text(GTK_ENTRY(iw[i].widget)));
	    break;
	case GAI_SPINBUTTON:
	    if(((int *)iw[i].result) != NULL)
		((int *)iw[i].result)[0] = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(iw[i].widget));
	    break;
	case GAI_SPINBUTTON_FLOAT:
	    if(((float *)iw[i].result) != NULL)
		((float *)iw[i].result)[0] = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(iw[i].widget));
	    break;
	case GAI_COLORSELECTOR:
	    ((GaiColor *)iw[i].result)[0] = ss[(int)iw[i].result2].color;
	    break;
	case GAI_OPTIONMENU:
	    if(((int *)iw[i].result) != NULL)
		((int *)iw[i].result)[0] = gtk_option_menu_get_history(GTK_OPTION_MENU(iw[i].widget));
	    
	    break;

	default:
	    break;
	}

    }


    if(GAI.applet_type == GAI_ROX){
	gai_size_change(gtk_adjustment_get_value(rox_adj), 0, 0, TRUE, 0);
	buff = g_strdup_printf("%s/rox_panel_size", GAI.applet.name);
	gai_save_int(buff, gtk_adjustment_get_value(rox_adj));
	g_free(buff);
    }

    gtk_widget_destroy(pref_window);
    pref_window = NULL;

    if(GAI.on_preferences_callback){
	GAI.restarting = TRUE;
	if (GAI.on_preferences_callback)
	    GAI.on_preferences_callback(GAI.on_preferences_userdata);

	GAI.restarting = FALSE;
    }


    return TRUE;
}

static gboolean on_help_button_clicked(GtkWidget *widget, gpointer d)
{
    GtkWidget *w;

    if(GAI.help_text == NULL)
	return FALSE;

    w = gtk_message_dialog_new (NULL,0,GTK_MESSAGE_INFO,GTK_BUTTONS_OK,
				GAI.help_text);
    gtk_widget_show (w);
    g_signal_connect_swapped (G_OBJECT(w), "response",
			      G_CALLBACK(gtk_widget_destroy),
			      G_OBJECT(w));

    return TRUE;
}

static GtkWidget *
gai_create_main_buttons(void)
{
    GtkWidget *buttonbar, *close_button, *help_button;

    buttonbar = gtk_hbox_new(FALSE, 0);

    help_button = gtk_button_new_from_stock("gtk-help");
    close_button = gtk_button_new_from_stock("gtk-close");

    gtk_container_set_border_width(GTK_CONTAINER(help_button), 5);
    gtk_container_set_border_width(GTK_CONTAINER(close_button), 5);

    gtk_widget_set_size_request(help_button, 95, 42);
    gtk_widget_set_size_request(close_button, 95, 42);

    gtk_box_pack_start(GTK_BOX(buttonbar), help_button, FALSE, FALSE, 5);
    gtk_box_pack_end(GTK_BOX(buttonbar), close_button, FALSE, FALSE, 5);

    g_signal_connect(G_OBJECT(help_button), "clicked",
		     G_CALLBACK(on_help_button_clicked), NULL);

    g_signal_connect(G_OBJECT(close_button), "clicked",
		     G_CALLBACK(on_close_button_clicked), NULL);


    return buttonbar;
}

void 
gai_make_preference_window2(const char *window_name, GaiPI *g)
{
    GtkWidget *page, *mainbox, *notebook, *label;
    GtkWidget *rox_box, *rox_frame, *rox_scale;
    int i, old_ptr, j, num_ss = 0, num_iw = 0;

    GAI_ENTER;

    if(pref_window != NULL){
	gtk_window_present(GTK_WINDOW(pref_window));
	GAI_LEAVE;
	return;
    }

    iw_ptr = 0;
    ss_ptr = 0;
    num_items = 0;
    num_notebooks=0;
    radio_group_number = 0;

    while(g[num_items].type != GAI_END){
	if(g[num_items].type == GAI_NOTEBOOK) 
	    num_notebooks++;
	num_items++;
    }


    for(i=0;i<num_items;i++){
	if((g[i].type&GAI_PREF_MASK) == GAI_CHECKBUTTON ||
	   (g[i].type&GAI_PREF_MASK) == GAI_TEXTENTRY ||
	   (g[i].type&GAI_PREF_MASK) == GAI_RADIOBUTTON ||
	   (g[i].type&GAI_PREF_MASK) == GAI_SPINBUTTON ||
	   (g[i].type&GAI_PREF_MASK) == GAI_COLORSELECTOR ||
	   (g[i].type&GAI_PREF_MASK) == GAI_FILESELECTOR ||
	   (g[i].type&GAI_PREF_MASK) == GAI_OPTIONMENU ||
	   (g[i].type&GAI_PREF_MASK) == GAI_BUTTON_TEXT ||
	   (g[i].type&GAI_PREF_MASK) == GAI_BUTTON_STOCK ||
	   (g[i].type&GAI_PREF_MASK) == GAI_BUTTON_IMAGE ||
	   (g[i].type&GAI_PREF_MASK) == GAI_PASSWORDENTRY ||
	   (g[i].type&GAI_PREF_MASK) == GAI_SPINBUTTON_FLOAT ||
	   (g[i].type&GAI_PREF_MASK) == GAI_COMBO)
	num_iw++;

	if((g[i].type&GAI_PREF_MASK) == GAI_COLORSELECTOR || 
	   (g[i].type&GAI_PREF_MASK) == GAI_FILESELECTOR)
	    num_ss++;


	if((g[i].type&GAI_PREF_MASK) == GAI_RADIOBUTTON){
	    for(j=0;;j++){
		if(((char **)g[i].name)[j] == NULL) break;
		num_iw++;
	    }
	    num_iw--;
	} 
	    

    }

    iw = g_malloc0(sizeof(GaiIW)*num_iw);
    ss = g_malloc0(sizeof(GaiSelS)*num_ss);

    ss_ptr = 0;

    pref_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(pref_window), window_name);

    mainbox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(pref_window), mainbox);


    if(num_notebooks){
	if(g[0].type != GAI_NOTEBOOK){
	    printf("If you want to use Notebooks, the first entry must be a GAI_NOTEBOOK\n");
	    GAI_NOTE("If you want to use Notebooks, the first entry must be a GAI_NOTEBOOK\n");
	    GAI_LEAVE;
	    return;
	}
	notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(mainbox), notebook, FALSE, FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER (notebook), 5);

	ptr = 1;
	old_ptr = 0;
	i = 0;
	while(g[ptr].type !=GAI_END){
	    page = gai_create_page(g);

	    gtk_container_add(GTK_CONTAINER(notebook), page);

	    label = gtk_label_new_with_mnemonic(g[old_ptr].name);

	    gtk_notebook_set_tab_label(GTK_NOTEBOOK(notebook), 
				       gtk_notebook_get_nth_page(GTK_NOTEBOOK (notebook), i), 
				       label);

	    old_ptr = ptr;
	    i++;

	    if(g[ptr].type == GAI_END)
		break;

	    if(g[ptr].type != GAI_NOTEBOOK){
		printf("A GAI_NOTEBOOK_E must be followed by a new GAI_NOTEBOOK!\n");
		GAI_NOTE("A GAI_NOTEBOOK_E must be followed by a new GAI_NOTEBOOK!\n");
		break;
	    }
	}
    } else {
	ptr = 0;
	page = gai_create_page(g);
	gtk_box_pack_start(GTK_BOX(mainbox), page, TRUE, TRUE, 5);
    }

    
    if(GAI.applet_type == GAI_ROX){

	rox_frame = gtk_frame_new(NULL);
	gtk_container_set_border_width(GTK_CONTAINER (rox_frame), 5);
	gtk_box_pack_start(GTK_BOX(mainbox), rox_frame, TRUE, TRUE, 5);
	gtk_frame_set_label_widget(GTK_FRAME(rox_frame), gtk_label_new("ROX settings"));

	rox_box = gtk_hbox_new(FALSE,2);
	gtk_box_pack_start(GTK_BOX(rox_box), gtk_label_new("Applet size:"), TRUE, TRUE, 5);

	if(GAI.orient == GAI_VERTICAL)
	    rox_adj = GTK_ADJUSTMENT(gtk_adjustment_new(GAI.width, 16 , 128, 0, 0, 0));
	else
	    rox_adj = GTK_ADJUSTMENT(gtk_adjustment_new(GAI.height, 16 , 128, 0, 0, 0));

	rox_scale = gtk_hscale_new(rox_adj);
	gtk_scale_set_digits(GTK_SCALE(rox_scale), 0);
	gtk_box_pack_start(GTK_BOX(rox_box), rox_scale, TRUE, TRUE, 5);

	gtk_container_add(GTK_CONTAINER(rox_frame), rox_box);

    }
    

    gtk_box_pack_end(GTK_BOX(mainbox), gai_create_main_buttons(), TRUE, TRUE, 5);

    gtk_widget_show_all(pref_window);

    GAI_LEAVE;
}
