namespace Galago
{
	using System;
	using System.Runtime.InteropServices;

	public class Account
		: Object
	{
		~Account()
		{
			Dispose();
		}

		public Account(IntPtr raw) : base(raw) {}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_account_new(IntPtr service, IntPtr person,
												string username);

		public Account(Service service, Person person, string username)
		{
			Raw = galago_account_new(service.Handle, person.Handle, username);
		}

		protected Account() : base() {}


		/*
		 * Connected
		 */
		[DllImport("libgalago.so")]
		static extern void galago_account_set_connected(IntPtr account,
														bool connected);

		[DllImport("libgalago.so")]
		static extern bool galago_account_is_connected(IntPtr account);

		public bool IsConnected
		{
			get { return galago_account_is_connected(Handle);  }
			set { galago_account_set_connected(Handle, value); }
		}

		/*
		 * Service
		 */
		[DllImport("libgalago.so")]
		static extern IntPtr galago_account_get_service(IntPtr account);

		public Service Service
		{
			get
			{
				IntPtr raw_ret = galago_account_get_service(Handle);
				return Object.Wrap(raw_ret) as Service;
			}
		}

		/*
		 * Person
		 */
		[DllImport("libgalago.so")]
		static extern IntPtr galago_account_get_person(IntPtr account);

		public Person Person
		{
			get
			{
				IntPtr raw_ret = galago_account_get_person(Handle);
				return Object.Wrap(raw_ret) as Person;
			}
		}

		/*
		 * Presence
		 */
		[DllImport("libgalago.so")]
		static extern void galago_account_set_presence(IntPtr account,
													   IntPtr presence);

		public Presence Presence
		{
			get { return GetPresence(true);                          }
			set { galago_account_set_presence(Handle, value.Handle); }
		}

		/*
		 * GetPresence
		 */
		[DllImport("libgalago.so")]
		static extern IntPtr galago_account_get_presence(IntPtr account,
														 bool query);

		public Presence GetPresence(bool query)
		{
			IntPtr raw_ret = galago_account_get_presence(Handle, query);

			return Object.Wrap(raw_ret) as Presence;
		}

		public Presence GetPresence()
		{
			return GetPresence(true);
		}

		/*
		 * Avatar
		 */
		[DllImport("libgalago.so")]
		static extern void galago_account_set_avatar(IntPtr account,
													 IntPtr avatar);

		public Avatar Avatar
		{
			get { return GetAvatar(true);                          }
			set { galago_account_set_avatar(Handle, value.Handle); }
		}

		/*
		 * GetAvatar
		 */
		[DllImport("libgalago.so")]
		static extern IntPtr galago_account_get_avatar(IntPtr account,
													   bool query);

		public Avatar GetAvatar(bool query)
		{
			IntPtr raw_ret = galago_account_get_avatar(Handle, query);

			return Object.Wrap(raw_ret) as Avatar;
		}

		public Avatar GetAvatar()
		{
			return GetAvatar(true);
		}

		/*
		 * UserName
		 */
		[DllImport("libgalago.so")]
		static extern IntPtr galago_account_get_username(IntPtr account);

		public string UserName
		{
			get
			{
				IntPtr raw_ret = galago_account_get_username(Handle);
				return Marshal.PtrToStringAnsi(raw_ret);
			}
		}

		/*
		 * Native
		 */
		[DllImport("libgalago.so")]
		static extern bool galago_account_is_native(IntPtr account);

		public bool Native
		{
			get { return galago_account_is_native(Handle); }
		}

		/*
		 * DisplayName
		 */
		[DllImport("libgalago.so")]
		static extern void galago_account_set_display_name(IntPtr account,
														   string display_name);

		[DllImport("libgalago.so")]
		static extern IntPtr galago_account_get_display_name(IntPtr account);

		public string DisplayName
		{
			get
			{
				IntPtr raw_ret = galago_account_get_display_name(Handle);
				return Marshal.PtrToStringAnsi(raw_ret);
			}

			set
			{
				galago_account_set_display_name(Handle, value);
			}
		}

		/*
		 * DisplayNameSet
		 */
		[DllImport("libgalago.so")]
		static extern bool galago_account_is_display_name_set(IntPtr account);

		public bool DisplayNameSet
		{
			get { return galago_account_is_display_name_set(Handle); }
		}

		/*
		 * AddContact
		 */
		[DllImport("libgalago.so")]
		static extern void galago_account_add_contact(IntPtr account,
													  IntPtr contact);

		public void AddContact(Account contact)
		{
			galago_account_add_contact(Handle, contact.Handle);
		}

		/*
		 * RemoveContact
		 */
		[DllImport("libgalago.so")]
		static extern void galago_account_remove_contact(IntPtr account,
														 IntPtr contact);

		public void RemoveContact(Account contact)
		{
			galago_account_remove_contact(Handle, contact.Handle);
		}

		/*
		 * GetContact
		 */
		[DllImport("libgalago.so")]
		static extern IntPtr galago_account_get_contact(IntPtr account,
														string username,
														bool query);

		public Account GetContact(string username, bool query)
		{
			IntPtr raw_ret = galago_account_get_contact(Handle, username,
														query);
			return Object.Wrap(raw_ret) as Account;
		}

		public Account GetContact(string username)
		{
			return GetContact(username, true);
		}

		/*
		 * GetChildren
		 */
		[DllImport("libgalago.so")]
		static extern IntPtr galago_account_get_contacts(IntPtr account,
														 bool query);

		public List GetContacts(bool query)
		{
			IntPtr raw_ret = galago_account_get_contacts(Handle, query);

			return new List(raw_ret);
		}

		public List GetContacts()
		{
			return GetContacts(true);
		}

		/*
		 * Children
		 */
		public Galago.List Children
		{
			get { return GetContacts(true); }
		}

		/*
		 * Signals
		 */
		[Galago.Signal("connected")]
		public event EventHandler OnConnected
		{
			add
			{
				AddSignalHandler("connected",
								 typeof(SignalMarshals.VoidVoidSignal),
								 typeof(System.EventArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("connected", value);
			}
		}

		[Galago.Signal("disconnected")]
		public event EventHandler OnDisconnected
		{
			add
			{
				AddSignalHandler("disconnected",
								 typeof(SignalMarshals.VoidVoidSignal),
								 typeof(System.EventArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("disconnected", value);
			}
		}

		[Galago.Signal("display-name-changed")]
		public event EventHandler OnDisplayNameChanged
		{
			add
			{
				AddSignalHandler("display-name-changed",
								 typeof(SignalMarshals.VoidVoidSignal),
								 typeof(System.EventArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("display-name-changed", value);
			}
		}

		[Galago.Signal("presence-set")]
		public event PresenceSetHandler OnPresenceSet
		{
			add
			{
				AddSignalHandler("presence-set",
								 typeof(SignalMarshals.VoidPointerSignal),
								 typeof(PresenceSetArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("presence-set", value);
			}
		}

		[Galago.Signal("avatar-set")]
		public event AvatarSetHandler OnAvatarSet
		{
			add
			{
				AddSignalHandler("avatar-set",
								 typeof(SignalMarshals.VoidPointerSignal),
								 typeof(AvatarSetArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("avatar-set", value);
			}
		}

		[Galago.Signal("contact-added")]
		public event AddedHandler OnContactAdded
		{
			add
			{
				AddSignalHandler("contact-added",
								 typeof(SignalMarshals.VoidPointerSignal),
								 typeof(AddedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("contact-added", value);
			}
		}

		[Galago.Signal("contact-removed")]
		public event RemovedHandler OnContactRemoved
		{
			add
			{
				AddSignalHandler("contact-removed",
								 typeof(SignalMarshals.VoidPointerSignal),
								 typeof(RemovedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("contact-removed", value);
			}
		}

		[Galago.Signal("updated")]
		public event EventHandler OnUpdated
		{
			add
			{
				AddSignalHandler("updated",
								 typeof(SignalMarshals.VoidVoidSignal),
								 typeof(EventArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("updated", value);
			}
		}
	}
}
