namespace Galago.Tests
{
	using NUnit.Framework;
	using System;

	public class Utils
	{
		public static Galago.Person MakeDummyPerson()
		{
			return Galago.Global.CreatePerson("dummy-person");
		}

		public static Galago.Service MakeDummyService()
		{
			return Galago.Global.CreateService("dummy-service",
											   "Dummy Service", 0);
		}

		public static Galago.Account MakeDummyAccount()
		{
			return MakeDummyService().CreateAccount(MakeDummyPerson(),
													"dummy-account");
		}

		public static Galago.Presence MakeDummyPresence()
		{
			return MakeDummyAccount().CreatePresence();
		}
	}

	[TestFixture]
	public class InitTest
	{
		[Test]
		public void TestInitReinit()
		{
			Assert.IsTrue(Galago.Global.Init("check-galago-sharp",
											 Galago.InitFlags.Feed),
						  "Unable to initialize Galago");
			Assert.IsTrue(Galago.Global.IsInitted,
						  "Galago.Global.Init succeeded, but IsInitted failed");
			Galago.Global.Uninit();
			Assert.IsFalse(Galago.Global.IsInitted,
						   "Still initted after Galago.Global.Uninit");

			Assert.IsTrue(Galago.Global.Init("check-galago-sharp"),
						  "Unable to initialize Galago");
			Assert.IsTrue(Galago.Global.IsInitted,
						  "Galago.Global.Init succeeded, but IsInitted failed");
			Galago.Global.Uninit();
			Assert.IsFalse(Galago.Global.IsInitted,
						   "Still initted after Galago.Global.Uninit");
		}
	}

	public class BaseTest
	{
		[SetUp]
		public void Init()
		{
			Assert.IsTrue(Galago.Global.Init("check-galago-python",
											 Galago.InitFlags.Feed),
						  "Unable to initialize Galago");
		}

		[TearDown]
		public void Uninit()
		{
			Galago.Global.Uninit();
			Assert.IsFalse(Galago.Global.IsInitted,
						   "Still initted after Galago.Global.Uninit");
		}
	}

	[TestFixture]
	public class ObjectTest : BaseTest
	{
		[Test]
		public void TestObjectsEqual()
		{
			Assert.AreEqual(Utils.MakeDummyService(),
							Utils.MakeDummyService());
			Assert.AreEqual(Utils.MakeDummyAccount(),
							Utils.MakeDummyAccount());
			Assert.AreEqual(Utils.MakeDummyPresence(),
							Utils.MakeDummyPresence());
		}

		[Test]
		public void TestCreatePerson()
		{
			CheckObject(Galago.Global.CreatePerson("dummy-person"));
			CheckObject(Galago.Global.CreatePerson());
		}

		[Test]
		public void TestCreateService()
		{
			CheckObject(Galago.Global.CreateService("dummy-service-1",
													"Dummy Service 1", 0));
			CheckObject(Galago.Global.CreateService("dummy-service-2",
													"Dummy Service 2"));
		}

		[Test]
		public void TestCreateAccount()
		{
			Galago.Person person;
			Galago.Service service;

			person = Galago.Global.CreatePerson("dummy-person-x");
			service = Galago.Global.CreateService("dummy-service-x",
												  "Dummy Service");
			CheckObject(person);
			CheckObject(service);
			CheckObject(service.CreateAccount(person, "dummy-account"));
		}

		[Test]
		public void TestCreatePresence()
		{
			CheckObject(Utils.MakeDummyPresence());
		}

		private void CheckObject(Galago.Object obj)
		{
			Assert.IsNotNull(obj, "Resulting object is null");
			Assert.IsFalse(obj.DBusPath == "",
						   "Resulting object's D-BUS path is empty");
		}
	}

	[TestFixture]
	public class ServiceTest : BaseTest
	{
		[Test]
		public void TestNormalize()
		{
			CheckNormalizeWith(0, "joebobsmith/home");
			CheckNormalizeWith(Galago.ServiceFlags.PreserveSpaces,
							   "joebob  smith/home");
			CheckNormalizeWith(Galago.ServiceFlags.PreserveCase,
							   "JoeBobSmith/Home");
			CheckNormalizeWith(Galago.ServiceFlags.StripSlash,
							   "joebobsmith");
			CheckNormalizeWith(Galago.ServiceFlags.PreserveSpaces |
							   Galago.ServiceFlags.PreserveCase |
							   Galago.ServiceFlags.StripSlash,
							   "JoeBob  Smith");
		}

		private void CheckNormalizeWith(Galago.ServiceFlags flags,
										string expectedStr)
		{
			Galago.Service service =
				Galago.Global.CreateService("test-service", "Test Service",
											flags);
			string username = service.Normalize("JoeBob  Smith/Home");
			Assert.IsTrue(username == expectedStr);
			service.Destroy();
		}
	}

	[TestFixture]
	public class PersonTest : BaseTest
	{
		[Test]
		public void TestPriorityAccounts()
		{
			Galago.Person  person  = Utils.MakeDummyPerson();
			Galago.Service service = Utils.MakeDummyService();
			Galago.Presence presence;

			Galago.Account account1 = service.CreateAccount(person,
															"account-1");
			presence = account1.CreatePresence();
			presence.SetIdle(true);

			Galago.Account account2 = service.CreateAccount(person,
															"account-2");
			presence = account2.CreatePresence();

			Galago.Account priorityAccount = person.PriorityAccount;

			Assert.IsNotNull(priorityAccount, "Priority account is null");
			Assert.AreEqual(priorityAccount, account2,
							"Priority account was not account 2");

			// Test with a dummy handler.
			Galago.Global.Core.CalcPriorityAccount += DummyCalcPriorityAccount;
			priorityAccount = person.PriorityAccount;
			Assert.IsNotNull(priorityAccount, "Priority account is null");
			Assert.AreEqual(priorityAccount, account2,
							"Priority account was not account 2");
			Galago.Global.Core.CalcPriorityAccount -= DummyCalcPriorityAccount;

			// Test with a custom handler.
			Galago.Global.Core.CalcPriorityAccount += CustomCalcPriorityAccount;

			priorityAccount = person.PriorityAccount;
			Assert.IsNotNull(priorityAccount, "Priority account is null");
			Assert.AreEqual(priorityAccount, account1,
							"Priority account was not account 1");
			Galago.Global.Core.CalcPriorityAccount -= CustomCalcPriorityAccount;
		}

		private void DummyCalcPriorityAccount(
			object obj, Galago.CalcPriorityAccountArgs args)
		{
			args.RetVal = null;
		}

		private void CustomCalcPriorityAccount(
			object obj, Galago.CalcPriorityAccountArgs args)
		{
			GLib.List accounts = args.Person.GetAccounts();
			Assert.IsTrue(accounts.Count > 0,
						  "Returned accounts list is empty");
			args.RetVal = accounts[0];
		}
	}
}
