/*********************************************
 *
 * $GAMGI/src/gtk/cell/gamgi_gtk_cell_link.c
 *
 * Copyright (C) 2004 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

#include "gamgi_engine.h"
#include "gamgi_gtk.h"
#include "gamgi_phys.h"
#include "gamgi_math.h"
#include "gamgi_io.h"
#include "gamgi_global.h"

#include "gamgi_engine_link.h"
#include "gamgi_engine_unlink.h"
#include "gamgi_engine_find.h"
#include "gamgi_gtk_dialog.h"
#include "gamgi_gtk_history.h"
#include "gamgi_gtk_object.h"
#include "gamgi_mesa_select.h"
#include "gamgi_phys_cell_link.h"
#include "gamgi_phys_space.h"
#include "gamgi_phys_rcp.h"
#include "gamgi_io_token.h"

static gamgi_enum static_class (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *button;
GtkWidget *combo;
const char *name;
int row;

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_global");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_cell");
name = gtk_entry_get_text (GTK_ENTRY (entry));

/*******************************************
 * when entries are empty, look for a cell *
 *******************************************/

if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == FALSE &&
gamgi_io_token_check (name) == FALSE) return GAMGI_ENGINE_CELL;

/********************
 * get object class *
 ********************/

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_above");
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == TRUE)
  {
  combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_above");
  row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
  if (row == 0) return GAMGI_ENGINE_LAYER;
  if (row == 1) return GAMGI_ENGINE_ASSEMBLY;
  }
else
  {
  combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_below");
  row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
  if (row == 0) return GAMGI_ENGINE_CLUSTER;
  if (row == 1) return GAMGI_ENGINE_MOLECULE;
  if (row == 2) return GAMGI_ENGINE_GROUP;
  if (row == 3) return GAMGI_ENGINE_PLANE;
  if (row == 4) return GAMGI_ENGINE_DIRECTION;
  if (row == 5) return GAMGI_ENGINE_ATOM;
  if (row == 6) return GAMGI_ENGINE_ORBITAL;
  if (row == 7) return GAMGI_ENGINE_TEXT;
  }

return FALSE;
}

static void static_number (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *button;

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_number");
button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_number");
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == TRUE)
  {
  gtk_widget_set_sensitive (entry, TRUE);
  }
else
  {
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  gtk_widget_set_sensitive (entry, FALSE);
  }
}

static void static_wyckoff_change (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
GtkWidget *entry;
const char **wyckoff;
const int *driver;
gamgi_bool used[3];
int site;

/***********************************
 * wyckoff data: positions, driver *
 ***********************************/

wyckoff = (const char **) g_object_get_data (G_OBJECT (dialog), "wyckoff");
driver = (const int *) g_object_get_data (G_OBJECT (dialog), "driver");

/**************************
 * determine wyckoff site *
 **************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_wyckoff");
site = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

/******************************************************************
 * determine which x,y,z entries are needed for this wyckoff site *
 *                                                                *
 * defensive programming: if wyckoff is NULL then driver should   *
 * be also NULL and according to the code above, site will be 0   *
 ******************************************************************/

if (wyckoff == NULL || driver == NULL || site == 0)
  { used[0] = TRUE; used[1] = TRUE; used[2] = TRUE; }
else
  gamgi_phys_space_used (wyckoff, driver, site, used);

/************************
 * update x,y,z entries *
 ************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
if (used[0] == TRUE) gtk_widget_set_sensitive (entry, TRUE);
else
  { 
  gtk_widget_set_sensitive (entry, FALSE);
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
if (used[1] == TRUE) gtk_widget_set_sensitive (entry, TRUE);
else
  { 
  gtk_widget_set_sensitive (entry, FALSE);
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
if (used[2] == TRUE) gtk_widget_set_sensitive (entry, TRUE);
else
  { 
  gtk_widget_set_sensitive (entry, FALSE);
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  }
}

static void static_wyckoff_start (gamgi_window *window, gamgi_cell *cell)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
GtkListStore *store;
GtkTreeIter iter;
const int *driver;
int site, n_sites, n_positions;
int group;
char token[GAMGI_ENGINE_TOKEN];
const char **wyckoff;
const char **symmetry;
char letter;

/******************************
 * get crystallographic group *
 ******************************/

if (cell == NULL) group = 0;
else group = cell->group;

/*****************************************************
 * create combo box list: the first option,          *
 * the basis, is always avalable. To guarantee that  *
 * the combo box is not resized, the Basis option    *
 * should be at least as long as the longest option, *
 * apparently group 221, with positions 4/mm.m       *
 *****************************************************/

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, " 1  Basis  1     ", -1);

/*********************************************
 * List all the wyckoff positions, according *
 * to  the group information available.      *
 *********************************************/

if (group > 0)
  {
  gamgi_phys_space_get (group, &wyckoff, &symmetry, &driver);
  g_object_set_data (G_OBJECT (dialog), "wyckoff", (void *) wyckoff);
  g_object_set_data (G_OBJECT (dialog), "driver", (void *) driver);

  n_sites = driver[0];
  for (site = 1; site <= n_sites; site++)
    {
    n_positions = gamgi_phys_space_multiplicity (driver, site);
    letter = gamgi_phys_space_letter (n_sites, site);
    sprintf (token, "%2d  %c  %s", n_positions, letter, symmetry[site - 1]);

    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter, 0, token, -1);
    }
  }
else
  {
  g_object_set_data (G_OBJECT (dialog), "wyckoff", NULL);
  g_object_set_data (G_OBJECT (dialog), "driver", NULL);
  }

/*********************************************************
 * set model (and automatically remove the previous one) *
 *********************************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_wyckoff");
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

/**********************************************************
 * When group information is available, the combo default *
   is the group general position, otherwise is the basis  *
 **********************************************************/

if (group == 0)
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
else
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 1);

static_wyckoff_change (NULL, window);
}

static void static_nodes_start (gamgi_window *window, gamgi_cell *cell)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *button_p, *button_i, *button_a, *button_b, *button_c, *button_r;
gamgi_enum lattice, type;

/*****************************
 * get cell lattice and type *
 *****************************/

if (cell == NULL)
  {
  lattice = FALSE;
  type = FALSE;
  }
else
  {
  lattice = cell->lattice;
  type = cell->type;
  }

/***********************************
 * get buttons for occupancy nodes *
 ***********************************/

button_p = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_p");
button_i = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_i");
button_a = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_a");
button_b = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_b");
button_c = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_c");
button_r = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_r");

/*******************************
 * set P,I,A,B,C,R buttons off *
 *******************************/

gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_p), FALSE);
gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_i), FALSE);
gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_a), FALSE);
gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_b), FALSE);
gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_c), FALSE);
gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_r), FALSE);

/**********************************
 * set P,I,A,B,C,R buttons active *
 **********************************/

gtk_widget_set_sensitive (button_p, TRUE);
gtk_widget_set_sensitive (button_i, TRUE);
gtk_widget_set_sensitive (button_a, TRUE);
gtk_widget_set_sensitive (button_b, TRUE);
gtk_widget_set_sensitive (button_c, TRUE);
gtk_widget_set_sensitive (button_r, TRUE);

if (lattice == FALSE) return;

/********************************************************
 * set P,I,A,B,C,R buttons according to Bravais lattice *
 ********************************************************/

gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_p), TRUE);

switch (lattice)
  {
  case GAMGI_PHYS_CUBIC_I: case GAMGI_PHYS_TETRAGONAL_I: 
  case GAMGI_PHYS_ORTHORHOMBIC_I:
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_i), TRUE);
  break;
 
  case GAMGI_PHYS_CUBIC_F: case GAMGI_PHYS_ORTHORHOMBIC_F:
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_a), TRUE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_b), TRUE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_c), TRUE);
  break;

  case GAMGI_PHYS_ORTHORHOMBIC_C: case GAMGI_PHYS_MONOCLINIC_C:
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_c), TRUE);
  break;

  case GAMGI_PHYS_HEXAGONAL_R:
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_r), TRUE);
  break;
  
  default:
  break;
  }

/************************************************************
 * buttons for nodes that users cannot select (because they *
 * do not exist for this cell lattice) or unselect (because *
 * this cell type does not allow it) are set insensitive    *
 ************************************************************/

if (type != GAMGI_PHYS_CONVENTIONAL)
  {
  gtk_widget_set_sensitive (button_p, FALSE);
  gtk_widget_set_sensitive (button_i, FALSE);
  gtk_widget_set_sensitive (button_a, FALSE);
  gtk_widget_set_sensitive (button_b, FALSE);
  gtk_widget_set_sensitive (button_c, FALSE);
  gtk_widget_set_sensitive (button_r, FALSE);
  }
else
  {
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_i)) == FALSE)
    gtk_widget_set_sensitive (button_i, FALSE);
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_a)) == FALSE)
    gtk_widget_set_sensitive (button_a, FALSE);
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_b)) == FALSE)
    gtk_widget_set_sensitive (button_b, FALSE);
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_c)) == FALSE)
    gtk_widget_set_sensitive (button_c, FALSE);
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_r)) == FALSE)
    gtk_widget_set_sensitive (button_r, FALSE);
  }
}

static void static_cell (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
gamgi_cell *cell;
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
int row;

/*****************
 * get link mode *
 *****************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_link");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

if (row == 1)
  {
  /*****************************************************
   * crystallographic link mode: show wyckoff,node     *
   * data (when cell is NULL, reset wyckoff,node data) *
   *****************************************************/

  cell = GAMGI_CAST_CELL gamgi_gtk_object_name_number (dialog,
  "entry_cell", GAMGI_ENGINE_CELL, window);

  static_wyckoff_start (window, cell);
  static_nodes_start (window, cell);
  }
}

static void static_start_position (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
GtkWidget *entry;
char token[GAMGI_ENGINE_TOKEN];

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_vectors");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
sprintf (token, "%.*f", gamgi->gamgi->length, GAMGI_PHYS_NODE_X);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
sprintf (token, "%.*f", gamgi->gamgi->length, GAMGI_PHYS_NODE_Y);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
sprintf (token, "%.*f", gamgi->gamgi->length, GAMGI_PHYS_NODE_Z);
gtk_entry_set_text (GTK_ENTRY (entry), token);
}

static void static_start_occupancy (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
GtkWidget *entry;

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_cutoff");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_a");
gtk_entry_set_text (GTK_ENTRY (entry), "*");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_b");
gtk_entry_set_text (GTK_ENTRY (entry), "*");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_c");
gtk_entry_set_text (GTK_ENTRY (entry), "*");
}

static void static_start_packing (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *button;
char token[GAMGI_ENGINE_TOKEN];

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_radius");
gtk_button_clicked (GTK_BUTTON (button));

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_time");
sprintf (token, "%.*f", 1, GAMGI_PHYS_RCP_TIME);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_seed");
sprintf (token, "%d", GAMGI_PHYS_RCP_SEED);
gtk_entry_set_text (GTK_ENTRY (entry), token);
}

static void static_link (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
GtkWidget *vbox_position, *vbox_occupancy, *vbox_packing;
int row;

/**********************
 * get link combo box *
 **********************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_link");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

/************
 * get data *
 ************/

vbox_position = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_position");
vbox_occupancy = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_occupancy");
vbox_packing = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_packing");

/*******************************
 * Object: link object to cell *
 *******************************/

if (row == 0)
  {
  gtk_widget_hide (vbox_position);
  gtk_widget_hide (vbox_occupancy);
  gtk_widget_hide (vbox_packing);
  }

/************************************************
 * Crystallographic: use crystallographic       *
 * information to link object to lattice nodes  *
 ************************************************/

if (row == 1)
  {
  gtk_widget_show (vbox_position);
  gtk_widget_show (vbox_occupancy);
  gtk_widget_hide (vbox_packing);

  static_cell (NULL, window);
  static_start_position (window);
  static_start_occupancy (window);
  }

/********************************************
 * Random Close Packing: use RCP algorithm  *
 * to build compact random sphere structure *
 ********************************************/

if (row == 2)
  {
  gtk_widget_hide (vbox_position);
  gtk_widget_hide (vbox_occupancy);
  gtk_widget_show (vbox_packing);

  static_start_packing (window);
  }

}

static void static_hierarchy (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *combo_below, *combo_above, *combo_link;
GtkWidget *button;
GtkWidget *entry;
gamgi_bool *sensitive;

combo_above = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_above");
combo_below = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_below");

combo_link = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_link");
sensitive = (gamgi_bool *) g_object_get_data (G_OBJECT (dialog), "sensitive_link");

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_above");
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == TRUE)
  {
  /***************************************************
   * the default for the object class above is Layer *
   ***************************************************/

  gtk_widget_show (combo_above);
  gtk_widget_hide (combo_below);
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo_above), 0);

  gtk_combo_box_set_active (GTK_COMBO_BOX (combo_link), 0);
  sensitive[1] = FALSE;
  sensitive[2] = FALSE;
  }
else
  {
  /***********************************************
   * the default for the combo box below is Atom *
   ***********************************************/

  gtk_widget_hide (combo_above);
  gtk_widget_show (combo_below);
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo_below), 5);

  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  }

/**********************
 * reset object entry *
 **********************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_object");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/********************
 * update mode data *
 ********************/

static_link (NULL, window);
}

/****************** internal function ****************
 *                                                   *
 *                     STATIC_RESET                  *
 *                                                   *
 * Reset the cell link dialog, cleaning all entries. *
 *                                                   *
 *****************************************************/

static void static_reset (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;

/*******************************
 * reset cell and object names *
 *******************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_cell");
gtk_entry_set_text (GTK_ENTRY (entry), "");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_object");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/********************
 * reset data pages *
 ********************/

static_cell (NULL, window);
static_start_position (window);
static_start_occupancy (window);
}

static void static_clean (gamgi_cell *cell, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *button;
GtkWidget *entry;

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_above");
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == TRUE)
  { static_reset (window); return; }

/***************************
 * keep cell, reset object *
 ***************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_object");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/********************
 * reset data pages *
 ********************/

static_cell (NULL, window);
static_start_position (window);
static_start_occupancy (window);
}

/************ internal function ************
 *                                         *
 *               STATIC_GLOBAL             *
 *                                         *
 * Makes the cell entry sensitive when the *
 * local button is pressed and insensitive *
 * when the global button is pressed.      *
 *                                         *
 *******************************************/

static void static_global (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *hbox, *button;

hbox = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "hbox_cell");
button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_global");

if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == FALSE)
  {
  /****************************
   * Choose the cell directly *
   ****************************/

  gtk_widget_set_sensitive (hbox, TRUE);
  }
else
  {
  /****************************************
   * Choose the cells previously selected *
   ****************************************/

  static_reset (window);
  gtk_widget_set_sensitive (hbox, FALSE);
  }
}

static void static_object (gamgi_cell *cell, 
gamgi_object *object, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *button;

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_above");
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == TRUE)
  gamgi_gtk_object_link (GAMGI_CAST_OBJECT cell, object, window);
else
  gamgi_gtk_object_link (object, GAMGI_CAST_OBJECT cell, window);

static_clean (cell, window);
}

static void static_crystallographic (gamgi_cell *cell, 
gamgi_object *object, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *button;
GtkWidget *combo;
const char **wyckoff;
const char *name, *rule_a, *rule_b, *rule_c;
char token[GAMGI_ENGINE_TOKEN];
double position[3];
gamgi_enum vectors, cutoff;
gamgi_bool nodes_p, nodes_i, nodes_a, nodes_b, nodes_c, nodes_r; 
const int *driver; 
int site;
int row;

/***************************************
 * Currently only atoms are supported, *
 * due to object orientation issues    *
 ***************************************/

if (object->class != GAMGI_ENGINE_ATOM)
  {
  gamgi_gtk_dialog_message_create ("Error", "Sorry, not supported yet", window);
  return;
  }

/*************************
 * get vectors combo box *
 *************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_vectors");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

vectors = FALSE;
if (row == GAMGI_PHYS_CONVENTIONAL - 1) vectors = GAMGI_PHYS_CONVENTIONAL;
if (row == GAMGI_PHYS_PRIMITIVE - 1) vectors = GAMGI_PHYS_PRIMITIVE;

/*************************
 * get wyckoff combo box *
 *************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_wyckoff");
site = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

wyckoff = (const char **) g_object_get_data (G_OBJECT (dialog), "wyckoff");
driver = (const int *) g_object_get_data (G_OBJECT (dialog), "driver");

/**************************
 * get vector coordinates *
 **************************/

position[0] = 0;
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (GTK_WIDGET_SENSITIVE (entry) == TRUE && 
gamgi_io_token_double_scan (name, &position[0], -DBL_MAX, DBL_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Vector Coordinates", window);
  return;
  }

position[1] = 0;
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (GTK_WIDGET_SENSITIVE (entry) == TRUE && 
gamgi_io_token_double_scan (name, &position[1], -DBL_MAX, DBL_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Vector Coordinates", window);
  return;
  }

position[2] = 0;
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (GTK_WIDGET_SENSITIVE (entry) == TRUE &&
gamgi_io_token_double_scan (name, &position[2], -DBL_MAX, DBL_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Vector Coordinates", window);
  return;
  }

/*********************************************************************
 * coordinates for referential vectors must be real, coordinates for *
 * conventional and primitive vectors must be in the range ]-1, +1[  *
 *********************************************************************/

if (vectors != FALSE && (abs (position[0]) >= 1.0 || 
abs (position[1]) >= 1.0 || abs (position[2]) >= 1.0))
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Vector Coordinates", window);
  return;
  }
 
/************************
 * get cutoff combo box *
 ************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_cutoff");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

cutoff = FALSE;
if (row == 0) cutoff = GAMGI_PHYS_CUTOFF_OBJECTS;
if (row == 1) cutoff = GAMGI_PHYS_CUTOFF_NODES;

/********************
 * get active nodes *
 ********************/

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_p");
nodes_p = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_i");
nodes_i = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_a");
nodes_a = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_b");
nodes_b = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_c");
nodes_c = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "check_button_r");
nodes_r = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));

/*************
 * get rules *
 *************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_a");
rule_a = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_alpha_scan (rule_a, token, 
GAMGI_IO_RULES, GAMGI_ENGINE_STRING) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Occupancy Rule", window);
  return;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_b");
rule_b = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_alpha_scan (rule_b, token, 
GAMGI_IO_RULES, GAMGI_ENGINE_STRING) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Occupancy Rule", window);
  return;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_c");
rule_c = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_alpha_scan (rule_c, token, 
GAMGI_IO_RULES, GAMGI_ENGINE_STRING) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Occupancy Rule", window);
  return;
  }

/***************************************
 * Save current layer as undo buffer   *
 * layer so it can be recovered later. *
 ***************************************/

gamgi_gtk_history_update (window->layer, window->layer->undo, GAMGI_ENGINE_UNDO);

/**********************************
 * link objects to cell and reset *
 **********************************/

gamgi_phys_cell_link (cell, GAMGI_CAST_ATOM object, vectors, wyckoff, 
driver, site, position, cutoff, nodes_p, nodes_i, nodes_a, nodes_b, 
nodes_c, nodes_r, rule_a, rule_b, rule_c, window);

static_clean (cell, window);

/************************
 * Redraw gl_area image *
 ************************/

gtk_widget_queue_draw (window->area);
}

static void static_random (gamgi_cell *cell, 
gamgi_object *object, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *text;
GtkWidget *entry;
const char *name;
double time;
int n_atoms;
int seed;

/****************************
 * Only atoms can be packed *
 ****************************/

if (object->class != GAMGI_ENGINE_ATOM)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Object Data", window);
  return;
  }

/***************************************************************
 * calculate number of atoms that will have approximately the  *
 * same radius as the template or get explicit number of atoms *
 ***************************************************************/

n_atoms = 0;
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_number");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (GTK_WIDGET_SENSITIVE (entry) == TRUE && gamgi_io_token_int_scan (name, 
&n_atoms, GAMGI_PHYS_RCP_ATOMS, INT_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Atom Number", window);
  return;
  }
  
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_time");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_double_scan (name, &time, 
GAMGI_PHYS_RCP_TIME_MIN, GAMGI_PHYS_RCP_TIME_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Relaxation Data", window);
  return;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_seed");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_int_scan (name, &seed, 1, INT_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Seed Data", window);
  return;
  }

/***************************************
 * Save current layer as undo buffer   *
 * layer so it can be recovered later. *
 ***************************************/

gamgi_gtk_history_update (window->layer, window->layer->undo, GAMGI_ENGINE_UNDO);

/**********************************
 * create and initialize text     *
 * widget, to show the RCP report *
 **********************************/

text = gamgi_gtk_dialog_text_create ();

/***********************
 * apply RCP algorithm *
 ***********************/

if (gamgi_phys_rcp (cell, GAMGI_CAST_ATOM object, 
n_atoms, seed, time, text) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Packing Data", window);
  gtk_widget_destroy (text);
  return;
  }

dialog = gamgi_gtk_dialog_report_create (text, 
"Random Close Packing", GAMGI_GTK_CELL_LINK_RCP, window);
gtk_widget_show (dialog);

/************************
 * Redraw gl_area image *
 ************************/

gtk_widget_queue_draw (window->area);
}

static void static_ok (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
gamgi_cell *cell;
gamgi_object *object;
int row;

/************
 * Get Cell *
 ************/

cell = GAMGI_CAST_CELL gamgi_gtk_object_name_number (dialog,
"entry_cell", GAMGI_ENGINE_CELL, window);
if (cell == NULL)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Cell", window);
  return;
  }

/*********************
 * Get global Object *
 *********************/

object = gamgi_gtk_object_name_number (dialog,
"entry_object", static_class (window), NULL);
if (object == NULL)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Object", window);
  return;
  }

/**********************
 * get link combo box *
 **********************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_link");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

/************************************************
 * get method: object, crystallographic, random *
 ************************************************/

if (row == 0) static_object (cell, object, window);
if (row == 1) static_crystallographic (cell, object, window);
if (row == 2) static_random (cell, object, window);
}

static void static_init (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *notebook;
GtkWidget *button;
GtkWidget *combo_above, *combo_below, *combo;
GtkRequisition size;
int width, height;

/***********************************************************
 * combo boxes above and below need different sizes: use   *
 * the largest size for both, so the size remains constant *
 ***********************************************************/

combo_above = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_above");
gtk_widget_size_request (GTK_WIDGET (combo_above), &size);
width = size.width; height = size.height;

combo_below = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_below");
gtk_widget_size_request (GTK_WIDGET (combo_below), &size);
if (size.width > width) width = size.width;
if (size.height > height) height = size.height;

gtk_widget_set_size_request (combo_above, width, height);
gtk_widget_set_size_request (combo_below, width, height);

/*************************************************
 * show combo box BEFORE measuring notebook size *
 *************************************************/

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_below");
gtk_button_clicked (GTK_BUTTON (button));

/***************************************************
 * Set notebook current size as its maximum size:  *
 * this is needed because pages are shown/hidden,  *
 * changing the currently needed size for dialog.  *
 * Fixing the dialog size only partly solves the   *
 * problem because the page size can still change. *
 ***************************************************/

notebook = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "notebook");
gtk_widget_size_request (GTK_WIDGET (notebook), &size);
gtk_widget_set_size_request (notebook, size.width, size.height);

/********************************************
 * hide pages AFTER measuring notebook size *
 ********************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_link");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
}

static void static_press (gamgi_object *object, void *data)
{
gamgi_window *window_dialog = gamgi->window_dialog;
GtkWidget *dialog = window_dialog->dialog0;
GtkWidget *button;
GtkWidget *entry;
char string[2 * GAMGI_ENGINE_TOKEN];
const char *name;

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_global");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_cell");
name = gtk_entry_get_text (GTK_ENTRY (entry));

if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == TRUE ||
gamgi_io_token_check (name) == TRUE)
  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_object");

sprintf (string, "%s %d", object->name, object->number);
gtk_entry_set_text (GTK_ENTRY (entry), string);
}

/***************** external function ****************
 *                                                  *
 *              GAMGI_GTK_CELL_LINK_PRESS           *
 *                                                  *
 ****************************************************/

void gamgi_gtk_cell_link_press (gamgi_window *window_mouse,
int x, int y, gamgi_window *window_dialog)
{
GtkWidget *dialog = window_dialog->dialog0;
GtkWidget *button;
GtkWidget *entry;
const char *name;

/***********************************************
 * global selection can be used only to select *
 * the second object, not the main object(s)   *
 ***********************************************/
 
button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_global");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_cell");
name = gtk_entry_get_text (GTK_ENTRY (entry));

if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == FALSE &&
gamgi_io_token_check (name) == FALSE && 
window_dialog != window_mouse) return;

/************************************************************************
 * window_dialog is the (local) window where the dialog window was open *
 * window_mouse is the (global) window where the user presses the mouse *
 ************************************************************************/

gamgi_mesa_select_object (window_mouse, x, y,
static_class (window_dialog), TRUE, static_press);
}

/*************** external function ******************
 *                                                  *
 *               GAMGI_GTK_CELL_LINK                *
 *                                                  *
 *  Creates the dialog window used to link cells.   *
 *                                                  *
 ****************************************************/

void gamgi_gtk_cell_link (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog;
GtkWidget *notebook;
GtkWidget *button;
GtkWidget *label;
GtkWidget *entry;
GtkWidget *frame;
GtkWidget *table;
GtkWidget *hbox_center, *hbox_left, *hbox_left_left;
GtkWidget *vbox_dialog, *vbox_page, *vbox_frame, *vbox_top, *vbox_top_top;
GtkWidget *combo;
GtkListStore *store;
GtkCellRenderer *renderer;
GtkTreeIter iter;
gamgi_bool *sensitive;

/******************
 * dialog level 0 *
 ******************/

dialog = gamgi_gtk_dialog_task0_create ("Cell Link", window);
window->action = GAMGI_GTK_CELL_LINK;
gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);

/********************
 * global container *
 ********************/

vbox_dialog = gtk_vbox_new (FALSE, 5);
gtk_container_add (GTK_CONTAINER (dialog), vbox_dialog);
gtk_widget_show (vbox_dialog);

/****************
 * Cell objects *
 ****************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_dialog), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 10);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

/*********
 * Local *
 *********/

hbox_left_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_left), hbox_left_left, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "hbox_cell", hbox_left_left);
gtk_widget_show (hbox_left_left);

label = gtk_label_new ("Cell");
gtk_box_pack_start (GTK_BOX (hbox_left_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
g_signal_connect (GTK_OBJECT (entry), "changed",
G_CALLBACK (static_cell), window);
g_object_set_data (G_OBJECT (dialog), "entry_cell", entry);
gtk_widget_show (entry);

/**********
 * Global *
 **********/

button = gtk_toggle_button_new_with_label ("Global");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_signal_connect (GTK_OBJECT (button), "toggled",
G_CALLBACK (static_global), window);
g_object_set_data (G_OBJECT (dialog), "button_global", button);
gtk_widget_show (button);

/************
 * notebook *
 ************/

notebook = gtk_notebook_new ();
gtk_box_pack_start (GTK_BOX (vbox_dialog), notebook, FALSE, FALSE, 0);
gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
g_object_set_data (G_OBJECT (dialog), "notebook", notebook);
gtk_widget_show (notebook);

/***************
 * Object page *
 ***************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new (" Object ");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 5);
gtk_widget_show (vbox_page);

frame = gtk_frame_new (NULL);
gtk_box_pack_start (GTK_BOX (vbox_page), frame, TRUE, TRUE, 0);
gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
gtk_widget_show (frame);

vbox_frame = gtk_vbox_new (TRUE, 0);
gtk_container_add (GTK_CONTAINER (frame), vbox_frame);
gtk_container_set_border_width (GTK_CONTAINER (vbox_frame), 5);
gtk_widget_show (vbox_frame);

vbox_top = gtk_vbox_new (FALSE, 15);
gtk_box_pack_start (GTK_BOX (vbox_frame), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_top), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

/*************
 * Hierarchy *
 *************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Hierarchy");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 5);
gtk_widget_show (label);

button = gtk_radio_button_new_with_label (NULL, "Above");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0); 
g_signal_connect (GTK_OBJECT (button), "clicked",
G_CALLBACK (static_hierarchy), window);
g_object_set_data (G_OBJECT (dialog), "button_above", button);
gtk_widget_show (button);

button = gtk_radio_button_new_with_label (
gtk_radio_button_get_group (GTK_RADIO_BUTTON (button)), "Below");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_signal_connect (GTK_OBJECT (button), "clicked",
G_CALLBACK (static_hierarchy), window);
g_object_set_data (G_OBJECT (dialog), "button_below", button);
gtk_widget_show (button);

/**********
 * object *
 **********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

/************************************
 * hide combo box for objects above *
 ************************************/

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "combo_above", combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Layer", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Assembly", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/************************************
 * hide combo box for objects below *
 ************************************/

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "combo_below", combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Cluster", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Molecule", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Group", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Plane", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Direction", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Atom", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Orbital", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Text", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), entry, FALSE, FALSE, 0); 
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
g_object_set_data (G_OBJECT (dialog), "entry_object", entry);
gtk_widget_show (entry);

/********
 * link *
 ********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Link");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_signal_connect (GTK_OBJECT (combo), "changed",
G_CALLBACK (static_link), window);
g_object_set_data (G_OBJECT (dialog), "combo_link", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Object", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Crystallographic", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Random", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

sensitive = gamgi_gtk_dialog_sensitive_create (3);
g_object_set_data (G_OBJECT (dialog), "sensitive_link", sensitive);
gtk_cell_layout_set_cell_data_func (GTK_CELL_LAYOUT (combo),
renderer, gamgi_gtk_dialog_sensitive_scan, sensitive, free);

/*****************
 * Position page *
 *****************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new ("Position");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 5);
g_object_set_data (G_OBJECT (dialog), "vbox_position", vbox_page);
gtk_widget_show (vbox_page);

frame = gtk_frame_new (NULL);
gtk_box_pack_start (GTK_BOX (vbox_page), frame, TRUE, TRUE, 0);
gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
gtk_widget_show (frame);

vbox_frame = gtk_vbox_new (TRUE, 0);
gtk_container_add (GTK_CONTAINER (frame), vbox_frame);
gtk_container_set_border_width (GTK_CONTAINER (vbox_frame), 5);
gtk_widget_show (vbox_frame);

vbox_top = gtk_vbox_new (FALSE, 10);
gtk_box_pack_start (GTK_BOX (vbox_frame), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 15);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

/*********************
 * Vectors combo box *
 *********************/

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_left), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

label = gtk_label_new ("Vectors");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (vbox_top_top), combo, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "combo_vectors", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Conventional", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Primitive", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Local", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/*********************
 * Wyckoff combo box *
 *********************/

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_left), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

label = gtk_label_new ("Wyckoff");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (vbox_top_top), combo, FALSE, FALSE, 0);
g_signal_connect (GTK_OBJECT (combo), "changed",
G_CALLBACK (static_wyckoff_change), window);
g_object_set_data (G_OBJECT (dialog), "combo_wyckoff", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, " 1  Basis  1     ", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/***************
 * Coordinates *
 ***************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

label = gtk_label_new ("Coordinates");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
gtk_widget_show (label);

table = gtk_table_new (1, 6, FALSE);
gtk_box_pack_start (GTK_BOX (vbox_top_top), table, FALSE, FALSE, 0);
gtk_widget_show (table);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_x", entry);
gtk_widget_show (entry);

label = gtk_label_new ("X");
gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_y", entry);
gtk_widget_show (entry);

label = gtk_label_new ("Y");
gtk_table_attach (GTK_TABLE (table), label, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 4, 5, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_z", entry);
gtk_widget_show (entry);

label = gtk_label_new ("Z");
gtk_table_attach (GTK_TABLE (table), label, 5, 6, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

/******************
 * Occupancy page *
 ******************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new ("Occupancy");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 5);
g_object_set_data (G_OBJECT (dialog), "vbox_occupancy", vbox_page);
gtk_widget_show (vbox_page);

frame = gtk_frame_new (NULL);
gtk_box_pack_start (GTK_BOX (vbox_page), frame, TRUE, TRUE, 0);
gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
gtk_widget_show (frame);

vbox_frame = gtk_vbox_new (TRUE, 0);
gtk_container_add (GTK_CONTAINER (frame), vbox_frame);
gtk_container_set_border_width (GTK_CONTAINER (vbox_frame), 5);
gtk_widget_show (vbox_frame);

vbox_top = gtk_vbox_new (FALSE, 10);
gtk_box_pack_start (GTK_BOX (vbox_frame), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

/********************
 * Cutoff combo box *
 ********************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Cutoff");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "combo_cutoff", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Objects", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Nodes", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/*********
 * Nodes *
 *********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Nodes");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

button = gtk_check_button_new_with_label ("P");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "check_button_p", button);
gtk_widget_show (button);

button = gtk_check_button_new_with_label ("I");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "check_button_i", button);
gtk_widget_show (button);

button = gtk_check_button_new_with_label ("A");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "check_button_a", button);
gtk_widget_show (button);

button = gtk_check_button_new_with_label ("B");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "check_button_b", button);
gtk_widget_show (button);

button = gtk_check_button_new_with_label ("C");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "check_button_c", button);
gtk_widget_show (button);

button = gtk_check_button_new_with_label ("R");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "check_button_r", button);
gtk_widget_show (button);

/*******************
 * Occupancy rules *
 *******************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

label = gtk_label_new ("Rules");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
gtk_widget_show (label);

table = gtk_table_new (2, 3, FALSE);
gtk_box_pack_start (GTK_BOX (vbox_top_top), table, FALSE, FALSE, 0);
gtk_widget_show (table);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_10);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_ENGINE_STRING);
g_object_set_data (G_OBJECT (dialog), "entry_a", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_10);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_ENGINE_STRING);
g_object_set_data (G_OBJECT (dialog), "entry_b", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_10);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_ENGINE_STRING);
g_object_set_data (G_OBJECT (dialog), "entry_c", entry);
gtk_widget_show (entry);

label = gtk_label_new ("A");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
gtk_widget_show (label);

label = gtk_label_new ("B");
gtk_table_attach (GTK_TABLE (table), label, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
gtk_widget_show (label);

label = gtk_label_new ("C");
gtk_table_attach (GTK_TABLE (table), label, 2, 3, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
gtk_widget_show (label);

/****************
 * Packing page *
 ****************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new ("Packing");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 5);
g_object_set_data (G_OBJECT (dialog), "vbox_packing", vbox_page);
gtk_widget_show (vbox_page);

frame = gtk_frame_new (NULL);
gtk_box_pack_start (GTK_BOX (vbox_page), frame, TRUE, TRUE, 0);
gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
gtk_widget_show (frame);

vbox_frame = gtk_vbox_new (TRUE, 0);
gtk_container_add (GTK_CONTAINER (frame), vbox_frame);
gtk_container_set_border_width (GTK_CONTAINER (vbox_frame), 5);
gtk_widget_show (vbox_frame);

vbox_top = gtk_vbox_new (FALSE, 15);
gtk_box_pack_start (GTK_BOX (vbox_frame), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

/******************
 * Radius, Number *
 ******************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

label = gtk_label_new ("Priority");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
gtk_widget_show (label);

/*******
 * row *
 *******/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Atom");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 10);
gtk_widget_show (label);

button = gtk_radio_button_new_with_label (NULL, "Radius");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_signal_connect (GTK_OBJECT (button), "clicked",
G_CALLBACK (static_number), window);
g_object_set_data (G_OBJECT (dialog), "button_radius", button);
gtk_widget_show (button);

button = gtk_radio_button_new_with_label (
gtk_radio_button_get_group (GTK_RADIO_BUTTON (button)), "Number");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_signal_connect (GTK_OBJECT (button), "clicked",
G_CALLBACK (static_number), window);
g_object_set_data (G_OBJECT (dialog), "button_number", button);
gtk_widget_show (button);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_number", entry);
gtk_widget_show (entry);

/********************
 * Relaxation, Seed *
 ********************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

/**********
 * column *
 **********/

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

label = gtk_label_new ("Relaxation");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
gtk_widget_show (label);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Time");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_time", entry);
gtk_widget_show (entry);

/**********
 * column *
 **********/

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

label = gtk_label_new ("Random");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
gtk_widget_show (label);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Seed");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_seed", entry);
gtk_widget_show (entry);

/*********************
 * Ok/Cancel buttons *
 *********************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_dialog), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

button = gamgi_gtk_dialog_button_create ("Ok", NULL);
gtk_box_pack_start (GTK_BOX (hbox_center), button, FALSE, FALSE, 0);
gtk_widget_set_size_request (button, GAMGI_GTK_BUTTON_WIDTH, -1);
GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
g_signal_connect (GTK_OBJECT (button), "clicked",
G_CALLBACK (static_ok), window);
gtk_widget_show (button);

button = gamgi_gtk_dialog_button_create ("Cancel", "red");
gtk_box_pack_start (GTK_BOX (hbox_center), button, FALSE, FALSE, 0);
gtk_widget_set_size_request (button, GAMGI_GTK_BUTTON_WIDTH, -1);
GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
gtk_widget_grab_focus (button);
g_signal_connect (GTK_OBJECT (button), "clicked",
G_CALLBACK (gamgi_gtk_dialog_task0_remove), window);
gtk_widget_show (button);

static_init (window);
gtk_widget_show (dialog);
}
