/*****************************************
 *
 * $GAMGI/src/expat/gamgi_expat_export.c
 *
 * Copyright (C) 2004 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

#include "gamgi_engine.h"
#include "gamgi_gtk.h"
#include "gamgi_io.h"
#include "gamgi_expat.h"
#include "gamgi_global.h"

#include "gamgi_engine_list.h"
#include "gamgi_engine_find.h"
#include "gamgi_gtk_dialog.h"
#include "gamgi_math_sort.h"
#include "gamgi_chem_bond.h"
#include "gamgi_chem_atom.h"
#include "gamgi_expat_export_config.h"
#include "gamgi_expat_export_object.h"
#include "gamgi_io_file.h"
#include "gamgi_io_x3d.h"

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_space (FILE *fp, int space, gamgi_bool *error)
{
while (space-- > 0) gamgi_io_file_fprintf (fp, " ", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static int static_end (FILE *fp, char *string, gamgi_bool *error)
{
static_space (fp, GAMGI_EXPAT_END, error);
gamgi_io_file_fprintf (fp, string, error);

return GAMGI_EXPAT_END;
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static int static_middle (FILE *fp, char *string, gamgi_bool *error)
{
static_space (fp, GAMGI_EXPAT_MIDDLE, error);
gamgi_io_file_fprintf (fp, string, error);

return GAMGI_EXPAT_MIDDLE;
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static int static_start (FILE *fp, char *string, int depth, gamgi_bool *error)
{
gamgi_io_file_fprintf (fp, "\n", error);
depth = gamgi_math_sort_min (depth, GAMGI_EXPAT_DEPTH_MAX);
static_space (fp, depth * GAMGI_EXPAT_START, error);
gamgi_io_file_fprintf (fp, string, error);

return depth * GAMGI_EXPAT_START;
}

/***************** external function *********************
 *                                                       *
 *********************************************************/

void gamgi_expat_export_attribute (FILE *fp,
char *string, int depth, int *column, gamgi_bool *error)
{
int length = strlen (string);

if (*column + length < GAMGI_EXPAT_LINE - GAMGI_EXPAT_END - 2)
  *column += length + static_middle (fp, string, error);
else
  *column = length + static_start (fp, string, depth, error);
}

/***************** external function *********************
 *                                                       *
 *********************************************************/

void gamgi_expat_export_element (FILE *fp, char *element,
char *string, int depth_parent, int *depth_last, int *column,
gamgi_bool *output, gamgi_bool *error)
{
int depth;

if (*output == FALSE)
  {
  *output = TRUE;
  depth = depth_parent + 1;
  if (depth > *depth_last) gamgi_io_file_fprintf (fp, ">", error);
  if (depth - *depth_last == 2) *column = strlen ("<gamgi>") + 
  static_start (fp, "<gamgi>", depth - 1, error);
  *column = strlen (element) + static_start (fp, element, depth, error);
  *depth_last = depth;
  }

gamgi_expat_export_attribute (fp, string, *depth_last, column, error);
}

void gamgi_expat_export_bond (gamgi_window *window, FILE *fp, 
int depth_parent, int *depth_last, gamgi_bool *error, 
int element1, int element2, double min, double max)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_bond_local (window, fp, depth_parent,
depth_last, &output, error, element1, element2, min, max);

if (output == TRUE) static_end (fp, "/>", error);
}

void gamgi_expat_export_atom (gamgi_window *window, FILE *fp,
int depth_parent, int *depth_last, gamgi_bool *error, int element,
double mass, double radius, float red, float green, float blue)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_atom_local (window, fp, depth_parent,
depth_last, &output, error, element, mass, radius, red, green, blue);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_help (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_help (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_text (gamgi_window *window, 
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_text (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_orbital (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_orbital (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_bond (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_bond_global (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_atom (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_atom_global (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_direction (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_direction (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_plane (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_plane (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_group (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_group (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_molecule (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_molecule (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_cluster (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_cluster (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_cell (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_cell (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_arrow (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_arrow (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_shape (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_shape (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_graph (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_graph (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_assembly (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_assembly (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_light (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_light (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_layer (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_layer (window, fp,
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_window (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;

gamgi_expat_export_config_window (window, fp, 
depth_parent, depth_last, &output, error);

if (output == TRUE) static_end (fp, "/>", error);
}

static void static_config_gamgi (gamgi_window *window,
FILE *fp, int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_bool output = FALSE;
int depth;

gamgi_expat_export_config_gamgi (window, fp,
depth_parent, depth_last, &output, error);

depth = depth_parent + 1;

static_config_window (window, fp, depth, depth_last, error);
static_config_layer (window, fp, depth, depth_last, error);
static_config_light (window, fp, depth, depth_last, error);
static_config_assembly (window, fp, depth, depth_last, error);
static_config_graph (window, fp, depth, depth_last, error);
static_config_shape (window, fp, depth, depth_last, error);
static_config_arrow (window, fp, depth, depth_last, error);
static_config_cell (window, fp, depth, depth_last, error);
static_config_cluster (window, fp, depth, depth_last, error);
static_config_molecule (window, fp, depth, depth_last, error);
static_config_group (window, fp, depth, depth_last, error);
static_config_plane (window, fp, depth, depth_last, error);
static_config_direction (window, fp, depth, depth_last, error);

static_config_atom (window, fp, depth, depth_last, error);
gamgi_chem_atom_export (window, fp, depth, depth_last, error);

static_config_bond (window, fp, depth, depth_last, error);
gamgi_chem_bond_export (window, fp, depth, depth_last, error);

static_config_orbital (window, fp, depth, depth_last, error);
static_config_text (window, fp, depth, depth_last, error);
static_config_help (window, fp, depth, depth_last, error);

if (*depth_last == depth)
  static_end (fp, "/>", error);
else if (*depth_last == depth + 1)
  static_start (fp, "</gamgi>", depth, error);
}

static void static_config (gamgi_window *window, FILE *fp, 
int depth_parent, int *depth_last, gamgi_bool *error)
{
static_config_gamgi (window, fp, depth_parent, depth_last, error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_text (gamgi_text *text, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<text") + static_start (fp, "<text", depth, error);
*depth_last = depth;

gamgi_expat_export_object_text (text, fp, depth, &column, id, error);

/*******************
 * write text list *
 *******************/

dlist = text->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</text>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_orbital (gamgi_orbital *orbital, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<orbital") + static_start (fp, "<orbital", depth, error);
*depth_last = depth;

gamgi_expat_export_object_orbital (orbital, fp, depth, &column, id, error);

/*******************
 * write text list *
 *******************/

dlist = orbital->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</orbital>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_bond (gamgi_bond *bond, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<bond") + static_start (fp, "<bond", depth, error);
*depth_last = depth;

gamgi_expat_export_object_bond (bond, fp, depth, &column, id, error);

/***************************************
 * clean object and dlist addresses    *
 * that were used to keep the atom ids *
 ***************************************/

bond->object.object = NULL;
bond->object.dlist = NULL;

/******************
 * child elements *
 ******************/

dlist = bond->orbital_start;
while (dlist != NULL)
  { static_object_orbital (GAMGI_CAST_ORBITAL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

dlist = bond->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</bond>", depth, error);
else
  static_end (fp, "/>", error);
}

static void static_bond_add (gamgi_object *parent, gamgi_bond *bond)
{
gamgi_direction *direction;
gamgi_plane *plane;
gamgi_group *group;
gamgi_molecule *molecule;
gamgi_cluster *cluster;
gamgi_cell *cell;
gamgi_assembly *assembly;
gamgi_layer *layer;
gamgi_dlist *dlist = NULL;

switch (parent->class)
  {
  case GAMGI_ENGINE_DIRECTION:
  direction = GAMGI_CAST_DIRECTION parent;
  dlist = gamgi_engine_dlist_add_end (direction->bond_end);
  direction->bond_end = dlist;
  if (dlist->before == NULL) direction->bond_start = dlist;
  break;

  case GAMGI_ENGINE_PLANE:
  plane = GAMGI_CAST_PLANE parent;
  dlist = gamgi_engine_dlist_add_end (plane->bond_end);
  plane->bond_end = dlist;
  if (dlist->before == NULL) plane->bond_start = dlist;
  break;

  case GAMGI_ENGINE_GROUP:
  group = GAMGI_CAST_GROUP parent;
  dlist = gamgi_engine_dlist_add_end (group->bond_end);
  group->bond_end = dlist;
  if (dlist->before == NULL) group->bond_start = dlist;
  break;

  case GAMGI_ENGINE_MOLECULE:
  molecule = GAMGI_CAST_MOLECULE parent;
  dlist = gamgi_engine_dlist_add_end (molecule->bond_end);
  molecule->bond_end = dlist;
  if (dlist->before == NULL) molecule->bond_start = dlist;
  break;

  case GAMGI_ENGINE_CLUSTER:
  cluster = GAMGI_CAST_CLUSTER parent;
  dlist = gamgi_engine_dlist_add_end (cluster->bond_end);
  cluster->bond_end = dlist;
  if (dlist->before == NULL) cluster->bond_start = dlist;
  break;

  case GAMGI_ENGINE_CELL:
  cell = GAMGI_CAST_CELL parent;
  dlist = gamgi_engine_dlist_add_end (cell->bond_end);
  cell->bond_end = dlist;
  if (dlist->before == NULL) cell->bond_start = dlist;
  break;

  case GAMGI_ENGINE_ASSEMBLY:
  assembly = GAMGI_CAST_ASSEMBLY parent;
  dlist = gamgi_engine_dlist_add_end (assembly->bond_end);
  assembly->bond_end = dlist;
  if (dlist->before == NULL) assembly->bond_start = dlist;
  break;

  case GAMGI_ENGINE_LAYER:
  layer = GAMGI_CAST_LAYER parent;
  dlist = gamgi_engine_dlist_add_end (layer->bond_end);
  layer->bond_end = dlist;
  if (dlist->before == NULL) layer->bond_start = dlist;
  break;

  /************************
   * Control should never *
   *  reach this point    *
   ************************/

  default:
  break;
  }

dlist->data = bond;
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_bond_parent (gamgi_bond *bond, 
gamgi_window *window, int *id, gamgi_bool *error)
{
gamgi_object *parent;

if (bond->first == TRUE)
  {
  parent = gamgi_engine_find_root (GAMGI_CAST_OBJECT bond->atom1,
  GAMGI_CAST_OBJECT bond->atom2, window->focus);
  if (parent != NULL)
    {
    /**********************************************
     * If the parent is out of range, the second  *
     * atom will not appear and this bond can be  *
     * safely discarded without further action.   *
     *                                            *
     *  If a common parent is found, then the     *
     *  second atom must necessarily show up:     *
     *  we don't need to care about half bonds.   *
     *                                            *
     * atom id numbers must be saved in a place   *
     * that the bond can reach, in order to allow *
     * the bond later on to identify its parent   *
     * atoms, when the bond is actually exported  *
     * (at the end of the common parent block)    *
     *                                            *
     *  the first and second atom ids are saved   *
     *  in the unused bond->object.object and     *
     *  bond->object.dlist addresses.             *
     *                                            *
     * atom->mark is used to flag when the atom   *
     * id identifiers must be written. If atoms   *
     * A,B form a bond between atoms B,A, atom A  *
     * is scanned first but marked last, so all   *
     * atoms must be marked in the first scan,    *
     * before their <atom> blocks are exported.   *
     **********************************************/

    bond->atom1->mark = TRUE;
    bond->atom2->mark = TRUE;
    static_bond_add (parent, bond);
    bond->object.object = GAMGI_INT_TO_POINTER (*id);
    bond->first = FALSE;
    }
  }
else
  {
  bond->object.dlist = GAMGI_INT_TO_POINTER (*id);
  bond->first = TRUE;
  }

}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_atom (gamgi_atom *atom, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<atom") + static_start (fp, "<atom", depth, error);
*depth_last = depth;

/*******************************************
 * bonds are written at the bottom of the  *
 * first block that contains the two atoms *
 *                                         *
 * bonds must be checked before printing   *
 * atom data, to see if ids must be added  *
 *******************************************/

dlist = atom->bond_start;
while (dlist != NULL)
  { static_bond_parent (GAMGI_CAST_BOND dlist->data, window, id, error);
    dlist = dlist->next; }

gamgi_expat_export_object_atom (atom, fp, depth, &column, id, error);

/******************
 * child elements *
 ******************/

dlist = atom->orbital_start;
while (dlist != NULL)
  { static_object_orbital (GAMGI_CAST_ORBITAL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

dlist = atom->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</atom>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_direction (gamgi_direction *direction, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<direction") + static_start (fp, "<direction", depth, error);
*depth_last = depth;

gamgi_expat_export_object_direction (direction, fp, depth, &column, id, error);

/*******************
 * write atom list *
 *******************/

dlist = direction->atom_start;
while (dlist != NULL)
  { static_object_atom (GAMGI_CAST_ATOM dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write bond list *
 *******************/

dlist = direction->bond_start;
while (dlist != NULL)
  { static_object_bond (GAMGI_CAST_BOND dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = gamgi_engine_dlist_remove_start (dlist); }

direction->bond_start = NULL;
direction->bond_end = NULL;

/**********************
 * write orbital list *
 **********************/

dlist = direction->orbital_start;
while (dlist != NULL)
  { static_object_orbital (GAMGI_CAST_ORBITAL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write text list *
 *******************/

dlist = direction->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</direction>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_plane (gamgi_plane *plane, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<plane") + static_start (fp, "<plane", depth, error);
*depth_last = depth;

gamgi_expat_export_object_plane (plane, fp, depth, &column, id, error);

/************************
 * write direction list *
 ************************/

dlist = plane->direction_start;
while (dlist != NULL)
  { static_object_direction (GAMGI_CAST_DIRECTION dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write atom list *
 *******************/

dlist = plane->atom_start;
while (dlist != NULL)
  { static_object_atom (GAMGI_CAST_ATOM dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write bond list *
 *******************/

dlist = plane->bond_start;
while (dlist != NULL)
  { static_object_bond (GAMGI_CAST_BOND dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = gamgi_engine_dlist_remove_start (dlist); }

plane->bond_start = NULL;
plane->bond_end = NULL;

/**********************
 * write orbital list *
 **********************/

dlist = plane->orbital_start;
while (dlist != NULL)
  { static_object_orbital (GAMGI_CAST_ORBITAL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write text list *
 *******************/

dlist = plane->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</plane>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_group (gamgi_group *group, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<group") + static_start (fp, "<group", depth, error);
*depth_last = depth;

gamgi_expat_export_object_group (group, fp, depth, &column, id, error);

/********************
 * write group list *
 ********************/

dlist = group->group_start;
while (dlist != NULL)
  { static_object_group (GAMGI_CAST_GROUP dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write plane list *
 ********************/

dlist = group->plane_start;
while (dlist != NULL)
  { static_object_plane (GAMGI_CAST_PLANE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/************************
 * write direction list *
 ************************/

dlist = group->direction_start;
while (dlist != NULL)
  { static_object_direction (GAMGI_CAST_DIRECTION dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write atom list *
 *******************/

dlist = group->atom_start;
while (dlist != NULL)
  { static_object_atom (GAMGI_CAST_ATOM dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write bond list *
 *******************/

dlist = group->bond_start;
while (dlist != NULL)
  { static_object_bond (GAMGI_CAST_BOND dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = gamgi_engine_dlist_remove_start (dlist); }

group->bond_start = NULL;
group->bond_end = NULL;

/**********************
 * write orbital list *
 **********************/

dlist = group->orbital_start;
while (dlist != NULL)
  { static_object_orbital (GAMGI_CAST_ORBITAL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write text list *
 *******************/

dlist = group->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</group>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_molecule (gamgi_molecule *molecule, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<molecule") + static_start (fp, "<molecule", depth, error);
*depth_last = depth;

gamgi_expat_export_object_molecule (molecule, fp, depth, &column, id, error);

/********************
 * write group list *
 ********************/

dlist = molecule->group_start;
while (dlist != NULL)
  { static_object_group (GAMGI_CAST_GROUP dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write plane list *
 ********************/

dlist = molecule->plane_start;
while (dlist != NULL)
  { static_object_plane (GAMGI_CAST_PLANE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/************************
 * write direction list *
 ************************/

dlist = molecule->direction_start;
while (dlist != NULL)
  { static_object_direction (GAMGI_CAST_DIRECTION dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write atom list *
 *******************/

dlist = molecule->atom_start;
while (dlist != NULL)
  { static_object_atom (GAMGI_CAST_ATOM dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write bond list *
 *******************/

dlist = molecule->bond_start;
while (dlist != NULL)
  { static_object_bond (GAMGI_CAST_BOND dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = gamgi_engine_dlist_remove_start (dlist); }

molecule->bond_start = NULL;
molecule->bond_end = NULL;

/**********************
 * write orbital list *
 **********************/

dlist = molecule->orbital_start;
while (dlist != NULL)
  { static_object_orbital (GAMGI_CAST_ORBITAL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write text list *
 *******************/

dlist = molecule->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</molecule>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_cluster (gamgi_cluster *cluster, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<cluster") + static_start (fp, "<cluster", depth, error);
*depth_last = depth;

gamgi_expat_export_object_cluster (cluster, fp, depth, &column, id, error);

/**********************
 * write cluster list *
 **********************/

dlist = cluster->cluster_start;
while (dlist != NULL)
  { static_object_cluster (GAMGI_CAST_CLUSTER dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/***********************
 * write molecule list *
 ***********************/

dlist = cluster->molecule_start;
while (dlist != NULL)
  { static_object_molecule (GAMGI_CAST_MOLECULE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write group list *
 ********************/

dlist = cluster->group_start;
while (dlist != NULL)
  { static_object_group (GAMGI_CAST_GROUP dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write plane list *
 ********************/

dlist = cluster->plane_start;
while (dlist != NULL)
  { static_object_plane (GAMGI_CAST_PLANE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/************************
 * write direction list *
 ************************/

dlist = cluster->direction_start;
while (dlist != NULL)
  { static_object_direction (GAMGI_CAST_DIRECTION dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write atom list *
 *******************/

dlist = cluster->atom_start;
while (dlist != NULL)
  { static_object_atom (GAMGI_CAST_ATOM dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write bond list *
 *******************/

dlist = cluster->bond_start;
while (dlist != NULL)
  { static_object_bond (GAMGI_CAST_BOND dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = gamgi_engine_dlist_remove_start (dlist); }

cluster->bond_start = NULL;
cluster->bond_end = NULL;

/**********************
 * write orbital list *
 **********************/

dlist = cluster->orbital_start;
while (dlist != NULL)
  { static_object_orbital (GAMGI_CAST_ORBITAL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write text list *
 *******************/

dlist = cluster->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</cluster>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_cell (gamgi_cell *cell, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<cell") + static_start (fp, "<cell", depth, error);
*depth_last = depth;

gamgi_expat_export_object_cell (cell, fp, depth, &column, id, error);

/**********************
 * write cluster list *
 **********************/

dlist = cell->cluster_start;
while (dlist != NULL)
  { static_object_cluster (GAMGI_CAST_CLUSTER dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/***********************
 * write molecule list *
 ***********************/

dlist = cell->molecule_start;
while (dlist != NULL)
  { static_object_molecule (GAMGI_CAST_MOLECULE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write group list *
 ********************/

dlist = cell->group_start;
while (dlist != NULL)
  { static_object_group (GAMGI_CAST_GROUP dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write plane list *
 ********************/

dlist = cell->plane_start;
while (dlist != NULL)
  { static_object_plane (GAMGI_CAST_PLANE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/************************
 * write direction list *
 ************************/

dlist = cell->direction_start;
while (dlist != NULL)
  { static_object_direction (GAMGI_CAST_DIRECTION dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write atom list *
 *******************/

dlist = cell->atom_start;
while (dlist != NULL)
  { static_object_atom (GAMGI_CAST_ATOM dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write bond list *
 *******************/

dlist = cell->bond_start;
while (dlist != NULL)
  { static_object_bond (GAMGI_CAST_BOND dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = gamgi_engine_dlist_remove_start (dlist); }

cell->bond_start = NULL;
cell->bond_end = NULL;

/**********************
 * write orbital list *
 **********************/

dlist = cell->orbital_start;
while (dlist != NULL)
  { static_object_orbital (GAMGI_CAST_ORBITAL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write text list *
 *******************/

dlist = cell->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</cell>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_arrow (gamgi_arrow *arrow, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<arrow") + static_start (fp, "<arrow", depth, error);
*depth_last = depth;

gamgi_expat_export_object_arrow (arrow, fp, depth, &column, id, error);

/*******************
 * write text list *
 *******************/

dlist = arrow->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</arrow>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_shape (gamgi_shape *shape, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<shape") + static_start (fp, "<shape", depth, error);
*depth_last = depth;

gamgi_expat_export_object_shape (shape, fp, depth, &column, id, error);

/*******************
 * write text list *
 *******************/

dlist = shape->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</shape>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_graph (gamgi_graph *graph, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<graph") + static_start (fp, "<graph", depth, error);
*depth_last = depth;

gamgi_expat_export_object_graph (graph, fp, depth, &column, id, error);

/*******************
 * write text list *
 *******************/

dlist = graph->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</graph>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_assembly (gamgi_assembly *assembly, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<assembly") + static_start (fp, "<assembly", depth, error);
*depth_last = depth;

gamgi_expat_export_object_assembly (assembly, fp, depth, &column, id, error);

/***********************
 * write assembly list *
 ***********************/

dlist = assembly->assembly_start;
while (dlist != NULL)
  { static_object_assembly (GAMGI_CAST_ASSEMBLY dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write graph list *
 ********************/

dlist = assembly->graph_start;
while (dlist != NULL)
  { static_object_graph (GAMGI_CAST_GRAPH dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write shape list *
 ********************/

dlist = assembly->shape_start;
while (dlist != NULL)
  { static_object_shape (GAMGI_CAST_SHAPE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write arrow list *
 ********************/

dlist = assembly->arrow_start;
while (dlist != NULL)
  { static_object_arrow (GAMGI_CAST_ARROW dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write cell list *
 *******************/

dlist = assembly->cell_start;
while (dlist != NULL)
  { static_object_cell (GAMGI_CAST_CELL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/**********************
 * write cluster list *
 **********************/

dlist = assembly->cluster_start;
while (dlist != NULL)
  { static_object_cluster (GAMGI_CAST_CLUSTER dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/***********************
 * write molecule list *
 ***********************/

dlist = assembly->molecule_start;
while (dlist != NULL)
  { static_object_molecule (GAMGI_CAST_MOLECULE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write group list *
 ********************/

dlist = assembly->group_start;
while (dlist != NULL)
  { static_object_group (GAMGI_CAST_GROUP dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write plane list *
 ********************/

dlist = assembly->plane_start;
while (dlist != NULL)
  { static_object_plane (GAMGI_CAST_PLANE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/************************
 * write direction list *
 ************************/

dlist = assembly->direction_start;
while (dlist != NULL)
  { static_object_direction (GAMGI_CAST_DIRECTION dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write atom list *
 *******************/

dlist = assembly->atom_start;
while (dlist != NULL)
  { static_object_atom (GAMGI_CAST_ATOM dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write bond list *
 *******************/

dlist = assembly->bond_start;
while (dlist != NULL)
  { static_object_bond (GAMGI_CAST_BOND dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = gamgi_engine_dlist_remove_start (dlist); }

assembly->bond_start = NULL;
assembly->bond_end = NULL;

/**********************
 * write orbital list *
 **********************/

dlist = assembly->orbital_start;
while (dlist != NULL)
  { static_object_orbital (GAMGI_CAST_ORBITAL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write text list *
 *******************/

dlist = assembly->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</assembly>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_light (gamgi_light *light, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<light") + static_start (fp, "<light", depth, error);
*depth_last = depth;

gamgi_expat_export_object_light (light, fp, depth, &column, id, error);

if (depth < *depth_last)
  static_start (fp, "</light>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_layer (gamgi_layer *layer, 
gamgi_window *window, FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<layer") + static_start (fp, "<layer", depth, error);
*depth_last = depth;

gamgi_expat_export_object_layer (layer, fp, depth, &column, id, error);

/********************
 * write light list *
 ********************/

dlist = layer->light_start;
while (dlist != NULL)
  { static_object_light (GAMGI_CAST_LIGHT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/***********************
 * write assembly list *
 ***********************/

dlist = layer->assembly_start;
while (dlist != NULL)
  { static_object_assembly (GAMGI_CAST_ASSEMBLY dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write graph list *
 ********************/

dlist = layer->graph_start;
while (dlist != NULL)
  { static_object_graph (GAMGI_CAST_GRAPH dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write shape list *
 ********************/

dlist = layer->shape_start;
while (dlist != NULL)
  { static_object_shape (GAMGI_CAST_SHAPE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write arrow list *
 ********************/

dlist = layer->arrow_start;
while (dlist != NULL)
  { static_object_arrow (GAMGI_CAST_ARROW dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write cell list *
 *******************/

dlist = layer->cell_start;
while (dlist != NULL)
  { static_object_cell (GAMGI_CAST_CELL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/**********************
 * write cluster list *
 **********************/

dlist = layer->cluster_start;
while (dlist != NULL)
  { static_object_cluster (GAMGI_CAST_CLUSTER dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/***********************
 * write molecule list *
 ***********************/

dlist = layer->molecule_start;
while (dlist != NULL)
  { static_object_molecule (GAMGI_CAST_MOLECULE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write group list *
 ********************/

dlist = layer->group_start;
while (dlist != NULL)
  { static_object_group (GAMGI_CAST_GROUP dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/********************
 * write plane list *
 ********************/

dlist = layer->plane_start;
while (dlist != NULL)
  { static_object_plane (GAMGI_CAST_PLANE dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/************************
 * write direction list *
 ************************/

dlist = layer->direction_start;
while (dlist != NULL)
  { static_object_direction (GAMGI_CAST_DIRECTION dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write atom list *
 *******************/

dlist = layer->atom_start;
while (dlist != NULL)
  { static_object_atom (GAMGI_CAST_ATOM dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write bond list *
 *******************/

dlist = layer->bond_start;
while (dlist != NULL)
  { static_object_bond (GAMGI_CAST_BOND dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = gamgi_engine_dlist_remove_start (dlist); }

layer->bond_start = NULL;
layer->bond_end = NULL;

/**********************
 * write orbital list *
 **********************/

dlist = layer->orbital_start;
while (dlist != NULL)
  { static_object_orbital (GAMGI_CAST_ORBITAL dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

/*******************
 * write text list *
 *******************/

dlist = layer->text_start;
while (dlist != NULL)
  { static_object_text (GAMGI_CAST_TEXT dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</layer>", depth, error);
else
  static_end (fp, "/>", error);
}

/***************** internal function *********************
 *                                                       *
 *********************************************************/

static void static_object_window (gamgi_window *window, FILE *fp, 
int depth_parent, int *depth_last, int *id, gamgi_bool *error)
{
gamgi_dlist *dlist;
int column, depth;

depth = depth_parent + 1;
if (depth > *depth_last)
  gamgi_io_file_fprintf (fp, ">", error);
column = strlen ("<window") + static_start (fp, "<window", depth, error);
*depth_last = depth;

gamgi_expat_export_object_window (window, fp, depth, &column, id, error);

/********************
 * write layer list *
 ********************/

dlist = window->layer_start;
while (dlist != NULL)
  { static_object_layer (GAMGI_CAST_LAYER dlist->data, 
    window, fp, depth, depth_last, id, error);
    dlist = dlist->next; }

if (depth < *depth_last)
  static_start (fp, "</window>", depth, error);
else
  static_end (fp, "/>", error);
}

static void static_object_gamgi (FILE *fp, int depth_parent, 
int *depth_last, int *id, gamgi_bool *error)
{
gamgi_window *window;

/*********************
 * write window list *
 *********************/

window = gamgi->window_start;
while (window != NULL)
  { static_object_window (window, fp, depth_parent, depth_last, id, error);
    window = window->next; }
}

static void static_object (gamgi_window *window, FILE *fp, 
int depth_parent, int *depth_last, gamgi_bool *error)
{
gamgi_object *object = window->focus;
int id = 1;

switch (object->class)
  {
  case GAMGI_ENGINE_TEXT:
  static_object_text (GAMGI_CAST_TEXT object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_ORBITAL:
  static_object_orbital (GAMGI_CAST_ORBITAL object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_ATOM:
  static_object_atom (GAMGI_CAST_ATOM object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_DIRECTION:
  static_object_direction (GAMGI_CAST_DIRECTION object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_PLANE:
  static_object_plane (GAMGI_CAST_PLANE object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_GROUP:
  static_object_group (GAMGI_CAST_GROUP object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_MOLECULE:
  static_object_molecule (GAMGI_CAST_MOLECULE object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_CLUSTER:
  static_object_cluster (GAMGI_CAST_CLUSTER object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_CELL:
  static_object_cell (GAMGI_CAST_CELL object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_ARROW:
  static_object_arrow (GAMGI_CAST_ARROW object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_SHAPE:
  static_object_shape (GAMGI_CAST_SHAPE object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_GRAPH:
  static_object_graph (GAMGI_CAST_GRAPH object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_ASSEMBLY:
  static_object_assembly (GAMGI_CAST_ASSEMBLY object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_LIGHT:
  static_object_light (GAMGI_CAST_LIGHT object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_LAYER:
  static_object_layer (GAMGI_CAST_LAYER object, 
  window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_WINDOW:
  static_object_window (window, fp, depth_parent, depth_last, &id, error);
  break;

  case GAMGI_ENGINE_GAMGI:
  static_object_gamgi (fp, depth_parent, depth_last, &id, error);
  break;
  }
}

static void static_header_xml (FILE *fp, gamgi_bool *error)
{
char string[GAMGI_ENGINE_LINE];
int column;

/********************************
 * first row: XML common header *
 ********************************/

gamgi_io_file_fprintf (fp, "<?xml", error);
column = strlen ("<?xml");

sprintf (string, "version=\"%s\"", GAMGI_EXPAT_VERSION);
gamgi_expat_export_attribute (fp, string, 0, &column, error);

sprintf (string, "encoding=\"%s\"", GAMGI_EXPAT_ENCODING);
gamgi_expat_export_attribute (fp, string, 0, &column, error);

sprintf (string, "standalone=\"%s\"", GAMGI_EXPAT_STANDALONE);
gamgi_expat_export_attribute (fp, string, 0, &column, error);

static_end (fp, "?>\n", error);
}

static void static_header_gml (FILE *fp, gamgi_bool *error)
{
char string[GAMGI_ENGINE_LINE];
int column;

/*************
 * first row *
 *************/

static_header_xml (fp, error);

/**************
 * second row *
 **************/

gamgi_io_file_fprintf (fp, "<!DOCTYPE", error);
column = strlen ("<!DOCTYPE");

gamgi_expat_export_attribute (fp, "gml", 0, &column, error);

gamgi_expat_export_attribute (fp, "SYSTEM", 0, &column, error);

sprintf (string, "\"%s\"", GAMGI_IO_GML_SYSTEM);
gamgi_expat_export_attribute (fp, string, 0, &column, error);

static_end (fp, ">\n", error);
}

static void static_header_x3d (FILE *fp, gamgi_bool *error)
{
char string[GAMGI_ENGINE_LINE];
int column;

/*************
 * first row *
 *************/

static_header_xml (fp, error);

/**************
 * second row *
 **************/

gamgi_io_file_fprintf (fp, "<!DOCTYPE", error);
column = strlen ("<!DOCTYPE");

gamgi_expat_export_attribute (fp, "X3D", 0, &column, error);

gamgi_expat_export_attribute (fp, "PUBLIC", 0, &column, error);

sprintf (string, "\"%s\"", GAMGI_IO_X3D_PUBLIC);
gamgi_expat_export_attribute (fp, string, 0, &column, error);

sprintf (string, "\"%s\"", GAMGI_IO_X3D_SYSTEM);
gamgi_expat_export_attribute (fp, string, 0, &column, error);

static_end (fp, ">\n", error);
}

int gamgi_expat_export_length (gamgi_text *text)
{
char *s;
int i;

/***********************************************
 * get buffer size (add 1 to count final '\0') *
 ***********************************************/

i = 0;
for (s = text->string; *s != '\0'; s++)
  {
  i++;
  if (*s == '<' || *s == '>') i += 3;
  else if (*s == '&') i += 4;
  else if (*s == '\'' || *s == '\"') i += 5;
  }

return i;
}

void gamgi_expat_export_string (gamgi_text *text, char *buffer)
{
char *s;
int i;

/******************************
 * translate string to buffer *
 ******************************/

i = 0;
for (s = text->string; *s != '\0'; s++)
  {
  switch (*s)
    {
    case '<':
    buffer[i++] = '&'; buffer[i++] = 'l';
    buffer[i++] = 't'; buffer[i++] = ';';
    break;

    case '>':
    buffer[i++] = '&'; buffer[i++] = 'g';
    buffer[i++] = 't'; buffer[i++] = ';';
    break;

    case '&':
    buffer[i++] = '&'; buffer[i++] = 'a';
    buffer[i++] = 'm'; buffer[i++] = 'p';
    buffer[i++] = ';';
    break;

    case '\'':
    buffer[i++] = '&'; buffer[i++] = 'a';
    buffer[i++] = 'p'; buffer[i++] = 'o';
    buffer[i++] = 's'; buffer[i++] = ';';
    break;

    case '\"':
    buffer[i++] = '&'; buffer[i++] = 'q';
    buffer[i++] = 'u'; buffer[i++] = 'o';
    buffer[i++] = 't'; buffer[i++] = ';';
    break;

    default:
    buffer[i++] = *s;
    break;
    }
  }
buffer[i] = '\0';
}

void gamgi_expat_export_gml (FILE *fp, gamgi_window *window, gamgi_bool *error)
{
int depth = GAMGI_EXPAT_DEPTH_MIN;
int depth_last = GAMGI_EXPAT_DEPTH_MIN;

/********************
 * write gml header *
 ********************/

if (GAMGI_EXPAT_HEADER_GML == TRUE) static_header_gml (fp, error);

/***********************
 * start gml root node *
 ***********************/

static_space (fp, depth * GAMGI_EXPAT_START, error);
gamgi_io_file_fprintf (fp, "<gml", error);

/************************************
 * write config BEFORE object data! *
 ************************************/

if (GAMGI_EXPAT_CONFIG == TRUE) 
  static_config (window, fp, depth, &depth_last, error);
if (GAMGI_EXPAT_OBJECT == TRUE)
  static_object (window, fp, depth, &depth_last, error);

/*********************
 * end gml root node *
 *********************/

if (depth_last == depth) gamgi_io_file_fprintf (fp, "/>", error);
else static_start (fp, "</gml>", depth, error);
gamgi_io_file_fprintf (fp, "\n", error);
}

void gamgi_expat_export_x3d (FILE *fp, gamgi_window *window, gamgi_bool *error)
{
char string[GAMGI_ENGINE_LINE];
int depth = 0;

/********************
 * write X3D header *
 ********************/

if (GAMGI_EXPAT_HEADER_X3D == TRUE) static_header_x3d (fp, error);

/***********************
 * start X3D root node *
 ***********************/

sprintf (string, "<X3D profile=\"%s\" version=\"%s\">\n", 
GAMGI_IO_X3D_PROFILE, GAMGI_IO_X3D_VERSION);
gamgi_io_file_fprintf (fp, string, error);

/********************************************************
 * export in X3D format everything in the current layer *
 ********************************************************/

gamgi_io_x3d (fp, window, error);

/*********************
 * end X3D root node *
 *********************/

static_start (fp, "</X3D>", depth, error);
gamgi_io_file_fprintf (fp, "\n", error);
}
