/**
 * File:          $RCSfile: 2vector.c,v $
 * Module:        Size 2 vectors (double precision)
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.20 $
 * Last edited:   $Date: 2003/01/31 18:57:10 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <math.h>
#define GAN_KEEP_VECDEFS
#include <gandalf/linalg/2vector.h>
#undef GAN_KEEP_VECDEFS
#include <gandalf/common/misc_defs.h>
#include <gandalf/common/misc_error.h>
#include <string.h>

/**
 * \defgroup LinearAlgebra Linear Algebra Package
 * \{
 */

/**
 * \defgroup FixedSizeMatVec Fixed Size Matrices and Vectors
 * \{
 */

/**
 * \defgroup FixedSizeVector Fixed Size Vectors
 * \{
 */

/* start off other groups */

/**
 * \defgroup FixedSizeVectorIO Fixed Size Vector I/O
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorFill Fill a Fixed Size Vector
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorCopy Copy Fixed Size Vectors
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorScale Multiply/Divide a Fixed Size Vector by a Scalar
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorAdd Add Fixed Size Vectors
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorSubtract Subtract Fixed Size Vectors
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorScalarProduct Scalar Product of Fixed Size Vectors
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorProduct Vector Product of Fixed Size Vectors
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorNorms Norms of Fixed Size Vectors
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorExtract Extract from a Fixed Size Vector
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorBuild Build a Fixed Size Vector
 * \{
 */

/**
 * \}
 */

/**
 * \defgroup FixedSizeVectorConvert Fixed Size Vector Conversion
 * \{
 */

/**
 * \}
 */

/* define all the standard small vector functions */
#include <gandalf/linalg/vectorf_noc.c>

/**
 * \addtogroup FixedSizeVectorIO
 * \{
 */

/**
 * \brief Print 2-vector to file
 *
 * Print 2-vector \a p to file pointer \a fp, with prefix string \a prefix,
 * indentation \a indent and floating-point format \a fmt.
 *
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 */
Gan_Bool
 gan_vec2_fprint ( FILE *fp, Gan_Vector2 *p,
                   const char *prefix, int indent,  const char *fmt )
{
   int i;

   /* print indentation */
   for ( i = indent-1; i >= 0; i-- ) fprintf ( fp, " " );
   fprintf ( fp, "%s: ", prefix ); fprintf ( fp, fmt, p->x );
   fprintf ( fp, " " );            fprintf ( fp, fmt, p->y );
   fprintf ( fp, "\n" );

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Read 2-vector from file
 *
 * Read 2-vector \a p from file pointer \a fp. The prefix string for
 * the vector is read from the file into the \a prefix string, up to the
 * maximum length \a prefix_len of the \a prefix string. Any remaining
 * characters after \a prefix has been filled are ignored. Pass \c NULL for
 * \a prefix and zero for \a prefix_len to ignore the prefix string.
 *
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 */
Gan_Bool
 gan_vec2_fscanf ( FILE *fp, Gan_Vector2 *p, char *prefix, int prefix_len )
{
   int ch, result;

   /* read indentation */
   for(;;)
      if ( (ch = getc(fp)) != ' ' || ch == EOF )
         break;

   /* end of file means corrupted file */
   if ( ch == EOF )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_vec2_fscanf", GAN_ERROR_CORRUPTED_FILE, "" );
      return GAN_FALSE;
   }

   /* rewind one character to reread the prefix string */
   fseek ( fp, -1, SEEK_CUR );

   /* read prefix string, terminated by ":" */
   for ( prefix_len--; prefix_len > 0; prefix_len-- )
   {
      ch = getc(fp);
      if ( ch == EOF || ch == ':' ) break;
      if ( prefix != NULL ) *prefix++ = (char)ch;
   }

   if ( ch != ':' )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_vec2_fscanf", GAN_ERROR_CORRUPTED_FILE, "" );
      return GAN_FALSE;
   }

   /* terminate string */
   if ( prefix != NULL ) *prefix = '\0';

   /* read rest of string if necessary */
   if ( prefix_len == 0 )
   {
      for(;;)
      {
         ch = getc(fp);
         if ( ch == EOF || ch == ':' ) break;
      }

      if ( ch != ':' )
      {
         gan_err_flush_trace();
         gan_err_register ( "gan_vec2_fscanf", GAN_ERROR_CORRUPTED_FILE, "" );
         return GAN_FALSE;
      }
   }

   /* read vector data */
   result = fscanf ( fp, "%lf%lf\n", &p->x, &p->y );
   if ( result != 2 )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_vec2_fscanf", GAN_ERROR_CORRUPTED_FILE, "" );
      return GAN_FALSE;
   }

   /* read end-of-line character */
   result = getc(fp);
   assert ( result == '\n' );

   /* success */
   return GAN_TRUE;
}

/**
 * \}
 */

/**
 * \addtogroup FixedSizeVectorFill
 * \{
 */

/**
 * \brief Return 2-vector filled with values.
 *
 * Fill 2-vector with values:
 * \f[
 *     \left(\begin{array}{c} X \\ Y \end{array}\right)
 * \f]
 *
 * \return filled 2-vector.
 */
Gan_Vector2
 gan_vec2_fill_s ( double X, double Y )
{
   Gan_Vector2 p;

   (void)gan_vec2_fill_q ( &p, X, Y );
   return p;
}

/**
 * \}
 */

/**
 * \addtogroup FixedSizeVectorConvert
 * \{
 */

/**
 * \brief Convert generic vector to 2-vector structure.
 *
 * Convert generic vector \a x to 2-vector structure \a a.
 *
 * \return Pointer to result 2-vector \a a.
 */
Gan_Vector2 *
 gan_vec2_from_vec_q ( Gan_Vector *x, Gan_Vector2 *a )
{
   if ( !gan_vec_read_va ( x, 2, &a->x, &a->y ) )
   {
      gan_err_register ( "gan_vec2_from_vec_q", GAN_ERROR_FAILURE, "" );
      return NULL;
   }

   return a;
}

/**
 * \}
 */

/**
 * \}
 */

/**
 * \}
 */

/**
 * \}
 */
