/* Copyright (C) 2000-2003 Damir Zucic */

/*=============================================================================

				plot.c

Purpose:
	Execute plot command: plot requested functions.  The command plot
	may be followed by a list of keywords. If keyword OFF is present,
	the main window  drawing mode  will be  changed  to default mode.
	Otherwise, for all recognized keywords the corresponding flags in
	RuntimeS structure will be set to one.

Input:
	(1) Pointer to MolComplexS structure.
	(2) The number of macromolecular complexes.
	(3) Pointer to RuntimeS structure.
	(4) Pointer to ConfigS structure.
	(5) Pointer to GUIS structure.
	(6) Pointer to NearestAtomS structure.
	(7) The number of pixels in the main window free area.
	(8) Pointer to refreshI.
	(9) Pointer to the remainder of the command string.  This command
	    may be given with a list of keywords or with keyword OFF.

Output:
	(1) The main window mode changed to 4 (default is zero).
	(2) Return value.

Return value:
	(1) Positive (command) code on success.
	(2) Negative (error) code on failure.

Notes:
	(1) This command reinitializes the NearestAtomS array,  except if
	    at least one of additional keywords is not recognized.

========includes:============================================================*/

#include <stdio.h>

#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "commands.h"
#include "typedefs.h"

/*======function prototypes:=================================================*/

void		InitNearest_ (NearestAtomS *, size_t);
int		ExtractTwoIntegers_ (int *, int *, char *);
size_t		MainRefresh_ (MolComplexS *, int,
			      RuntimeS *, ConfigS *, GUIS *,
			      NearestAtomS *, size_t, unsigned int);
int		ControlRefresh_ (MolComplexS *, ConfigS *, GUIS *);

/*======execute plot command:================================================*/

int Plot_ (MolComplexS *mol_complexSP, int mol_complexesN,
	   RuntimeS *runtimeSP, ConfigS *configSP, GUIS *guiSP,
	   NearestAtomS *nearest_atomSP, size_t pixelsN,
	   unsigned int *refreshIP, char *stringP)
{
char		*P;
int		n;
int		residue1I, residue2I;

/* Reset flags: */
runtimeSP->average_hydrophobicityF = 0;
runtimeSP->hydrophobic_momentF     = 0;

/* If keyword OFF is present, switch to default drawing mode: */
if (strstr (stringP, "OFF"))
	{
	/* Reset drawing mode index: */
	guiSP->main_window_modeI = 0;

	/* Reinitialize the NearestAtomS array: */
	InitNearest_ (nearest_atomSP, pixelsN);
	*refreshIP = 1;

	/* Refresh the main window: */
	(*refreshIP)++;
	MainRefresh_ (mol_complexSP, mol_complexesN,
		      runtimeSP, configSP, guiSP,
		      nearest_atomSP, pixelsN, *refreshIP);

	/* Refresh the control window: */
	ControlRefresh_ (mol_complexSP + runtimeSP->default_complexI,
			 configSP, guiSP);

	/* Return the command code: */
	return COMMAND_PLOT;
	}

/* The sliding window should not be wider than the sequence: */
if (runtimeSP->sliding_window_width > runtimeSP->residuesN)
	{
	strcpy (runtimeSP->messageA,
		"The sliding window is wider than the sequence!");
	runtimeSP->message_length = strlen (runtimeSP->messageA);
	return ERROR_PLOT;
	}

/* If keyword  HYD (short for HYDROPHOBICITY) is present,  set the */
/* flag which signals that average hydrophobicity should be drawn: */
if (strstr (stringP, "HYD"))
	{
	runtimeSP->average_hydrophobicityF = 1;
	}

/* If keyword MOM (short for MOMENT) is present, set the flag */
/* which signals  that  hydrophobic moment  should be  drawn: */
if (strstr (stringP, "MOM"))
	{
	runtimeSP->hydrophobic_momentF = 1;
	}

/* At least one flag should be set to one: */
if ((runtimeSP->average_hydrophobicityF == 0) &&
    (runtimeSP->hydrophobic_momentF      == 0))
	{
	strcpy (runtimeSP->messageA,
		"Failed to recognize what should be drawn!");
	runtimeSP->message_length = strlen (runtimeSP->messageA);
	return ERROR_PLOT;
	}

/* Replace each minus and colon in the input string with space: */
P = stringP;
while ((n = *P++) != '\0')
	{
	if      (n == '-') *(P - 1) = ' ';
	else if (n == ':') *(P - 1) = ' ';
	}

/* Try to extract two indices: */
if (ExtractTwoIntegers_ (&residue1I, &residue2I, stringP) > 0)
	{
	/* Check indices: */
	if (residue2I < residue1I)
		{
		strcpy (runtimeSP->messageA, "Bad range (check indices)!");
		runtimeSP->message_length = strlen (runtimeSP->messageA);
		return ERROR_PLOT;
		}

	/* Store the extracted indices: */
	runtimeSP->range_startI = (size_t) residue1I;
	runtimeSP->range_endI   = (size_t) residue2I;
	}

/* If failed to extract indices, use default values: */
else
	{
	runtimeSP->range_startI = *runtimeSP->serialIP;
	runtimeSP->range_endI   = *(runtimeSP->serialIP +
				    runtimeSP->residuesN - 1);
	}

/* Set the main window drawing mode index: */
guiSP->main_window_modeI = 4;

/* Reinitialize the NearestAtomS array and refresh index: */
InitNearest_ (nearest_atomSP, pixelsN);
*refreshIP = 1;

/* Refresh the main window: */
(*refreshIP)++;
MainRefresh_ (mol_complexSP, mol_complexesN,
	      runtimeSP, configSP, guiSP,
	      nearest_atomSP, pixelsN, *refreshIP);

/* Refresh the control window: */
ControlRefresh_ (mol_complexSP + runtimeSP->default_complexI, configSP, guiSP);

/* Return the command code: */
return COMMAND_PLOT;
}

/*===========================================================================*/


